// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/nursery_step_gen.cpp,v 1.2 2001/08/03 03:13:42 xli18 Exp $
// 

#include "nursery_step_gen.h"
#include "gc_space.h"
#include "card_table.h"
#include "gc_hooks.h"
#include "gc_plan.h"
#include "gc_consts.h"
#include "orp_threads.h"
#include "orp_synch.h"


#ifdef GC_ZERO_THREAD

#include "nursery.h"


DWORD WINAPI 
zero_thread_func(LPVOID ptr_to_this)
{

    Step_Plus_Nursery_Generation *my_step = (Step_Plus_Nursery_Generation *)ptr_to_this;
    assert(my_step);

    my_step->zero_thread_is_sleeping = false;

    Nursery *previous_nursery_zeroed = NULL;
    int num_zeroed = 0;
    volatile int nursery_clear_state;

    // The main loop. Keep asking for nurseries and keep zeroing them.

    while(true) {

        // Check if I must die...
        if (my_step->zero_thread_die_signal) { 
            // printf("zero thread dying...\n");
            ExitThread(0);
        }

        Nursery *nursery_to_zero = 
             my_step->get_me_next_nursery_to_zero(previous_nursery_zeroed);

        if (nursery_to_zero == NULL) { 
      
            goto suspend_thread_and_continue;
        }
    
        // Got a valid nursery???
        nursery_clear_state = nursery_to_zero->_is_nursery_clear;

        if (nursery_clear_state == NURSERY_CLEARING_IN_PROGRESS) {
            // I lost out.
            continue;
        }

        if (nursery_clear_state == NURSERY_CLEARED) {

            if (nursery_to_zero->p_next_nursery != NULL) { 
                // Doing a second round here....suspend myself.
                goto suspend_thread_and_continue;
            }
        }

        // I need to clear the nursery. So, lock it first.
        nursery_clear_state = 
            
            (int) InterlockedCompareExchangePointer((PVOID *)&(nursery_to_zero->_is_nursery_clear), 
                                                            (PVOID)NURSERY_CLEARING_IN_PROGRESS, 
                                                            (PVOID)NURSERY_NOT_CLEARED);

        if (nursery_clear_state == NURSERY_CLEARING_IN_PROGRESS) { 
            // I lost out again....
            continue;
        } 
        
        if (nursery_clear_state == NURSERY_CLEARED) {

            // Unlikely case, but still a possibility to be covered.
            if (nursery_to_zero->p_next_nursery != NULL) { 
                // Doing a second round here....suspend myself.
                goto suspend_thread_and_continue;            
            } else {
                // Check next nursery
                continue;
            }
        }
        
        // I grabbed it!!!
        assert(nursery_clear_state == NURSERY_NOT_CLEARED);
        num_zeroed++;

        // Now I zero it after asserting that no thread has started allocating from this nursery.
        assert(nursery_to_zero->_p_free == nursery_to_zero->_p_base);

        memset(nursery_to_zero->_p_free, 0, 
               ((POINTER_SIZE_INT)(nursery_to_zero->_p_ceiling) - (POINTER_SIZE_INT)(nursery_to_zero->_p_free)));

        nursery_to_zero->_is_nursery_clear = NURSERY_CLEARED;
        previous_nursery_zeroed = nursery_to_zero;
        // All done
        continue;

suspend_thread_and_continue:

        assert(my_step->zero_thread_is_sleeping == false);
//        printf("Zero thread going to sleep :  %d nurseries zeroed...\n", num_zeroed);            
        my_step->zero_thread_is_sleeping = true;
        SuspendThread(my_step->Zero_Thread_Handle);
        my_step->zero_thread_is_sleeping = false;
//        printf("zero thread just awoke....\n");
        num_zeroed = 0;
        continue;
    }

    return 1;
}

#endif // GC_ZERO_THREAD



Step_Plus_Nursery_Generation::
    Step_Plus_Nursery_Generation(unsigned char generation_number,
                                 Gc_Fast_Hooks *p_gc_hooks,
                                 Gc_Plan       *p_gc_plan,
					             Gc_Interface  *p_container,
								 Generation    *p_superior,
                                 Card_Table    *p_card_table,
								 Block_Store   *p_bs)

                : Step_Generation(generation_number,
                                  p_gc_hooks,
                                  p_gc_plan,
				                  p_container,
				  			      p_superior,
                                  p_card_table,
								  p_bs) 
{
	Step *p_superior_step = _p_step[0];
	//
	// Initialize the count of nurseries allocated each cycle.
	//
	_allocated_nurseries = 0;
//	_spent_nurseries     = 0;

    _number_of_nurseries = 0;
	//
	// Find out how many nurseries we plan to have from the GC plan.
	//
	unsigned long number_of_large_nurseries =
		_p_gc_plan->number_of_large_nurseries();

	int large_nursery_block_size_bytes = _p_gc_plan->large_nursery_block_size_bytes();

#if (GC_DEBUG>2)
	assert(number_of_large_nurseries < MAXIMUM_NURSERIES);
#endif

    for (unsigned long nursery_index = 0; 
	                   nursery_index < number_of_large_nurseries; 
					   nursery_index++) {

        _p_nurseries[nursery_index] = 
			new Nursery(nursery_index,
                        p_gc_hooks,
                        _p_gc_plan,
			            this,
						p_superior_step,
                        p_card_table,
			            p_bs,
						large_nursery_block_size_bytes);

        _number_of_nurseries++;				   
	}

    _current_nursery = 0;

	// Later will chain all spent nurseries to this hook:
    //    orp_cout << "_p_spent_nurseries set to NULL" << endl;

	_p_spent_nurseries = NULL;

    //    orp_cout << "_p_free_nurseries set to NULL" << endl;
    _p_free_nurseries  = NULL;

	//
	// Take all the new nurseries and link them up into the free list.
	//
	_link_free_nurseries(0);

    // Nursery list such as the free list and the spent are updated atomically using
    // InterlockedCompareExchangePointer so there is no need for critical sections.

#ifdef GC_ZERO_THREAD    

    // This is a good place to start the GC zero thread for this particular step.
    
    DWORD ThreadId;

#ifdef ORP_POSIX
    orp_cout << "GC_ZERO_THREAD not yet implemented on Linux" << endl;
    orp_exit(-1);
#endif

    // Boolean indicating if Zero thread has suspended itself.
    zero_thread_is_sleeping = true;
 
    // The Zero thread will die when this boolean is set...(it is set to true by the step destructor)
    zero_thread_die_signal = false;

    Zero_Thread_Handle = CreateThread(
                                 NULL, 
                                 0, 
                                 zero_thread_func, 
                                 (LPVOID) this, 
                                 CREATE_SUSPENDED, 
                                 &ThreadId
                                 );
    
    if (Zero_Thread_Handle == NULL) { 
            orp_cout << "Step_Plus_Nursery_Generation: CreateThread() failed...exiting...\n";
            orp_exit(-1);
    }

    // Start the thread.
    if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
        orp_cout << "Step_Plus_Nursery_Generation: ResumeThread() failed...exiting...\n";
        orp_exit(-1);
    }

    zero_thread_is_sleeping = false;

#endif // GC_ZERO_THREAD



}

Step_Plus_Nursery_Generation::~Step_Plus_Nursery_Generation() 
{
	for (unsigned long idx = 0; idx < _number_of_nurseries; idx++) {
		delete _p_nurseries[idx];
	}

#ifdef GC_ZERO_THREAD

    // Bring down the zero thread....

    // Set the die signal 
    zero_thread_die_signal = true;
  
    if (zero_thread_is_sleeping) { 
        
        // Restart the sleeping thread so that it can kill itself.
        if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
            orp_cout << "~Step_Plus_Nursery_Generation: ResumeThread() failed...exiting...\n";
            orp_exit(-1);
        }
    }

#endif // GC_ZERO_THREAD

}


#ifdef GC_ZERO_THREAD

Nursery *
Step_Plus_Nursery_Generation::get_me_next_nursery_to_zero(Nursery *previous_nursery_zeroed)
{
    if (previous_nursery_zeroed != NULL) {

        volatile Nursery *next_nursery = previous_nursery_zeroed->p_next_nursery;
        // Just return the next nursery from the free list.
        // If next_nursery is NULL it means that has been allocated by now and
        // "previous_nursery_zeroed" thus has its p_next_nursery field set to NULL:
        
        if (next_nursery != NULL)
            return (Nursery *) next_nursery;
    }
    
    // If we fall through till here, it means that we have to return the nursery at 
    // the head of the free list since it is the first available non-allocated free nursery.

    return (Nursery *) _p_free_nurseries;
}


#endif // GC_ZERO_THREAD


//
// We probably ran out of nurseries because there were more threads
// than nurseries. In this case, we need to try to add more.
//
void
Step_Plus_Nursery_Generation::add_nurseries()
{
	unsigned int saved_number_of_nurseries = _number_of_nurseries;

    unsigned long new_number_of_nurseries =
        _number_of_nurseries + NURSERY_INCREMENT;

	int nursery_block_size_bytes = _p_gc_plan->nursery_block_size_bytes();

    for (unsigned long nursery_index = _number_of_nurseries; 
                       nursery_index < new_number_of_nurseries; 
                       nursery_index++) {
        _p_nurseries[nursery_index] = 

			new Nursery(nursery_index,
                        _p_gc_hooks,
                        _p_gc_plan,
			            this,
						_p_step[0],
                        _p_card_table,
			            _p_block_store,
						nursery_block_size_bytes);

        _number_of_nurseries++;
    }
	_link_free_nurseries(saved_number_of_nurseries);
}
#ifdef GC_SAPPHIRE
//
// sapphire_move_spent_nurseries_into_focus_car
//
// Move the blocks in the spent nurseries into the focus car and refill
// the nurseries with empty blocks. Link the free nurseries onto the free_list.
//
bool Step_Plus_Nursery_Generation::sapphire_evict_spent_nurseries_into_focus_car(Car *focus_car)
{
    // First get the spent nurseries, removing them from the _p_spent_nurseries list.
    Nursery  *spent_nursery = (Nursery *)_p_spent_nurseries;
    while (spent_nursery != (Nursery *)InterlockedCompareExchangePointer((PVOID *)&_p_spent_nurseries, 
                                                            (PVOID)NULL, 
                                                            (PVOID)spent_nursery)) {
        // Compare Exchange failed, some other thread changed _p_spent_nurseries, try again.
        spent_nursery = (Nursery *)_p_spent_nurseries;
    }
    assert (nurseries_are_orphans (spent_nursery));

    Nursery  *first_spent_nursery = spent_nursery;
    while (spent_nursery) {
        // try to evict blocks and and reload with fresh heap space.
        if (! spent_nursery->sapphire_evict_blocks(focus_car)) {
            assert (0); // stop and debug..... We have a lot of orphaned nurseries we need to deal with.
            return false;
       }
       spent_nursery = spent_nursery->p_next_nursery;
    }
    // Move the nursery from the spent to the available list.
    _unlink_spent_nurseries (first_spent_nursery);
    return true;
}


#endif // GC_SAPPHIRE

// 
// Move the blocks in the spent nursery into the step and refill the
// nurseries with empty blocks. 
// Return true if you were successful and a full collection is not needed.
// otherwise return false. 
//

// GC_FALSE_COLLECTION

bool Step_Plus_Nursery_Generation::move_spent_nurseries_into_step()
{
    
//    orp_cout << " We have the following spent nurseries" << endl;
    // Grab the spent nurseries atomically.

    Nursery  *spent_nursery = (Nursery *)_p_spent_nurseries;
    while (spent_nursery != (Nursery *)InterlockedCompareExchangePointer((PVOID *)&_p_spent_nurseries, 
                                                            (PVOID)NULL, 
                                                            (PVOID)spent_nursery)) {
        // Compare Exchange failed, some other thread changed _p_spent_nurseries, try again.
        spent_nursery = (Nursery *)_p_spent_nurseries;
    }
    
    assert (nurseries_are_orphans (spent_nursery));

    // _p_spent_nurseries will be NULL so nothing will be linked onto these nurseries.
    int count = 0;
    Nursery *first_spent_nursery = spent_nursery;
    while (spent_nursery) {
        // try to evict blocks and and reload with fresh heap space.
       if (!spent_nursery->evict_blocks()) {
           assert (0); // RLH April 28, 2000 
           // Stop and debug we should be sure we have enough space
           // before we attempt this.

           // couldn't evict blocks so bail out and do full collection.
           // First we need to relink up the spent nurseries.
           return false;   
       }
       spent_nursery = spent_nursery->p_next_nursery;
       count++;
    }
    if (count == 0) {
//        orp_cout << "Evicted nursery count is " << count;
    }
    
//    orp_cout << "Evicted nursery count is " << count;

    assert (nurseries_are_orphans (first_spent_nursery));

    // now move the nursery to the available list.
    _unlink_spent_nurseries (first_spent_nursery);

#ifdef GC_ZERO_THREAD

    if (zero_thread_is_sleeping) { 

        //printf("move_spent_nurseries_into_step(): restarting zero thread\n");

        if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
            orp_cout << "move_spent_nurseries_into_step: ResumeThread() failed...exiting...\n";
            orp_exit(-1);
        }
           
    }    

#endif // GC_ZERO_THREAD

    return true;
}
 
//
// After a minor stop-the-world collection, Mrl_Gc_Vxx gives us
// an opportunity to do any cleanups. This is done before the
// incremental collection of mature space.
//
void
Step_Plus_Nursery_Generation::cleanup()
{
#ifndef GC_FIXED_V1 // ndef - ignore if doing simple fixed collection.
	//
	// Clean up the steps;
	//
	Step_Generation::cleanup();
    
	//
	// Clean up nurseries. Turn this off if you turn any of the
	// following debug thingies on.
	//

    // *All* nurseries will get filled with a pattern in cleanup if debug is on.
	for (unsigned long n_idx = 0; n_idx < _number_of_nurseries; n_idx++) {
		_p_nurseries[n_idx]->cleanup();
	}

	//
	// Reset number of used nurseries.
	//
	_current_nursery = 0;
	
	//
	// Undo the chain of linked spent nurseries.
	//

    // Unlink spent list.
    
    Nursery *old_spent_nurseries = (Nursery *)_p_spent_nurseries;
    while (old_spent_nurseries != (Nursery *)InterlockedCompareExchangePointer((PVOID *)&_p_spent_nurseries, 
                                                            (PVOID)NULL, 
                                                            (PVOID)old_spent_nurseries)) {
        old_spent_nurseries = (Nursery *)_p_spent_nurseries;
    }
    // If this collection was due to pinned allocation spent_nurseries could be NULL.
	_unlink_spent_nurseries((Nursery *)old_spent_nurseries);

#ifdef GC_ZERO_THREAD

    // Restart the sleeping GC zero thread.
    if (zero_thread_is_sleeping) { 

        //printf("cleanup(): restarting zero thread\n");

        if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
            orp_cout << "move_spent_nurseries_into_step: ResumeThread() failed...exiting...\n";
            orp_exit(-1);
        }
    }    

#endif // GC_ZERO_THREAD

    return;

#endif
}

#if (GC_DEBUG>1)
void
Step_Plus_Nursery_Generation::debug_list_all_nurseries()
{
	orp_cout << "Dumping all nurseries" << endl;
	for (unsigned int i=0;i<_number_of_nurseries;i++) {
		orp_cout << "    " << _p_nurseries[i] << endl;
	}
	orp_cout << endl;
}

void
Step_Plus_Nursery_Generation::debug_list_free_nurseries()
{
	Nursery *p_nursery = _p_free_nurseries;

	cout << "Listing free nurseries" << endl;
	if (p_nursery) {
		while (p_nursery) {
			cout << "    " << p_nursery << endl;
			p_nursery = p_nursery->p_next_nursery;
		}
		orp_cout << endl;
	} else {
		orp_cout << "No free nurseries" << endl;
	}
}

void
Step_Plus_Nursery_Generation::debug_list_spent_nurseries()
{
	Nursery *p_nursery = _p_spent_nurseries;

	orp_cout << "Listing spent nurseries" << endl;
	if (p_nursery) {
		while (p_nursery) {
			orp_cout << "    " << p_nursery << endl;
			p_nursery = p_nursery->p_next_nursery;
		}
		orp_cout << endl;
	} else {
		orp_cout << "No spent nurseries" << endl;
	}
} 

void
Step_Plus_Nursery_Generation::dump_nursery_stats()
{
	//bool stat = p_gc_lock->_lock_enum_or_null(false);
	cout << endl << endl;
	cout << "DUMPING NURSERY STATS for thread " << GetCurrentThreadId() << endl;
	debug_list_all_nurseries();
	debug_list_spent_nurseries();
	debug_list_free_nurseries();
	cout << "END DUMPING NURSERY STATS" << endl;
	cout << endl << endl;
	//p_gc_lock->_unlock_enum_or_null();
}

#endif // GC_DEBUG>4

#if (GC_DEBUG>4)
//
// This debug routine leaves the old nurseries intact (for the ORP to
// access if it wants), instead to allocate new nursery blocks.
//
void
Step_Plus_Nursery_Generation::_debug_replace_nurseries_intact()
{
	for (unsigned long n_idx = 0; n_idx < _number_of_nurseries; n_idx++) {
		//
		// Write protect this nursery.
		//
		_p_nurseries[n_idx]->debug_replace_intact();
	}
}
#endif // _DEBUG

#if (GC_DEBUG>4)
//
// This debug routine leaves the old nurseries intact (for the ORP to
// access if it wants), instead to allocate new nursery blocks.
//
void
Step_Plus_Nursery_Generation::_debug_replace_nurseries_cleared()
{
	for (unsigned long n_idx = 0; n_idx < _number_of_nurseries; n_idx++) {
		//
		// Write protect this nursery.
		//
		_p_nurseries[n_idx]->debug_replace_cleared();
	}
}
#endif // _DEBUG

#if (GC_DEBUG>4)
//
// Debug time routine to write protect and save spent nurseries
// to detect any ensuing access from the ORP that is incorrect.
//
void
Step_Plus_Nursery_Generation::_debug_write_protect_and_replace_nurseries()
{
	for (unsigned long n_idx = 0; n_idx < _number_of_nurseries; n_idx++) {
		//
		// Write protect this nursery.
		//
		_p_nurseries[n_idx]->debug_write_protect();
	}
}
#endif // _DEBUG

#ifdef GC_SAPPHIRE

Nursery *
Step_Plus_Nursery_Generation::p_sapphire_release_nursery(Nursery *p_old_nursery,
                                                         bool returnNullOnFail)
{
#if (GC_DEBUG>0) 
	// Make sure all nurseries on the free list are empty.
    Nursery *temp = (Nursery *)_p_free_nurseries;
    while (temp != NULL) {
        assert (p_old_nursery != temp); // the old nursery cant be on the free list.
    	temp->verify_empty_nursery();
	    temp = temp->p_next_nursery;
    } 
    // Make sure p_old_nursery is not on the spent list.
    int count = 0;
    temp = (Nursery *)_p_spent_nurseries;
    while (temp != NULL) {
        assert (p_old_nursery != temp); // the old nursery cant be on the free list.
	    temp = temp->p_next_nursery;
        count++;
    } 
    int fcount = 0;
    temp = (Nursery *)_p_free_nurseries;
    while (temp != NULL) {
        assert (p_old_nursery != temp); // the old nursery cant be on the free list.
	    temp = temp->p_next_nursery;
        fcount++;
    } 
    orp_cout << "Number of spent nurseries is " << count << endl;

    orp_cout << "Number of free nurseries is " << fcount << endl;
#endif // (GC_DEBUG>0)
    // The thread associated with this nursery is at a safepoint
    // and the sapphire driver thread holds the p_gc_lock.
    Nursery *p_nursery = NULL; 
    assert (returnNullOnFail);

	if (_p_free_nurseries == NULL) {
		if (returnNullOnFail) {
			return NULL;
		}
        assert (0);

		//
		// Notify enclosing GC Interface that I'm out of space
		//
		_p_container->notify_out_of_space(0);   // 0 signifies that you need 
                                                // nurseries and not
                                                // LOS space. 
	} else {
		//
		// Fresh nurseries remain
		// 
		if (p_old_nursery) {
            // We hold p_gc_lock.
			store_spent_nursery(p_old_nursery);
		}
		//
		// Get a new nursery from the free list.
		//
		p_nursery        = (Nursery *)_p_free_nurseries;
#if (GC_DEBUG>0)
		p_nursery->verify_empty_nursery();
        Nursery *temp = (Nursery *)_p_spent_nurseries;
        while (temp != NULL) {
            assert (p_nursery != temp); // the old nursery cant be on the free list. 
	        temp = temp->p_next_nursery;
        } 
#endif // (GC_DEBUG>0)
		_p_free_nurseries         = _p_free_nurseries->p_next_nursery; 

		p_nursery->p_next_nursery = NULL;
		_allocated_nurseries++;
	}
    assert (p_nursery);
	return p_nursery;
} // p_sapphire_release_nursery

#endif // GC_SAPPHIRE


bool orp_enable_gc();
bool orp_disable_gc();

// This is not thread safe by any means....
bool 
Step_Plus_Nursery_Generation::verify_nurseries ()
{
#if 0 // 0 (GC_DEBUG>0)
    // Make sure all nurseries on the free list are empty.
    Nursery *temp = (Nursery *)_p_free_nurseries;
    int count = 0;
    while (temp != NULL) { 
        count++;
        assert (count < 10000); // If you assert here then you either have a
                                // lot of threads and this isn't a bug 
                                // or the free list is circular and broken.
        temp->verify_empty_nursery();
        temp = temp->p_next_nursery;
    } 
    
    // Make sure old nursery is not on spend list.
    Nursery *temp_spent = (Nursery *)_p_spent_nurseries;
    count = 0;
    while (temp_spent != NULL) { 
        Nursery *temp = (Nursery *)_p_free_nurseries;
        while (temp != NULL) { 
            assert (temp_spent != temp);
            temp = temp->p_next_nursery;
        } 
        count++;
        assert (count < 10000);
        temp_spent = temp_spent->p_next_nursery;
    }
#endif // GC_DEBUG>0
    
    return true;
}

//
// The thread just ran out of another nursery. Give it a fresh one.
//
Nursery *
Step_Plus_Nursery_Generation::p_cycle_nursery(Nursery *p_old_nursery,
											  bool returnNullOnFail)
{
    assert (nurseries_are_orphans (p_old_nursery));
    Nursery *result = NULL;
    Nursery *temp;

    if (p_old_nursery) {
        assert (p_old_nursery->p_next_nursery == NULL);
        assert (p_old_nursery != result);
        // Move the old nursery to the spent nursery list.
        store_spent_nursery(p_old_nursery);
    }

    while (result == NULL) {
        if (_p_free_nurseries == NULL) {
		    if (returnNullOnFail) {
			    return NULL;
		    }
            orp_enable_gc();
            bool stat = p_gc_lock->_lock_enum_or_null(returnNullOnFail);
            orp_disable_gc();
        	assert (stat == true);
            //
	    	// Notify GC I'm (still) out of space
		    // 
            if (_p_free_nurseries == NULL) { 
                // This thread holds lock to notify GC
                _p_container->notify_out_of_space(0);   // 0 signifies that you need 
                                                // nurseries and not
                                                // LOS space.
            }
            p_gc_lock->_unlock_enum_or_null ();
            // another thread can force _p_free_nurseries to be NULL here.
        }   // or here.
        // GC is disabled so we will be forced out of here if the GC code 
        // ever tries to redo _p_free_nurseries.
        // Sapphire will have an enabled gc.      

        temp = (Nursery *)_p_free_nurseries;
        if (temp != NULL) {
            Nursery *next = temp->p_next_nursery;
            assert (temp != next);
            result = 
                (Nursery *)InterlockedCompareExchangePointer ((PVOID *)&_p_free_nurseries, 
                                                               next, 
                                                               temp);
            if (result != temp) {
                result = NULL;
                // Try again
            } else {
//                orp_cout << " Good " << endl;
            }
        } 
    }
    result->p_next_nursery = NULL;
#if (GC_DEBUG>0)
    result->verify_empty_nursery();
#endif
    assert (nurseries_are_orphans(result)); // The result is owned by noone.
    // This is where we should clear out the nursery.........
    return result;
}

#if (GC_DEBUG>3)
//
// Routine to display the contents of the generation.
//
void Step_Plus_Nursery_Generation::inspect(unsigned int level)
{
	if (level == 0) {
		Step_Generation::inspect(level);
		cout << "Inspecting Nurseries of Step plus Nursery Generation" << endl;
		cout << "    There are " << _number_of_nurseries << " nurseries." << endl;
		Nursery *p_nursery = _p_spent_nurseries;
		while (p_nursery != NULL) {
			p_nursery->inspect(level);
			p_nursery = p_nursery->p_next_nursery;
		}
	} else {
		cout << "Error: inspect level " << level << " unimplemented" << endl;
		orp_exit(1); // unimplemented level
	}
}
#endif // _DEBUG

// We assume that the nurseries going onto the free list on not on the
// spent list or the free list at this time. We also assume they are not
// associated with any thread. Check this assumption.

bool Step_Plus_Nursery_Generation::nurseries_are_orphans (Nursery *nurseries)
{
#if (GC_DEBUG>0)
    Nursery *p_nursery = nurseries;
    int to_be_free_count = 0;
    while (p_nursery) {
        Nursery *temp = (Nursery *)_p_free_nurseries;
        while (temp != NULL) {
            assert (temp != p_nursery);
            temp = temp->p_next_nursery;
        }
        temp = (Nursery *)_p_spent_nurseries;
        while (temp != NULL) {
            assert (temp != p_nursery);
            temp = temp->p_next_nursery;
        }
        ORP_thread *p_thr = p_active_threads_list;
        while (p_thr != NULL) {
            assert (p_thr->p_nursery != p_nursery);

            // Due to race conditions that do not cause a GC and therefore
            // do not force us out of this code. A nursery can be moved from the
            // free list into an active thread after we have temp but before we
            // do the check in the while loop. 
            // 

            // Make sure no active threads have nurseries on on spent list.
            temp = (Nursery *)_p_spent_nurseries;
            while (temp != NULL) {
                assert (temp != p_thr->p_nursery);
                temp = temp->p_next_nursery;
            }
            p_thr = p_thr->p_active;
        }
        to_be_free_count++;
        p_nursery = p_nursery->p_next_nursery;
    }
//    orp_cout << " To be free count is " << to_be_free_count << endl;
#endif
    return true;
}

void 
Step_Plus_Nursery_Generation::link_onto_free_list (Nursery *first_nursery, Nursery *last_nursery)
{
    assert (last_nursery->p_next_nursery == NULL);
    assert (nurseries_are_orphans (first_nursery));
    Nursery *old_nursery = (Nursery *)_p_free_nurseries;
	last_nursery->p_next_nursery = old_nursery;
    while (old_nursery != (Nursery *)InterlockedCompareExchangePointer((PVOID *)&_p_free_nurseries, 
                                                            (PVOID)first_nursery, 
                                                            (PVOID)old_nursery)) {
        // Compare Exchange failed, some other thread changed _p_free_nurseries, try again.
        old_nursery = (Nursery *)_p_free_nurseries;
    	last_nursery->p_next_nursery = old_nursery;
    }
//    orp_cout << " link_onto_free_list " << first_nursery << endl;
    return;
}

//
// Just after creating the nurseries, they are linked up
// into a free list.
//
void
Step_Plus_Nursery_Generation::_link_free_nurseries(unsigned int start_index)
{
    if (_number_of_nurseries==0) {
        //
        // We want to accomodate weird setups with zero nurseries.
        //
        return;
    }

	//
	// Link each nursery to the next higher one.
	//
	for (unsigned long n_idx = start_index; 
	                   n_idx < (_number_of_nurseries - 1);
					   n_idx++) {
		_p_nurseries[n_idx]->p_next_nursery =
					_p_nurseries[n_idx + 1];
        
#if (GC_DEBUG>0)
        _p_nurseries[n_idx]->verify_empty_nursery();

#endif
                       }
    // Link the new nurseries in atomically.
    Nursery *first_new_nursery = _p_nurseries[start_index];    
    Nursery *last_new_nursery = _p_nurseries[_number_of_nurseries - 1];
    link_onto_free_list (first_new_nursery, last_new_nursery);
    
//    orp_cout << "_p_free_nurseries set to list starting at " << start_index << endl;
}

//
// A thread just used up another nursery. Store this away till it is
// time to garbage collect.
//
void Step_Plus_Nursery_Generation::store_spent_nursery(Nursery *p_nursery)
{
    assert (p_nursery->p_next_nursery == NULL);
    assert (nurseries_are_orphans(p_nursery));
    // Put it on the spent list.
#ifndef GC_SAPPHIRE
    // For sapphire this isn't true....
    assert(p_TLS_orpthread->gc_enabled_status == disabled);
#endif
    Nursery  *old_nursery = (Nursery *)_p_spent_nurseries;
    p_nursery->p_next_nursery = old_nursery;
    while (old_nursery != (Nursery *)InterlockedCompareExchangePointer((PVOID *)&_p_spent_nurseries, 
                                                            (PVOID)p_nursery, 
                                                            (PVOID)old_nursery)) {
        // Compare Exchange failed, some other thread changed _p_spent_nurseries, try again.
        old_nursery = (Nursery *)_p_spent_nurseries;
        p_nursery->p_next_nursery = old_nursery;
    }
//    _spent_nurseries++;

    return;
}

//
// Unlink the spent nurseries. We are called *after* collecting young
// space, but before incrementally collecting mature space.
// We take the spent nurseries and put them on the free list.
//
void
Step_Plus_Nursery_Generation::_unlink_spent_nurseries(Nursery *spent_nurseries) 
{
    if (_number_of_nurseries < 1) {
        return;
    }
    if (spent_nurseries == NULL) { // This case is common if GC is done due to pinned areas.
        return;
    }
    Nursery *last_nursery = spent_nurseries;
    while (last_nursery->p_next_nursery) {
        last_nursery = last_nursery->p_next_nursery;
    }
    link_onto_free_list (spent_nurseries, last_nursery);
    return;
}

#if (GC_DEBUG>3)
//
// Routine to verify the consistency of all the spaces of this generation.
//
bool Step_Plus_Nursery_Generation::verify_generation()
{
	Step_Generation::verify_generation();

	for (unsigned long idx = 0; idx < _current_nursery; idx++) {
		if (_p_nurseries[idx]->verify_space()) {
			continue;
		} else {
			return false;
		}
	}
	return true;
}
#endif // _DEBUG


int Step_Plus_Nursery_Generation::get_number_of_spent_nurseries () 
{
    int count = 0;
    Nursery *temp_n = (Nursery *)_p_spent_nurseries;
    while (temp_n)
    {
        temp_n = temp_n->p_next_nursery;
        count++;
    }
    return count;
}

// end file nursery_step_gen.cpp
