// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/step.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#ifndef _step_H_
#define _step_H_

//
// Each generation consists of a variable number of steps which are used
// to age objects before promoting them to an older generation. Steps
// are built out of blocks.
//
//
// A Step is a Gc_Space. It is a container, or a repository of
// Java objects. Along with Cars, it inherits from Block_List, which
// implements spaces that are built out of non-contiguous blocks.
//
//                          Gc_Space (abstract)
//                              |
//             ------------------------------------
//             |                                  |
//             |                                  |
//        Block_List (abstract)                   |
//      ---------------------                     |
//      |                    |                 Nursery
//     Car                  Step
//
//

#include "platform.h"

#include "gc_header.h"
#include "remembered_set.h"
#include "Block_Store.h"
#include "generation.h"
#include "descendents.h"
#include "gc_hooks.h"

#include "block_list.h"

#ifdef GC_COPY_V2
// I am not sure why this is set to a constant but for a simple copy collector
// it needs to be very big. This will cause us to have very big arrays
// to hold this information and this needs to be dealt with. - RLH 8-17-98
#define MAXIMUM_BLOCKS_IN_STEP 16384

#else 

#define MAXIMUM_BLOCKS_IN_STEP 32768

#endif

class Card_Table;
class Gc_Fast_Hooks;
class Gc_Plan;

class Step : public Block_List {
public:
	//
	// Create a step (done by Step_Plus_Nursery_Generation).
	//
    Step(unsigned long step_id,
         Gc_Fast_Hooks *p_gc_hooks,
         Gc_Plan       *p_gc_plan,
		 Generation    *p_container,
		 Gc_Space      *p_superior,
         Card_Table    *p_card_table,
		 Block_Store   *p_block_store);

	//
	// Delete a step. Currently unused. May need to collapse steps later.
	//
	virtual ~Step()
		// BUG BUG: NEED TO CALL ANCESTOR DESTRUCTOR : ~Block_List() {
		//
		// Delete only the FROM blocks. (The ancestor will take
		// care of the rest.
		//
	{
		unsigned char block_index;
		for (block_index = 0; 
			 block_index < _number_of_from_blocks; 
			 block_index++) {

			_p_block_store->release_super_block(_p_from_block[block_index]);
		}

	}
#if (GC_DEBUG>3)
    //
    // Instead of re-cycling blocks, read protect them.
    //
    void debug_read_protect_from_space();
#endif 

#if (GC_DEBUG>3)
    //
    // Instead of re-cycling, set them to obsolete space.
    //
    void debug_set_obsolete_from_space();
#endif 
	//
	// A live object was traced down to my container.
	//
	virtual Java_java_lang_Object *p_evict_object(Java_java_lang_Object **pp_obj, bool doing_mos_collection);

    //
	// Beforp the stop-the-world collection, set up cheney spaces.
	//
	void set_up_cheney_spaces();

	//
	// Do any required cleanup at the end of the collection.
	//
	virtual void cleanup();

    void _walk_from_space(bool (*func)(Object_Gc_Header *,
		                               Remembered_Set *));

	//
	// routines for introspection
	//

	virtual bool is_step() {
		return true;
	}

#if (GC_DEBUG>3)
	//
	// Debug routine to inspect details.
	//
	virtual void inspect(unsigned int level);
#endif // _DEBUG

	//
	// Debug time: ensure that the object being evicted is not in TO space.
	// 
	bool not_in_to_block(Java_java_lang_Object *p_obj);
	//
	// Merge the entries in the incoming remembered set with entry.
    // This is only valid for GC Spaces with their own remembered sets.
    // (Such as cars, and unlike nurseries and steps.)
	//
	virtual void update_remembered_set(Java_java_lang_Object **pp_obj) {
		cout << "Error: attempt to add entry to a step's rs (I don't have one!)" << endl;
        orp_exit(1);
	}

	inline virtual int my_block_size_bytes() {
		return _step_block_size_bytes;
	}

protected:
	//
	// The following routine should be implemented by all containers
	// that implement block lists. The value returned by the container
	// reflects a generation encoding, which is used by the block list
	// object for marking all sub blocks in the block store. This is
	// used later for doing fast checks of generations.
	//
	virtual unsigned long _get_generation_encoding() {
		//
		// We return zero signifying that all steps are currently
		// part of the young generation.
		//
		return YOS_GEN_NUMBER;
	}

private:

    //
    // The size of the blocks allocated to a step when it needs more space
    // If blocks are moved from nurseries then they could be of differing sizes.
    //
	int _step_block_size_bytes;

    Step();


	void _scavenge_descendents(Java_java_lang_Object *p_obj);

    // WARNING WARNING
    // These need to be extensible arrays and not statically declared.

    void *_p_from_block[MAXIMUM_BLOCKS_IN_STEP];
	Object_Gc_Header *_p_from_block_end[MAXIMUM_BLOCKS_IN_STEP];

    unsigned long _number_of_from_blocks;
};



#endif // _step_H_

