// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/car.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#ifndef _car_H_
#define _car_H_


//
// A Car is comprised of multiple non-contiguous blocks. Therefore it
// inherits from Block_List.. 
//
//                          Gc_Space (abstract)
//                              |
//             ------------------------------------
//             |                                  |
//             |                                  |
//        Block_List (abstract)                   |
//      ---------------------                     |
//      |                    |                 Nursery
//     Car                  Step
//
//

#include "remembered_set.h"
#include "object_layout.h"
#include "train_generation.h"

#include "block_list.h"
#include "gc_plan.h"

class Object_List;
class Car;
class Block_Store;
class Card_Table;
class Train;
class Gc_Fast_Hooks;
class Gc_Plan;

class Car : public Block_List {
public:

    Car(unsigned long    car_id,
        Gc_Fast_Hooks    *p_gc_hooks,
        Gc_Plan          *p_gc_plan,
		Train            *p_container,
        Train_Generation *p_train_generation,
        Card_Table       *p_card_table,
		Block_Store      *p_block_store);

    virtual ~Car();
    //
	// Somebody is notifying us of an interesting incoming reference.
	// We need to find out if it requires an update of our write barrier.
    //
    void check_update_car_write_barrier(Java_java_lang_Object **pp_obj_ref);

	//
	// Do any required cleanup at the end of the collection.
	//
	virtual void cleanup();
#if (GC_DEBUG>3)

	void dump_container_write_barrier();
#endif // _DEBUG

    //
    // The remembered set of all pointers from younger MOS areas into this car.
    // This is either built directly or from scanning cards.
    //

    Remembered_Set *get_car_remembered_set ();

    unsigned long get_car_id() {
        return this->get_id();
    }

	unsigned long get_train_id();

    virtual Train *p_get_train() {
#if (GC_DEBUG>2)
    assert(_p_train != NULL);
#endif // _DEBUG
        return _p_train;
    }

	//
	// routines for introspection
	//
	virtual bool is_car() {
		return true;
	}

#if (GC_DEBUG>3)

    //
    // Return my name and ID in a printable string for debugging.
    //
    virtual void my_name();

#endif // _DEBUG

	//
	// Are there any pointers into this car from a younger train?
	// We aren't interested in pointers from younger space, or from
	// the ORP, since that is assumed to have been taken care of previously.
	//
	bool inter_train_refs_exist();

    //
    // Our container train was asked to reclaim the oldest train.
    // It turned out that the entire train wasn't garbage. Therefore
    // the train has singled me out as the oldest car, and asked 
    // me to reclaim myself.
    //
  	void reclaim_car(Remembered_Set *p_outside_refs,
                     Remembered_Set *p_weak_refs);

#if (GC_DEBUG>0)
	//
	// Debug routine to inspect details.
	//
	virtual void inspect(unsigned int level);
#endif // _DEBUG

	Java_java_lang_Object *p_reserve_space(Java_java_lang_Object **pp_obj_ref);
	Java_java_lang_Object *p_reserve_bytes(unsigned size);

	//
	// Wrapper for the extra work we do during scavenging, compared
	// to parent Block_List.
	//
	virtual Java_java_lang_Object *p_scavenge_object(Java_java_lang_Object **pp_obj);

	//
	// Need to be able to update car ids after reclaiming trains.
	//
	void set_id(unsigned id) {
		identity = id;
	}

	//
	// Overload operator== to compare cars of objects.
	//
	bool operator==(Car &car2) {
		if (identity == car2.get_car_id())
        	return true;

		return false;
	}

	bool operator<(Car &car2) {
		if (identity < car2.get_car_id())
        	return true;

		return false;
	}

	bool operator>(Car &car2) {
		if (identity > car2.get_car_id())
        	return true;

		return false;
	}

    //
    // The young generation is notifying us that a finalizable,
    // but still reachable, object has just been tenured into 
    // our car.
    //
    void record_finalizable_object(Java_java_lang_Object *p_obj);
    //
    // Run all finalizers at exit time if required by app.
    //
    void run_all_finalizers();

	// enqueue the ref on the list, this car holds the referent.
	virtual void enqueue_soft_ref (java_lang_ref_Reference *soft_reference);
	virtual void enqueue_weak_ref (java_lang_ref_Reference *weak_reference);
	virtual void enqueue_phantom_ref (java_lang_ref_Reference *phantom_reference);

    //
    // Debug routine to verify the integrity of the contents of this car's write barrier
    // Verify that none of the incoming pointers are from obsolete space.
    //
    void verify_car_write_barrier();

	//
	// Merge the entries in the incoming remembered set with entry.
    // This is only valid for GC Spaces with their own remembered sets.
    // (Such as cars, and unlike nurseries and steps.)
	//
	virtual void update_container_write_barrier(Java_java_lang_Object **pp_obj_ref);							 

    // Given an address is it in this car. Used to see if object is in
    // focus car.

    bool is_address_in_this_car(void *p_addr);

   	//
	// Walk the car and apply function argument to all 
	// recognized objects.
	//
	bool walk_car(bool(*func)(Object_Gc_Header *, Remembered_Set *));

	virtual int my_block_size_bytes() {
		return _car_block_size_bytes;
	}

#ifdef GC_SAPPHIRE
	// Move nursery blocks into the "from" C space focus car, leave fresh nurseries in U space.
	void sapphire_absorb_nurseries();
#endif // GC_SAPPHIRE

    void enumerate_reference_queues ();
    
    // If the slot is not in the same train as this car evict it to the appropriate place.
    Java_java_lang_Object *p_evict_if_out_of_train_object(Java_java_lang_Object **pp_obj_ref);

    Java_java_lang_Object *p_evict_out_of_mos_ref(Java_java_lang_Object **pp_obj_ref);

protected:
	//
	// The following routine should be implemented by all containers
	// that implement block lists. The value returned by the container
	// reflects a generation encoding, which is used by the block list
	// object for marking all sub blocks in the block store. This is
	// used later for doing fast checks of generations.
	//
	virtual unsigned long _get_generation_encoding() {

		//
		// Return an encoding of the car and the train.
		//
        // Since YOS_GEN_NUMBER is 0 and unallocated is -1
        // return something reasonable different here.
        //
        // Since this is a car return 1;

        return 1;
        // someday come up with useful train car encoding. remembering
        // the train id and car ids can both be 0 so be careful that you
        // don't use that to create an encoding the same as YOS_GEN_NUM.
        // If we add one to the train ids then wen can leave the car id's alone.
        unsigned long result = ((get_train_id() + 1) << 16) | (get_car_id());

        assert (result != YOS_GEN_NUMBER); 

        return result;
	}

private:

	int _car_block_size_bytes;

    Car();
	//
	// A reclaim of this car: take care of the pointers
	// from within this generation.
	//
	void _evict_inside_generation_refsxxx();
    
	//
	// A live object was traced down to my container (the focus_car).
	//
	virtual Java_java_lang_Object *p_evict_object(Java_java_lang_Object **pp_obj, bool doing_mos_collection);

    bool _is_slot_interesting(Java_java_lang_Object **pp_obj_ref);
    //
    // A reclaim of this car: take care of
    // slots not in the focus train.
    //
    void _evict_out_of_mos_refs(Remembered_Set *p_incoming_refs);

    //
    // A reclaim of this car: take care of
    // slots in the focus train but not in the focus car.

    void _evict_out_of_train_refs(Remembered_Set *p_incoming_refs);
    //
    // A reclaim of this car: take care of
    // slots in the focus train but not in the focus car.

    void _evict_out_of_car_refs(Remembered_Set *p_incoming_refs);

    //
    // We have finished reclaiming the focus car. Beforp we return 
    // we need to scan for unreachable 
    // objects that have just become candidates for finalization.
    //
    void _locate_and_process_finalizable_objects(bool doing_mos_collection);

    Object_List *_p_finalizable_object_list;

    //
	// Lists to deal with weak reference objects.
	// These need to be enumerated for the GC. Each car has a set of these
	// queues as does each step (or step_nursery) generation.
	//
    java_lang_ref_Reference *_p_weak_object_list;

    java_lang_ref_Reference *_p_soft_object_list;

    java_lang_ref_Reference *_p_phantom_object_list;

	void remove_weak_ref_Reference (java_lang_ref_Reference *p_ref_Reference,
		java_lang_ref_Reference *previous);
	void remove_soft_ref_Reference (java_lang_ref_Reference *p_ref_Reference,
		java_lang_ref_Reference *previous);	
	void remove_phantom_ref_Reference (java_lang_ref_Reference *p_ref_Reference,
		java_lang_ref_Reference *previous);	

    //
	// The maximum size a car can grow to. This has to be limited
    // in order to minimize pause times. Tune this appropriately.
	//
	unsigned long _maximum_blocks_in_car;

	//
	// Is the specified address contained in my car.
	//
	bool _is_in_my_car(void *p_addr);
    //
    // Is the specified object in my car
    //
    bool _is_object_in_my_car(Java_java_lang_Object *p_obj) {
        return _is_in_my_car((void *)p_obj);
    }
	//
	// Is the specified address not contained in my car.
	//
	bool _is_not_in_my_car(void *p_addr);

    // Remembered set of slots holding interesting pointers, 
    // (pointers to YOS or older trains or older cars in this train)
    Remembered_Set *_p_car_remembered_set;

#ifdef PER_CAR_BARRIERS
    //
    // Performance enhancement: maintain the old-to-young write barrier
    // entries per-car. That way it is easier to purge the entries 
    // specific to a car when that car is reclaimed.
    //
    Remembered_Set *_p_old_to_young;
#endif

	//
	// My containing train.
	//
    Train *_p_train;


    //
    // My containing generation.
    //
    Train_Generation *_p_train_generation;

	void _update_phantom_references(bool doing_mos_collection);
	void _update_soft_references(bool low_on_space, bool doing_mos_collection);
	void _update_weak_references(bool doing_mos_collection);

	bool _will_fit_in_car(Java_java_lang_Object **pp_obj_ref);
	bool _will_bytes_fit_in_car(unsigned size);
 
	bool _is_reference_into_this_car(Java_java_lang_Object **pp_obj_ref);
};

#endif // _car_H_

