// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/register_allocator.h,v 1.2 2001/08/13 10:00:42 xhshi Exp $
//



#ifndef _REGISTER_ALLOCATOR_H
#define _REGISTER_ALLOCATOR_H

/* The "Register Allocator" class description.

   It (currently) consists of an abstract class with 3 subclasses:
     Register_Allocator_None (no register allocation)
     Register_Allocator_Simple
     Register_Allocator_Priority (what was formerly called "Chow")
   
 */

#include "jit.h"
#include "Mem_Manager.h"
#include "x86.h"
#include "runtime_register_allocator.h"

class Register_Allocator
{
public:
    // Return the register allocated to the given variable at the given
    // byte code.  Returns n_reg on failure.
    virtual X86_Reg_No reg_var_allocated_to(int index, const unsigned char *where) = 0;
    
    // Return the variable held in the given register at the given byte code.
    // Returns -1 on failure.
    virtual unsigned var_in_reg(X86_Reg_No reg_no, const unsigned char *where) = 0;
    
    virtual int need_sorted_gc_sites() = 0;
    
    // Returns true if the variable is register-allocated and is live on
    // entry to the function; i.e., whether we need to load the input
    // argument into the register in the prolog.
    virtual int var_is_live_on_entry(unsigned var) = 0;
    
    // Returns the number of bytes of extra space needed in the method_info.
    // This would be due to a variable-length array.
    virtual int space_in_method_info(Jit_Method_Info *method_info) = 0;
    
    virtual void set_call_site_info(Jit_Method_Info *new_mi, unsigned size) = 0;
    virtual void set_register_saved_info(Jit_Method_Info *new_mi) = 0;
    
    virtual unsigned callee_saved_regs(int ebp_based) = 0;

    virtual void update_method_info(Jit_Method_Info *mi) = 0;
    
    virtual unsigned register_allocate(class CG_Prepass *prepass,
        Method_Handle methodHandle,
        Class_Handle classHandle,
        unsigned max_locals,
        const unsigned char *bc,
        unsigned code_length) = 0;
    
    virtual unsigned char registers_available_in_codegen_bb(const unsigned char *bc, int &hint) = 0;

    virtual void init_rra(struct Runtime_Register_Allocator &rra) = 0;

   	void *operator new(size_t sz,Mem_Manager& m) { return m.alloc(sz); }

protected:
    Mem_Manager *mm;

};

class Register_Allocator_None : public Register_Allocator
{
public:
    Register_Allocator_None(Mem_Manager &m) { mm = &m; }
    int id() { return 0; }
    X86_Reg_No reg_var_allocated_to(int index, const unsigned char *where);
    unsigned var_in_reg(X86_Reg_No reg_no, const unsigned char *where);
    int need_sorted_gc_sites() { return 0; }
    int var_is_live_on_entry(unsigned var) { return 0; }
    int space_in_method_info(Jit_Method_Info *method_info) { return 0; }
    void set_call_site_info(Jit_Method_Info *new_mi, unsigned size) {}
    void set_register_saved_info(Jit_Method_Info *new_mi) {}
    unsigned callee_saved_regs(int ebp_based);
    void update_method_info(Jit_Method_Info *mi) {}
    unsigned register_allocate(class CG_Prepass *prepass,
        Method_Handle methodHandle,
        Class_Handle classHandle,
        unsigned max_locals,
        const unsigned char *bc,
        unsigned code_length) { return (method_get_num_handlers(methodHandle) == 0) ? 0 : 3; }
    unsigned char registers_available_in_codegen_bb(const unsigned char *bc, int &hint) { return 0; }
    void init_rra(struct Runtime_Register_Allocator &rra) {
        rra.get_register_mapping_and_liveness = get_register_mapping_and_liveness;
        rra.should_restore_callee = should_restore_callee;
        rra.should_restore_ebp = should_restore_ebp;
    }
private:
    static void get_register_mapping_and_liveness(unsigned k, Call_Site_Info *csi,
        Jit_Method_Info *mi, X86_Reg_No &reg, int &live) { reg = n_reg; live = 1; }
    static int should_restore_callee(Jit_Method_Info *mi, X86_Reg_No reg);
    static void should_restore_ebp(Jit_Method_Info *method_info, int out_args, int &restore, int &off);
};

class Register_Allocator_Simple : public Register_Allocator
{
public:
    Register_Allocator_Simple(Mem_Manager &m) { mm = &m; esi_local = edi_local = ebx_local = ebp_local = -1; }
    int id() { return 1; }
    X86_Reg_No reg_var_allocated_to(int index, const unsigned char *where);
    unsigned var_in_reg(X86_Reg_No reg_no, const unsigned char *where);
    int need_sorted_gc_sites() { return 1; }
    int var_is_live_on_entry(unsigned var) { return 1; }
    int space_in_method_info(Jit_Method_Info *method_info) { return 0; }
    void set_call_site_info(Jit_Method_Info *new_mi, unsigned size) {}
    void set_register_saved_info(Jit_Method_Info *new_mi) {}
    unsigned callee_saved_regs(int ebp_based);
    void update_method_info(Jit_Method_Info *mi);
    unsigned register_allocate(class CG_Prepass *prepass,
        Method_Handle methodHandle,
        Class_Handle classHandle,
        unsigned max_locals,
        const unsigned char *bc,
        unsigned code_length);
    unsigned char registers_available_in_codegen_bb(const unsigned char *bc, int &hint) { return 0; }
    void init_rra(struct Runtime_Register_Allocator &rra) {
        rra.get_register_mapping_and_liveness = get_register_mapping_and_liveness;
        rra.should_restore_callee = should_restore_callee;
        rra.should_restore_ebp = should_restore_ebp;
    }
private:
    static void get_register_mapping_and_liveness(unsigned k, Call_Site_Info *csi,
        Jit_Method_Info *mi, X86_Reg_No &reg, int &live);
    static int should_restore_callee(Jit_Method_Info *mi, X86_Reg_No reg);
    static void should_restore_ebp(Jit_Method_Info *method_info, int out_args, int &restore, int &off);
   	short esi_local;							// mapping of regs to local vars
	short edi_local;
	short ebx_local;
	short ebp_local;
};

class Register_Allocator_Priority : public Register_Allocator
{
public:
    Register_Allocator_Priority(Mem_Manager &m) { mm = &m; fg = NULL; }
    int id() { return 2; }
    X86_Reg_No reg_var_allocated_to(int index,
        const unsigned char *where);
    unsigned var_in_reg(X86_Reg_No reg_no, const unsigned char *where);
    int need_sorted_gc_sites() { return 1; }
    int var_is_live_on_entry(unsigned var);
    int space_in_method_info(struct Jit_Method_Info *method_info) { return (method_info->num_in_args + method_info->num_vars + 7) / 8; }
    void set_call_site_info(struct Jit_Method_Info *new_mi, unsigned size);
    void set_register_saved_info(struct Jit_Method_Info *new_mi);
    unsigned callee_saved_regs(int ebp_based);
    void update_method_info(struct Jit_Method_Info *mi) {}
    unsigned char registers_available_in_codegen_bb(const unsigned char *bc, int &hint);
    unsigned register_allocate(class CG_Prepass *prepass,
        Method_Handle methodHandle,
        Class_Handle classHandle,
        unsigned max_locals,
        const unsigned char *bc,
        unsigned code_length);
    void init_rra(struct Runtime_Register_Allocator &rra) {
        rra.get_register_mapping_and_liveness = get_register_mapping_and_liveness;
        rra.should_restore_callee = should_restore_callee;
        rra.should_restore_ebp = should_restore_ebp;
    }
private:
    static void get_register_mapping_and_liveness(unsigned k, Call_Site_Info *csi,
        Jit_Method_Info *mi, X86_Reg_No &reg, int &live);
    static int should_restore_callee(Jit_Method_Info *mi, X86_Reg_No reg);
    static void should_restore_ebp(Jit_Method_Info *method_info, int out_args, int &restore, int &off);
    class Jcfg *fg;
};

#endif // _REGISTER_ALLOCATOR_H
