// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/cg_conversion.cpp,v 1.2 2001/08/13 09:59:51 xhshi Exp $
//

#include "defines.h"
#include "jit_intf.h"
#include "jit_runtime_support.h"
#include <iostream.h>
#include "code_emitter.h"
#include "stack.h"
#include "operand.h"
#include "lazy_code_selector.h"
#include "cg_conversion.h"
#include "fp_compatibility.h"

//
//	mov  ecx, eax
//	sar  ecx, 31
//
void gen_i2l(Code_Emitter& emitter,Stack& stack) {
	Reg_Operand *dst_reg;
	Operand *src = stack.pop();
	// src is not a local reg
	if (!src->is_reg() ||
		!src->hold_local_reg(stack.reg_manager.local_regs())) {
		src->free_opnd(&stack.reg_manager);
		dst_reg = stack.reg_manager.get_reg();
		src->emit_mov_to_reg(emitter,&dst_reg->opnd);
	} else
		dst_reg = (Reg_Operand*)src;
	//
	// high 32 bit
	Reg_Operand *reg = stack.reg_manager.get_reg();
	dst_reg->emit_mov_to_reg(emitter,&reg->opnd);
	emitter.emit_shift(sar_opc,&reg->opnd,&Imm_Opnd(31));
	stack.push64(dst_reg,reg);
}

//
//  fild  mem_opnd		; load operand to fp stack
//  fst	  off[esp]		; store result back to spill area
//
void gen_int2fp(Mem_Manager&            mem_manager,
		        Code_Emitter&           emitter,
                Stack&                  stack,
				Pre_Alloc_Operand_Pool& op_pool,
				bool                    is_dbl) {
	Operand *src = stack.pop();
	Stack_Operand *stk_src = op_pool.nth_stack(stack.depth()+is_dbl);
	//
	// Because fild takes only mem operand, we spill src if src is
	// not mem type
	//
	if (!src->is_mem()) {
		gen_store32(emitter,stack,&stk_src->opnd,src);
		src = stk_src;
	} else
		src->free_opnd(&stack.reg_manager);

    //
    // Too much precision causes fp comparison to fail.  Here
    // we force the result to be stored back to memory.
    //
    if (true || stack.fp_strict_mode)
    {
        ((Mem_Operand*)src)->emit_fild(emitter,0);
	    emitter.emit_fst(&stk_src->opnd,is_dbl,1);
	    if (!is_dbl)  // float
		    stack.push(stk_src);
	    else  // double
		    stack.push64(op_pool.nth_stack(stack.depth()+1),
					     op_pool.nth_stack(stack.depth()));
    }  
    else
    {
        if (stack.fp_check_stack(is_dbl, op_pool, emitter)) {
            stack.fp_inc_cnt();
        }
        ((Mem_Operand*)src)->emit_fild(emitter,0);
    
        result_on_fp_stack(mem_manager, stack, is_dbl);
    }
}

//
//  fild  mem_opnd		; load operand to fp stack
//  fst	  off[esp]		; store result back to spill area
//
void gen_long2fp(Mem_Manager&            mem_manager,
                 Code_Emitter&           emitter,
                 Stack&                  stack,
				 Pre_Alloc_Operand_Pool& op_pool,
				 bool                    is_dbl) {
	Operand *src_lo,*src_hi;
	stack.pop64(src_lo,src_hi);
	Stack_Operand *stk_lo = op_pool.nth_stack(stack.depth()+1);
	Stack_Operand *stk_hi = op_pool.nth_stack(stack.depth());

	if (!src_lo->is_mem()) {
		gen_store32(emitter,stack,&stk_lo->opnd,src_lo);
		src_lo = stk_lo;
	}
	if (!src_hi->is_mem()) {
		gen_store32(emitter,stack,&stk_hi->opnd,src_hi);
		src_hi = stk_hi;
	}
	src_lo->free_opnd(&stack.reg_manager);
	src_hi->free_opnd(&stack.reg_manager);

    //
    // Too much precision causes fp comparison to fail.  Here
    // we force the result to be stored back to memory.
    //
    if (true || stack.fp_strict_mode)
    {
        ((Mem_Operand*)src_lo)->emit_fild(emitter,1);
	    //
	    // The result is stored back to stk_hi because the stack_depth is 
	    // increased by 1 (not 2) after the result is pushed onto stack.
	    //
	    if (!is_dbl) { // float
		    emitter.emit_fst(&stk_hi->opnd,0,1);
		    stack.push(stk_hi);
	    } else { // double
		    emitter.emit_fst(&stk_lo->opnd,1,1);
		    stack.push64(stk_lo,stk_hi);
	    }
    }
    else
    {
        if (stack.fp_check_stack(is_dbl, op_pool, emitter)) {
            stack.fp_inc_cnt();
        }
        ((Mem_Operand*)src_lo)->emit_fild(emitter,1);

        result_on_fp_stack(mem_manager, stack, is_dbl);
    }
}

void gen_float2int(Mem_Manager&   mem_manager,
				   Code_Emitter&  emitter,
                   Stack&         stack,
				   Code_Patch*&   code_patch_list,
				   ORP_RT_SUPPORT help_func) {
	stack.call_home(1); 
	Operand *src = stack.pop();
	src->emit_push(emitter); 
	void *addr = orp_get_rt_support_addr(help_func);
	unsigned patch_offset = emitter.get_offset()+1;
	emitter.emit_call((char *)addr);
	code_patch_list = 
		new(mem_manager) Call_Patch(code_patch_list,patch_offset,(char*)addr);
	src->free_opnd(&stack.reg_manager);

	if (help_func == ORP_RT_F2I) 
		stack.push(stack.reg_manager.get_reg(eax_reg));
	else if (help_func == ORP_RT_F2L) 
		stack.push64(stack.reg_manager.get_reg(eax_reg),
					 stack.reg_manager.get_reg(edx_reg));
	else assert(0);
}

//
// push  hi-32bit
// push  low-32bit
// call   ORP_RT_D2I help func
//
void gen_double2int(Mem_Manager&   mem_manager,
				    Code_Emitter&  emitter,
                    Stack&         stack,
				    Code_Patch*&   code_patch_list,
				    ORP_RT_SUPPORT help_func) {
	stack.call_home(2);
	Operand *src_lo, *src_hi;
	stack.pop64(src_lo,src_hi);
	src_hi->emit_push(emitter); 
	src_lo->emit_push(emitter,1); 
	void *addr = orp_get_rt_support_addr(help_func);
	unsigned patch_offset = emitter.get_offset()+1;
	emitter.emit_call((char *)addr);
	code_patch_list = 
		new(mem_manager) Call_Patch(code_patch_list,patch_offset,(char*)addr);
	src_lo->free_opnd(&stack.reg_manager);
	src_hi->free_opnd(&stack.reg_manager);

	if (help_func == ORP_RT_D2I)
		stack.push(stack.reg_manager.get_reg(eax_reg));
	else if (help_func == ORP_RT_D2L)
		stack.push64(stack.reg_manager.get_reg(eax_reg),
					 stack.reg_manager.get_reg(edx_reg));
	else assert(0);
}

//
//  fld  mem_opnd		; load operand to fp stack
//  fst	  off[esp]		; store result back to spill area
//
void gen_f2d(Mem_Manager&            mem_manager,
             Code_Emitter&           emitter,
             Stack&                  stack,
			 Pre_Alloc_Operand_Pool& op_pool) {
	Operand *src = stack.pop();
	Stack_Operand *stk_src = op_pool.nth_stack(stack.depth()+1);

    if (stack.fp_strict_mode)
    {
	    if (!src->is_mem()) {
		    gen_store32(emitter,stack,&stk_src->opnd,src);
		    src = stk_src;
	    } else
		    src->free_opnd(&stack.reg_manager);
        ((Mem_Operand*)src)->emit_fld(emitter,0);
        emitter.emit_fst(&stk_src->opnd,1,1);
	    stack.push64(stk_src,op_pool.nth_stack(stack.depth()));
    }
    else
    {
        load_onto_fp_stack(stack, src, false);
        result_on_fp_stack(mem_manager, stack, true);
    }
}

//
//  fld  mem_opnd		; load operand to fp stack
//  fst	  off[esp]		; store result back to spill area
//
void gen_d2f(Mem_Manager&            mem_manager,
             Code_Emitter&           emitter,
             Stack&                  stack,
			 Pre_Alloc_Operand_Pool& op_pool) {


	Operand *src_lo, *src_hi;
	stack.pop64(src_lo,src_hi);
	Stack_Operand *stk_lo = op_pool.nth_stack(stack.depth()+1);
	Stack_Operand *stk_hi = op_pool.nth_stack(stack.depth());

    //
    // gen_d2f() is not contracting doubles to non-strict floats correctly.
    // The bad contract values can exceed legal float-extended-exponent range.
    // Always go through the "strict mode" code.  
    //
    if (true || stack.fp_strict_mode) {
	    if (!src_lo->is_mem()) {
		    gen_store32(emitter,stack,&stk_lo->opnd,src_lo);
		    src_lo = stk_lo;
	    }

	    if (!src_hi->is_mem()) {
		    gen_store32(emitter,stack,&stk_hi->opnd,src_hi);
		    src_hi = stk_hi;
	    }

	    src_lo->free_opnd(&stack.reg_manager);
	    src_hi->free_opnd(&stack.reg_manager);
        ((Mem_Operand*)src_lo)->emit_fld(emitter,1);
        emitter.emit_fst(&stk_hi->opnd,0,1);
	    stack.push(stk_hi);
    }
    else
    {
        load_onto_fp_stack(stack, src_lo, true);
        result_on_fp_stack(mem_manager, stack, false);
    }
}

//
// int to byte, char, or short
void gen_int2bcs(Code_Emitter& emitter,Stack& stack,
				 unsigned is_signed,unsigned is_half){
	Operand *src = stack.pop();
	//
	// if src is a register, then we have to make sure that the register 
	// is byte register, i.e., eax, edx, ecx, and ebx, because MOVSX reads 
	// the contents of the effective address or register as a byte.
	//
	if ( !src->is_mem() && 
		(!src->is_reg() || !((Reg_Operand*)src)->is_byte_reg())) {
		src->free_opnd(&stack.reg_manager);
		Reg_Operand *reg = stack.reg_manager.get_byte_reg();

		src->emit_mov_to_reg(emitter,&reg->opnd);
		src = reg;
	}
	src->free_opnd(&stack.reg_manager);
	Reg_Operand *dst_reg = stack.reg_manager.get_reg();
	src->emit_widen(emitter,&dst_reg->opnd,is_signed,is_half);
	stack.push(dst_reg);
}
