// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/train_generation.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//


#ifndef _train_generation_H_
#define _train_generation_H_

//
// Generations which have trains instead of steps. Train_Generations
// inherit from Generation.
//
//                Generation
//                    |
//                    |
//       ---------------------------
//       |                         |
//       |                         |
//    Step_Generation              |
//       |                         |
//       |                  Train_Generation
//       |
//  Step_Plus_Nursery_Generation
//



class Mrl_Gc_V1;
#include <iostream.h>
#include "generation.h"
#include "mrl_gc_v1.h"

#define MAXIMUM_TRAINS 16

class Car;
class Card_Table;
class Gc_Space;
class Nursery;
class Train;
class Step_Plus_Nursery_Generation;
class Gc_Fast_Hooks;
class Gc_Plan;

class Train_Generation : public Generation {
public:

    Train_Generation(unsigned char generation_number,
                     Gc_Fast_Hooks  *p_gc_hooks,
                     Gc_Plan        *p_gc_plan,
		             Gc_Interface   *p_container,
					 Generation     *p_superior,
                     Card_Table     *p_card_table,
		             Block_Store    *p_bs);

	virtual ~Train_Generation();

    void _execute_policy (Java_java_lang_Object **pp_child_ref, bool doing_mos_collection);
	//
	// Every generation needs to define a policy for handling objects
	// found during cheney scans. Train generations evacuate all followers
	// into the train they are referenced from.
	//
	virtual void cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection);

//#ifdef GC_OFFSET_LOGIC
	virtual void cheney_scan_array_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection);
//#endif // GC_OFFSET_LOGIC

	//
	// The GC interface gives each generation an opportunity to
	// clean up after a collection of that generation.
	//
	virtual void cleanup() {
		orp_exit(1); // stub
	}

	//
	// In its dying gasp, a contained car is notifying us that the
	// old-to-young references corresponding to that car should be
	// flushed.
	//
	void flush_rs_entries_of_car(Car *p_car);
	//
	// The young generation is tenuring an object 
	// into my space.
	//
	Java_java_lang_Object *p_import_object_from_yos(Java_java_lang_Object **pp_obj);
    
    // Move an object from the focus car to a car in another train.
  	Java_java_lang_Object *p_move_train_object(Java_java_lang_Object **pp_obj);

    // Move an object from the focus car to another car in the focus train
  	Java_java_lang_Object *p_move_car_object(Java_java_lang_Object **pp_obj);
    
    void set_executing_mature_generation_collection (bool value); 

	//
	// Return the focus car.
	//
	virtual Car *p_nominate_focus_car();
	//
	// Return the oldest train in my generation.
	//
	virtual Train *p_nominate_focus_train();
	//
	// Check if car is the focus / from / oldest car.
	//
    
	bool is_focus_car(Car *p_car); 
	//
	// Check if specified object is in the focus / from / oldest car.
	//
	bool is_object_in_focus_car(Java_java_lang_Object *p_object) {
		Gc_Space *p_container = _p_block_store->p_get_object_container(p_object);
#if (GC_DEBUG>2)
	assert(p_container->is_obsolete() != true);
#endif // _DEBUG
		if (p_container->is_car()) {
			if (is_focus_car((Car *)p_container)) {
				return true;
			}
		}
		return false;
	}

	//
	// We are asked to reclaim our generation with a remembered set
	// of references into the oldest focus car, along with a boolean
	// that tells us if any pointers into the oldest train were found
	// when collecting the younger generation. We can use this to
	// optimize reclamation of garbage-filled trains.
	//
	virtual Remembered_Set *
		p_reclaim_generation(Remembered_Set *p_young_to_old_rs,
                             Remembered_Set *p_weak_refs,
		                     bool train_is_alive);

    //
    // At shutdown time, may need to run all finalizers.
    //
    virtual void run_all_finalizers();

    virtual bool is_train_generation() {
        return true;
    }

    virtual bool is_step_generation() {
        return false;
    }

    virtual bool is_step_plus_nursery_generation() {
        return false;
    }

    // Given a pointer to a slot in a train return the train.

    static Train *slot_to_train (Java_java_lang_Object **pp_obj_ref);

    // Given a pointer to a slot get the car holding the slot.
    static Car *slot_to_car (Java_java_lang_Object **pp_obj_ref);

    //
    // A contained car is telling us to move this object to a specific
    // train, because the referencing pointer is in that train.
    //
	Java_java_lang_Object *move_object_to_referencing_train(Java_java_lang_Object **pp_ref);

#if (GC_DEBUG>3)
	//
	// Routine to display the contents of the generation.
	//
	virtual void inspect(unsigned int level);
#endif // _DEBUG

	//
	// Train Generations don't have nurseries.
	//
	virtual Nursery *p_cycle_nursery(Nursery *p_nursery,
		                             bool returnNullOnFail) {
		orp_exit(1); // illegal for this generation type.
		return NULL;
	}

	//
	// Used by the GC Interface container to set a pointer to younger
	// generation.
	//
	void set_young_generation(Step_Plus_Nursery_Generation *p_gen) {
		_p_young_generation = p_gen;
	}
    //
	// Some object movement is necessitating a re-computation of the
	// inter-car write barriers in this mature space.
    //
    virtual void add_entry_to_generation_write_barriers(Java_java_lang_Object **pp_obj_ref,
                                                        Java_java_lang_Object *p_obj_ref);
    //
    // This is a fast version (train-specific, non-virtual) version
    // of add_entry_to_generation_write_barriers. Retire the other
    // if this works well.
    //
    void update_car_write_barriers(Java_java_lang_Object **pp_obj_ref,
                                   Java_java_lang_Object *p_obj);

#if (GC_DEBUG>0)
    //
    // Debug routine to verify all the cars write barriers in all the trains.
    //
    void verify_trains_cars_write_barriers();
#endif // _DEBUG

//
// Scan cheney spaces determines if any of the cars have objects 
// that need to be scanned. If so it scans them and returns true
// to indicate that more work might need to be done.
// Otherwise it returns false.
// If all the generations return false then there is no more cheney 
// scanning to be done.
bool scan_cheney_generation (bool doing_mos_collection); 
bool cheney_scan_pending();

#if (GC_DEBUG>3)
	//
	// Routine to verify the consistency of all the spaces of this generation.
	//
	virtual bool verify_generation();
#endif // _DEBUG

#if (GC_DEBUG>3)
	//
	// At debug time we do a painfully slow scan of all of mature
	// space to determine if there are any mature to young pointers
	// that didn't get recorded by the write barriers.
	//
	void verify_all_mature_to_young_references_are_recorded();
#endif
    // The oldest train holds not cars so release it.
    void free_empty_oldest_train();

	Train *p_get_youngest_train();



    void enumerate_reference_queues ();

    virtual void enqueue_soft_ref (java_lang_ref_Reference *a_reference);
	virtual void enqueue_weak_ref (java_lang_ref_Reference *a_reference);
	virtual void enqueue_phantom_ref (java_lang_ref_Reference *a_reference);

private:
    Train_Generation();

	//
	// Add a new train to this generation.
	//
	void _add_new_train();

    //
    // Reclaim an entire train that has been deemed to be junk.
    //
    void _free_oldest_train();
	//
	// This routine implements the plan for placing imported objects 
	// into optimal trains.
	//
	Java_java_lang_Object *
		_heuristic_place_yos_object(Java_java_lang_Object **pp_obj_ref);

    unsigned long _number_of_trains;

    Train *_p_train[MAXIMUM_TRAINS]; 
	//
	// A pointer to my younger generation.
	//
	Step_Plus_Nursery_Generation *_p_young_generation;
};

#endif // _train_generation_H_
