// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/block_list.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//


#ifndef _block_list_H_
#define _block_list_H_



// A Block List is a GC space (i.e. a container) which consists of a list
// of blocks. This is an abstraction of steps and cars. While a Nursery
// consists of a list of contiguous blocks, descendents of Block_List consist
// of a list of non-contiguous blocks. Block_List is an abstract class.
//
//
//                          Gc_Space (abstract)
//                              |
//             ------------------------------------
//             |                                  |
//             |                                  |
//        Block_List (abstract)                   |
//             |                                  |
//      ---------------------                     |
//      |                    |                 Nursery
//     Car                  Step
//
//


#include "platform.h"
#include "gc_hooks.h"
#include "gc_header.h"
#include "remembered_set.h"
#include "Block_Store.h"
#include "generation.h"
#include "descendents.h"

#include "gc_space.h"
#include "gc_consts.h"

class Card_Table;
class Gc_Fast_Hooks;
class Gc_Plan;

class Block_List : public Gc_Space {
public:
    //
    // Create a step (done by Step_Plus_Nursery_Generation).
    //
    Block_List(unsigned long space_id,
        Gc_Fast_Hooks *p_gc_hooks,
        Gc_Plan       *p_gc_plan,
        Generation    *p_container,
        Gc_Space      *p_superior,
        Card_Table    *p_card_table,
        Block_Store   *p_block_store)
        
        : Gc_Space(space_id, 
        p_gc_hooks,
        p_gc_plan,
        p_container, 
        p_superior, 
        p_card_table,
        p_block_store)	{
#if (GC_DEBUG>2)
        assert(_p_block_store);
#endif // _DEBUG
        
        _number_of_blocks        = 0;
        
        assert(sizeof(_current_block_id) == 4);
        _current_block_id = 0xFFFFFFFF;
        //
        // Don't create a default block. The first cheney copy will do this.
        //
        _p_base = _p_free = _p_scan = NULL;
    }
    
    //
    // Deletion is child space-specific.
    //
    virtual ~Block_List() {
        unsigned int block_index;
        for (block_index = 0; 
        block_index < _number_of_blocks; 
        block_index++) {
            
            _p_block_store->release_super_block(_p_block[block_index]);
        }
    }
    
    void scavenge_block(void *the_block, 
        int the_size,
        void *block_end);
    
    
    // Check if this space owns this block
    bool debug_owns_block(void *p_block)
    {
        for (unsigned int idx = 0; idx < _number_of_blocks; idx++) {
            if (p_block == _p_block[idx]) {
                return true;
            }
        }
        return false;
    }
    
    //
    // Should use inspect instead.
    //
    //
    // At debug time, when a car has been reclaimed, or when
    // the FROM blocks are being reclaimed, we may choose to
    // avoid re-using the blocks, instead marking them as
    // in an obsolete container.
    //
    void debug_set_blocks_obsolete();
    void debug_write_protect_blocks();
    
    //	
    // Given a pointer to an object in a forpign step/,
    // nursery,import it and set the original forwarded.
    //    
    virtual
        Java_java_lang_Object *p_scavenge_object(Java_java_lang_Object **pp_obj);
    
    // 
    // Given a block list do a cheney scan on the list, This could be
    // a car or a step.
    //
    bool Block_List::cheney_scan_block_list(bool doing_mos_collection);
    bool Block_List::cheney_scan_pending();
    
    //
    // Do any required cleanup at the end of the collection.
    //
    virtual void cleanup();
    
#if (GC_DEBUG>3)
    //
    // Each space provides a routine to verify that it is consistent.
    // i.e. contains only well-formed objects.
    //
    virtual bool verify_space();
#endif // _DEBUG
    
    //
    // Walk the entire list in the container and apply the function header.
    //
    void walk_list(bool (*func)(Object_Gc_Header *,
        Remembered_Set *));
    
    
protected:
    
    //
    // The following routine should be implemented by all containers
    // that implement block lists. The value returned by the container
    // reflects a generation encoding, which is used by the block list
    // object for marking all sub blocks in the block store. This is
    // used later for doing fast checks of generations.
    //
    virtual unsigned long _get_generation_encoding() = YOS_GEN_NUMBER;
    
    
    Block_List();
    //
    // Helper routine to add a new block into a step.
    //
    void _add_block();
    
    //
    // Bump the scan pointer forward past an object, return true if you have a new object,
    // otherwise return false.
    //
    inline bool
        Block_List::_advance_scan_pointer(unsigned int real_object_size_bytes)
    {
#if (GC_DEBUG>2)
        _debug_check_block_integrity(); // DEBUG
#endif // _DEBUG
        
        if (_p_free == _p_scan) {
            //
            // Done with scan.
            //
            assert(_current_block_id == (_number_of_blocks - 1));
            assert(_number_of_blocks > 0);
            return false;
        }
        
        assert (_current_block_id >= 0);
        assert(_p_block_end[_current_block_id] < _p_block_store->p_get_address_super_block_end_address(_p_scan));
        assert(_p_scan <= _p_block_end[_current_block_id]);
        assert (_current_block_id < _number_of_blocks); // _current_block_id is 0 based.
        
        if (_p_scan < _p_block_end[_current_block_id]) {
            //
            // OK, bumping the scan pointer won't go past the last object
            // in this block, the current block.
            //
            
#ifdef OBJECT_SPLITTING
            extern bool is_object_gc_header(void *p_thing);
            assert(is_object_gc_header(_p_scan));
            
            _p_scan = (Object_Gc_Header *)((char *)_p_scan + (real_object_size_bytes / 2));
            
            if (_p_scan != _p_free) {
                assert(is_object_gc_header(_p_scan));
            }
#else
            _p_scan = 
                (Object_Gc_Header *)((char *)_p_scan + 
                real_object_size_bytes);
#endif // OBJECT_SPLITTING
            
            goto success;
        } else if (_p_scan == _p_block_end[_current_block_id]) {
            //
            // We are pointing to the last object in this block.
            //
            if (_current_block_id < (_number_of_blocks - 1)) {
                //
                // Go to the next block in this step.
                //
                _p_scan = (Object_Gc_Header *)_p_block[_current_block_id + 1];
                _current_block_id++;
#if (GC_DEBUG>2)
                _debug_check_block_integrity();
#endif // DEBUG
                
                goto success;
            } else {
                //
                // Ok, its the last block and we are at the end. 
                // Signal end of scan.
                //
#ifdef OBJECT_SPLITTING
                assert ( ((char *)_p_scan + (real_object_size_bytes / 2)) == _p_free);
                _p_scan = (Object_Gc_Header *)_p_free;
#else
                assert ( ((char *)_p_scan + real_object_size_bytes) == _p_free);
                _p_scan = (Object_Gc_Header *)_p_free;
#endif // OBJECT_SPLITTING
                
#if (GC_DEBUG>1)
                _debug_check_block_integrity(); // DEBUG
#endif
                return false; 
            }
        } else {
            //
            // this means _p_scan has gone past the last object in this block.
            //
            orp_cout << "Error advancing scan pointer during cheney scan" << endl;
            assert (0);
            orp_exit(1); // shouldn't get here
            return false;
        }
success:
#if (GC_DEBUG>1)
        _debug_check_block_integrity();
#endif
        
        return true;
    }
    
    
    bool _will_fit_in_sub_block (void *p_free_ptr, int size);
    
    Object_Gc_Header *p_fast_reserve_space (int real_object_size_bytes);
    
    Java_java_lang_Object *p_reserve_space (Java_java_lang_Object **pp_obj_ref);
    Java_java_lang_Object *p_reserve_bytes (unsigned size);
    
    Java_java_lang_Object *_add_into_block(Java_java_lang_Object **pp_obj);
    
    //
    // Identifies current block being cheney scanned.
    //
    unsigned long _current_block_id;
    
    //
    // Number of blocks in this list.
    //
    unsigned long _number_of_blocks;
    
    //
    // Debug time integrity checking routine.
    //
    void _debug_check_block_integrity();
    
    void _scavenge_descendents(Java_java_lang_Object *p_obj);
    
    void *_p_block[MAXIMUM_BLOCKS_IN_LIST];
    Object_Gc_Header *_p_block_end[MAXIMUM_BLOCKS_IN_LIST];
    
#if (GC_DEBUG>0)
    void _inspect_block(void             *p_block, 
        Object_Gc_Header *p_block_end, 
        unsigned long    block_idx) {
        cout << "        start = " << p_block ;
        cout << " || end   = " << p_block_end ;
        cout << " || ceiling = ";
        cout << (void *)((int)p_block+(int)my_block_size_bytes()) << endl;
        return; // STUB
    }
#endif 
    
    //
    // Verify all objects in specified block are consistent.
    //
    bool _verify_block(void *p_block_start,
        void *p_block_end);
    //
    // This routine takes a function argument that accepts a GC header
    // and walks the entire block applying that function to each
    // object. The walk terminates if the function returns false.
    //
    void _walk_block(void *p_block_start, 
        void *p_block_end,
        bool (*func)(Object_Gc_Header *,
        Remembered_Set *));
    
    bool _will_not_fit_in_block(Java_java_lang_Object **pp_obj_ref);
    
    bool _will_not_fit_in_block(unsigned long real_object_size_bytes);
};



#endif // _block_list_H_

