// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/compile.cpp,v 1.4 2001/11/14 23:31:29 michal Exp $
//

#include "platform.h"
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "orp_types.h"
#include "Class.h"
#include "environment.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "exceptions.h"
#include "jit_intf.h"
#include "jit_intf_cpp.h"
#include "root_set_enum.h"
#include "object_layout.h"
//#include "find_natives.h"
#include "nogc.h"
 
#include "gc_for_orp.h"
 
#include "orp_utils.h"
#include "orp_synch.h"
#include "orp_threads.h"
#include "ini.h"
#include "orp_vtune.h"
#include "orp_stats.h"

#include "compile.h"

#ifdef CLI_TESTING
#include "cli_misc.h"
#endif

#ifdef DUMP_JIT
#include "dump_jit.h"
#endif // DUMP_JIT


// This should probably be in jni.h
//extern const struct JNINativeInterface_ *jni_native_intf;
extern const struct JNIEnv_ *jni_native_intf;

JIT_Result prepare_native_method(Method &method,
                                 JIT_Flags flags
                                 );


void orp_init(Global_Env *env, Loader_Exception *exception);



void init_object_handles_for_args(Method          *method,
                                  J2N_Saved_State *ljf,
                                  uint32          *repushed_args
                                  )
{
    unsigned ref_arg_num = 0;
    unsigned num_ref_args = method->get_num_ref_args();
    if(method->is_static()) {
        num_ref_args++;
    }
    assert(num_ref_args);   // We only call this function if there exist ref args.
    Object_Handle_Struct *object_handles =
        ((Object_Handle_Struct *)ljf) - num_ref_args;
    Object_Handle_Struct *h;
    Object_Handle_Struct *sentinel = &(object_handles[num_ref_args]);

    // First connect the handles into a list and mark them as stack allocated.
    Object_Handle_Struct *prev = 0;
    for(h = object_handles; h < sentinel; h++) {
        h->java_reference = 0;
        h->prev = prev;
        h->next = (h + 1);  // This is wrong for the last iteration.  See below.
        h->allocated_on_the_stack = TRUE;
        prev = h;
    }
    object_handles[num_ref_args - 1].next = 0;  // Undo the incorrect assignment above.

    unsigned num_arg_bytes = method->get_num_arg_bytes();

    h = object_handles;

    // For static methods this is the extra jclass argument,
    // otherwise it's the this pointer.
    h->java_reference = (Java_java_lang_Object *)*repushed_args;
    assert(*repushed_args);
    *repushed_args++ = (uint32)h;
    h++;

    // Skip over either the this pointer or the extra arg for static methods.
    //repushed_args++;

    Arg_List_Iterator iter = method->get_argument_list();
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_LONG:
        case JAVA_TYPE_DOUBLE:
            repushed_args += 2;
            break;
        case JAVA_TYPE_CLASS:
        case JAVA_TYPE_ARRAY:
            h->java_reference = (Java_java_lang_Object *)*repushed_args;
            if(*repushed_args) {
                *repushed_args = (uint32)h;
            }
            repushed_args++;
            h++;
            break;
        default:
            repushed_args++;
            break;
        }
        iter = advance_arg_iterator(iter);
    }
    assert(h == sentinel);
    assert(repushed_args == (uint32 *)object_handles);

    ljf->local_object_handles = object_handles;
} //init_object_handles_for_args



void free_local_object_handles(Object_Handle head,
                               Object_Handle tail
                               )
{
    Object_Handle next;
    for(; head != tail; head = next) {
        assert(head);
        next = head->next;
        orp_free_object_handle(head);
    }
} //free_local_object_handles



void run_main_entry_point(Method *entry_point, void *java_args)
{
    J_Value arg;
    arg.r = java_args;

    // This method is called only once - during ORP start. we make sure
    // that the app_status for the current thread is "birthing"
	assert(p_TLS_orpthread->app_status == thread_is_birthing);

    // Now, we set it to "running" just before entering Java code.
	p_TLS_orpthread->app_status = thread_is_running;

	// Moved from run_main_from_class() to here
	if(!entry_point) {
        orp_cout << "Couldn't find main" << endl;
        return;
    }

    orp_execute_java_method_array(entry_point, 0, &arg);

    Java_java_lang_Throwable *exc = (Java_java_lang_Throwable *)get_current_thread_exception();
    set_current_thread_exception(0);
    if(exc) {
        print_uncaught_exception_message(stderr, exc);
        //orp_cout << "Uncaught exception:" << exc->vt->clss->name->bytes << endl;
        //print_stack_trace(exc);
    }
} //run_main_entry_point



void run_main_from_class(Class *clss, Signature *main_sig, void *java_args)
{
    assert(main_sig);
    Method *entry_point = class_lookup_method(clss, main_sig);
/*	 Should move the "Couldn't find main" block to a later place, if we run a class without main() method,
	 Consider the following sequence in main2()
	        int result = run_java_main(classname, argv + arg_num, java_argc, p_env);
			assert(p_TLS_orpthread->app_status == thread_is_running);
	  But if we exit here, we haven't set p_TLS_orpthread->app_status to thread_is_running;
	  Then assert...
*/
/*    if(!entry_point) {
        orp_cout << "Couldn't find main" << endl;
        return;
    }
*/
    run_main_entry_point(entry_point, java_args);
} //run_main_from_class



//
// This function finds the static main([LString;)V method and executes it.
//
void compile_and_run_class(Class *clss, void *java_args)
{
    Signature *main_sig = ORP_Global_State::loader_env->Main_Signature;
    run_main_from_class(clss, main_sig, java_args);
} //compile_and_run_class



#ifdef CLI_TESTING

void cli_compile_and_run_class(Class *clss, void *java_args)
{
    //printf("cli_compile_and_run_class: %s\n", clss->name->bytes);
    Global_Env *env = ORP_Global_State::loader_env;
    String *name = env->string_pool.lookup("Main");
    String *descriptor = env->string_pool.lookup("()V");
    Signature *main_sig = env->sig_table.lookup(name, descriptor);
    run_main_from_class(clss, main_sig, java_args);
} //cli_compile_and_run_class

#endif



Method *find_entry_point(String *classname, Global_Env *p_env)
{
	Loader_Exception exception = LD_NoException;
    Method *entry_point = 0;

	orp_init(p_env, &exception);
    assert(exception == LD_NoException);

	Class *clss = load_class(p_env, classname, &exception);
    if(clss == NULL) {
#ifdef CLI_TESTING
        // Interpret classname as a file name
        Method *entry_point = find_entry_point_in_pe_file(classname->bytes);
        if(entry_point) {
            return entry_point;
        }
#endif
        switch (exception) {
        case LD_ClassCircularityError:
			break;
		case LD_NoClassDefFoundError:
			break;
		case LD_ClassFormatError:
			break;
    	}
        printf("Couldn't load class: %s\n", classname->bytes);
        orp_exit(1);
    }

	//
	// Signal the GC that the ORP is now completely 
	// initialized. Hence GC stop-the-world events
	// can occur at any time, since the ORP is now
	// ready to enumerate live references when asked.
	//

    gc_orp_initialized();
  
#ifdef CLI_TESTING
    if(clss->cli_type_def) {
        Loader_Result res = cli_class_prepare(clss);
        assert(res == LD_OK);
        cli_class_initialize(clss);
    } else {
#endif
    class_prepare(clss);
    class_initialize(clss);
#ifdef CLI_TESTING
    }
#endif

    Signature *main_sig = 0;
#ifdef CLI_TESTING
        if(clss->cli_type_def) {
            Global_Env *env = ORP_Global_State::loader_env;
            String *name = env->string_pool.lookup("Main");
            String *descriptor = env->string_pool.lookup("()V");
            main_sig = env->sig_table.lookup(name, descriptor);
            //cli_compile_and_run_class(clss, java_argv);
        } else {
#endif
            main_sig = ORP_Global_State::loader_env->Main_Signature;
            //compile_and_run_class(clss, java_argv);
#ifdef CLI_TESTING
        }
#endif
    assert(main_sig);
    entry_point = class_lookup_method(clss, main_sig);
    return entry_point;
} //find_entry_point



////////////////////////////////////////////////////////////////////////
// begin JIT_Specific_Info


JIT_Specific_Info::JIT_Specific_Info()
{
#ifdef ORP_VTUNE_SUPPORT
    _has_been_loaded_for_vtune = false;
#endif
#ifdef ORP_STATS
    num_throws  = 0;
    num_catches = 0;
    num_unwind_java_frames_gc = 0;
    num_unwind_java_frames_non_gc = 0;
#endif
} //JIT_Specific_Info::JIT_Specific_Info


// end JIT_Specific_Info
////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////
// begin JIT management


JIT *jit_compilers[] = {0, 0, 0, 0, 0};


void orp_add_jit(JIT *jit)
{
    int max_jit_num = sizeof(jit_compilers) / sizeof(JIT *) - 2;
    if(jit_compilers[max_jit_num]) {
        assert(0);
        return;
    }

    // Shift the jits
    for(int i = max_jit_num; i > 0; i--) {
        jit_compilers[i] = jit_compilers[i - 1];
    }

    jit_compilers[0] = jit;
    assert(jit_compilers[max_jit_num + 1] == 0);
} //orp_add_jit



void orp_swap_jits(unsigned idx1, unsigned idx2)
{
    assert(jit_compilers[idx1]);
    assert(jit_compilers[idx2]);
    JIT *tmp = jit_compilers[idx1];
    jit_compilers[idx1] = jit_compilers[idx2];
    jit_compilers[idx2] = tmp;
} //orp_swap_jits



#ifdef CLI_TESTING

JIT *jit_compilers_cli[] = {0, 0, 0, 0, 0};


void orp_add_jit_cli(JIT *jit)
{
    int max_jit_num = sizeof(jit_compilers_cli) / sizeof(JIT *) - 2;
    if(jit_compilers_cli[max_jit_num]) {
        assert(0);
        return;
    }

    // Shift the jits
    for(int i = max_jit_num; i > 0; i--) {
        jit_compilers_cli[i] = jit_compilers_cli[i - 1];
    }

    jit_compilers_cli[0] = jit;
    assert(jit_compilers_cli[max_jit_num + 1] == 0);
} //orp_add_jit_cli

#endif


void orp_delete_all_jits()
{
    JIT **jit;
    for(jit = jit_compilers; *jit; jit++) {
        delete (*jit);
        *jit = 0;
    }
#ifdef CLI_TESTING
    for(jit = jit_compilers_cli; *jit; jit++) {
        delete (*jit);
        *jit = 0;
    }
#endif
} //orp_delete_all_jits


void JIT::init()
{

    Compilation_Handle *comp_handle = new Compilation_Handle();
    comp_handle->method = 0;
    comp_handle->env    = ORP_Global_State::loader_env;
    comp_handle->ld_exc = LD_NoException;
    comp_handle->jit    = this;
    global_compile_handle = comp_handle;
    jit_flags.insert_write_barriers = (gc_requires_barriers() != GC_NO_BARRIER);
} //JIT::init


// end JIT management
////////////////////////////////////////////////////////////////////////


