/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


//---------------------------------------------------------------------------

#include <vcl.h>
#pragma hdrstop
#include <d3d.h>
#include <vcl\registry.hpp>

#include "main.h"
#include "D3DRender.h"
#include "OpenGLRender.h"
#include "SoftRender.h"
#include "groupfile.h"
#include "map.h"
#include "DirectInput.h"
#include "art.h"
#include "sound.h"
#include "DSoundEngine.h"
#include "console.h"
#include "music.h"
#include "RTSFile.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"

#define PI 3.14159265358979323846264338327950288

struct
{
    char mapFile[16];
    char mapName[32];
    char mapMusic[32];
} DefaultMapInfo[]=
    {{"e1l1.map","HOLLYWOOD HOLOCAUST","STALKER.MID"},
     {"e1l2.map","RED LIGHT DISTRICT","DETHTOLL.MID"},
     {"e1l3.map","DEATH ROW","STREETS.MID"},
     {"e1l4.map","TOXIC DUMP","WATRWLD1.MID"},
     {"e1l5.map","THE ABYSS","SNAKE1.MID"},
     {"e1l6.map","LAUNCH FACILITY","THECALL.MID"},
     {"e1l7.map","FACES OF DEATH","AHGEEZ.MID"},
     {"e1l8.map","USER MAP","DETHTOLL.MID"},
     {"e1l9.map","MULTIPLAYER 3","STREETS.MID"},
     {"e1l10.map","MULTIPLAYER 4","WATRWLD1.MID"},
     {"e1l11.map","MULTIPLAYER 5","SNAKE1.MID"},
     {"e2l1.map","SPACEPORT","FUTURMIL.MID"},
     {"e2l2.map","INCUBATOR","STORM.MID"},
     {"e2l3.map","WARP FACTOR","GUTWRNCH.MID"},
     {"e2l4.map","FUSION STATION","ROBOCREP.MID"},
     {"e2l5.map","OCCUPIED TERRITORY","STALAG.MID"},
     {"e2l6.map","TIBERIUS STATION","PIZZED.MID"},
     {"e2l7.map","LUNAR REACTOR","ALIENZ.MID"},
     {"e2l8.map","DARK SIDE","XPLASMA.MID"},
     {"e2l9.map","OVERLORD","ALFREDH.MID"},
     {"e2l10.map","SPIN CYCLE","GLOOMY.MID"},
     {"e2l11.map","LUNATIC FRINGE","INTENTS.MID"},
     {"e3l1.map","RAW MEAT","INHIDING.MID"},
     {"e3l2.map","BANK ROLL","FATCMDR.MID"},
     {"e3l3.map","FLOOD ZONE","NAMES.MID"},
     {"e3l4.map","L.A. RUMBLE","SUBWAY.MID"},
     {"e3l5.map","MOVIE SET","INVADER.MID"},
     {"e3l6.map","RABID TRANSIT","GOTHAM.MID"},
     {"e3l7.map","FAHRENHEIT","233C.MID"},
     {"e3l8.map","HOTEL HELL","LORDOFLA.MID"},
     {"e3l9.map","STADIUM","URBAN.MID"},
     {"e3l10.map","TIER DROPS","SPOOK.MID"},
     {"e3l11.map","FREEWAY","WHOMP.MID"},
     {"e4l1.map","IT'S IMPOSSIBLE","MISSIMP.MID"},
     {"e4l2.map","DUKE-BURGER","PREPD.MID"},
     {"e4l3.map","SHOP-N-BAG","BAKEDGDS.MID"},
     {"e4l4.map","BABE LAND","CF.MID"},
     {"e4l5.map","PIGSTY","LEMCHILL.MID"},
     {"e4l6.map","GOING POSTAL","POB.MID"},
     {"e4l7.map","XXX-STACY","WAREHAUS.MID"},
     {"e4l8.map","CRITICAL MASS","LAYERS.MID"},
     {"e4l9.map","DERELICT","FLOGHORN.MID"},
     {"e4l10.map","THE QUEEN","DEPART.MID"},
     {"e4l11.map","AREA 51","RESTRICT.MID"},
     {"",""}};

TForm1 *Form1;

// Menu functions, simply calls the corresponding menu
// function in the main form's class
void YesNoItem() { Form1->YesNoItem(); }
void NewGame() { Form1->NewGame(); }
void Options() { Form1->Options(); }
void SaveGame() { Form1->SaveGame(); }
void LoadGame() { Form1->LoadGame(); }
void Help() { Form1->Help(); }
void QuitToTitle() { Form1->QuitToTitle(); }
void Quit() { Form1->Quit(); }
void Episode1() { Form1->Episode1(); }
void Episode2() { Form1->Episode2(); }
void Episode3() { Form1->Episode3(); }
void Skill1() { Form1->Skill1(); }
void Skill2() { Form1->Skill2(); }
void Skill3() { Form1->Skill3(); }
void Skill4() { Form1->Skill4(); }
void WindowedOption() { Form1->WindowedOption(); }
void ResolutionOption() { Form1->ResolutionOption(); }
void BitDepthOption() { Form1->BitDepthOption(); }
void DriverOption() { Form1->DriverOption(); }
void SoundOption() { Form1->SoundOption(); }
void MusicOption() { Form1->MusicOption(); }
void Multiplayer() { Form1->Multiplayer(); }
void FindServers() { Form1->FindServers(); }
void ConnectToServer() { Form1->ConnectToServer(); }
void StartNewServer() { Form1->StartNewServer(); }
void MapsToCycle() { Form1->MapsToCycle(); }
void ServerGameOptions() { Form1->ServerGameOptions(); }
void CreateServer() { Form1->CreateServer(); }
void JoinServer() { Form1->JoinServer(); }

// Menu structures
// First item defines text within header and texture
// number of header, terminate the menu items with an
// item with NULL as the name
Menu mainMenu[]={{"",NULL,TEXTURE_MENU_HEADER,MENUTYPE_NORMAL,0},
                 {"NEW GAME",NewGame,48,MENUITEM_NORMAL,0},
                 {"MULTIPLAYER",Multiplayer,64,MENUITEM_NORMAL,0},
                 {"OPTIONS",Options,80,MENUITEM_NORMAL,0},
                 {"LOAD GAME",LoadGame,96,MENUITEM_NORMAL,0},
                 {"HELP",Help,112,MENUITEM_NORMAL,0},
                 {"QUIT",Quit,128,MENUITEM_NORMAL,0},
                 {NULL,NULL,0,0,0}};
Menu multiplayerMenu[]={{"MULTIPLAYER",NULL,TEXTURE_MENU_TEXTHEADER,MENUTYPE_NORMAL,0},
                        {"FIND SERVERS",FindServers,48,MENUITEM_NORMAL,0},
                        {"CONNECT TO SERVER",ConnectToServer,68,MENUITEM_NORMAL,0},
                        {"START NEW SERVER",StartNewServer,88,MENUITEM_NORMAL,0},
                        {NULL,NULL,0,0,0}};
char serverAddr[32]="";
Menu connectMenu[]={{"CONNECT TO SERVER",NULL,TEXTURE_MENU_TEXTHEADER,MENUTYPE_OPTIONS,100},
                    {"IP",NULL,60,MENUITEM_NAME,(int)serverAddr},
                    {"PORT",NULL,80,MENUITEM_PORT,1234},
                    {"CONNECT AND JOIN GAME",JoinServer,100,MENUITEM_NORMAL,0},
                    {NULL,NULL,0,0,0}};
char serverName[32]="Untitled";
Menu newServerMenu[]={{"START SERVER",NULL,TEXTURE_MENU_TEXTHEADER,MENUTYPE_OPTIONS,152},
                      {"NAME",NULL,48,MENUITEM_NAME,(int)serverName},
                      {"LIST OF MAPS TO CYCLE",MapsToCycle,64,MENUITEM_NORMAL,0},
                      {"GAME OPTIONS",ServerGameOptions,80,MENUITEM_NORMAL,0},
                      {"PLAYERS",NULL,96,MENUITEM_PLAYERS,16},
                      {"PUBLIC",YesNoItem,112,MENUITEM_YESNO,1},
                      {"DEDICATED",YesNoItem,128,MENUITEM_YESNO,0},
                      {"PORT",NULL,144,MENUITEM_PORT,1234},
                      {"START NEW SERVER",CreateServer,160,MENUITEM_NORMAL,0},
                      {NULL,NULL,0,0,0}};
Menu pauseMenu[]={{"",NULL,TEXTURE_MENU_HEADER,MENUTYPE_NORMAL,0},
                  {"NEW GAME",NewGame,48,MENUITEM_NORMAL,0},
                  {"MULTIPLAYER",Multiplayer,64,MENUITEM_NORMAL,0},
                  {"SAVE GAME",SaveGame,80,MENUITEM_NORMAL,0},
                  {"LOAD GAME",LoadGame,96,MENUITEM_NORMAL,0},
                  {"OPTIONS",Options,112,MENUITEM_NORMAL,0},
                  {"HELP",Help,128,MENUITEM_NORMAL,0},
                  {"QUIT TO TITLE",QuitToTitle,144,MENUITEM_NORMAL,0},
                  {"QUIT GAME",Quit,160,MENUITEM_NORMAL,0},
                  {NULL,NULL,0,0,0}};
Menu episodeMenu[]={{"SELECT AN EPISODE",NULL,TEXTURE_MENU_TEXTHEADER,MENUTYPE_NORMAL,0},
                    {"L.A. MELTDOWN",Episode1,48,MENUITEM_NORMAL,0},
                    {"LUNAR APOCALYPSE",Episode2,68,MENUITEM_NORMAL,0},
                    {"SHRAPNEL CITY",Episode3,88,MENUITEM_NORMAL,0},
                    {NULL,NULL,0,0,0}};
Menu skillMenu[]={{"SELECT SKILL",NULL,TEXTURE_MENU_TEXTHEADER,MENUTYPE_NORMAL,0},
                  {"PIECE OF CAKE",Skill1,60,MENUITEM_NORMAL,0},
                  {"LET'S ROCK",Skill2,80,MENUITEM_NORMAL,0},
                  {"COME GET SOME",Skill3,100,MENUITEM_NORMAL,0},
                  {"DAMN I'M GOOD",Skill4,120,MENUITEM_NORMAL,0},
                  {NULL,NULL,0,0,0}};
#define WINDOWED_OPTION optionsMenu[1]
#define RESOLUTION_OPTION optionsMenu[2]
#define DEPTH_OPTION optionsMenu[3]
#define RENDERER_OPTION optionsMenu[4]
#define SOUND_OPTION optionsMenu[5]
#define MUSIC_OPTION optionsMenu[6]
Menu optionsMenu[]={{"OPTIONS",NULL,TEXTURE_MENU_TEXTHEADER,MENUTYPE_OPTIONS,172},
                    {"WINDOWED",WindowedOption,48,MENUITEM_YESNO,1},
                    {"RESOLUTION",ResolutionOption,64,MENUITEM_MODELIST,0},
                    {"BIT DEPTH",BitDepthOption,80,MENUITEM_DISABLED,0},
                    {"DRIVER",DriverOption,96,MENUITEM_DRIVERLIST,2},
                    {"SOUND",SoundOption,112,MENUITEM_ONOFF,1},
                    {"MUSIC",MusicOption,128,MENUITEM_ONOFF,1},
                    {NULL,NULL,0,0,0}};

#define DRIVER_COUNT 3
char *driverNames[DRIVER_COUNT]={"DIRECT3D","OPENGL","SOFTWARE"};

Render *render;

ModeList modes16,modes32,modesMerged;
ModeList stdModes={{{640,480},{800,600},{1024,768},{1280,1024},{1600,1200}},5};

extern HINSTANCE hInst;

extern int scrnX,scrnY;
int scrnDepth;

int curSect=-1;

extern int graphicalConsole;
extern HANDLE consoleScreenBuf;
//---------------------------------------------------------------------------
__fastcall TForm1::TForm1(TComponent* Owner)
    : TForm(Owner)
{
    // If the console is text based, go ahead and
    // initialize the console so that early debug
    // messages can be viewed
    if (!graphicalConsole)
        CreateConsole(NULL,0,0,NULL);
    // Set variables to their default values
    curMap=NULL;
    frames=0; fps=0; showfps=1; showstats=1;
    vid_width=640; vid_height=480; vid_depth=0;
    markcursect=0;
    titlePalNum=-1;
    minimized=false;
    strcpy(directory,"c:\\duke3d");
    // Read settings from the registry
    TRegistry *reg=new TRegistry;
    reg->RootKey=HKEY_LOCAL_MACHINE;
    reg->OpenKey("\\Software\\ACZ\\Duke Nukem 3D",true);
    if (reg->ValueExists("vid_width"))
        vid_width=reg->ReadInteger("vid_width");
    if (reg->ValueExists("vid_height"))
        vid_height=reg->ReadInteger("vid_height");
    if (reg->ValueExists("vid_depth"))
        vid_depth=reg->ReadInteger("vid_depth");
    if (reg->ValueExists("renderer"))
        RENDERER_OPTION.value=reg->ReadInteger("renderer");
    if (reg->ValueExists("showfps"))
        showfps=reg->ReadInteger("showfps");
    if (reg->ValueExists("showstats"))
        showstats=reg->ReadInteger("showstats");
    if (reg->ValueExists("sound"))
        SOUND_OPTION.value=reg->ReadInteger("sound");
    if (reg->ValueExists("music"))
        MUSIC_OPTION.value=reg->ReadInteger("music");
    if (reg->ValueExists("directory"))
        strcpy(directory,reg->ReadString("directory").c_str());
    // Open the duke3d.grp file
    group=new GroupFile;
    if (!group->Open("duke3d.grp"))
    {
        char file[256];
        if (directory[strlen(directory)-1]=='\\')
            sprintf(file,"%sduke3d.grp",directory);
        else
            sprintf(file,"%s\\duke3d.grp",directory);
        if (!group->Open(file))
        {
            if (!group->Open("c:\\duke3d\\duke3d.grp"))
            {
                TOpenDialog *dlog=new TOpenDialog(this);
                dlog->FileName="duke3d.grp";
                dlog->Filter="Duke Nukem 3D Data File|DUKE3D.GRP";
                dlog->FilterIndex=1;
                dlog->Options.Clear();
                dlog->Options=dlog->Options<<ofPathMustExist<<ofFileMustExist;
                dlog->Title="Locate DUKE3D.GRP in your Duke Nukem 3D directory";
                if (!dlog->Execute())
                {
                    delete reg;
                    exit(1);
                }
                char drive[8],folder[256];
                fnsplit(dlog->FileName.c_str(),drive,folder,NULL,NULL);
                delete dlog;
                sprintf(directory,"%s%s",drive,folder);
                char file[256];
                if (directory[strlen(directory)-1]=='\\')
                    sprintf(file,"%sduke3d.grp",directory);
                else
                    sprintf(file,"%s\\duke3d.grp",directory);
                if (!group->Open(file))
                {
                    MessageBox(NULL,"Could not find DUKE3D.GRP","Error",MB_OK);
                    delete reg;
                    exit(1);
                }
                reg->WriteString("directory",directory);
            }
        }
    }
    delete reg;
    cprintf("Opened duke3d.grp\n");
    // Open the default RTS sound file
    rts=new RTSFile;
    if (!rts->Open("duke.rts"))
    {
        char file[256];
        if (directory[strlen(directory)-1]=='\\')
            sprintf(file,"%sduke.rts",directory);
        else
            sprintf(file,"%s\\duke.rts",directory);
        if (!rts->Open(file))
        {
            if (!rts->Open("c:\\duke3d\\duke.rts"))
            {
                delete rts;
                rts=NULL;
            }
            else
                cprintf("Opened duke.rts\n");
        }
        else
            cprintf("Opened duke.rts\n");
    }
    else
        cprintf("Opened duke.rts\n");
    curAnim=NULL;
}
//---------------------------------------------------------------------------
void __fastcall TForm1::FormCreate(TObject *Sender)
{
    // Start the renderer in a window with the default
    // form size
    ClientWidth=vid_width;
    ClientHeight=vid_height;
    scrnDepth=vid_depth;
    if (vid_depth!=0)
    {
        if (!graphicalConsole)
        {
            // Non-graphical consoles are not available
            // in full screen mode, so make sure the
            // console is graphical
            graphicalConsole=1;
        }
        // Full screen mode requires the renderer to use
        // the hidden "application window" that C++Builder
        // creates.  This window's only purpose in normal
        // applications is the taskbar icon. However,
        // DirectX requires full screen mode to use this
        // window since it is the application's true
        // main window.
        ShowWindow(Application->Handle,SW_SHOW);
    }
    switch (RENDERER_OPTION.value)
    {
        case 0: render=new D3DRender(hInst,vid_depth?Application->Handle:Handle,vid_width,vid_height,vid_depth); break;
        case 1: render=new OpenGLRender(hInst,vid_depth?Application->Handle:Handle,vid_width,vid_height,vid_depth); break;
        case 2: render=new SoftRender(hInst,vid_depth?Application->Handle:Handle,vid_width,vid_height,vid_depth); break;
    }
    if (!render->Initialized())
    {
        RENDERER_OPTION.value=2;
        render=new SoftRender(hInst,Handle,vid_width,vid_height,vid_depth);
    }
    cprintf("Renderer initialized\n");
    // Set up a new message handler so that we can
    // receive the MCI message which is generated when
    // the currently playing music reaches the end
    WindowProc=MCIWindowProc;
    // Obtain a list of valid video modes
    render->EnumModes(&modes16,&modes32);
    // Construct a sorted merged list of video modes
    memcpy(modesMerged.mode,stdModes.mode,stdModes.count*sizeof(modes16.mode[0]));
    memcpy(&modesMerged.mode[stdModes.count],modes16.mode,modes16.count*sizeof(modes16.mode[0]));
    memcpy(&modesMerged.mode[stdModes.count+modes16.count],modes32.mode,modes32.count*sizeof(modes32.mode[0]));
    modesMerged.count=stdModes.count+modes16.count+modes32.count;
    for (int i=0;i<modesMerged.count;i++)
    {
        for (int j=i+1;j<modesMerged.count;j++)
        {
            if ((modesMerged.mode[i].w>modesMerged.mode[j].w)||
               ((modesMerged.mode[i].w==modesMerged.mode[j].w)&&
                (modesMerged.mode[i].h>modesMerged.mode[j].h)))
            {
                int temp=modesMerged.mode[i].w;
                modesMerged.mode[i].w=modesMerged.mode[j].w;
                modesMerged.mode[j].w=temp;
                temp=modesMerged.mode[i].h;
                modesMerged.mode[i].h=modesMerged.mode[j].h;
                modesMerged.mode[j].h=temp;
            }
        }
    }
    // Remove duplicates in the merged mode list
    for (int i=0;i<modesMerged.count-1;i++)
    {
        if ((modesMerged.mode[i].w==modesMerged.mode[i+1].w)&&
            (modesMerged.mode[i].h==modesMerged.mode[i+1].h))
        {
            for (int j=i;j<modesMerged.count-1;j++)
            {
                modesMerged.mode[j].w=modesMerged.mode[j+1].w;
                modesMerged.mode[j].h=modesMerged.mode[j+1].h;
            }
            modesMerged.count--;
            i--;
        }
    }
    // Locate default resolution in the mode list and
    // store its index
    for (int i=0;i<modesMerged.count;i++)
    {
        if ((modesMerged.mode[i].w==vid_width)&&(modesMerged.mode[i].h==vid_height))
        {
            RESOLUTION_OPTION.value=i;
            break;
        }
    }
    if (vid_depth!=0)
    {
        WINDOWED_OPTION.value=0;
        DEPTH_OPTION.type=MENUITEM_DEPTHLIST;
    }
    if (vid_depth==16)
        DEPTH_OPTION.value=0;
    else if (vid_depth==32)
        DEPTH_OPTION.value=1;
    CreateConsole(group,ClientWidth,ClientHeight,&art);
    // Initialize the sound engine and set up the
    // sound channels
    sound=new DirectSoundEngine(Handle);
    sound->SoundMute(!SOUND_OPTION.value);
    MusicMute(!MUSIC_OPTION.value);
    sound->SetChannelType(0,CHANNEL_RTS);
    sound->SetChannelType(1,CHANNEL_ANIM);
    // Load the RTS sound files if the RTS file existed
    if (rts)
    {
        for (int i=0;i<10;i++)
            rtsSound[i]=LoadRTSSound(rts,i+1);
    }
    // Load commonly used sounds
    secRtsSound[0]=LoadSound(group,"HOLYCW01.VOC");
    secRtsSound[1]=LoadSound(group,"GETSOM1A.VOC");
    secRtsSound[2]=LoadSound(group,"GOTHRT01.VOC");
    secRtsSound[3]=LoadSound(group,"IMGOOD12.VOC");
    secRtsSound[4]=LoadSound(group,"PIECE02.VOC");
    secRtsSound[5]=LoadSound(group,"WAITIN03.VOC");
    secRtsSound[6]=LoadSound(group,"AMESS06.VOC");
    secRtsSound[7]=LoadSound(group,"LETSRK03.VOC");
    secRtsSound[8]=LoadSound(group,"PISSIN01.VOC");
    secRtsSound[9]=LoadSound(group,"MYSELF3A.VOC");
    animSound1=LoadSound(group,"FLYBY.VOC");
    animSound2=LoadSound(group,"BOMBEXPL.VOC");
    menuMoveSound=LoadSound(group,"KICKHIT.VOC");
    menuSelectSound=LoadSound(group,"BULITHIT.VOC");
    menuExitSound=LoadSound(group,"ITEM15.VOC");
    skillSound[0]=LoadSound(group,"PIECE02.VOC");
    skillSound[1]=LoadSound(group,"LETSRK03.VOC");
    skillSound[2]=LoadSound(group,"GETSOM1A.VOC");
    skillSound[3]=LoadSound(group,"IMGOOD12.VOC");
    cprintf("Default sounds cached\n");
    // Load the palette and cache the texture information
    art.LoadPalette(0,group,"PALETTE.DAT");
    // Make a copy of the palette so that special textures
    // (text characters, etc.) can have both a version
    // optimized for bitmap drawing and a version compatible
    // with drawing in levels loaded at the same time
    art.LoadPalette(1,group,"PALETTE.DAT");
    art.ImportAll(group);
    cprintf("Textures cached\n");
    // Load the intro animation
    introAnim=new ANMFile(group,"LOGO.ANM",&art);
    cprintf("Intro animation loaded\n");
    animStarted=0;
    state=INTRO_ANIM;
    subState=INTRO_ANIM_PLAY;
    strcpy(mapLoadName,"E1L1.MAP");
    // Cache commonly used textures
    art.RequestTexture(TEXTURE_STATUS_BAR,1,TEXTYPE_BITMAP);
    art.RequestTexture(TEXTURE_MENU_HEADER,1,TEXTYPE_BITMAP_NOBORDER);
    art.RequestTexture(TEXTURE_MENU_TEXTHEADER,1,TEXTYPE_BITMAP_NOBORDER);
    for (int i=0;i<=9;i++)
        art.RequestTexture(TEXTURE_DIGITAL_NUMBERS+i,1,TEXTYPE_BITMAP_NOBORDER);
    for (int i=0;i<12;i++)
        art.RequestTexture(TEXTURE_SMALL_FONT+i,1,TEXTYPE_BITMAP_NOBORDER);
    for (int i=0;i<80;i++)
        art.RequestTexture(TEXTURE_LARGE_FONT+i,1,TEXTYPE_BITMAP_NOBORDER);
    menuAnimFrame=0;
    for (int i=0;i<8;i++)
        art.RequestTexture(TEXTURE_MENU_SELECT+i,1,TEXTYPE_BITMAP_NOBORDER);
    cprintf("Common textures loaded\n");
    ConsoleGameReady();
    // Set up application event handlers
    Application->OnIdle=OnIdle;
    Application->OnActivate=FormActivate;
    Application->OnDeactivate=FormDeactivate;
    Application->OnMinimize=FormMinimize;
    Application->OnRestore=FormRestore;
    // Initialize DirectInput
    di=new CDirectInput(hInst,Handle);
    useDirectInput=0;
}

void TForm1::DrawCenteredDigitalNumber(int x,int y,int num)
{
    char str[8];
    sprintf(str,"%d",num);
    int width=0;
    for (int i=0;i<strlen(str);i++)
        width+=art.origSizeX[str[i]-'0'+TEXTURE_DIGITAL_NUMBERS]+1;
    x-=width/2;
    for (int i=0;i<strlen(str);i++)
    {
        render->RenderBitmap(x,y,&art,str[i]-'0'+TEXTURE_DIGITAL_NUMBERS,RGBA_MAKE(128,128,128,255));
        x+=art.origSizeX[str[i]-'0'+TEXTURE_DIGITAL_NUMBERS]+1;
    }
}

void TForm1::DrawSmallText(int x,int y,char *text,unsigned long color)
{
    for (int i=0;i<strlen(text);i++)
    {
        if (text[i]==' ')
            x+=4;
        else if (text[i]==':')
        {
            render->RenderBitmap(x,y+1,&art,TEXTURE_SMALL_FONT+10,color);
            x+=2;
        }
        else if (text[i]=='/')
        {
            render->RenderBitmap(x,y,&art,TEXTURE_SMALL_FONT+11,color);
            x+=5;
        }
        else
        {
            render->RenderBitmap(x,y,&art,TEXTURE_SMALL_FONT+text[i]-'0',color);
            x+=4;
        }
    }
}

void TForm1::DrawCenteredLargeText(int x,int y,char *str,unsigned long color)
{
    int width=0;
    for (int i=0;i<strlen(str);i++)
    {
        int tex;
        if (str[i]==' ')
        {
            width+=8;
            continue;
        }
        if ((str[i]>='0')&&(str[i]<='9'))
            tex=TEXTURE_LARGE_FONT+str[i]-'0';
        else if ((str[i]>='A')&&(str[i]<='Z'))
            tex=TEXTURE_LARGE_FONT+str[i]-'A'+10;
        else if ((str[i]>='a')&&(str[i]<='z'))
            tex=TEXTURE_LARGE_FONT+str[i]-'a'+10;
        else if (str[i]=='.')
            tex=TEXTURE_LARGE_FONT+72;
        else if (str[i]==',')
            tex=TEXTURE_LARGE_FONT+73;
        else if (str[i]=='\'')
            tex=TEXTURE_LARGE_FONT+73;
        else if (str[i]=='!')
            tex=TEXTURE_LARGE_FONT+74;
        else if (str[i]=='?')
            tex=TEXTURE_LARGE_FONT+75;
        else if (str[i]==';')
            tex=TEXTURE_LARGE_FONT+76;
        else if (str[i]==':')
            tex=TEXTURE_LARGE_FONT+77;
        else if (str[i]=='/')
            tex=TEXTURE_LARGE_FONT+78;
        else if (str[i]=='%')
            tex=TEXTURE_LARGE_FONT+79;
        else
            continue;
        width+=art.origSizeX[tex]+1;
    }
    DrawLargeText(x-(width/2),y,str,color);
}

void TForm1::DrawLargeText(int x,int y,char *str,unsigned long color)
{
    for (int i=0;i<strlen(str);i++)
    {
        int tex;
        int yflipped=0;
        if (str[i]==' ')
        {
            x+=8;
            continue;
        }
        if ((str[i]>='0')&&(str[i]<='9'))
            tex=TEXTURE_LARGE_FONT+str[i]-'0';
        else if ((str[i]>='A')&&(str[i]<='Z'))
            tex=TEXTURE_LARGE_FONT+str[i]-'A'+10;
        else if ((str[i]>='a')&&(str[i]<='z'))
        {
            tex=TEXTURE_LARGE_FONT+str[i]-'a'+10;
            render->RenderScaledBitmap(x,y+art.origSizeY[tex]-
                (art.origSizeY[tex]*4/5),art.origSizeX[tex]*4/5,
                art.origSizeY[tex]*4/5,&art,tex,color);
            x+=(art.origSizeX[tex]*4/5)+1;
            continue;
        }
        else if (str[i]=='.')
            tex=TEXTURE_LARGE_FONT+72;
        else if (str[i]==',')
            tex=TEXTURE_LARGE_FONT+73;
        else if (str[i]=='\'')
        {
            tex=TEXTURE_LARGE_FONT+73;
            yflipped=1;
        }
        else if (str[i]=='!')
            tex=TEXTURE_LARGE_FONT+74;
        else if (str[i]=='?')
            tex=TEXTURE_LARGE_FONT+75;
        else if (str[i]==';')
            tex=TEXTURE_LARGE_FONT+76;
        else if (str[i]==':')
            tex=TEXTURE_LARGE_FONT+77;
        else if (str[i]=='/')
            tex=TEXTURE_LARGE_FONT+78;
        else if (str[i]=='%')
            tex=TEXTURE_LARGE_FONT+79;
        else if (str[i]=='_')
        {
            render->RenderScaledBitmap(x,y,12,art.origSizeY[TEXTURE_LARGE_FONT+72],
                &art,TEXTURE_LARGE_FONT+72,color);
            x+=13;
            continue;
        }
        else
            continue;
        if (yflipped)
            render->RenderBitmapYFlipped(x,y,&art,tex,color);
        else
            render->RenderBitmap(x,y,&art,tex,color);
        x+=art.origSizeX[tex]+1;
    }
}

void TForm1::IntroAnimState()
{
    int color=0xff808080;
    if (subState==INTRO_ANIM_PLAY)
    {
        if (!animStarted)
        {
            // If this is the first frame of the animation,
            // save the current time for the timing code
            animStarted=1;
            lastTime=GetTickCount();
        }
        // Ten frames per second
        while ((GetTickCount()-lastTime)>100)
        {
            lastTime+=100;
            // Get next frame of intro animation
            if (!introAnim->NextFrame())
            {
                // Animation has ended, so fade out
                subState=INTRO_ANIM_FADE_OUT;
            }
            else if (introAnim->FrameNum()==1)
                sound->PlaySound(animSound1,CHANNEL_MAIN,0,0);
            else if (introAnim->FrameNum()==18)
                sound->PlaySound(animSound2,CHANNEL_MAIN,0,0);
        }
    }
    else
    {
        int time=GetTickCount()-lastTime;
        if (time>250)
        {
            // Fade out lasts 1/4 second
            if (titlePalNum==-1)
            {
                unsigned char *titlePalette=new char[768];
                // Load the title palette
                group->LoadFileB("LOOKUP.DAT",(char*)&titlePalette[0],7962,768);
                titlePalNum=art.FindFreePalette();
                art.LoadPalette(titlePalNum,titlePalette);
                delete[] titlePalette;
            }
            // Request the texture with the correct palette
            art.RequestTexture(TEXTURE_TITLE,titlePalNum,TEXTYPE_BITMAP);
            art.RequestTexture(TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,TEXTYPE_BITMAP_NOBORDER);
            art.RequestTexture(TEXTURE_TITLE_3D,titlePalNum,TEXTYPE_BITMAP_NOBORDER);
            // Go to the title screen
            state=DUKE_LOGO;
            subState=DUKE_LOGO_FADE_IN;
            RequestMusic(group,"GRABBAG.MID");
            lastTime=GetTickCount();
            return;
        }
        else
        {
            // Calculate color value for the fade out
            int bright=128*(250-time)/250;
            color=0xff000000|(bright<<16)|(bright<<8)|bright;
        }
    }
    // Draw the current animation frame to the screen
    render->BeginBitmaps();
    render->RenderBitmap(0,0,introAnim->GetTexture(),320,200,color);
    render->EndBitmaps();
    // Render the console
    UpdateConsolePos();
    RenderConsole();
}

void TForm1::PlayAnimState()
{
    if (!animStarted)
    {
        // If this is the first frame of the animation,
        // save the current time for the timing code
        animStarted=1;
        lastTime=GetTickCount();
    }
    // Ten frames per second
    while ((GetTickCount()-lastTime)>100)
    {
        lastTime+=100;
        // Get next frame of animation
        if (!curAnim->NextFrame())
        {
            delete curAnim;
            curAnim=NULL;
            state=GAME;
            return;
        }
    }
    // Draw the current animation frame to the screen
    render->BeginBitmaps();
    render->RenderBitmap(0,0,curAnim->GetTexture(),320,200,RGBA_MAKE(128,128,128,255));
    render->EndBitmaps();
    // Render the console
    UpdateConsolePos();
    RenderConsole();
}

void TForm1::DukeLogoState()
{
    // Draw the title screen bitmap
    render->BeginBitmaps();
    if ((subState!=DUKE_LOGO_FADE_IN)&&(subState!=DUKE_LOGO_FADE_OUT))
        render->RenderBitmap(0,0,&art,TEXTURE_TITLE,titlePalNum,RGBA_MAKE(128,128,128,255));
    if (subState==DUKE_LOGO_FADE_IN)
    {
        int time=GetTickCount()-lastTime;
        if (time>250)
        {
            // Fade in lasts for 1/4 second
            subState=DUKE_LOGO_START;
            lastTime=GetTickCount();
            render->RenderBitmap(0,0,&art,TEXTURE_TITLE,titlePalNum,RGBA_MAKE(128,128,128,255));
        }
        else
        {
            // Calculate color value for the fade in
            int bright=128*time/250;
            int color=0xff000000|(bright<<16)|(bright<<8)|bright;
            render->RenderBitmap(0,0,&art,TEXTURE_TITLE,titlePalNum,color);
        }
    }
    else if (subState==DUKE_LOGO_START)
    {
        // Wait 1.5 seconds before starting "Duke Nukem"
        // overlay animation
        if ((GetTickCount()-lastTime)>1500)
        {
            subState=DUKE_LOGO_DUKENUKEM;
            lastTime=GetTickCount();
            sound->PlaySound(animSound2,CHANNEL_MAIN,0,0);
        }
    }
    else if (subState==DUKE_LOGO_DUKENUKEM)
    {
        int time=GetTickCount()-lastTime;
        if (time>1000)
        {
            // "Duke Nukem" overlay animation plus the
            // wait for the "3D" overlay is a total
            // of 1 second
            render->RenderScaledBitmap(72,84,176,39,&art,TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,RGBA_MAKE(128,128,128,255));
            subState=DUKE_LOGO_3D;
            lastTime=GetTickCount();
            sound->PlaySound(animSound2,CHANNEL_MAIN,0,0);
        }
        else if (time>500) // Animation lasts for 1/2 second
            render->RenderScaledBitmap(72,84,176,39,&art,TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,RGBA_MAKE(128,128,128,255));
        else
            render->RenderScaledBitmap(160-(88*time/500),103-(19*time/500),176*time/500,39*time/500,&art,TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,RGBA_MAKE(128,128,128,255));
    }
    else if (subState==DUKE_LOGO_3D)
    {
        render->RenderScaledBitmap(72,84,176,39,&art,TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,RGBA_MAKE(128,128,128,255));
        int time=GetTickCount()-lastTime;
        if (time>250)
        {
            // "3D" overlay animation lasts for 1/4 second
            render->RenderScaledBitmap(129,117,62,24,&art,TEXTURE_TITLE_3D,titlePalNum,RGBA_MAKE(128,128,128,255));
            subState=DUKE_LOGO_END;
            lastTime=GetTickCount();
        }
        else
            render->RenderScaledBitmap(160-(31*time/250),129-(12*time/250),62*time/250,24*time/250,&art,TEXTURE_TITLE_3D,titlePalNum,RGBA_MAKE(128,128,128,255));
    }
    else if (subState==DUKE_LOGO_END)
    {
        int time=GetTickCount()-lastTime;
        if (time>5000)
        {
            // After 5 seconds, end title screen
            subState=DUKE_LOGO_FADE_OUT;
            lastTime=GetTickCount();
        }
        render->RenderScaledBitmap(72,84,176,39,&art,TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,RGBA_MAKE(128,128,128,255));
        render->RenderScaledBitmap(129,117,62,24,&art,TEXTURE_TITLE_3D,titlePalNum,RGBA_MAKE(128,128,128,255));
    }
    else if (subState==DUKE_LOGO_FADE_OUT)
    {
        int time=GetTickCount()-lastTime;
        if (time>250)
        {
            // Fade out lasts for 1/4 second
            state=MENU;
            subState=MENU_SELECT;
            menu=mainMenu;
            menuSelected=0;
            prevCount=0;
            render->RenderBitmap(0,0,&art,TEXTURE_TITLE,titlePalNum,RGBA_MAKE(0,0,0,255));
        }
        else
        {
            // Calculate color value for the fade out
            int bright=128*(250-time)/250;
            int color=0xff000000|(bright<<16)|(bright<<8)|bright;
            render->RenderBitmap(0,0,&art,TEXTURE_TITLE,titlePalNum,color);
            render->RenderScaledBitmap(72,84,176,39,&art,TEXTURE_TITLE_DUKE_NUKEM,titlePalNum,color);
            render->RenderScaledBitmap(129,117,62,24,&art,TEXTURE_TITLE_3D,titlePalNum,color);
        }
    }
    render->EndBitmaps();
    // Render the console
    UpdateConsolePos();
    RenderConsole();
}

void TForm1::MapLoadState()
{
    StopMusic();
    // Render the loading screen
    introAnim->Seek(30);
    render->BeginBitmaps();
    render->RenderBitmap(0,0,introAnim->GetTexture(),320,200,RGBA_MAKE(128,128,128,255));
    render->EndBitmaps();
    // Draw the text
    render->BeginBitmaps();
    DrawCenteredLargeText(160,80,"ENTERING",RGBA_MAKE(192,192,192,255));
    char name[256],cmpName[256];
    strcpy(name,mapLoadName);
    strcpy(cmpName,mapLoadName);
    strlwr(cmpName);
    for (int i=0;DefaultMapInfo[i].mapFile[0];i++)
    {
        if (!strcmp(DefaultMapInfo[i].mapFile,cmpName))
            strcpy(name,DefaultMapInfo[i].mapName);
    }
    DrawCenteredLargeText(160,100,name,RGBA_MAKE(192,192,192,255));
    render->EndBitmaps();
    // End the frame
    render->EndRender();
    Map *newMap;
    try
    {
        // Attempt to load the new map
        newMap=new Map(group,mapLoadName,&art);
    }
    catch(char *error)
    {
        // Caught an error, print it to the console
        cprintf(error);
        state=oldState;
        return;
    }
    // Start music for level
    for (int i=0;DefaultMapInfo[i].mapFile[0];i++)
    {
        if (!strcmp(DefaultMapInfo[i].mapFile,cmpName))
        {
            RequestMusic(group,DefaultMapInfo[i].mapMusic);
            break;
        }
    }
    // Delete the old map object and set the current
    // map pointer to the new map
    delete curMap;
    curMap=newMap;
    // Set the current position to the starting location
    // from the map
    render->SetXPos(curMap->startX/64.0);
    render->SetYPos(-curMap->startZ/1024.0);
    render->SetZPos(-curMap->startY/64.0);
    curSect=curMap->startSect;
    // Start the game
    state=GAME;
    subState=GAME_PLAYING;
    useDirectInput=1;
    di->Acquire(DINPUT_MOUSE);
    render->BeginRender();
    // Render the loading screen
    introAnim->Seek(30);
    render->BeginBitmaps();
    render->RenderBitmap(0,0,introAnim->GetTexture(),320,200,RGBA_MAKE(128,128,128,255));
    render->EndBitmaps();
    // Draw the text
    render->BeginBitmaps();
    DrawCenteredLargeText(160,80,"ENTERING",RGBA_MAKE(192,192,192,255));
    strcpy(name,mapLoadName);
    strcpy(cmpName,mapLoadName);
    strlwr(cmpName);
    for (int i=0;DefaultMapInfo[i].mapFile[0];i++)
    {
        if (!strcmp(DefaultMapInfo[i].mapFile,cmpName))
            strcpy(name,DefaultMapInfo[i].mapName);
    }
    DrawCenteredLargeText(160,100,name,RGBA_MAKE(192,192,192,255));
    render->EndBitmaps();
}

void TForm1::DrawMenu()
{
    // Draw the menu header
    render->BeginBitmaps();
    render->RenderBitmap(160-(art.origSizeX[menu[0].y]/2),8,&art,menu[0].y,RGBA_MAKE(128,128,128,255));
    DrawCenteredLargeText(160,10,menu[0].item,RGBA_MAKE(192,192,192,255));
    if (menu[0].type==MENUTYPE_NORMAL)
    {
        // Draw the menu items
        for (int i=0;menu[i+1].item;i++)
        {
            if (i==menuSelected)
                DrawCenteredLargeText(160,menu[i+1].y,menu[i+1].item,RGBA_MAKE(192,192,192,255));
            else
                DrawCenteredLargeText(160,menu[i+1].y,menu[i+1].item,RGBA_MAKE(144,144,144,255));
        }
    }
    else if (menu[0].type==MENUTYPE_OPTIONS)
    {
        int x=menu[0].value;
        // Draw the menu items
        for (int i=0;menu[i+1].item;i++)
        {
            int color;
            if (i==menuSelected)
                color=RGBA_MAKE(192,192,192,255);
            else
                color=RGBA_MAKE(144,144,144,255);
            if (menu[i+1].type==MENUITEM_DISABLED)
                color=RGBA_MAKE(64,64,64,255);
            DrawLargeText(32,menu[i+1].y,menu[i+1].item,color);
            if (menu[i+1].type==MENUITEM_YESNO)
            {
                if (menu[i+1].value)
                    DrawLargeText(x,menu[i+1].y,"YES",color);
                else
                    DrawLargeText(x,menu[i+1].y,"NO",color);
            }
            else if (menu[i+1].type==MENUITEM_ONOFF)
            {
                if (menu[i+1].value)
                    DrawLargeText(x,menu[i+1].y,"ON",color);
                else
                    DrawLargeText(x,menu[i+1].y,"OFF",color);
            }
            else if (menu[i+1].type==MENUITEM_MODELIST)
            {
                char mode[16];
                sprintf(mode,"%dX%d",modesMerged.mode[menu[i+1].value].w,
                    modesMerged.mode[menu[i+1].value].h);
                DrawLargeText(x,menu[i+1].y,mode,color);
            }
            else if (menu[i+1].type==MENUITEM_DEPTHLIST)
            {
                if (menu[i+1].value)
                    DrawLargeText(x,menu[i+1].y,"32",color);
                else
                    DrawLargeText(x,menu[i+1].y,"16",color);
            }
            else if (menu[i+1].type==MENUITEM_DRIVERLIST)
            {
                DrawLargeText(x,menu[i+1].y,driverNames[menu[i+1].value],color);
            }
            else if ((menu[i+1].type==MENUITEM_PLAYERS)||
                     (menu[i+1].type==MENUITEM_PORT))
            {
                char str[16];
                sprintf(str,"%d",menu[i+1].value);
                DrawLargeText(x,menu[i+1].y,str,color);
            }
            else if (menu[i+1].type==MENUITEM_NAME)
            {
                char str[32];
                if (i==menuSelected)
                    sprintf(str,"%s_",(char*)menu[i+1].value);
                else
                    sprintf(str,"%s",(char*)menu[i+1].value);
                DrawLargeText(x,menu[i+1].y,str,color);
            }
        }
    }
    if (subState==MENU_WAIT_FOR_SOUND)
    {
        if ((GetTickCount()-lastTime)>2500)
            state=MAP_LOAD;
    }
    else
    {
        // Update menu select icon animation
        if ((GetTickCount()-lastTime)>50)
        {
            lastTime=GetTickCount();
            menuAnimFrame=(menuAnimFrame+1)%7;
        }
    }
    // Draw menu select icons
    if (menu[0].type==MENUTYPE_NORMAL)
    {
        render->RenderBitmap(40-(art.origSizeX[TEXTURE_MENU_SELECT+menuAnimFrame]/2),
            menu[menuSelected+1].y-3,&art,TEXTURE_MENU_SELECT+menuAnimFrame,
            RGBA_MAKE(128,128,128,255));
        render->RenderBitmap(280-(art.origSizeX[TEXTURE_MENU_SELECT+menuAnimFrame]/2),
            menu[menuSelected+1].y-3,&art,TEXTURE_MENU_SELECT+menuAnimFrame,
            RGBA_MAKE(128,128,128,255));
    }
    else
    {
        render->RenderBitmap(14-(art.origSizeX[TEXTURE_MENU_SELECT+menuAnimFrame]/2),
            menu[menuSelected+1].y-3,&art,TEXTURE_MENU_SELECT+menuAnimFrame,
            RGBA_MAKE(128,128,128,255));
        render->RenderBitmap(306-(art.origSizeX[TEXTURE_MENU_SELECT+menuAnimFrame]/2),
            menu[menuSelected+1].y-3,&art,TEXTURE_MENU_SELECT+menuAnimFrame,
            RGBA_MAKE(128,128,128,255));
    }
    render->EndBitmaps();
}

void TForm1::MenuState()
{
    // Render the background image
    introAnim->Seek(30);
    render->BeginBitmaps();
    render->RenderBitmap(0,0,introAnim->GetTexture(),320,200,RGBA_MAKE(128,128,128,255));
    render->EndBitmaps();
    DrawMenu();
    // Render the console
    UpdateConsolePos();
    RenderConsole();
}

void TForm1::GameState()
{
    int axis[4],button[8];
    static int b1=0,b2=0;
    if (useDirectInput)
    {
        // Get the data for mouse input
        di->GetInput(DINPUT_MOUSE,axis,button);
        // Check the buttons
        if (button[0+MB_DOWN]) b1=1;
        if (button[0+MB_UP]) b1=0;
        if (button[1+MB_DOWN]) b2=1;
        if (button[1+MB_UP]) b2=0;
    }
    else
    {
        axis[0]=0;
        axis[1]=0;
    }
    if ((b1)&&(!b2))
    {
        // If only the left button is held, movement
        // is forward/back/turn
        render->AddYRotate(-axis[0]/200.0);
        render->GoForward(-axis[1]/4.0);
    }
    else if (b1&&b2)
    {
        // If both buttons are held, movement is
        // up/down/side to side
        render->GoSide(axis[0]/4.0);
        render->AddYPos(-axis[1]/4.0);
    }
    else
    {
        // Otherwise, the mouse is used to look around
        render->AddXRotate(-axis[1]/200.0);
        if (render->GetXRotate()>(PI/2.0))
            render->SetXRotate(PI/2.0);
        if (render->GetXRotate()<(-PI/2.0))
            render->SetXRotate(-PI/2.0);
        render->AddYRotate(-axis[0]/200.0);
    }
    // Handle flickering lights
    int randNum=rand()&127;
    for (int i=0;i<curMap->numSprites;i++)
    {
        if ((curMap->sprite[i].picnum==1)&&(curMap->sprite[i].lotag==4))
        {
            // Sprite #1 (sectoreffector), lotag 4 designates
            // random lighting
            // Calculate shade value when light is on
            int shade=curMap->sprite[i].shade*4;
            if (shade<-128) shade=-128;
            if (shade>127) shade=127;
            unsigned char light=~(shade+0x80);
            int newColor=0xff000000|(light<<16)|(light<<8)|light;
            // Go through each polygon in the current sector
            // and reset the lighting level
            int sect=curMap->sprite[i].sectnum;
            PolyListElem *ptr=curMap->sectorPolys[sect].first;
            while (ptr)
            {
                for (int j=0;j<ptr->poly.nVert;j++)
                {
                    if (randNum<16)
                        ptr->poly.v[j].color=newColor;
                    else
                        ptr->poly.v[j].color=ptr->poly.origColor[j];
                }
                ptr=ptr->next;
            }
        }
    }
    // If we are still in the sector curSect, there is no
    // need to recalculate the current sector
    if (!curMap->IsInSector(curSect,render->GetXPos()*64,-render->GetZPos()*64,-render->GetYPos()*1024))
    {
        // No longer in sector curSect, so if current sector
        // marking in on, un-highlight the sector
        if ((markcursect)&&(curSect!=-1))
        {
            PolyListElem *ptr=curMap->sectorPolys[curSect].first;
            while (ptr)
            {
                for (int i=0;i<ptr->poly.nVert;i++)
                    ptr->poly.v[i].color=ptr->poly.origColor[i];
                ptr=ptr->next;
            }
        }
        // Find which sector we are now in
        curSect=-1;
        for (int i=0;i<curMap->numSectors;i++)
        {
            if (curMap->IsInSector(i,render->GetXPos()*64,-render->GetZPos()*64,-render->GetYPos()*1024))
            {
                curSect=i;
                break;
            }
        }
    }
    // Highlight the current sector if sector highlighting
    // is enabled
    if ((markcursect)&&(curSect!=-1))
    {
        PolyListElem *ptr=curMap->sectorPolys[curSect].first;
        while (ptr)
        {
            for (int i=0;i<ptr->poly.nVert;i++)
                ptr->poly.v[i].color=0xffff0000;
            ptr=ptr->next;
        }
    }
    // Render the map
    int renderedSectors=render->RenderMap(curMap,curSect);
    // Show statistics if the indicator is enabled
    if (showstats)
    {
        render->UseOriginView();
        render->BeginScene();
        render->StartText();
        char str[32];
        sprintf(str,"Sectors drawn: %d of %d",renderedSectors,curMap->numSectors);
        render->RenderText(0,24,str,RGBA_MAKE(255,255,255,255));
        RenderStats stats;
        render->GetStats(&stats);
        sprintf(str,"Polygons drawn: %d",stats.polyDrawn);
        render->RenderText(0,36,str,RGBA_MAKE(255,255,255,255));
        sprintf(str,"Current Sector #: %d",curSect);
        render->RenderText(0,48,str,RGBA_MAKE(255,255,255,255));
        render->EndText();
        render->EndScene();
        render->UseNormalView();
    }
    render->BeginBitmaps();
    render->RenderBitmap(0,166,&art,TEXTURE_STATUS_BAR,RGBA_MAKE(128,128,128,255));
    DrawCenteredDigitalNumber(31,183,100);
    DrawCenteredDigitalNumber(63,183,0);
    DrawCenteredDigitalNumber(208,183,50);
    DrawSmallText(88,179,"1:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(88,185,"2:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(88,191,"3:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(128,179,"4:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(128,185,"5:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(128,191,"6:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(160,179,"7:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(160,185,"8:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(160,191,"9:",RGBA_MAKE(120,100,32,255));
    DrawSmallText(94,179," 50/200",RGBA_MAKE(150,144,212,255));
    DrawSmallText(94,185,"  0/50",RGBA_MAKE(75,75,108,255));
    DrawSmallText(94,191,"  0/200",RGBA_MAKE(75,75,108,255));
    DrawSmallText(134,179," 0/50",RGBA_MAKE(75,75,108,255));
    DrawSmallText(134,185," 0/50",RGBA_MAKE(75,75,108,255));
    DrawSmallText(134,191," 0/50",RGBA_MAKE(75,75,108,255));
    DrawSmallText(166,179," 0/99",RGBA_MAKE(75,75,108,255));
    DrawSmallText(166,185," 0/10",RGBA_MAKE(75,75,108,255));
    DrawSmallText(166,191," 0/99",RGBA_MAKE(75,75,108,255));
    render->EndBitmaps();
    if ((subState==GAME_PAUSED)||(subState==MENU_WAIT_FOR_SOUND))
        DrawMenu();
    // Render the console
    UpdateConsolePos();
    RenderConsole();
}
//---------------------------------------------------------------------------
void __fastcall TForm1::OnIdle(TObject *,bool& done)
{
    // Note: Put all network communication/update code
    // here, before the function returns in the case
    // of a minimized window
    if (minimized)
    {
        // Pause rendering when window is minimized
        Sleep(10);
        done=false;
        return;
    }
    // Start the new frame of rendering
    render->BeginRender();
    switch (state)
    {
        case INTRO_ANIM: IntroAnimState(); break;
        case PLAY_ANIM: PlayAnimState(); break;
        case DUKE_LOGO: DukeLogoState(); break;
        case MAP_LOAD: MapLoadState(); break;
        case MENU: MenuState(); break;
        case GAME: GameState(); break;
    }
    if (!graphicalConsole)
    {
        // If not a graphical console, check for any
        // on the text-based console
        CheckNonGraphicalConsoleInput(Handle);
        // Check to see if a command is ready
        if (IsCommandReady())
        {
            // Parse the string into command and parameter
            char *str=CommandString();
            int i=0;
            // Strip leading spaces
            for (;(str[i]==' ')&&(str[i]!=0);) i++;
            // Return if no command
            if (!str[i]) return;
            int first=i;
            // Split the command and the parameter
            for (;str[i];i++)
            {
                if (str[i]==' ')
                {
                    str[i++]=0;
                    for (;(str[i]==' ')&&(str[i]!=0);) i++;
                    // Call the command handler
                    ExecuteConsoleCommand(&str[first],&str[i]);
                    return;
                }
            }
            // If the loop finishes, there was no parameter,
            // call the command handler
            ExecuteConsoleCommand(&str[first],NULL);
        }
    }
    // Update frame counter
    frames++;
    // Show frames per second if the indicator is enabled
    if (showfps)
    {
        render->UseOriginView();
        render->BeginScene();
        render->StartText();
        char str[32];
        sprintf(str,"%d FPS",fps);
        render->RenderText(0,0,str,RGBA_MAKE(255,255,255,255));
        render->EndText();
        render->EndScene();
        render->UseNormalView();
    }
    // End the frame
    render->EndRender();
    ProcessMusicRequest(Handle);
    // Set the "done" flag to false so that OnIdle is
    // still called when there are no messages
    done=false;
}
//---------------------------------------------------------------------------

void __fastcall TForm1::FormActivate(TObject *Sender)
{
    if (useDirectInput)
        di->Acquire(DINPUT_MOUSE);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::FormDeactivate(TObject *Sender)
{
    if (useDirectInput)
        di->Unacquire(DINPUT_MOUSE);
}
//---------------------------------------------------------------------------

void TForm1::DynamicResChange()
{
    // Restarts video system with new resolution and/or
    // bit depth settings
    // Save the current position and rotation values
    Vector p=Vector(render->GetXPos(),render->GetYPos(),render->GetZPos());
    Vector r=Vector(render->GetXRotate(),render->GetYRotate(),render->GetZRotate());
    // Save the old settings in case the new settings fail
    int oldScrnX=scrnX;
    int oldScrnY=scrnY;
    int oldScrnDepth=scrnDepth;
    // Change the settings
    scrnX=vid_width;
    scrnY=vid_height;
    scrnDepth=vid_depth;
    ClientWidth=scrnX;
    ClientHeight=scrnY;
    if (vid_depth==0)
    {
        // Windowed mode, so activate the regular form
        ShowWindow(Application->Handle,SW_SHOWMINNOACTIVE);
        ShowWindow(Handle,SW_SHOW);
        render->SetRenderWnd(Handle);
        ClientWidth=scrnX;
        ClientHeight=scrnY;
    }
    else
    {
        if (!graphicalConsole)
        {
            // Non-graphical consoles are not available
            // in full screen mode, so make sure the
            // console is graphical
            graphicalConsole=1;
            FreeConsole();
            ConsoleGameReady();
        }
        // Full screen mode requires the renderer to use
        // the hidden "application window" that C++Builder
        // creates.  This window's only purpose in normal
        // applications is the taskbar icon. However,
        // DirectX requires full screen mode to use this
        // window since it is the application's true
        // main window.
        ShowWindow(Application->Handle,SW_SHOW);
        render->SetRenderWnd(Application->Handle);
    }
    try
    {
        // Attempt to restart the video system with
        // the new settings
        render->ChangeResolution(scrnX,scrnY,vid_depth);
    }
    catch (char *msg)
    {
        // Caught an error, print it to the console
        cprintf(msg);
        // Restore old settings
        scrnX=oldScrnX;
        scrnY=oldScrnY;
        scrnDepth=oldScrnDepth;
        // Restart video system with old settings
        if (scrnDepth==0)
        {
            ShowWindow(Application->Handle,SW_SHOWMINNOACTIVE);
            ShowWindow(Handle,SW_SHOW);
            render->SetRenderWnd(Handle);
            ClientWidth=scrnX;
            ClientHeight=scrnY;
        }
        else
        {
            ShowWindow(Application->Handle,SW_SHOW);
            render->SetRenderWnd(Application->Handle);
        }
        render->ChangeResolution(scrnX,scrnY,scrnDepth);
    }
    ClientWidth=scrnX;
    ClientHeight=scrnY;
    // Reload textures into texture memory
    art.ReloadTextures();
    // Resize console to the new screen size
    ResizeConsole(scrnX,scrnY);
    // Restore old position and rotation values
    render->SetXPos(p.x); render->SetYPos(p.y); render->SetZPos(p.z);
    render->SetXRotate(r.x); render->SetYRotate(r.y); render->SetZRotate(r.z);
    if (useDirectInput)
    {
        // Re-enable DirectInput
        di->Acquire(DINPUT_MOUSE);
    }
}

void TForm1::ExecuteConsoleCommand(char *action,char *param)
{
    strlwr(action);
    if (!strcmp(action,"exit"))
        Close();
    else if (!strcmp(action,"map"))
    {
        if (!param)
        {
            cprintf("Map name required\n");
            return;
        }
        oldState=state;
        state=MAP_LOAD;
        strcpy(mapLoadName,param);
    }
    else if (!strcmp(action,"music"))
    {
        if (!param)
        {
            // If no parameter was specified, stop music
            StopMusic();
            return;
        }
        try
        {
            // Start the new music file
            RequestMusic(group,param);
            ProcessMusicRequest(Handle);
        }
        catch(char *error)
        {
            // Caught an error, print it to the console
            cprintf(error);
            return;
        }
    }
    else if (!strcmp(action,"showfps"))
    {
        if (param)
            showfps=atoi(param);
        cprintf("showfps = %d\n",showfps);
    }
    else if (!strcmp(action,"showstats"))
    {
        if (param)
            showstats=atoi(param);
        cprintf("showstats = %d\n",showstats);
    }
    else if (!strcmp(action,"vid_width"))
    {
        if (param)
            vid_width=atoi(param);
        cprintf("vid_width = %d\n",vid_width);
    }
    else if (!strcmp(action,"vid_height"))
    {
        if (param)
            vid_height=atoi(param);
        cprintf("vid_height = %d\n",vid_height);
    }
    else if (!strcmp(action,"vid_depth"))
    {
        if (param)
            vid_depth=atoi(param);
        cprintf("vid_depth = %d\n",vid_depth);
    }
    else if (!strcmp(action,"vid_restart"))
        DynamicResChange();
    else if (!strcmp(action,"markcursect"))
    {
        // If set to one, the current sector will be
        // highlighted in red.
        // Clear any highlights before changing the setting
        if ((markcursect)&&(curSect!=-1))
        {
            PolyListElem *ptr=curMap->sectorPolys[curSect].first;
            while (ptr)
            {
                for (int i=0;i<ptr->poly.nVert;i++)
                    ptr->poly.v[i].color=ptr->poly.origColor[i];
                ptr=ptr->next;
            }
        }
        // If a parameter was given, set the new value
        if (param)
            markcursect=atoi(param);
        // If the feature is enabled, mark the current sector
        if ((markcursect)&&(curSect!=-1))
        {
            PolyListElem *ptr=curMap->sectorPolys[curSect].first;
            while (ptr)
            {
                for (int i=0;i<ptr->poly.nVert;i++)
                    ptr->poly.v[i].color=0xffff0000;
                ptr=ptr->next;
            }
        }
        // Print out the current value to the console
        cprintf("markcursect = %d\n",markcursect);
    }
    else if (!strcmp(action,"wallcoord"))
    {
        // Debug command to show coordinates of all
        // walls in the current sector
        cprintf("Wall#  X      Y      Next\n");
        for (int i=curMap->sector[curSect].wallptr;i<curMap->sector[curSect].wallptr+curMap->sector[curSect].wallnum;i++)
            cprintf("%-6d %-6d %-6d %-6d\n",i,curMap->wall[i].x,curMap->wall[i].y,curMap->wall[i].point2);
    }
    else if (!strcmp(action,"tex_count"))
    {
        if (param)
            cprintf("%d textures loaded with palette %d\n",art.GetTextureCount(atoi(param)),atoi(param));
        else
            cprintf("%d textures loaded\n",art.GetTextureCount());
    }
    else if (!strcmp(action,"anim"))
    {
        if (state==PLAY_ANIM)
        {
            delete curAnim;
            curAnim=NULL;
            state=GAME;
        }
        if (!param)
        {
            cprintf("Animation name required\n");
            return;
        }
        Map *newMap;
        try
        {
            // Attempt to load the animation
            curAnim=new ANMFile(group,param,&art);
        }
        catch(char *error)
        {
            // Caught an error, print it to the console
            cprintf(error);
            return;
        }
        animStarted=0;
        state=PLAY_ANIM;
        ToggleConsoleImmediate();
    }
    else if (!strcmp(action,"help"))
    {
        cprintf("Valid commands:\n");
        cprintf("map name.map    - Change map\n");
        cprintf("vid_width n     - Set video mode width for next vid_restart\n");
        cprintf("vid_height n    - Set video mode height for next vid_restart\n");
        cprintf("vid_depth n     - Set video mode bit depth for next vid_restart (0=windowed)\n");
        cprintf("vid_restart     - Restart video system, changing resolution if needed\n");
        cprintf("music name.mid  - Change music (omit name to stop music)\n");
        cprintf("showfps 0/1     - Toggle FPS indicator\n");
        cprintf("showstats 0/1   - Toggle statistics\n");
        cprintf("anim name.anm   - Play animation\n");
        cprintf("exit            - Exit game\n");
    }
    else
        cprintf("Unrecognized command %s\n",action);
}

void __fastcall TForm1::FormKeyPress(TObject *Sender, char &Key)
{
    // If the console is graphical, pass the key on
    // to the console's keyboard handler
    int consoleUsedChar=0;
    if (graphicalConsole)
        consoleUsedChar=ConsoleTypeChar(Key);
    // Check to see if a command is ready
    if (IsCommandReady())
    {
        // Parse the string into command and parameter
        char *str=CommandString();
        int i=0;
        // Strip leading spaces
        for (;(str[i]==' ')&&(str[i]!=0);) i++;
        // Return if no command
        if (!str[i]) return;
        int first=i;
        // Split the command and the parameter
        for (;str[i];i++)
        {
            if (str[i]==' ')
            {
                str[i++]=0;
                for (;(str[i]==' ')&&(str[i]!=0);) i++;
                // Call the command handler
                ExecuteConsoleCommand(&str[first],&str[i]);
                return;
            }
        }
        // If the loop finishes, there was no parameter,
        // call the command handler
        ExecuteConsoleCommand(&str[first],NULL);
    }
    if (!consoleUsedChar)
    {
        if ((state==MENU)||((state==GAME)&&(subState==GAME_PAUSED)))
        {
            if ((((Key>='0')&&(Key<='9'))||((Key>='A')&&(Key<='Z'))||
                 ((Key>='a')&&(Key<='z'))||(Key=='.')||(Key==',')||
                 (Key=='\'')||(Key=='!')||(Key=='?')||(Key==':')||
                 (Key==';')||(Key=='/')||(Key=='%')||(Key==' '))&&
                 (menu[menuSelected+1].type==MENUITEM_NAME))
            {
                sound->PlaySound(menuMoveSound,CHANNEL_MAIN,0,0);
                char *str=(char*)menu[menuSelected+1].value;
                str[strlen(str)+1]=0;
                str[strlen(str)]=Key;
            }
            else if ((Key==8)&&(menu[menuSelected+1].type==MENUITEM_NAME))
            {
                sound->PlaySound(menuMoveSound,CHANNEL_MAIN,0,0);
                char *str=(char*)menu[menuSelected+1].value;
                if (strlen(str)>0)
                    str[strlen(str)-1]=0;
            }
        }
    }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::Timer1Timer(TObject *Sender)
{
    // Called every 1 second, update frames per second value
    fps=frames;
    frames=0;
}
//---------------------------------------------------------------------------

void TForm1::MenuKeyHandler(int Key)
{
    if (Key==VK_UP)
    {
        sound->PlaySound(menuMoveSound,CHANNEL_MAIN,0,0);
        menuSelected--;
        if (menuSelected<0)
        {
            for (menuSelected=0;menu[menuSelected+1].item;)
                menuSelected++;
            menuSelected--;
        }
    }
    else if (Key==VK_DOWN)
    {
        sound->PlaySound(menuMoveSound,CHANNEL_MAIN,0,0);
        menuSelected++;
        if (!menu[menuSelected+1].item)
            menuSelected=0;
    }
    else if (Key==VK_LEFT)
    {
        sound->PlaySound(menuMoveSound,CHANNEL_MAIN,0,0);
        if (menu[menuSelected+1].type==MENUITEM_MODELIST)
        {
            menu[menuSelected+1].value--;
            if (menu[menuSelected+1].value<0)
                menu[menuSelected+1].value=modesMerged.count-1;
        }
        else if ((menu[menuSelected+1].type==MENUITEM_YESNO)||
                 (menu[menuSelected+1].type==MENUITEM_ONOFF)||
                 (menu[menuSelected+1].type==MENUITEM_DEPTHLIST))
        {
            menuItem=&menu[menuSelected+1];
            if (menu[menuSelected+1].func)
                menu[menuSelected+1].func();
        }
        else if (menu[menuSelected+1].type==MENUITEM_DRIVERLIST)
        {
            menu[menuSelected+1].value--;
            if (menu[menuSelected+1].value<0)
                menu[menuSelected+1].value=DRIVER_COUNT-1;
        }
        else if (menu[menuSelected+1].type==MENUITEM_PLAYERS)
        {
            menu[menuSelected+1].value--;
            if (menu[menuSelected+1].value<2)
                menu[menuSelected+1].value=2;
        }
        else if (menu[menuSelected+1].type==MENUITEM_PORT)
        {
            menu[menuSelected+1].value--;
            if (menu[menuSelected+1].value<0)
                menu[menuSelected+1].value=65535;
        }
        else if (menu[menuSelected+1].type==MENUITEM_NAME)
        {
            char *str=(char*)menu[menuSelected+1].value;
            if (strlen(str)>0)
                str[strlen(str)-1]=0;
        }
    }
    else if (Key==VK_RIGHT)
    {
        sound->PlaySound(menuMoveSound,CHANNEL_MAIN,0,0);
        if (menu[menuSelected+1].type==MENUITEM_MODELIST)
        {
            menu[menuSelected+1].value++;
            if (menu[menuSelected+1].value>=modesMerged.count)
                menu[menuSelected+1].value=0;
        }
        else if ((menu[menuSelected+1].type==MENUITEM_YESNO)||
                 (menu[menuSelected+1].type==MENUITEM_ONOFF)||
                 (menu[menuSelected+1].type==MENUITEM_DEPTHLIST))
        {
            menuItem=&menu[menuSelected+1];
            if (menu[menuSelected+1].func)
                menu[menuSelected+1].func();
        }
        else if (menu[menuSelected+1].type==MENUITEM_DRIVERLIST)
        {
            menu[menuSelected+1].value++;
            if (menu[menuSelected+1].value>=DRIVER_COUNT)
                menu[menuSelected+1].value=0;
        }
        else if (menu[menuSelected+1].type==MENUITEM_PLAYERS)
        {
            menu[menuSelected+1].value++;
            if (menu[menuSelected+1].value>32)
                menu[menuSelected+1].value=32;
        }
        else if (menu[menuSelected+1].type==MENUITEM_PORT)
        {
            menu[menuSelected+1].value++;
            if (menu[menuSelected+1].value>65535)
                menu[menuSelected+1].value=0;
        }
    }
    else if (Key==VK_RETURN)
    {
        sound->PlaySound(menuSelectSound,CHANNEL_MAIN,0,0);
        menuItem=&menu[menuSelected+1];
        if (menu[menuSelected+1].func)
            menu[menuSelected+1].func();
    }
    else if ((Key==VK_ESCAPE)&&(prevCount>0))
    {
        sound->PlaySound(menuExitSound,CHANNEL_MAIN,0,0);
        prevCount--;
        menu=prevMenu[prevCount];
        menuSelected=prevMenuSelected[prevCount];
    }
}

void __fastcall TForm1::FormKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
    if (Key==VkKeyScan('`'))
    {
        ToggleConsole();
        return;
    }
    if (!IsConsoleDown())
    {
        if (state==INTRO_ANIM)
        {
            subState=INTRO_ANIM_FADE_OUT;
            lastTime=GetTickCount();
        }
        else if (state==DUKE_LOGO)
        {
            subState=DUKE_LOGO_FADE_OUT;
            lastTime=GetTickCount();
        }
        else if (state==PLAY_ANIM)
        {
            delete curAnim;
            curAnim=NULL;
            state=GAME;
        }
        else if (state==MENU)
            MenuKeyHandler(Key);
        else if (state==GAME)
        {
            if (subState==GAME_PLAYING)
            {
                if (Key==VK_ESCAPE)
                {
                    sound->PlaySound(menuSelectSound,CHANNEL_MAIN,0,0);
                    subState=GAME_PAUSED;
                    menu=pauseMenu;
                    menuSelected=0;
                    useDirectInput=0;
                    prevCount=0;
                    di->Unacquire(DINPUT_MOUSE);
                }
            }
            else if (subState==GAME_PAUSED)
            {
                if ((Key==VK_ESCAPE)&&(prevCount==0))
                {
                    sound->PlaySound(menuExitSound,CHANNEL_MAIN,0,0);
                    subState=GAME_PLAYING;
                    useDirectInput=1;
                    di->Acquire(DINPUT_MOUSE);
                }
                else
                    MenuKeyHandler(Key);
            }
        }
    }
    if ((Key>=VK_F1)&&(Key<=VK_F10)&&(Shift.Contains(ssAlt)))
    {
        sound->PlaySound(rtsSound[Key-VK_F1],CHANNEL_RTS,0,0);
        Key=0;
    }
    else if ((Key>=VK_F1)&&(Key<=VK_F10)&&(Shift.Contains(ssCtrl)))
    {
        sound->PlaySound(secRtsSound[Key-VK_F1],CHANNEL_RTS,0,0);
        Key=0;
    }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::FormClose(TObject *Sender, TCloseAction &Action)
{
    StopMusic();
    // Save settings to the registry
    TRegistry *reg=new TRegistry;
    reg->RootKey=HKEY_LOCAL_MACHINE;
    reg->OpenKey("\\Software\\ACZ\\Duke Nukem 3D",true);
    reg->WriteInteger("vid_width",vid_width);
    reg->WriteInteger("vid_height",vid_height);
    reg->WriteInteger("vid_depth",vid_depth);
    reg->WriteInteger("renderer",RENDERER_OPTION.value);
    reg->WriteInteger("showfps",showfps);
    reg->WriteInteger("showstats",showstats);
    reg->WriteInteger("sound",SOUND_OPTION.value);
    reg->WriteInteger("music",MUSIC_OPTION.value);
    delete reg;
}
//---------------------------------------------------------------------------
void __fastcall TForm1::MCIWindowProc(TMessage& msg)
{
    if ((msg.Msg==MM_MCINOTIFY)&&(IsMusicPlaying()))
    {
        // Restart music if the music reached the end
    	char strRet[256];
	    mciSendString("status duke mode",strRet,256,Handle);
        if (!strcmp(strRet,"stopped"))
    	    mciSendString("play duke from 0 notify",strRet,256,Handle);
    }
    else
        WndProc(msg);
}

void __fastcall TForm1::FormClick(TObject *Sender)
{
    if (useDirectInput)
    {
        // If the form gets a click that DirectInput didn't
        // catch first, DirectInput must be disabled for
        // some reason when it should be enabled.
        di->Acquire(DINPUT_MOUSE);
    }
}

void __fastcall TForm1::FormMinimize(TObject *)
{
    minimized=true;
}

void __fastcall TForm1::FormRestore(TObject *)
{
    minimized=false;
}
//---------------------------------------------------------------------------

void TForm1::YesNoItem()
{
    menuItem->value=!menuItem->value;
}

void TForm1::NewGame()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=episodeMenu;
    menuSelected=0;
}

void TForm1::Episode1()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=skillMenu;
    menuSelected=1;
    episode=1;
}

void TForm1::Episode2()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=skillMenu;
    menuSelected=1;
    episode=2;
}

void TForm1::Episode3()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=skillMenu;
    menuSelected=1;
    episode=3;
}

void TForm1::Skill1()
{
    StopMusic();
    subState=MENU_WAIT_FOR_SOUND;
    lastTime=GetTickCount();
    sound->PlaySound(skillSound[0],CHANNEL_MAIN,0,0);
    sprintf(mapLoadName,"E%dL1.MAP",episode);
    prevCount=0;
}

void TForm1::Skill2()
{
    StopMusic();
    subState=MENU_WAIT_FOR_SOUND;
    lastTime=GetTickCount();
    sound->PlaySound(skillSound[1],CHANNEL_MAIN,0,0);
    sprintf(mapLoadName,"E%dL1.MAP",episode);
    prevCount=0;
}

void TForm1::Skill3()
{
    StopMusic();
    subState=MENU_WAIT_FOR_SOUND;
    lastTime=GetTickCount();
    sound->PlaySound(skillSound[2],CHANNEL_MAIN,0,0);
    sprintf(mapLoadName,"E%dL1.MAP",episode);
    prevCount=0;
}

void TForm1::Skill4()
{
    StopMusic();
    subState=MENU_WAIT_FOR_SOUND;
    lastTime=GetTickCount();
    sound->PlaySound(skillSound[3],CHANNEL_MAIN,0,0);
    sprintf(mapLoadName,"E%dL1.MAP",episode);
    prevCount=0;
}

void TForm1::Options()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=optionsMenu;
    menuSelected=0;
}

void TForm1::SaveGame()
{
}

void TForm1::LoadGame()
{
}

void TForm1::Help()
{
}

void TForm1::Quit()
{
    Close();
}

void TForm1::QuitToTitle()
{
    StopMusic();
    state=INTRO_ANIM;
    subState=INTRO_ANIM_PLAY;
    introAnim->Seek(0);
}

void TForm1::WindowedOption()
{
    menuItem->value=!menuItem->value;
    if (menuItem->value)
        DEPTH_OPTION.type=MENUITEM_DISABLED;
    else
        DEPTH_OPTION.type=MENUITEM_DEPTHLIST;
    vid_width=modesMerged.mode[RESOLUTION_OPTION.value].w;
    vid_height=modesMerged.mode[RESOLUTION_OPTION.value].h;
    vid_depth=menuItem->value?0:(DEPTH_OPTION.value?32:16);
    DynamicResChange();
}

void TForm1::ResolutionOption()
{
    vid_width=modesMerged.mode[menuItem->value].w;
    vid_height=modesMerged.mode[menuItem->value].h;
    vid_depth=WINDOWED_OPTION.value?0:(DEPTH_OPTION.value?32:16);
    DynamicResChange();
}

void TForm1::BitDepthOption()
{
    if (menuItem->type==MENUITEM_DISABLED)
        return;
    menuItem->value=1-menuItem->value;
    vid_width=modesMerged.mode[RESOLUTION_OPTION.value].w;
    vid_height=modesMerged.mode[RESOLUTION_OPTION.value].h;
    vid_depth=WINDOWED_OPTION.value?0:(menuItem->value?32:16);
    DynamicResChange();
}

void TForm1::DriverOption()
{
    // Save the current position and rotation values
    Vector p=Vector(render->GetXPos(),render->GetYPos(),render->GetZPos());
    Vector r=Vector(render->GetXRotate(),render->GetYRotate(),render->GetZRotate());
    // Free textures
    art.FreeTextureMem();
    introAnim->FreeTextureMem();
    if (curAnim) curAnim->FreeTextureMem();
    // Delete the current renderer object
    delete render;
    // Update video settings from menu options
    vid_width=modesMerged.mode[RESOLUTION_OPTION.value].w;
    vid_height=modesMerged.mode[RESOLUTION_OPTION.value].h;
    vid_depth=WINDOWED_OPTION.value?0:(DEPTH_OPTION.value?32:16);
    // Create the new renderer object
    switch(menuItem->value)
    {
        case 0: render=new D3DRender(hInst,Handle,vid_width,vid_height,vid_depth); break;
        case 1: render=new OpenGLRender(hInst,Handle,vid_width,vid_height,vid_depth); break;
        case 2: render=new SoftRender(hInst,Handle,vid_width,vid_height,vid_depth); break;
    }
    ClientWidth=scrnX;
    ClientHeight=scrnY;
    // Reload the textures into texture memory
    art.RecreateTextures();
    introAnim->RecreateTexture();
    if (curAnim) curAnim->RecreateTexture();
    art.ReloadPalettes();
    // Resize console to the new screen size
    ResizeConsole(scrnX,scrnY);
    // Restore old position and rotation values
    render->SetXPos(p.x); render->SetYPos(p.y); render->SetZPos(p.z);
    render->SetXRotate(r.x); render->SetYRotate(r.y); render->SetZRotate(r.z);
    if (useDirectInput)
    {
        // Re-enable DirectInput
        di->Acquire(DINPUT_MOUSE);
    }
}

void TForm1::SoundOption()
{
    menuItem->value=!menuItem->value;
    sound->SoundMute(!menuItem->value);
}

void TForm1::MusicOption()
{
    menuItem->value=!menuItem->value;
    MusicMute(!menuItem->value);
}

void TForm1::Multiplayer()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=multiplayerMenu;
    menuSelected=0;
}

void TForm1::FindServers()
{
}

void TForm1::ConnectToServer()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=connectMenu;
    menuSelected=0;
}

void TForm1::StartNewServer()
{
    prevMenu[prevCount]=menu;
    prevMenuSelected[prevCount++]=menuSelected;
    menu=newServerMenu;
    menuSelected=0;
}

void TForm1::MapsToCycle()
{
}

void TForm1::ServerGameOptions()
{
}

void TForm1::CreateServer()
{
}

void TForm1::JoinServer()
{
}

