/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.masterfs;

import org.openide.util.Utilities;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.io.File;
import org.netbeans.modules.masterfs.filebasedfs.utils.FileInfo;

/**
 * Encapsulates independent view of hierarchical resource path.
 * Implemented as immutable.
 *
 * @author Radek Matous
 */
final class ResourcePath {
    private final String resourcePath;

    private static final ResourcePath root = new ResourcePath("/");

    ResourcePath(StringBuilder sb) {
        resourcePath = sb.toString();
    }

    ResourcePath(final String resourcePath) {
        assert resourcePath != null;
        this.resourcePath = getNormalizedPath(resourcePath);
        assert this.resourcePath.length() > 0 && this.resourcePath.charAt(0) == '/';
    }

    static ResourcePath getRoot() {
        return root;
    }

    boolean isRoot() {
        return (getParent() == null) ? true : false;
    }

    File getFile () {         
        // XXX this is ugly and fragile; revisit
        if ((Utilities.isWindows () || (Utilities.getOperatingSystem () == Utilities.OS_OS2))) {
            File retVal = null;             
            // Windows: resourcePath is e.g. /d:/temp/foo.txt
            if (!isRoot()) {
                String filePath = resourcePath.substring(1).replace('/', File.separatorChar);//NOI18N
                if (getParent().isRoot() && !filePath.endsWith("/")) { //NOI18N
                    filePath = filePath + "/";//NOI18N     
                } 
                retVal = new File(filePath);
            }
            
            return retVal;
        } else {
            // Unix (or similar): resourcePath is e.g. /tmp/foo.txt
            return new File(resourcePath);
        }
    }
    
    final String getNormalizedPath() {
        return resourcePath;
    }

    Enumeration getElements () {
        StringTokenizer sTokens = new StringTokenizer(resourcePath, "/");
        return sTokens;
    }

    ResourcePath getChild(final String nameExt) {
        ResourcePath retVal = null;
        StringBuilder sb = new StringBuilder(resourcePath);
        if (!resourcePath.endsWith("/")) sb.append('/');//NOI18N
        sb.append(nameExt);
        retVal = new ResourcePath(sb);
        return retVal;
    }

    ResourcePath getChild(final String name, final String ext) {
        final StringBuilder sb = new StringBuilder(resourcePath);
        if (!resourcePath.endsWith("/")) sb.append('/');//NOI18N
        sb.append(name);
        if (ext != null && ext.length() != 0)
            sb.append('.').append(ext);//NOI18N
        return new ResourcePath(sb.toString());
    }

    ResourcePath getParent() {
        final int idx = resourcePath.lastIndexOf('/');
        if (idx == 0 && resourcePath.length() == 1) return null;
        return new ResourcePath((idx <= 0) ? "/" : resourcePath.substring(0, idx));//NOI18N
    }

    String getExt() {
        return FileInfo.getExt(getNameExt());        
    }

    /**
     * Implements abstract FileObject.getName()
     */
    String getName() {
        return FileInfo.getName(getNameExt());
    }

    /**
     * Overrides FileObject.getNameExt() to return only substring rather than newly created String
     */
    String getNameExt() {
        int idx0 = resourcePath.lastIndexOf('/');//NOI18N
        assert idx0 != -1 : resourcePath;
        idx0++;
        return resourcePath.substring(idx0);
    }

    String getPath() {
        return resourcePath.substring(1);
    }

    /** Adds slash at first position if necessary and removes slash from last position */
    static String getNormalizedPath(String resPath) {
        if (resPath == null) return resPath;
        resPath = resPath.replace('\\', '/');//NOI18N

        if (!resPath.startsWith("/")) //NOI18N
            resPath = "/" + resPath; //NOI18N

        if (resPath.endsWith("/") && resPath.length() != "/".length()) //NOI18N
            resPath = resPath.substring(0, resPath.length() - 1);

        return resPath;
    }

    public int hashCode() {
        return resourcePath.hashCode();
    }

    public boolean equals(final Object obj) {
        String resPath = null;
        if (obj instanceof String) {
            resPath = (String) obj;
        } else if (obj instanceof ResourcePath) {
            resPath = ((ResourcePath) obj).getNormalizedPath();
        }
        return resourcePath.equals(resPath);
    }

    public String toString() {
        return getNormalizedPath();
    }
}
