/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.bpel.xpath.view.visitor;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.xpath.model.nodes.CanvasNode;

import org.netbeans.modules.bpel.xpath.model.nodes.Node;
import org.netbeans.modules.bpel.xpath.model.nodes.NodeFactory;
import org.netbeans.modules.bpel.xpath.model.nodes.PartNode;
import org.netbeans.modules.bpel.xpath.model.nodes.TreeNode;
import org.netbeans.modules.bpel.xpath.model.nodes.VariableNode;
import org.netbeans.modules.bpel.xpath.model.nodes.impl.OperandPlaceHolderNodeImpl;
import org.netbeans.modules.bpel.xpath.util.Duration;
import org.netbeans.modules.bpel.xpath.view.VariableExpressionCommand;
import org.netbeans.modules.bpel.xpath.view.expression.impl.PredicatesManager;
import org.netbeans.modules.bpel.xpath.view.palette.IPaletteConstants;
import org.netbeans.modules.bpel.xpath.view.palette.XpathPaletteManager;
import org.netbeans.modules.xml.xpath.XPathCoreFunction;
import org.netbeans.modules.xml.xpath.XPathCoreOperation;
import org.netbeans.modules.xml.xpath.XPathExpression;
import org.netbeans.modules.xml.xpath.XPathExpressionPath;
import org.netbeans.modules.xml.xpath.XPathExtensionFunction;
import org.netbeans.modules.xml.xpath.XPathNumericLiteral;
import org.netbeans.modules.xml.xpath.XPathOperationOrFuntion;
import org.netbeans.modules.xml.xpath.XPathStringLiteral;
import org.netbeans.modules.xml.xpath.XPathVariableReference;
import org.netbeans.modules.xml.xpath.visitor.AbstractXPathVisitor;

import org.netbeans.modules.soa.mapper.common.ui.palette.IPaletteItem;
import org.netbeans.modules.soa.mapper.common.basicmapper.IBasicMapper;
import org.netbeans.modules.soa.mapper.common.basicmapper.methoid.IFieldNode;
import org.netbeans.modules.soa.mapper.common.basicmapper.methoid.IMethoid;
import org.netbeans.modules.soa.mapper.common.basicmapper.methoid.IMethoidNode;
import org.netbeans.modules.soa.mapper.common.basicmapper.palette.IPaletteView;
import org.netbeans.modules.soa.mapper.common.basicmapper.tree.IMapperTreeNode;
import org.netbeans.modules.soa.mapper.common.basicmapper.tree.IMapperTreeView;
import org.netbeans.modules.soa.mapper.common.IMapperNode;
import org.netbeans.modules.soa.mapper.common.IMapperViewModel;
import org.openide.util.NbBundle;

/**
 *
 * @author radval
 *
 */
public class NodeCreatorVisitor extends AbstractXPathVisitor {
    
    private static final Logger LOGGER = Logger.getLogger(NodeCreatorVisitor.class.getName());
    
    private IBasicMapper mMapper;
    private PredicatesManager mPredicatesManager;
    private IPaletteView mPaletteView = null;
    private List mIMethoidNodes = new ArrayList();
    private Stack mStack = new Stack();
    private Node mRootNode = null;
    private BpelEntity mHoldingEntity;
    
    
    public NodeCreatorVisitor(
            IBasicMapper mapper,
            PredicatesManager predicatesManger,
            BpelEntity holdingEntity) {
        this.mMapper = mapper;
        this.mPredicatesManager = predicatesManger;
        mPaletteView = this.mMapper.getMapperViewManager().getPaletteView();
        this.mHoldingEntity = holdingEntity;
    }
    
    private IBasicMapper getMapper() {
        return this.mMapper;
    }
    
    public Node getRootNode() {
        return this.mRootNode;
    }
    
    public List getAllNodes() {
        return this.mIMethoidNodes;
    }
    
    public void visit(XPathStringLiteral expr) {
        IMethoidNode node = null;
        try {
            String literal = expr.getExpressionString();
            Duration.parse(literal);
            node = createNewIMethoidNode(expr, IPaletteConstants.DURATION_LITERAL);
        } catch (Exception e) {
            // Not a duration literal
            node = null;
        }
        if (null == node) {
            node = createNewIMethoidNode(expr, IPaletteConstants.STRING_LITERAL);
        }
        IFieldNode fieldNode = (IFieldNode) node.getOutputFieldNodes().get(0);
        
        Object no = fieldNode.getNodeObject();
        
        handleNewNode(node, this.mRootNode == null);
        
        fieldNode.setLiteralName(expr.getExpressionString());
    }
    
    public void visit(XPathNumericLiteral expr) {
        IMethoidNode node = createNewIMethoidNode(expr, IPaletteConstants.NUMBER_LITERAL);
        IFieldNode fieldNode = (IFieldNode) node.getOutputFieldNodes().get(0);
        
        handleNewNode(node, this.mRootNode == null);
        
        fieldNode.setLiteralName(expr.getExpressionString());
    }
    
    public void visit(XPathCoreOperation expr) {
        visitXPathOperatorOrFunction(expr);
    }
    
    public void visit(XPathCoreFunction expr) {
        visitXPathOperatorOrFunction(expr);
    }
    
    private void visitXPathOperatorOrFunction(XPathOperationOrFuntion operator) {
        IMethoidNode node = createNewIMethoidNode(operator, operator.getName());
        handleNewNode(node, true);
        push(node);
        visitChildren(operator);
        pop();
    }
    
    public void visit(XPathExtensionFunction expr) {
        XPathExtensionFunctionNodeCreatorVisitor extNodeCreator =
                new XPathExtensionFunctionNodeCreatorVisitor(this.mMapper.getMapperViewManager().getSourceView(),
                this.mMapper,
                this.mPredicatesManager,
                this.mHoldingEntity);
        expr.accept(extNodeCreator);
        
        IMapperTreeNode mapperNode = extNodeCreator.getMapperNode();
        if (mapperNode != null) {
            handleTreeNode(mapperNode);
        } else {
            visitXPathOperatorOrFunction(expr);
        }
    }
    
    public void visit(XPathExpressionPath expressionPath) {
        IMapperTreeNode mapperNode = null;
        
        try {
            VariableExpressionCommand cmd = new VariableExpressionCommand(expressionPath.getExpressionString());
            cmd.execute();
            String variableName = cmd.getContainerName();
            String partName = cmd.getPartName();
            String query = cmd.getSubQuery();
            XPathExpression expr = cmd.getXPathExpression();
            IMapperTreeView sourceView = this.mMapper.getMapperViewManager().getSourceView();
            
            VariableNode variableNode = null;
            if (variableName == null && expr instanceof XPathVariableReference) {
                XPathVariableReference variableReference = (XPathVariableReference) expr;
                variableName = variableReference.getVariableName().toString();
            }
            variableNode = MapperTreeUtility.findVariable(variableName, sourceView.getTree().getModel());
            
            if (variableNode != null && query != null) {
                PartNode partNode = variableNode.getPartNode(partName);
                FindMapperTreeNodeVisitor fmtVisitor =
                        new FindMapperTreeNodeVisitor(
                        sourceView,
                        this.mPredicatesManager,
                        variableNode,
                        partNode,
                        this.mHoldingEntity);
                expressionPath.accept(fmtVisitor);
                mapperNode = fmtVisitor.getMapperTreeNode();
                
            } else if (variableName != null && partName != null) {
                if (variableNode != null) {
                    PartNode partNode = variableNode.getPartNode(partName);
                    mapperNode = FindMapperTreeNodeWithoutQuery.findPartMapperTreeNode(partNode, variableNode, sourceView);
                }
                
            } else if (variableName != null) {
                mapperNode  =
                        FindMapperTreeNodeWithoutQuery.findVariableMapperTreeNode(variableNode, sourceView);
            }
        } catch(Exception ex) {
            LOGGER.log(Level.WARNING,
                    NbBundle.getMessage(NodeCreatorVisitor.class,
                    "STR_FAILED_TO_GET_MAPPER_TREE_NODE_FOR_EXPRESSION_X",  // NOI18N
                    expressionPath.getExpressionString()), ex);
        }
        
        if (mapperNode != null) {
            handleTreeNode(mapperNode);
        }
    }
    
    
    public void visit(XPathVariableReference vReference) {
        IMapperTreeNode mapperNode = null;
        
        try {
            
            VariableExpressionCommand cmd = new VariableExpressionCommand(vReference.getExpressionString());
            cmd.execute();
            String variableName = cmd.getContainerName();
            String partName = cmd.getPartName();
            XPathExpression expr = cmd.getXPathExpression();
            
            IMapperTreeView sourceView = this.mMapper.getMapperViewManager().getSourceView();
            
            VariableNode variableNode = null;
            if (variableName == null && expr instanceof XPathVariableReference) {
                XPathVariableReference variableReference = (XPathVariableReference) expr;
                variableName = variableReference.getVariableName().toString();
            }
            variableNode = MapperTreeUtility.findVariable(variableName, sourceView.getTree().getModel());
            
            if (variableName != null && partName != null) {
                if (variableNode != null) {
                    PartNode partNode = variableNode.getPartNode(partName);
                    mapperNode = FindMapperTreeNodeWithoutQuery.findPartMapperTreeNode(partNode, variableNode, sourceView);
                }
            }  else if(variableName != null) {
            
                mapperNode  =
                        FindMapperTreeNodeWithoutQuery.findVariableMapperTreeNode(variableNode, sourceView);
            }
            } catch(Exception ex) {
            LOGGER.log(Level.WARNING,
                    NbBundle.getMessage(NodeCreatorVisitor.class,
                    "STR_FAILED_TO_GET_MAPPER_TREE_NODE_FOR_EXPRESSION_X",  // NOI18N
                    vReference.getExpressionString()), ex);
        }
        
        if (mapperNode != null) {
            handleTreeNode(mapperNode);
        } else if (vReference.toString().startsWith(OperandPlaceHolderNodeImpl.UNCONNECTED_PLACEHOLDER)){
            handleUnconnectedField();
        }
        
    }
    
    
    private IMethoidNode createNewIMethoidNode(XPathExpression expr, String name) {
        IMethoidNode node = null;
        IMethoid methoid = null;
        IPaletteItem pItem = null;
        
        pItem = XpathPaletteManager.getDefault().getItemByMethodName(name);
        LOGGER.fine(NbBundle.getMessage(NodeCreatorVisitor.class,
                "STR_CREATE_NEW_METHOID_NODE_FOR_X_FOUND_PALETTE_ITEM_Y",   // NOI18N
                name, pItem));
        
        methoid = (IMethoid) this.mPaletteView.findMethoid(pItem);
        node = this.mMapper.createMethoidNode(methoid);
        
        IFieldNode fieldNode = (IFieldNode) node.getOutputFieldNodes().get(0);
        Node operatorNode = NodeFactory.getInstance().createNode(expr);
        operatorNode.setMapperNode(node);
        fieldNode.setNodeObject(operatorNode);
        node.setNodeObject(operatorNode);
        
        return node;
    }
    
    private void push(IMethoidNode node) {
        IFieldNode fieldNode = (IFieldNode) node.getOutputFieldNodes().get(0);
        Node operatorNode = (Node) fieldNode.getNodeObject();
        mStack.push(operatorNode);
    }
    
    private void pop() {
        if (!this.mStack.empty()) {
            this.mStack.pop();
        }
    }
    
    private Node peek() {
        if (!this.mStack.empty()) {
            return (Node) this.mStack.peek();
        }
        return null;
    }
    
    private void handleNewNode(IMethoidNode node, boolean addToModel) {
        CanvasNode canvasNode = (CanvasNode) node.getNodeObject();
        canvasNode.setMapperNode(node);
        
        Node n = (Node) peek();
        if (n != null) {
            if (n instanceof CanvasNode) {
                ((CanvasNode) n).addInput(canvasNode);
            }
            canvasNode.addOutput(n);
        }
        
        if (addToModel) {
            addNode(node);
        }
        
        if (this.mRootNode == null) {
            this.mRootNode = canvasNode;
        }
    }
    private void handleUnconnectedField(){
        Node n = peek();
        if (n != null) {
            if (n instanceof CanvasNode) {
                ((CanvasNode) n).addInput(new OperandPlaceHolderNodeImpl());
            }
        }
        
    }
    private void handleTreeNode(IMapperTreeNode node) {
        TreeNode treeNode = (TreeNode) node.getNodeObject();
        if (treeNode != null) {
            treeNode.setMapperNode(node);
            
            Node n = (Node) peek();
            if (n != null) {
                if (n instanceof CanvasNode) {
                    ((CanvasNode) n).addInput(treeNode);
                }
            }
            
            if (this.mRootNode == null) {
                this.mRootNode = treeNode;
            }
        }
    }
    
    private void addNode(IMapperNode node) {
        IBasicMapper mapper = getMapper();
        if (
                (mapper.getMapperViewManager().getMapperModel() != null) &&
                (mapper.getMapperViewManager().getMapperModel().getSelectedViewModel() != null)) {
            IMapperViewModel viewModel = mapper.getMapperViewManager().getMapperModel().getSelectedViewModel();
            viewModel.addNode(node);
        }
    }
}
