/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2007 Sun Microsystems, Inc.
 */
package org.netbeans.modules.editor.settings.storage.codetemplates;

import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.netbeans.api.editor.mimelookup.MimeLookup;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.CodeTemplateDescription;
import org.netbeans.api.editor.settings.CodeTemplateSettings;
import org.netbeans.core.startup.Main;
import org.netbeans.junit.NbTestCase;
import org.netbeans.modules.editor.settings.storage.EditorTestLookup;

/**
 *
 * @author vita
 */
public class CodeTemplateSettingsImplTest extends NbTestCase {

    public CodeTemplateSettingsImplTest(String name) {
        super(name);
    }

    @Override
    protected void setUp() throws Exception {
        super.setUp();
    
        EditorTestLookup.setLookup(
            new URL[] {
                getClass().getClassLoader().getResource(
                    "org/netbeans/modules/editor/settings/storage/codetemplates/test-layer.xml"),
                getClass().getClassLoader().getResource(
                    "org/netbeans/modules/editor/settings/storage/layer.xml"),
                getClass().getClassLoader().getResource(
                    "org/netbeans/core/resources/mf-layer.xml"), // for MIMEResolverImpl to work
            },
            getWorkDir(),
            new Object[] {},
            getClass().getClassLoader()
        );

        // This is here to initialize Nb URL factory (org.netbeans.core.startup),
        // which is needed by Nb EntityCatalog (org.netbeans.core).
        // Also see the test dependencies in project.xml
        Main.initializeURLFactory();
    }

    public void testSimple() {
        CodeTemplateSettingsImpl ctsi = CodeTemplateSettingsImpl.get(MimePath.EMPTY);
        Map<String, CodeTemplateDescription> map = ctsi.getCodeTemplates();
        
        assertNotNull("CodeTemplates map should not be null", map);
        assertEquals("Wrong number of code templates", 3, map.size());
        
        checkCodeTemplate(map, "sout", "System Out Print Line", "System.out.println(${cursor});", "not-a-real-uuid", "aaa", "bbb");
        checkCodeTemplate(map, "hello", null, "  Hello World!  ", null);
        checkCodeTemplate(map, "xyz", "xyz", "<xyz>${cursor}</xyz>", null, "xyz");
    }

    public void testUserChangesOverrideDefaults() {
        MimePath mimePath = MimePath.parse("text/x-type-A");
        CodeTemplateSettingsImpl ctsi = CodeTemplateSettingsImpl.get(mimePath);
        Map<String, CodeTemplateDescription> map = ctsi.getCodeTemplates();
        
        assertNotNull("CodeTemplates map should not be null", map);
        assertEquals("Wrong number of code templates", 2, map.size());
        
        checkCodeTemplate(map, "module1", null, "module1", null);
        checkCodeTemplate(map, "user1", null, "user1", null);
        
        CodeTemplateDescription ct = map.get("module2");
        assertNull("'module2' code template should be removed", ct);
    }
    
    public void testSimpleWrite() {
        String abbrev = "ct";
        String desc = "Code Template";
        String text = "<code-template>${cursor}";
        String uuid = "test-uuid";
        List<String> contexts = Arrays.asList(new String [] { "ct-ctx1" });
        CodeTemplateDescription ct = new CodeTemplateDescription(abbrev, desc, text, contexts, uuid);
        
        MimePath mimePath = MimePath.parse("text/x-empty");
        CodeTemplateSettingsImpl ctsi = CodeTemplateSettingsImpl.get(mimePath);
        
        // Write the code template
        ctsi.setCodeTemplates(Collections.singletonMap(abbrev, ct));
        
        // Force loading from the files
        Map<String, CodeTemplateDescription> loadedMap = CodeTemplatesStorage.load(mimePath, false);
        assertNotNull("Can't load the map", loadedMap);
        assertEquals("Wrong number of code templates", 1, loadedMap.size());
        checkCodeTemplate(loadedMap, abbrev, desc, text, uuid, contexts.toArray(new String [contexts.size()]));
    }

    public void testRemoveAll() {
        MimePath mimePath = MimePath.parse("text/x-type-A");
        CodeTemplateSettingsImpl ctsi = CodeTemplateSettingsImpl.get(mimePath);
        Map<String, CodeTemplateDescription> map = ctsi.getCodeTemplates();
        
        assertNotNull("CodeTemplates map should not be null", map);
        assertTrue("Code templates map should not be empty", map.size() > 0);
        
        // Remove all code templates
        ctsi.setCodeTemplates(Collections.<String, CodeTemplateDescription>emptyMap());
        
        // Force loading from the files
        Map<String, CodeTemplateDescription> loadedMap = CodeTemplatesStorage.load(mimePath, false);
        assertNotNull("Can't load the map", loadedMap);
        assertEquals("Some template were not removed", 0, loadedMap.size());
    }
    
    public void testLoadLegacy() {
        MimePath mimePath = MimePath.parse("text/x-legacy");
        CodeTemplateSettingsImpl ctsi = CodeTemplateSettingsImpl.get(mimePath);
        Map<String, CodeTemplateDescription> map = ctsi.getCodeTemplates();
        
        assertNotNull("CodeTemplates map should not be null", map);
        assertEquals("Wrong number of code templates", 1, map.size());
        
        checkCodeTemplate(map, "tglb", null, "<%@taglib uri=\"${cursor}\"%>", null);
    }
    
    public void testMimeLookup() {
        MimePath mimePath = MimePath.parse("text/x-type-B");
        CodeTemplateSettings cts = MimeLookup.getLookup(mimePath).lookup(CodeTemplateSettings.class);
        
        assertNotNull("No CodeTemplateSettings in MimeLookup", cts);
        
        Collection<CodeTemplateDescription> codeTemplates = cts.getCodeTemplateDescriptions();
        assertNotNull("CodeTemplates should not be null", codeTemplates);
        assertEquals("Wrong number of code templates", 2, codeTemplates.size());
        
        Map<String, CodeTemplateDescription> map = toMap(codeTemplates);
        
        checkCodeTemplate(map, "module1", null, "module1", null);
        checkCodeTemplate(map, "user1", null, "user1", null);
        
        CodeTemplateDescription ct = map.get("module2");
        assertNull("'module2' code template should be removed", ct);
    }
    
    private void checkCodeTemplate(
        Map<String, CodeTemplateDescription> map, 
        String abbrev, 
        String desc, 
        String text, 
        String uuid,
        String... contexts
    ) {
        CodeTemplateDescription ct = map.get(abbrev);
        
        assertNotNull("Can't find code template for '" + abbrev + "'", ct);
        assertEquals("Wrong abbreviation", abbrev, ct.getAbbreviation());
        assertEquals("Wrong description", desc, ct.getDescription());
        assertEquals("Wrong text", text, ct.getParametrizedText());
        assertEquals("Wrong unique id", uuid, ct.getUniqueId());
        
        List<String> ctxs = ct.getContexts();
        assertEquals("Wrong number of contexts", contexts.length, ctxs.size());
        
        for(String c : contexts) {
            assertTrue("Code template does not contain context '" + c + "'", ctxs.contains(c));
        }
    }
    
    private static Map<String, CodeTemplateDescription> toMap(Collection<CodeTemplateDescription> c) {
        Map<String, CodeTemplateDescription> m = new HashMap<String, CodeTemplateDescription>();
        for(CodeTemplateDescription ct : c) {
            m.put(ct.getAbbreviation(), ct);
        }
        return m;
    }
}
