/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.editor.settings.storage;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.openide.filesystems.FileLock;

import org.openide.filesystems.FileObject;
import org.openide.xml.EntityCatalog;
import org.openide.xml.XMLUtil;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;


public final class XMLStorage {

    private static final Logger LOG = Logger.getLogger(XMLStorage.class.getName());
    
    private XMLStorage() {
        
    }
    
    public static void save(FileObject fo, Document doc) {
        assert fo != null : "FileObject can't be null"; //NOI18N
        assert doc != null : "Document can't be null"; //NOI18N
        
        try {
            FileLock lock = fo.lock();
            try {
                OutputStream os = fo.getOutputStream(lock);
                try {
                    XMLUtil.write(doc, os, "UTF-8"); //NOI18N
                } finally {
                    os.close();
                }
            } finally {
                lock.releaseLock();
            }
        } catch (IOException ex) {
            LOG.log(Level.WARNING, "Can't save editor settings to " + fo.getPath(), ex); //NOI18N
        }
    }
    
    public static Object load(FileObject fo, Handler handler) {
        assert fo != null : "Settings file must not be null"; //NOI18N
        
        handler.setProcessedFile(fo);
        try {
            XMLReader reader = XMLUtil.createXMLReader(true);
            reader.setEntityResolver(EntityCatalog.getDefault());
            reader.setContentHandler(handler);
            reader.setErrorHandler(handler);
            reader.setProperty("http://xml.org/sax/properties/lexical-handler", handler); //NOI18N
            
            InputStream is = fo.getInputStream();
            try {
                reader.parse(new InputSource(is));
            } finally {
                is.close();
            }
        } catch (Exception ex) {
            LOG.log(Level.WARNING, "Invalid or corrupted file: " + fo.getPath(), ex); //NOI18N
        } finally {
            handler.setProcessedFile(null);
        }
        
        return handler.getResult();
    }
    
    public static class Handler extends DefaultHandler implements LexicalHandler {

        // DefaultHandler implementation
        
        @Override
        public void warning(SAXParseException e) throws SAXException {
            log("warning", e); //NOI18N
        }

        @Override
        public void error(SAXParseException e) throws SAXException {
            log("error", e); //NOI18N
        }

        @Override
        public void fatalError(SAXParseException e) throws SAXException {
            log("fatal error", e); //NOI18N
            throw e;
        }

        @Override
        public InputSource resolveEntity(String pubid, String sysid) {
            return new InputSource(
                new java.io.ByteArrayInputStream(new byte [0])
            );
        }
        
        // LexicalHandler implementation
        
        public void startCDATA() throws SAXException {
        }
        
        public void endCDATA() throws SAXException {
        }

        public void comment(char[] ch, int start, int length) throws SAXException {
        }

        public void endDTD() throws SAXException {
        }

        public void endEntity(String name) throws SAXException {
        }

        public void startDTD(String name, String publicId, String systemId) throws SAXException {
        }

        public void startEntity(String name) throws SAXException {
        }
        
        // XMLStorage.Handler interface
        
        public void setResult (Object result) {
            this.result = result;
        }
        
        public Object getResult () {
            return result;
        }
        
        
        protected final FileObject getProcessedFile() {
            return this.file;
        }
        
        protected final boolean isModuleFile() {
            return this.isModuleFile;
        }

        protected final boolean isDefaultProfile() {
            return isDefaultProfile;
        }

        // Private implementation
        
        private FileObject file;
        private boolean isModuleFile;
        private boolean isDefaultProfile;
        
        private Object result;
        
        private void log(String errorType, SAXParseException e) {
            if (file == null) {
                LOG.log(Level.FINE, "XML parser " + errorType, e); //NOI18N
            } else {
                Level level;
                if (isModuleFile()) { //NOI18N
                    level = Level.WARNING; // warnings for module layer supplied files
                } else {
                    level = Level.FINE; // user files, can be from previous versions
                }
                
                LOG.log(level, "XML parser " + errorType + " in file " + file.getPath(), e); //NOI18N
            }
        }
        
        private void setProcessedFile(FileObject f) {
            this.file = f;
            this.isModuleFile = false;
            this.isDefaultProfile = false;
            
            if (this.file != null) {
                FileObject parent = this.file.getParent();
                if (parent != null) {
                    this.isModuleFile = parent.getNameExt().contains("Default"); //NOI18N
                    parent = parent.getParent();
                    if (parent != null) {
                        this.isDefaultProfile = parent.getNameExt().contains(EditorSettingsImpl.DEFAULT_PROFILE);
                    }
                }
            }
        }
    } // End of Handler class
}
