/*
 
 
    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
 
 
 */

package com.sap.dbtech.util;


import java.util.ResourceBundle;
import java.util.MissingResourceException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Locale;

/**
 * Handles lookup of the properties for the message and the messageformat calls.
 */
public class MessageTranslator {
    
    private static final String MESSAGES="com.sap.dbtech.jdbc.messages";  // name of message resource
    private static HashMap messagebundles=new HashMap();                  // messages for different locales 
    private static Object lock=new Object();                              // lock for init
    
    /**
     * Translation without parameters.
     * @param key key of the message
     * @return localized message
     */
    public static String translate(String key)
    {
        return translate(key, null);
    }

    /**
     * Translates message and formats it.
     * @param key key of the message
     * @param o1 parameter {0}
     * @return localized message
     */   
    public static String translate(String key, Object o1)
    {
        Object [] args={ o1 };
        return translate(key, args);
    }
    
    /**
     * Translates message and formats it.
     * @param key key of the message
     * @param o1 parameter {0}
     * @param o2 parameter {1}
     * @return localized message
     */   
    public static String translate(String key, 
                                   Object o1,
                                   Object o2)
    {
        Object [] args={ o1, o2 };
        return translate(key, args);
    }

         
    /**
     * Translates message and formats it.
     * @param key key of the message
     * @param o1 parameter {0}
     * @param o2 parameter {1}
     * @param o3 parameter {2}
     * @return localized message
     */   
    public static String translate(String key, 
                                   Object o1,
                                   Object o2,
                                   Object o3)
    {
        Object [] args={ o1, o2, o3 };
        return translate(key, args);
    }


    /**
     * Getter method for the messages bundle that performs
     * also the 1st time initialisation (loading the bundle).
     * @return the message resource bundle of the driver locale
     */
    public static ResourceBundle getMessages()
    {
        Object messages=messagebundles.get(Locale.getDefault().toString());
        if(messages==null) {
            synchronized(lock) {
                messages=messagebundles.get(Locale.getDefault().toString());
                if(messages==null) {
                    try {
                        messages=ResourceBundle.getBundle(MESSAGES);
                        messagebundles.put(Locale.getDefault().toString(), messages);
                    } catch(MissingResourceException missingResEx) {
                        // Ignore. If a locale is not supported it will find
                        // English defaults ... otherwise it is really messed up.
                    }
                }
            }
        }
        return (ResourceBundle)messages;
    }

    /**
     * Translates a given key and object array into a descriptive message.
     * @param key The message key.
     * @param args The additional arguments.
     * @return The composed message, in the locale of the installation.
     */
    public static String translate(String key,
                                   Object[] args) {
        try {
            // retrieve text and format it
            String messageText=getMessages().getString(key);
            return MessageFormat.format(messageText, args);
        } catch(MissingResourceException missingResEx) {
            
            // emergency - create an informative message in this case at least
            StringBuffer result=new StringBuffer("No message available for locale ");
            result.append(getMessages().getLocale().toString());
            result.append(", key ");
            result.append(key);
            // if arguments given append them
            if(args==null || args.length==0) {
                result.append(".");
            } else {
                result.append(", arguments [");
                for(int i=0; i<args.length-1; ++i) {
                    result.append(args[i].toString());
                    result.append(", ");
                }
                result.append(args[args.length-1].toString());
                result.append("].");
            }
            return result.toString();
        } 
        catch(NullPointerException nullpointerex) {
            StringBuffer result=new StringBuffer("No message available for default locale ");
            result.append("for key ");
            result.append(key);
            // if arguments given append them
            if(args==null || args.length==0) {
                result.append(".");
            } else {
                result.append(", arguments [");
                for(int i=0; i<args.length-1; ++i) {
                    result.append(args[i].toString());
                    result.append(", ");
                }
                result.append(args[args.length-1].toString());
                result.append("].");
            }
            return result.toString();
        }
    }
}
