/* WorksheetWindow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.worksheet;

import be.ugent.caagt.swirl.SwirlUtilities;
import be.ugent.caagt.swirl.actions.HideWindow;
import be.ugent.caagt.swirl.actions.SimpleAction;
import be.ugent.caagt.swirl.actions.ToolBarButton;
import be.ugent.caagt.swirl.menus.MenuBuilder;
import be.ugent.caagt.swirl.subwindows.Subwindow;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.swing.AbstractButton;
import javax.swing.ActionMap;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;

import org.grinvin.conjecture.history.ConjectureHistoryOverview;
import org.grinvin.conjecture.engine.EngineRunner.Status;
import org.grinvin.worksheet.WorksheetWindowModel.SubwindowType;
import org.grinvin.worksheet.actions.DeleteConjectureHistoryAction;
import org.grinvin.worksheet.actions.SaveConjectureHistoryAction;
import org.grinvin.gui.components.FilterListComponent;
import org.grinvin.gui.components.InvariantListComponent;
import org.grinvin.help.HelpManager;
import org.grinvin.gui.components.ExpressionListComponent;
import org.grinvin.gui.components.FilterListComponent.CombineSelectedAND;
import org.grinvin.gui.components.FilterListComponent.CombineSelectedOR;
import org.grinvin.gui.components.FilterListComponent.NegateSelectedElements;
import org.grinvin.gui.components.FilterListComponent.SplitSelectedElements;
import org.grinvin.gui.components.GeneratorInstanceListComponent;
import org.grinvin.gui.components.GraphListMultiPanelComponent;
import org.grinvin.help.actions.DisplayHelpAfterTracking;
import org.grinvin.help.actions.LaunchHelp;
import org.grinvin.list.graphs.GraphList;

import org.grinvin.list.actions.RunConjecturing;
import org.grinvin.list.actions.WorksheetRenameOne;
import org.grinvin.list.actions.GraphProperties;
import org.grinvin.list.actions.GraphRename;
import org.grinvin.list.actions.ExportGraph;
import org.grinvin.list.actions.ExportGraphImage;
import org.grinvin.list.actions.ExportGraphList;
import org.grinvin.list.actions.ImportGraphBundle;
import org.grinvin.preferences.GrinvinPreferences;

/**
 * Window associated to a conjecturing engine. Consists of
 * <ul>
 * <li>An invariant table which contains the data to be used by the engine.</li>
 * <li>A status and configuration panel for the engine.</li>
 * <li>A history panel displaying the generated conjectures.</li>
 * </ul>
 *
 */
public class WorksheetWindow extends JFrame
        implements WorksheetModelListener {

    private static final String BUNDLE_NAME = "org.grinvin.worksheet.worksheetwindow";
    private static final String WORKSHEET_BUNDLE_NAME = "org.grinvin.worksheet.worksheet-menus";

    // TODO: split off into separate subcomponents
    //
    private final WorksheetWindowModel worksheetWindowModel;

    //
    public WorksheetWindowModel getModel() {
        return worksheetWindowModel;
    }
    //
    private final JLabel statusLabel;
    //
    private JPanel configPanel;
    //
    private final JPanel enginePanel;
    //
    private final GraphListMultiPanelComponent graphlistcomponent;

    //
    public void setStatus(Status status) {
        ResourceBundle bundle = ResourceBundle.getBundle(BUNDLE_NAME);
        statusLabel.setText(bundle.getString("enginestatus." + status));
    }

    //
    public WorksheetWindow(WorksheetWindowModel worksheetWindowModel) {
        setTitle(worksheetWindowModel.getName());
        Container contentPane = getContentPane();
        contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));

        this.worksheetWindowModel = worksheetWindowModel;
        worksheetWindowModel.addWorksheetModelListener(this);

        ListSelectionModel selectionModel = new DefaultListSelectionModel();

        ResourceBundle bundle = ResourceBundle.getBundle(BUNDLE_NAME);

        // invariants
        JScrollPane iList = new JScrollPane(new InvariantListComponent(worksheetWindowModel.getInvariantListModel()));
        iList.setPreferredSize(new Dimension(500, 100));
        iList.setName(bundle.getString("invariant.subwindow.title"));
        MySubwindow invariantWindow = new MySubwindow(iList, worksheetWindowModel.getSubwindowModel(SubwindowType.INVARIANTLIST));
        invariantWindow.setTitleBarColor(new Color(215, 170, 170));
        invariantWindow.setTitleBarTextColor(Color.BLACK);
        contentPane.add(invariantWindow);

        // graphs
        graphlistcomponent = new GraphListMultiPanelComponent(worksheetWindowModel, selectionModel);
        graphlistcomponent.setPreferredSize(new Dimension(500, 200));
        graphlistcomponent.setName(bundle.getString("multipanel.subwindow.title"));
        MySubwindow multipanelWindow = new MySubwindow(graphlistcomponent, worksheetWindowModel.getSubwindowModel(SubwindowType.GRAPHLIST));
        multipanelWindow.setTitleBarColor(new Color(170, 215, 170));
        multipanelWindow.setTitleBarTextColor(Color.BLACK);
        contentPane.add(multipanelWindow);

        // generator instances
        JScrollPane geninstlist = new JScrollPane(new GeneratorInstanceListComponent(worksheetWindowModel.getGeneratorInstanceListModel()));
        geninstlist.setPreferredSize(new Dimension(500, 100));
        geninstlist.setName(bundle.getString("geninst.subwindow.title"));
        MySubwindow geninstWindow = new MySubwindow(geninstlist, worksheetWindowModel.getSubwindowModel(SubwindowType.GENERATORINSTANCELIST));
        geninstWindow.setTitleBarColor(new Color(181, 147, 113));
        geninstWindow.setTitleBarTextColor(Color.BLACK);
        contentPane.add(geninstWindow);

        // filter
        FilterListComponent filterListComponent = new FilterListComponent(worksheetWindowModel.getFilterListModel());
        JScrollPane filterList = new JScrollPane(filterListComponent);
        filterList.setPreferredSize(new Dimension(500, 100));
        filterList.setName(bundle.getString("filter.subwindow.title"));
        MySubwindow filterWindow = new MySubwindow(filterList, worksheetWindowModel.getSubwindowModel(SubwindowType.FILTERLIST));
        filterWindow.setTitleBarColor(new Color(181, 147, 181));
        filterWindow.setTitleBarTextColor(Color.BLACK);
        contentPane.add(filterWindow);

        GraphList list = worksheetWindowModel.getGraphList();

        //
        MenuBuilder menuBuilder = new MenuBuilder();
        menuBuilder.load("/org/grinvin/worksheet/worksheet-menus.xml",WORKSHEET_BUNDLE_NAME);
        if (GrinvinPreferences.getInstance().getOsType() == GrinvinPreferences.OsType.MAC_OS_X) {
            menuBuilder.setPredicate("mac");
        }
        ActionMap actionMap = menuBuilder.getActionMap();

        // create and register individual actions with this manager
        actionMap.put("graph.rename", new GraphRename(list, selectionModel));

        actionMap.put("graph.properties", new GraphProperties(list, selectionModel));

        actionMap.put("graph.exportgraph", new ExportGraph(this, list, selectionModel));

        actionMap.put("graph.exportgraphimage", new ExportGraphImage(this, list, selectionModel));


        actionMap.put("graph.import", new ImportGraphBundle(list, this));

        actionMap.put("graph.exportlist", new ExportGraphList(this, list));


        actionMap.put("conjecture.make", new RunConjecturing(this));

        actionMap.put("worksheet.close", new HideWindow(this));


        actionMap.put("worksheet.rename", new WorksheetRenameOne(getModel()));


        // TODO: these actions also exist in GrinvinWindow
        actionMap.put("LaunchHelp", new LaunchHelp());

        actionMap.put("DisplayHelpAfterTracking", new DisplayHelpAfterTracking());

        // menu
        JMenuBar menuBar = menuBuilder.createJMenuBar("main.menu");
        setJMenuBar(menuBar);
        // note: also configures the 'conjecture.make' action which
        // is needed below, hence this must be done early
        // TODO: adapt SWIRL to remove this dependency
        
        // conjecturing engine configuration window
        statusLabel = new JLabel(bundle.getString("enginestatus." + Status.STOPPED));
        statusLabel.setHorizontalAlignment(JLabel.TRAILING);

        enginePanel = new JPanel();
        enginePanel.setLayout(new BoxLayout(enginePanel, BoxLayout.LINE_AXIS));

        configPanel = worksheetWindowModel.getEngineRunner().getEngine().getConfigurationPanel(worksheetWindowModel);
        enginePanel.add(configPanel);

        JPanel statusPanel = new JPanel(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 10, 5, 10);
        gbc.anchor = GridBagConstraints.WEST;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 0.0;
        gbc.gridheight = 1;
        gbc.gridx = 1;
        gbc.anchor = GridBagConstraints.EAST;
        statusPanel.add(new ProgressBar(), gbc);
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 1.0;
        gbc.gridy = 1;
        gbc.gridx = 0;
        gbc.gridwidth = 2;
        statusPanel.add(statusLabel, gbc);
        gbc.weightx = 0.0;
        gbc.gridy = 0;
        gbc.gridheight = 2;
        gbc.gridwidth = 1;
        gbc.gridx = 2;
        AbstractButton button = new ToolBarButton(actionMap.get("conjecture.make"));

        statusPanel.add(button, gbc);

        enginePanel.add(statusPanel);
        enginePanel.setMaximumSize(new Dimension(Short.MAX_VALUE, statusPanel.getPreferredSize().height));
        // should not stretch when other windows collapse
        enginePanel.setName(bundle.getString("configuration.subwindow.title"));
        MySubwindow engineWindow = new MySubwindow(enginePanel, worksheetWindowModel.getSubwindowModel(SubwindowType.ENGINEPANEL));
        //engineWindow.setTitleBarColor (new Color (215, 170, 170));
        engineWindow.setTitleBarTextColor(Color.BLACK);
        contentPane.add(engineWindow);

        // expressions
        JScrollPane scrollPane = new JScrollPane(
                new ExpressionListComponent(worksheetWindowModel.getConjectureListModel()));
        scrollPane.setPreferredSize(new Dimension(300, 100));
        scrollPane.setName(bundle.getString("expression.subwindow.title"));
        MySubwindow expressionWindow = new MySubwindow(scrollPane, worksheetWindowModel.getSubwindowModel(SubwindowType.CONJECTURELIST));
        expressionWindow.setTitleBarColor(new Color(215, 215, 170));
        expressionWindow.setTitleBarTextColor(Color.BLACK);
        worksheetWindowModel.getConjectureListModel().addListDataListener(
                new ListDataListener() {

                    public void intervalAdded(ListDataEvent e) {
                        changed();
                    }

                    public void intervalRemoved(ListDataEvent e) {
                        changed();
                    }

                    public void contentsChanged(ListDataEvent e) {
                        changed();
                    }

                    private void changed() {
                        getModel().getSubwindowModel(SubwindowType.CONJECTURELIST).setVisible(true);
                        getModel().getSubwindowModel(SubwindowType.CONJECTURELIST).setCollapsed(false);
                    }
                });
        contentPane.add(expressionWindow);

        contentPane.add(Box.createVerticalGlue());

        // TODO: also install these actions using menuBuilder

        JMenu graph = (JMenu) SwirlUtilities.getChildWithName(menuBar, "menu.graph");
        new MyListener(getModel().getSubwindowModel(SubwindowType.GRAPHLIST), graph);

        JMenu filter = (JMenu) SwirlUtilities.getChildWithName(menuBar, "menu.filter");
        filter.add(new CombineSelectedAND(filterListComponent));
        filter.add(new CombineSelectedOR(filterListComponent));
        filter.add(new NegateSelectedElements(filterListComponent));
        filter.add(new SplitSelectedElements(filterListComponent));
        new MyListener(getModel().getSubwindowModel(SubwindowType.FILTERLIST), filter);

        JMenu engine = (JMenu) SwirlUtilities.getChildWithName(menuBar, "menu.conjecture");
        engine.add(new ViewHistory());
        engine.add(new SaveConjectureHistoryAction(worksheetWindowModel));
        engine.add(new DeleteConjectureHistoryAction(worksheetWindowModel));
        new MyListener(getModel().getSubwindowModel(SubwindowType.ENGINEPANEL), engine);

        JMenu view = (JMenu) SwirlUtilities.getChildWithName(menuBar, "menu.view");
        view.add(new HideSubwindowMenuItem(invariantWindow, invariantWindow.getModel()));
        view.add(new HideSubwindowMenuItem(multipanelWindow, multipanelWindow.getModel()));
        view.add(new HideSubwindowMenuItem(geninstWindow, geninstWindow.getModel()));
        view.add(new HideSubwindowMenuItem(filterWindow, filterWindow.getModel()));
        view.add(new HideSubwindowMenuItem(engineWindow, engineWindow.getModel()));
        view.add(new HideSubwindowMenuItem(expressionWindow, expressionWindow.getModel()));
        
        pack();
        
        // help
        HelpManager.setHelpIDString(this, "org.grinvin.window.worksheet");
        //HelpManager.setHelpIDString(button, "org.grinvin.window.conjecturing.conjecturebutton");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "menu.worksheet"), "org.grinvin.window.worksheet.menu.worksheet");
        HelpManager.setHelpIDString(graph, "org.grinvin.window.worksheet.menu.graph");
        HelpManager.setHelpIDString(filter, "org.grinvin.window.worksheet.menu.filter");
        HelpManager.setHelpIDString(engine, "org.grinvin.window.worksheet.menu.conjecture");
        HelpManager.setHelpIDString(view, "org.grinvin.window.worksheet.menu.view");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "menu.help"), "org.grinvin.window.worksheet.menu.help");
        HelpManager.enableHelpKey(this.getRootPane(), "org.grinvin.window.worksheet");
    }

    public GraphListMultiPanelComponent getGraphListComponent() {
        return graphlistcomponent;
    }

    public void engineChanged() {
        enginePanel.remove(configPanel);
        configPanel = worksheetWindowModel.getEngineRunner().getEngine().getConfigurationPanel(worksheetWindowModel);
        enginePanel.add(configPanel, 0);
        enginePanel.validate();
    }

    public void valueChanged(ListSelectionEvent e) {
    // intentionally left blank
    }

    // TODO: use delegation?
    public void intervalAdded(ListDataEvent e) {
    // intentionally left blank
    }

    public void intervalRemoved(ListDataEvent e) {
    // intentionally left blank
    }

    public void contentsChanged(ListDataEvent e) {
    // intentionally left blank
    }

    public void engineConfigurationChanged() {
    // intentionally left blank
    }

    public void filterChanged() {
    // intentionally left blank
    }

    public void worksheetModelNameChanged() {
        this.setTitle(worksheetWindowModel.getName());
    }

    //
    private static class MySubwindow extends Subwindow implements SubwindowModelListener {

        private final SubwindowModel model;

        public MySubwindow(JComponent child, SubwindowModel model) {
            super(child, model.isCollapsed());
            this.model = model;
            setVisible(model.isVisible());
            model.addSubwindowModelListener(this);
        }

        public void visibleStateChanged() {
            setVisible(model.isVisible());
        }

        //
        @Override
        public void setCollapsed(boolean collapsed) {
            model.setCollapsed(collapsed);
        }

        public void collapsedStateChanged() {
            super.setCollapsed(model.isCollapsed());
        }

        public SubwindowModel getModel() {
            return model;
        }
    }

    private static class MyListener implements SubwindowModelListener {

        //
        private final SubwindowModel model;
        //
        private final JMenu menu;

        public MyListener(SubwindowModel model, JMenu menu) {
            this.model = model;
            this.menu = menu;
            menu.setEnabled(model.isVisible());
            model.addSubwindowModelListener(this);
        }

        public void visibleStateChanged() {
            menu.setEnabled(model.isVisible());
        }

        public void collapsedStateChanged() {
        //ignore
        }
    }

    //
    private class ProgressBar extends JProgressBar implements ChangeListener {

        public ProgressBar() {
            super();
            worksheetWindowModel.getEngineRunner().addChangeListener(this);
            setPreferredSize(new Dimension(60, getPreferredSize().height));
        }

        public void stateChanged(ChangeEvent e) {
            if (worksheetWindowModel.getEngineRunner().getStatus() == Status.STOPPED) {
                configPanel.setEnabled(true);
                setIndeterminate(false);
            } else {
                configPanel.setEnabled(false);
                try {
                    Thread.sleep(500);
                    setIndeterminate(true);
                } catch (InterruptedException ex) {
                //TODO: document this
                }
            }
        }
    }

    private class ViewHistory extends SimpleAction {

        public ViewHistory() {
            super(ResourceBundle.getBundle(WORKSHEET_BUNDLE_NAME),
                    "menu.conjecture.viewhistory.text", null);
        }

        public void actionPerformed(ActionEvent e) {
            ConjectureHistoryOverview.getView(worksheetWindowModel.getConjectureHistoryListModel()).setVisible(true);
        }
    }
}
