/* SvgIconManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.icons;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import javax.imageio.ImageIO;
import javax.swing.Icon;
import javax.swing.ImageIcon;

import org.apache.batik.transcoder.TranscoderException;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.image.ImageTranscoder;

import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;

/**
 * Handles creation of icons from svg files.
 */
public class SvgIconManager {
    
    /**
     * Default constructor. Can be overridden, but should not
     * be called directly.
     * @see #getInstance
     */
    protected SvgIconManager() { }
    
    //
    private static final SvgIconManager SINGLETON
            = new SvgIconManager();
    
    /**
     * Return the sole instance of this class.
     */
    public static SvgIconManager getInstance() {
        return SINGLETON;
    }

    //
    public SelectUnselectIcon createSelectUnselectIcon(String iconPath, String selectedIconPath, int size) {
        Icon icon = getIcon(iconPath, size);
        Icon selectedIcon = getIcon(selectedIconPath, size);
        return new DefaultSelectUnselectIcon(icon, selectedIcon);
    }
    
    /**
     * Returns an imageicon with the given size for the svg file given.
     * @param iconPath Absolute classpath to the svg file
     * @param size Size in pixels of the resulting icon
     */
    public ImageIcon getIcon(String iconPath, int size) {
        return getIcon(iconPath, size, size);
    }
    
    //
    public ImageIcon getIcon(String iconPath, int width, int height) {
      return new ImageIcon(getImage(iconPath, width, height));
    }

    //
    private Image getImage(String imagePath, int width, int height) {
        Image image = getCachedImage(imagePath, width, height);
        if (image != null) {
            return image;
        } else {
            BufferedImage bufferedImage = renderImage(SvgIconManager.class.getResource(imagePath), width, height);
            putCachedImage(bufferedImage, imagePath, width, height);
            return bufferedImage;
        }
    }
    
    //
    private BufferedImage renderImage(URL imageURL, int width, int height) {
        BufferedImageTranscoder t = new BufferedImageTranscoder();
        t.setDimensions(width, height);
        try {
            URI uri = imageURL.toURI();
            if (uri != null)
                t.transcode(new TranscoderInput(uri.toString()), null);
        } catch (URISyntaxException ex) {
            //TODO: log
        } catch (TranscoderException ex) {
            //TODO: log
        }
        return t.getBufferedImage();
    }
    
    private String getCachedFilename(String imagePath, int width, int height) {
        // drop leading / and replace the others with _
        return imagePath.substring(1).replace('/','_') + "-" + width + "-" + height + ".png";
    }
    
    private File getCachedFile(String imagePath, int width, int height) {
        return new File (getCacheDir() + "/" + getCachedFilename(imagePath, width, height));
    }
    
    public InputStream getImageAsStream(String imagePath, int width, int height){
        getImage(imagePath, width, height); //make sure cached file exists
        File cachedFile = getCachedFile(imagePath, width, height);
        if(cachedFile.exists())
            try {
                return new FileInputStream(cachedFile);
            } catch (FileNotFoundException ex) {
                //
            }
        //when it is possible to disable caching, the image should be returned as stream here
        return null;
    }
    
    //
    private Image getCachedImage(String imagePath, int width, int height) {
        File cachedFile = getCachedFile(imagePath, width, height);
        if (cachedFile.exists()) {
            try {
                Image image = ImageIO.read(cachedFile.getAbsoluteFile());
                return image;
            } catch (IOException ex) {
                //TODO: log/handle
                return null;
            }
        } else {
            return null;
        }
    }
    
    //
    private void putCachedImage(BufferedImage image, String imagePath, int width, int height) {
        if (imagePath.charAt(0) != '/') {
            return; //only cache absolute path images
        }
        if (checkCacheDir()) {
            File cachedFile = getCachedFile(imagePath, width, height);
            try {
                ImageIO.write(image, "png", cachedFile.getAbsoluteFile());
            } catch (IOException ex) {
                //TODO: handle exception
            }
        }
    }
    
    private String getCacheDir() {
        return GrinvinPreferences.getInstance().getStringPreference(Preference.GRINVIN_CACHE_DIR)+"/icons";
    }
    
    //
    private boolean checkCacheDir() {
        File cacheDir = new File(getCacheDir());
        if (cacheDir.exists()) {
            return cacheDir.isDirectory();
        } else {
            return cacheDir.mkdirs();
        }
    }
    
    //
    private static class BufferedImageTranscoder extends ImageTranscoder {
        
        //
        private BufferedImage bufferedImage;
        
        //
        public BufferedImage createImage(int width, int height) {
            return new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        }
        
        //
        public void writeImage(BufferedImage img, TranscoderOutput output) throws TranscoderException {
            bufferedImage = img;
        }
        
        //
        public BufferedImage getBufferedImage() {
            return bufferedImage;
        }
        
        //
        public void setDimensions(int w, int h) {
            hints.put(KEY_WIDTH, Float.valueOf(w));
            hints.put(KEY_HEIGHT, Float.valueOf(h));
        }
    }

}
