/* JTabbedPaneMouseListener.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui;

import java.awt.Component;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.HashMap;
import java.util.Map;
import javax.swing.JPopupMenu;
import javax.swing.JTabbedPane;
import javax.swing.plaf.TabbedPaneUI;

/**
 * Class that listens to mouse clicks on a JTabbedPane and displays
 * a popup menu corresponding to the clicked tab.
 */
public class JTabbedPaneMouseListener implements MouseListener {
    
    //
    private JTabbedPane tabbedPane;
    
    //
    private Map<Component,JPopupMenu> popups = new HashMap<Component,JPopupMenu>();
    
    //
    private JPopupMenu defaultPopup;
    
    /**
     * Creates a new instance of JTabbedPaneMouseListener and
     * registers it as listener with tabbedPane.
     */
    public JTabbedPaneMouseListener(JTabbedPane tabbedPane) {
        this.tabbedPane = tabbedPane;
        tabbedPane.addMouseListener(this);
    }
    
    /**
     * add a popup for the tab corresponding to the Component in the JTabbedpane.
     */
    public void addPopup(Component comp, JPopupMenu popup){
        if(tabbedPane.indexOfComponent(comp)!=-1)
            popups.put(comp, popup);
    }
    
    /**
     * sets the default popup to be shown when the user clicks on a tab with no popup
     * menu or doesn't click on the header of a tab.
     */
    public void setDefaultPopup(JPopupMenu defaultPopup){
        this.defaultPopup = defaultPopup;
    }
    
    private void displayPopup(MouseEvent e){
        TabbedPaneUI ui = tabbedPane.getUI();
        boolean showDefault = true;
        if (ui != null) {
            int index = ui.tabForCoordinate(tabbedPane, e.getX(), e.getY());
            if (index != -1) {
                JPopupMenu popup = popups.get(tabbedPane.getComponentAt(index));
                if(popup!=null){
                    popup.show(tabbedPane, e.getX(), e.getY());
                    showDefault = false;
                }
            }
        }
        if(showDefault && defaultPopup!=null)
            defaultPopup.show(tabbedPane, e.getX(), e.getY());
    }
    
    public void mouseClicked(MouseEvent e) {
    }
    
    public void mousePressed(MouseEvent e) {
        if (e.isPopupTrigger() && tabbedPane.equals(e.getComponent())) {
            displayPopup(e);
        }
    }
    
    public void mouseReleased(MouseEvent e) {
        if (e.isPopupTrigger() && tabbedPane.equals(e.getComponent())) {
            displayPopup(e);
        }
    }
    
    public void mouseEntered(MouseEvent e) {
    }
    
    public void mouseExited(MouseEvent e) {
    }
    
}
