package com.ibm.bsf.util.event;

import java.util.Hashtable;
import com.ibm.bsf.util.event.generator.EventAdapterGenerator;

/**
 * The <em>EventAdapterRegistry</em> is the registry of event adapters.
 * If a desired adapter is not found, the adapter will be dynamically
 * generated when lookup is attempted. Set the <code>dynamic</code> property
 * to <code>false</code> to disable this feature.
 * <p>
 * This implementation first looks for an adapter in its lookup table
 * and if it doesn't find one looks for a standard implementation of
 * that adapter in the com.ibm.bsf.util.event.adapters package with a
 * standard naming convention. The naming convention it assumes is the
 * following: for event listener type <tt>a.b.c.FooListener</tt>, 
 * it loads an adapter of type 
 * <tt>com.ibm.bsf.util.event.adapters.a_b_c_FooAdapter</tt>.
 * If both the loading and the dynamic generation fail, then a
 * <code>null</code> is returned.
 * <p>
 * 
 * @author   Sanjiva Weerawarana
 * @author   Matthew J. Duftler
 * @see      EventAdapter
 */
public class EventAdapterRegistry {
  private static Hashtable reg = new Hashtable ();
  private static ClassLoader cl = null;
  private static String adapterPackage = "com.ibm.bsf.util.event.adapters";
  private static String adapterSuffix = "Adapter";
  private static boolean dynamic = true;

  /**
   * Class loader to use to load event adapter classes.
   */
  public static void setClassLoader (ClassLoader cloader) {
    cl = cloader;
  }

  /**
   * Indicates whether or not to dynamically generate adapters; default is
   * <code>true</code>.
   * <p>
   * If the <code>dynamic</code> property is set to true, and the
   * <code>ClassLoader</code> is unable to resolve an adapter, one will be
   * dynamically generated.
   * 
   * @param dynamic whether or not to dynamically generate adapters.
   */
  public static void setDynamic (boolean dynamic) {
    EventAdapterRegistry.dynamic = dynamic;
  }

  public static void register (Class listenerType, Class eventAdapterClass) {
    String key = listenerType.getName().replace('.', '_');
    reg.put (key, eventAdapterClass);
  }

  public static Class lookup (Class listenerType) {
    String key = listenerType.getName().replace ('.', '_');
    Class adapterClass = (Class) reg.get (key);
    
    if (adapterClass == null) {
      String en = key.substring (0, key.lastIndexOf ("Listener"));
      String cn = adapterPackage + "." + en + adapterSuffix;

      try {
        // Try to resolve one.
        adapterClass = (cl != null) ? cl.loadClass (cn) : Class.forName (cn);
      } catch (ClassNotFoundException e) {
        if (dynamic) {
          // Unable to resolve one, try to generate one.
          adapterClass =
            EventAdapterGenerator.makeEventAdapterClass (listenerType, false);
        }
      }

      if (adapterClass != null) {
        reg.put (key, adapterClass);
      }
    }

    return adapterClass;
  }
}