/*********************************************************************
 *
 *	Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Utilities
 * MODULE:      CacheMap Test
 * FILE:        TestCacheMap.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      01/09/99        Initial version of TestIntHashtable
 *      nf      10/20/99        Changed to test CacheTable
 *      nf      01/12/02        Changed to test CacheMap
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the CacheMap class.
 *
 * $Id: TestCacheMap.java,v 1.1 2002/01/12 20:11:14 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.util.test;

import com.bluemarsh.util.CacheMap;
import java.util.*;

/**
 * Test-wrapper for the CacheMap class.
 *
 * @author  Nathan Fiedler
 */
public class TestCacheMap {
    /** You may have to change this value as the JVM may simply clear
     * out everything that is removable, leaving us with nothing. */
    private static final int MAX_ENTRIES = 20000;

    /**
     * Test a large map of elements.
     */
    protected static void testLarge() {
        Runtime rt = Runtime.getRuntime();
        System.out.println("! available memory (bytes): " + rt.freeMemory());
        System.out.println("! calling Runtime.gc()...");
        rt.gc();
        System.out.println("! available memory (bytes): " + rt.freeMemory());

        System.out.println("! inserting " + MAX_ENTRIES + " elements...");

        CacheMap table = new CacheMap();
        for (int ii = 1; ii <= MAX_ENTRIES; ii++) {
            table.put(new Integer(ii), String.valueOf(ii * 2));
        }

        System.out.println("! retrieving " + MAX_ENTRIES + " elements...");
        int good = 0;
        int bad = 0;
        for (int ii = 1; ii <= MAX_ENTRIES; ii++) {
            Object o = table.get(new Integer(ii));
            if (o != null) {
                good++;
            } else {
                bad++;
            }
        }
        System.out.println("! good (non-null) entries: " + good);
        System.out.println("! bad (null) entries: " + bad);
        System.out.println("! available memory (bytes): " + rt.freeMemory());
    } // testLarge

    /**
     * Test the maximum size property of the map.
     *
     * @param  size  maximum size.
     */
    protected static void testMaxSize(CacheMap table, int size) {
        table.clear();
        table.setMaximumSize(size);
        int more = size * 5;
        for (int ii = 1; ii <= more; ii++) {
            table.put(new Integer(ii), String.valueOf(ii * 2));
        }
    } // testMaxSize

    /**
     * Main method, tests the CacheMap class.
     *
     * @param  args  command-line string arguments
     */
    public static void main(String[] args) {
        System.out.println();
        System.out.println("  + = good");
        System.out.println("  - = bad");
        System.out.println("  ! = information");
        System.out.println("  ? = verify manually");
        System.out.println();

        CacheMap table = new CacheMap();
        if (table.isEmpty()) {
            System.out.println("+ table is empty");
        } else {
            System.out.println("- table is not empty, what's up with that?");
            return;
        }

        System.out.println("! populating table with 1000 elements");
        for (int ii = 1; ii <= 1000; ii++) {
            // create the key object
            Integer intObj = new Integer(ii);
            // create a value object that is the reverse of twice the key
            String strObj = String.valueOf(ii * 2);
            StringBuffer buf = new StringBuffer(strObj);
            buf.reverse();
            table.put(intObj, buf.toString());
        }

        if (!table.isEmpty()) {
            System.out.println("+ table is not empty");
        } else {
            System.out.println("- table is empty, what's up with that?");
            return;
        }

        System.out.println("? table hash = " + table.hashCode());
        if (table.equals(table)) {
            System.out.println("+ table equals itself");
        } else {
            System.out.println("- table does not equal itself, eh?");
            return;
        }

        if (table.size() == 1000) {
            System.out.println("+ table has 1000 elements");
        } else {
            System.out.println("- table count is not 1000");
            return;
        }

        System.out.println("! searching for all 1000 elements");
        boolean success = true;
        for (int ii = 1; ii <= 1000; ii++) {
            String str = (String) table.get(new Integer(ii));
            if (str == null) {
                System.out.println("- could not find element " + ii);
                success = false;
                // This may be a garbage collected value.
            }
        }
        if (success) {
            System.out.println("+ found all 1000 elements");
        } else {
            return;
        }

        System.out.println("! checking for 1000 keys in key set");
        Set keys = table.keySet();
        if (!keys.isEmpty() && keys.size() == 1000) {
            System.out.println("+ key set has 1000 keys");
        } else {
            return;
        }

        if (table.containsKey(new Integer(55))) {
            System.out.println("+ containsKey() seems to work");
        } else {
            System.out.println("- containsKey() failed");
            return;
        }

        if (table.containsValue(new String("011"))) {
            System.out.println("+ containsValue() seems to work");
        } else {
            System.out.println("- containsValue() failed to find '011'");
            return;
        }

        Set entries = table.entrySet();
        if (entries.size() == 1000) {
            System.out.println("+ entrySet() seems to work");
        } else {
            System.out.println("- entrySet() failed");
            return;
        }

        Collection values = table.values();
        if (values.size() == 1000) {
            System.out.println("+ values() seems to work");
        } else {
            System.out.println("- values() failed");
            return;
        }

        System.out.println("! removing every fifth element...");
        for (int ii = 1; ii <= 1000; ii += 5) {
            Integer igr = new Integer(ii);
            table.remove(igr);
        }
        if (table.size() == 800) {
            System.out.println("+ size is now 800");
        } else {
            System.out.println("- size is not 800");
            return;
        }

        System.out.println("! adding 100 more elements...");
        Map content = new HashMap();
        for (int ii = 1001; ii < 1101; ii++) {
            content.put(new Integer(ii), String.valueOf(ii * 2));
        }
        table.putAll(content);
        if (table.size() == 900) {
            System.out.println("+ size is now 900");
        } else {
            System.out.println("- size is not 900");
            return;
        }

        System.out.println("! clearing table");
        table.clear();
        if (table.size() == 0 && table.isEmpty()) {
            System.out.println("+ table is empty");
        } else {
            System.out.println("- table is not empty, warning, warning!!!");
            return;
        }

        testLarge();

        System.out.println("! testing maximum size of table");
        testMaxSize(table, 10);
        if (table.size() != 10) {
            System.out.println("- table is wrong size, should be 10, is " +
                               table.size());
            return;
        }
        testMaxSize(table, 5);
        if (table.size() != 5) {
            System.out.println("- table is wrong size, should be 5, is " +
                               table.size());
            return;
        }
        testMaxSize(table, 20);
        if (table.size() != 20) {
            System.out.println("- table is wrong size, should be 20, is " +
                               table.size());
            return;
        }
        System.out.println("+ maximum size feature seems to work");

        // Do this again after the heap has grown.
        testLarge();
    } // main
} // TestCacheMap
