/********************************************************************
 *
 *	Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Test Priority List
 * FILE:        TestPriorityList.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      10/02/99        Initial version
 *      nf      01/05/02        Greatly improved
 *
 * DESCRIPTION:
 *      Implements the TestPriorityList class to test the PriorityList
 *      class defined in PriorityList.java.
 *
 * $Id: TestPriorityList.java,v 1.4 2002/01/06 07:12:18 nfiedler Exp $
 *
 *******************************************************************/

package com.bluemarsh.adt.test;

import com.bluemarsh.adt.PriorityList;
import java.util.*;

/**
 * Test wrapper for the priority list class.
 *
 * @author  Nathan Fiedler
 */
public class TestPriorityList {

    /**
     * Test wrapper for PriorityList class.
     *
     * @param  args  array of string arguments
     */
    public static void main(String args[]) {
        System.out.println();
        System.out.println("  + = good");
        System.out.println("  - = bad");
        System.out.println("  ! = information");
        System.out.println("  ? = verify manually");
        System.out.println();

        PriorityList list = new PriorityList();
        if (list.isEmpty() && list.size() == 0) {
            System.out.println("+ list is empty");
        } else {
            System.out.println("- list is not empty, what's up with that?");
            return;
        }

        System.out.println("! populating list");
        for (int i = 0; i < 1000; i++) {
            list.add(new Integer(i), i);
        }

        if (!list.isEmpty()) {
            System.out.println("+ list is not empty");
        } else {
            System.out.println("- list is empty, what's up with that?");
            return;
        }

        System.out.println("? list hash = " + list.hashCode());
        if (list.equals(list)) {
            System.out.println("+ list equals itself");
        } else {
            System.out.println("- list does not equal itself, eh?");
            return;
        }

        if (list.size() == 1000) {
            System.out.println("+ list has 1000 elements");
        } else {
            System.out.println("- list count is not 1000");
            return;
        }

        Integer igr;
        System.out.println("! searching for all 1000 elements");
        boolean success = true;
        for (int i = 0; i < 1000; i++) {
            igr = (Integer) list.get(i);
            if (igr == null || igr.intValue() != (999 - i)) {
                System.out.println("- could not find element " + i);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ found all 1000 elements");
        } else {
            return;
        }

        igr = new Integer(500);
        int idx = list.indexOf(igr);
        int lidx = list.lastIndexOf(igr);
        if (idx == 499) {
            System.out.println("+ indexOf() seems to work");
        } else {
            System.out.println("- indexOf() did not get 499: got " + idx);
            return;
        }
        if (lidx == 499) {
            System.out.println("+ lastIndexOf() seems to work");
        } else {
            System.out.println("- lastIndexOf() did not get 499: got " + lidx);
            return;
        }

        if (list.contains(igr) && !list.contains(new Integer(12345))) {
            System.out.println("+ contains() seems to work");
        } else {
            System.out.println("- contains() failed");
            return;
        }

        List contents = new ArrayList();
        for (int ii = 0; ii < 1000; ii += 10) {
            contents.add(new Integer(ii));
        }
        if (list.containsAll(contents)) {
            System.out.println("+ containsAll() seems to work");
        } else {
            System.out.println("- containsAll() failed");
            return;
        }

        System.out.println("! testing clone()");
        PriorityList clone = (PriorityList) list.clone();
        if (!clone.isEmpty()) {
            System.out.println("+ clone is not empty");
        } else {
            System.out.println("- clone is empty, what's up with that?");
            return;
        }

        System.out.println("? clone hash = " + clone.hashCode());
        if (clone.equals(list)) {
            System.out.println("- clone equals original, eh?");
            return;
        } else {
            System.out.println("+ clone does not equal original");
        }

        if (clone.size() == 1000) {
            System.out.println("+ clone has 1000 elements");
        } else {
            System.out.println("- clone count is not 1000");
            return;
        }

        System.out.println("! clearing the clone");
        clone.clear();
        if (clone.isEmpty() && clone.size() == 0) {
            System.out.println("+ clone is empty");
        } else {
            System.out.println("- clone is not empty, what's up with that?");
            return;
        }

        list.addAll(contents, 12345);
        if (list.size() == 1100) {
            System.out.println("+ addAll(Collection, int) seems to work");
        } else {
            System.out.println("- addAll(Collection, int) failed");
            return;
        }
        list.remove(12345);
        if (list.size() == 1000) {
            System.out.println("+ remove(int) seems to work");
        } else {
            System.out.println("- remove(int) failed");
            return;
        }
        igr = (Integer) list.get(500);
        System.out.println("! removed element " + igr);
        list.remove(igr);
        if (list.size() == 999) {
            System.out.println("+ remove(Object) seems to work");
        } else {
            System.out.println("- remove(Object) failed");
            return;
        }
        list.removeAll(contents);
        if (list.size() == 899) {
            System.out.println("+ removeAll() seems to work");
        } else {
            System.out.println("- removeAll() failed: size = " + list.size());
            return;
        }

        System.out.println("! recreating list of 100 elements");
        list = new PriorityList();
        for (int i = 0; i < 100; i++) {
            list.add(new Integer(i), i);
        }

        List sub = list.subList(30, 80);
        if (sub.isEmpty() || sub.size() != 50) {
            System.out.println("- subList size is not 50");
        }
        for (int ii = sub.size() - 1, jj = 20; ii >= 0; ii--, jj++) {
            igr = (Integer) sub.get(ii);
            if (igr == null || igr.intValue() != jj) {
                System.out.println("- subList element incorrect: expected " +
                                   jj + " at " + ii + ", got " + igr);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ subList() seems to work");
        } else {
            System.out.println("- subList() failed");
            return;
        }

        System.out.println("! testing toArray()...");
        Object[] arr = list.toArray();
        for (int ii = 0; ii < arr.length; ii++) {
            if (((Integer) arr[ii]).intValue() != (arr.length - ii - 1)) {
                System.out.println("- array element incorrect " + arr[ii] +
                                   " at " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ toArray() seems to work");
        } else {
            System.out.println("- toArray() failed");
            return;
        }

        System.out.println("! testing toArray(Object[])...");
        Integer[] irr = (Integer[]) list.toArray(new Integer[100]);
        for (int ii = 0; ii < irr.length; ii++) {
            if (irr[ii].intValue() != (arr.length - ii - 1)) {
                System.out.println("- irray element incorrect " + irr[ii] +
                                   " at " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ toArray(Object[]) seems to work");
        } else {
            System.out.println("- toArray(Object[]) failed");
            return;
        }

        Iterator iter = list.iterator();
        int jj = 99;
        while (iter.hasNext()) {
            igr = (Integer) iter.next();
            if (igr.intValue() != jj) {
                System.out.println("- iterator element incorrect at " + jj);
                success = false;
                break;
            }
            jj--;
        }
        if (success) {
            System.out.println("+ iterator() seems to work");
        } else {
            return;
        }

        ListIterator liter = list.listIterator();
        jj = 0;
        while (liter.hasNext()) {
            if (liter.nextIndex() != jj) {
                System.out.println("- listIterator nextIndex incorrect at " +
                                   jj + ", got " + liter.nextIndex());
                success = false;
                break;
            }
            igr = (Integer) liter.next();
            if (igr.intValue() != (99 - jj)) {
                System.out.println("- listIterator next element incorrect at " + jj);
                success = false;
                break;
            }
            jj++;
        }
        if (success) {
            System.out.println("+ listIterator() next seems to work");
        } else {
            return;
        }

        while (liter.hasPrevious()) {
            jj--;
            if (liter.previousIndex() != jj) {
                System.out.println("- listIterator previousIndex incorrect at " + jj);
                success = false;
                break;
            }
            igr = (Integer) liter.previous();
            if (igr.intValue() != (99 - jj)) {
                System.out.println("- listIterator previous element incorrect at " + jj);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ listIterator() previous seems to work");
        } else {
            return;
        }

        liter = list.listIterator(10);
        if (liter.nextIndex() == 10) {
            System.out.println("+ listIterator(int) seems to work");
        } else {
            System.out.println("- listIterator(int) failed");
            return;
        }
        liter.next();
        liter.remove();
        if (list.size() == 99) {
            System.out.println("+ ListIterator.remove() seems to work");
        } else {
            System.out.println("- ListIterator.remove() failed");
            return;
        }

        igr = (Integer) list.set(30, new Integer(321));
        Integer nigr = (Integer) list.get(30);
        if (igr != null && igr.intValue() == 68 &&
            nigr != null && nigr.intValue() == 321) {
            System.out.println("+ set() seems to work");
        } else {
            System.out.println("- set() failed: igr = " + igr +
                               ", nigr = " + nigr);
            return;
        }

        System.out.println();
        System.out.println("+ Tests completed successfully.");
    } // main
} // TestPriorityList
