/*********************************************************************
 *
 *	Copyright (C) 2000 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Utilities
 * MODULE:      StringTokenizer Test
 * FILE:        TestStringTokenizer.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      07/16/00        Initial version
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the StringTokenizer class.
 *
 * $Id: TestStringTokenizer.java,v 1.3 2000/07/17 01:16:26 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.util.test;

import com.bluemarsh.util.StringTokenizer;

/**
 * Test-wrapper for the StringTokenizer class.
 *
 * @author  Nathan Fiedler
 * @version 1.0  7/16/00
 */
public class TestStringTokenizer {

    /**
     * Shows the count of tokens and lists the tokens from each
     * string tokenizer.
     *
     * @param  mine    My StringTokenizer.
     * @param  theirs  The java.util.StringTokenizer.
     */
    public static void test(StringTokenizer mine,
                            java.util.StringTokenizer theirs) {
        System.out.println("my    count = " + mine.countTokens());
        System.out.println("their count = " + theirs.countTokens());
        System.out.print("my    tokens: ");
        while (mine.hasMoreTokens()) {
            System.out.print(mine.nextToken());
            System.out.print(", ");
        }
        System.out.println();
        System.out.print("their tokens: ");
        while (theirs.hasMoreTokens()) {
            System.out.print(theirs.nextToken());
            System.out.print(", ");
        }
        System.out.println();
    } // test

    /**
     * Main method, tests the StringTokenizer class.
     *
     * @param  args  command-line string arguments
     */
    public static void main(String[] args) {
        String teststr = "test string \r\f\nyippee";
        StringTokenizer mine = new StringTokenizer(teststr, null);
        java.util.StringTokenizer theirs =
            new java.util.StringTokenizer(teststr);
        test(mine, theirs);

        // Average string, separated by single delimiters.
        teststr = ":yes:no:maybe:";
        mine = new StringTokenizer(teststr, ":");
        theirs = new java.util.StringTokenizer(teststr, ":");
        test(mine, theirs);

        // String with doubled delimiters.
        teststr = ";;yes\\no:maybe::";
        mine = new StringTokenizer(teststr, ":\\;");
        theirs = new java.util.StringTokenizer(teststr, ":\\;");
        test(mine, theirs);

        // String with no valid tokens.
        teststr = ";;\\:::";
        mine = new StringTokenizer(teststr, ":\\;");
        theirs = new java.util.StringTokenizer(teststr, ":\\;");
        test(mine, theirs);

        // String with one token.
        teststr = "token";
        mine = new StringTokenizer(teststr);
        theirs = new java.util.StringTokenizer(teststr);
        test(mine, theirs);

        // Empty string.
        teststr = "";
        mine = new StringTokenizer(teststr);
        theirs = new java.util.StringTokenizer(teststr);
        test(mine, theirs);

        // Null string.
        teststr = null;
        mine = new StringTokenizer(teststr);
        try {
            theirs = new java.util.StringTokenizer(teststr);
        } catch (NullPointerException npe) {
            theirs = new java.util.StringTokenizer("");
        }
        test(mine, theirs);

        // Average string, separated by single delimiters.
        // Returns delimiters.
        teststr = ":yes:no:maybe:";
        mine = new StringTokenizer(teststr, ":", true);
        theirs = new java.util.StringTokenizer(teststr, ":", true);
        test(mine, theirs);

        // Test using the nextToken(String) method.
        teststr = "first:yes;no$maybe";
        mine = new StringTokenizer(teststr, ":");
        theirs = new java.util.StringTokenizer(teststr, ":");
        System.out.println("my    count = " + mine.countTokens());
        System.out.println("their count = " + theirs.countTokens());
        System.out.print("my    tokens: ");
        System.out.print(mine.nextToken());
        System.out.print(", ");
        System.out.print(mine.nextToken(";"));
        System.out.print(", ");
        System.out.print(mine.nextToken("$"));
        System.out.print(", ");
        System.out.println(mine.nextToken());
        System.out.print("their tokens: ");
        System.out.print(theirs.nextToken());
        System.out.print(", ");
        System.out.print(theirs.nextToken(";"));
        System.out.print(", ");
        System.out.print(theirs.nextToken("$"));
        System.out.print(", ");
        System.out.println(theirs.nextToken());

        // Test the peek function.
        mine = new StringTokenizer("yes:no:maybe", ":");
        System.out.println("peek = " + mine.peek());
        System.out.println("next = " + mine.nextToken());
        System.out.println("peek = " + mine.peek());
        System.out.println("next = " + mine.nextToken());
        System.out.println("peek = " + mine.peek());
        System.out.println("next = " + mine.nextToken());

        // Test the rest function.
        mine = new StringTokenizer("first$second$third$fourth$fifth$$$", "$");
        System.out.println("first = " + mine.nextToken());
        System.out.println("rest = " + mine.rest());
        mine = new StringTokenizer("first$second$third$fourth$fifth$$$", "$");
        System.out.println("first = " + mine.nextToken());
        System.out.println("rest (trimmed) = " + mine.rest(true));

        // Test the restTrim function.
        mine = new StringTokenizer("first$  second$third$fourth$fifth  ", "$");
        System.out.println("first = " + mine.nextToken());
        System.out.println("rest = ***" + mine.restTrim() + "***");
    } // main
} // TestStringTokenizer
