/*
 * Apple built-in iSight firmware loader
 *
 * Copyright © 2006 Ronald S. Bultje <rbultje@ronald.bitfreak.net>
 * Copyright © 2007 Étienne Bersac <bersace@gmail.com>
 *
 *
 * Special thanks to Johannes Berg <johannes@sipsolutions.net> for helping
 * to find out how to load the firmware; see his website on
 * http://johannes.sipsolutions.net/MacBook/iSight for details.
 * Basic structure of firmware data: <len:2><off:2><data:len>, where
 * the $data of size $len is to be put at position $off in the device.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <hal/libhal.h>

#include "load.h"

gint
main(gint argc, gchar *argv[])
{
	struct usb_device *udev;
	/* TODO: parametize. Use fdi? */
	gchar* firmware = "/lib/firmware/isight.fw";
	const gchar *udi = g_getenv("UDI");
	LibHalContext *hal;
	DBusError derr;
	const gchar *bus, *dev;

	openlog("ift-callout", LOG_PERROR, LOG_USER);

#if ENABLE_NLS
	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);
#endif

	gchar *appname = _("HAL built-in iSight firmware loader");

	g_set_application_name(appname);
	g_set_prgname("ift-callout");

	if (g_access(firmware, R_OK)) {
		ift_error(_("Unable to read firmware %s."), firmware);
	}

	dbus_error_init(&derr);
	hal = libhal_ctx_init_direct(&derr);

	/* DEVICE IDS */
	/* this is ubuntu/debian specific */
	dbus_error_init(&derr);
	bus = g_strdup_printf("%03d",
			      libhal_device_get_property_int(hal, udi,
							     "usb.bus_number",
							     &derr));
	dbus_error_init(&derr);
	dev = g_strdup_printf("%03d",
			      libhal_device_get_property_int(hal, udi,
							     "usb.linux.device_number",
							     &derr));

	ift_debug("%s:%s", bus, dev);

	/* init usb */
	usb_init();
	if (usb_find_busses() == 0) {
		ift_error(_("No USB busses found"));
	} else if (usb_find_devices() == 0) {
		ift_error(_("No USB devices found"));
	}


	udev = find_usb_device(bus, dev);
	if (!dev)
		ift_error(_("Device %s:%s not found"), bus, dev);

	if (load_firmware(udev, firmware) == -1)
		ift_error(_("Failed to upload firmware to %s:%s (%x:%x)"),
			  bus, dev,
			  libhal_device_get_property_int(hal, udi, "usb.vendor_id", NULL),
			  libhal_device_get_property_int(hal, udi, "usb.product_id", NULL));
	else
		ift_message(_("Firmware loaded succesfully to %s:%s"),
			    bus, dev);

	closelog();

	return 0;
}

