/*
 * simple file browser
 * (c) 2001 Gerd Knorr <kraxel@bytesex.org>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/extensions/XShm.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/ScrolledW.h>
#include <Xm/SelectioB.h>

#include "RegEdit.h"
#include "ida.h"
#include "loader.h"
#include "viewer.h"
#include "browser.h"
#include "filter.h"
#include "x11.h"
#include "dither.h"
#include "selections.h"

static int icon_width  = 64;
static int icon_height = 48;

/*----------------------------------------------------------------------*/

struct browser_file {
    struct dirent dirent;
    char *filename;
    Widget rowcol,icon,name;
    int mode,unknown;
};

struct browser_handle {
    char *dirname;
    char *filter;
    Widget shell,viewport,rowcol,status;

    struct browser_file *list;
    int files;
    int dirs,sfiles,afiles;

    XtWorkProcId      wproc;
    struct ida_loader *loader;
    void              *wdata;
    struct ida_image  wimg;
    struct ida_image  simg;
    int i,y,state;
};

/*----------------------------------------------------------------------*/

static void browser_cd(Widget widget, XtPointer clientdata,
		       XtPointer call_data);
static void browser_file(Widget widget, XtPointer clientdata,
			 XtPointer call_data);
static void browser_drag(Widget widget, XtPointer clientdata,
			 XtPointer call_data);

/*----------------------------------------------------------------------*/

struct browser_cache {
    struct browser_cache *next;
    char *path;
    Pixmap pix;
};

static struct browser_cache *bcache = NULL;;

static void browser_cache_add(char *path, Pixmap pix)
{
    struct browser_cache *item;

    item = malloc(sizeof(*item));
    memset(item,0,sizeof(*item));
    item->next = bcache;
    item->path = strdup(path);
    item->pix = pix;
    bcache = item;
}

static Pixmap browser_cache_get(char *path)
{
    struct browser_cache *item;

    for (item = bcache; item != NULL; item = item->next)
	if (0 == strcmp(path,item->path))
	    return item->pix;
    return 0;
}

/*----------------------------------------------------------------------*/

static int browser_cmpfile(const void *a, const void *b)
{
    const struct browser_file *aa = a;
    const struct browser_file *bb = b;

    if (S_ISDIR(aa->mode) != S_ISDIR(bb->mode))
	return S_ISDIR(aa->mode) ? -1 : 1;
    return strcmp(aa->dirent.d_name,bb->dirent.d_name);
}

static int browser_addfile(struct browser_handle *h,
			   struct browser_file *l)
{
    XmString str;

    str = XmStringGenerate(l->dirent.d_name, NULL, XmMULTIBYTE_TEXT, NULL);
    l->rowcol = XtVaCreateWidget(l->dirent.d_name,
				 xmRowColumnWidgetClass,h->rowcol,
				 NULL);
    l->icon = XtVaCreateManagedWidget("icon",xmPushButtonWidgetClass,l->rowcol,
				      XtNwidth,icon_width+4,
				      XtNheight,icon_height+4,
				      XmNrecomputeSize,False,
				      NULL);
    l->name = XtVaCreateManagedWidget("name",xmLabelWidgetClass,l->rowcol,
				      XmNlabelString,str,
				      NULL);
    XmStringFree(str);
    return 0;
}

static Boolean
browser_readfiles(XtPointer clientdata)
{
    struct browser_handle *h = clientdata;
    struct op_resize_parm resize;
    struct ida_rect rect;
    struct ARGS save;
    XmString str;
    char blk[512],line[512];
    FILE *fp;
    Pixmap pix;
    int l;
    float xs,ys,scale;

    if (h->i == h->files) {
	/* all done */
	h->wproc = 0;
	if (h->wimg.data)
	    free(h->wimg.data);
	h->wimg.data = NULL;
	if (h->simg.data)
	    free(h->simg.data);
	h->simg.data = NULL;

	if (h->filter) {
	    sprintf(line,"%d dirs, %d/%d files [%s]",
		    h->dirs,h->sfiles,h->afiles,h->filter);
	} else {
	    sprintf(line,"%d dirs, %d files",
		    h->dirs,h->afiles);
	}
	str  = XmStringGenerate(line, NULL, XmMULTIBYTE_TEXT, NULL);
	XtVaSetValues(h->status,XmNlabelString,str,NULL);
	XmStringFree(str);

#if 0
	/* unmap unknown regular files */
	{
	    WidgetList list;
	    int i,n;
	    
	    list = malloc(sizeof(Widget*)*h->files);
	    for (i = 0, n = 0; i < h->files; i++)
		if (h->list[i].unknown)
		    list[n++] = h->list[i].rowcol;
	    XtUnmanageChildren(list,n);
	    free(list);
	}
#endif
	
	return TRUE;
    }

    switch (h->state) {
    case 0:
	if ((h->list[h->i].mode & S_IFMT) != S_IFREG)
	    goto next;

	/* check cache */
	pix = browser_cache_get(h->list[h->i].filename);
	if (0 != pix) {
	    XtVaSetValues(h->list[h->i].icon,
			  XmNlabelPixmap,pix,
			  XmNlabelType,XmPIXMAP,
			  NULL);
	    goto next;
	}

	/* open file */
	if (NULL == (fp = fopen(h->list[h->i].filename, "r"))) {
	    if (debug)
		fprintf(stderr,"open %s: %s\n",h->list[h->i].filename,
			strerror(errno));
	    goto unknown;
	}
	if (debug)
	    fprintf(stderr,"OPENED: %s\n",h->list[h->i].filename);
	memset(blk,0,sizeof(blk));
	fread(blk,1,sizeof(blk),fp);
	rewind(fp);
	
	/* pick loader */
	for (l = 0;; l++) {
	    h->loader = loaders[l];
	    if (NULL == h->loader) {
		if (debug)
		    fprintf(stderr,"%s: unknown format\n",
			    h->list[h->i].filename);
		fclose(fp);
		goto unknown;
	    }
	    if (NULL == h->loader->magic)
		continue;
	    if (0 == memcmp(blk+h->loader->moff,h->loader->magic,
			    h->loader->mlen))
		break;
	}
	
	/* load image */
	save = args;
	args.pcd_res = 1;
	pcd_res = args.pcd_res;
	h->wdata = h->loader->init(fp,h->list[h->i].filename,&h->wimg.i);
	args = save;
	pcd_res = args.pcd_res;
	if (NULL == h->wdata) {
	    if (debug)
		fprintf(stderr,"loading %s [%s] FAILED\n",
			h->list[h->i].filename,h->loader->name);
	    goto unknown;
	}
	if (h->wimg.data)
	    free(h->wimg.data);
	h->wimg.data = malloc(h->wimg.i.width * h->wimg.i.height * 3);

	sprintf(line,"reading %.450s [%dx%d] ...",
		strrchr(h->list[h->i].filename,'/')+1,
		h->wimg.i.width, h->wimg.i.height);
	str = XmStringGenerate(line, NULL, XmMULTIBYTE_TEXT, NULL);
	XtVaSetValues(h->status,XmNlabelString,str,NULL);
	XmStringFree(str);

	h->state = 1;
	h->y = 0;
	return FALSE;

    case 1:
	if (h->y < h->wimg.i.height) {
	    h->loader->read(h->wimg.data + 3*h->y*h->wimg.i.width,
			    h->y, h->wdata);
	    h->y++;
	    return FALSE;
	}
	h->loader->done(h->wdata);
	if (debug)
	    fprintf(stderr,"LOADED: %s [%dx%d]\n",
		    h->list[h->i].filename,h->wimg.i.width,h->wimg.i.height);
	
	/* resize image */
	xs = (float)icon_width  / h->wimg.i.width;
	ys = (float)icon_height / h->wimg.i.height;
	scale = (xs < ys) ? xs : ys;
	resize.width  = h->wimg.i.width  * scale;
	resize.height = h->wimg.i.height * scale;
	if (0 == resize.width)
	    resize.width = 1;
	if (0 == resize.height)
	    resize.height = 1;
	
	rect.x1 = 0;
	rect.x2 = h->wimg.i.width;
	rect.y1 = 0;
	rect.y2 = h->wimg.i.height;
	h->wdata = desc_resize.init(&h->wimg,&rect,&h->simg.i,&resize);
	if (h->simg.data)
	    free(h->simg.data);
	h->simg.data = malloc(h->simg.i.width * h->simg.i.height * 3);

	h->state = 2;
	h->y = 0;
	return FALSE;

    case 2:
	if (h->y < h->simg.i.height) {
	    desc_resize.work(&h->wimg,&rect,h->simg.data+3*h->simg.i.width*h->y,
			     h->y, h->wdata);
	    h->y++;
	    return FALSE;
	}
	desc_resize.done(h->wdata);
	if (debug)
	    fprintf(stderr,"SCALED: %s [%dx%d]\n",
		    h->list[h->i].filename,h->simg.i.width,h->simg.i.height);
	
	/* build, cache + install pixmap */
	pix = image_to_pixmap(&h->simg);
	browser_cache_add(h->list[h->i].filename,pix);
	XtVaSetValues(h->list[h->i].icon,
		      XmNlabelPixmap,pix,
		      XmNlabelType,XmPIXMAP,
		      NULL);
	h->state = 0;
	goto next;

    default:
	/* shouldn't happen */
	fprintf(stderr,"Oops: %s:%d\n",__FILE__,__LINE__);
	exit(1);
    }

 unknown:
    /* generic file icon */
    h->list[h->i].unknown = 1;
    pix = XmGetPixmap(XtScreen(h->list[h->i].icon),"unknown",0,0);
    XtVaSetValues(h->list[h->i].icon,
		  XmNlabelPixmap,pix,
		  XmNlabelType,XmPIXMAP,
		  NULL);
    
 next:
    h->i++;
    return FALSE;
}

static Boolean
browser_statfiles(XtPointer clientdata)
{
    struct browser_handle *h = clientdata;
    Pixmap pix;
    char *type;

    if (h->i == h->files) {
	/* done => read thumbnails now */
	h->wproc = 0;
	h->i = 0;
	h->wproc = XtAppAddWorkProc(app_context,browser_readfiles,h);
	return TRUE;
    }

    /* handle file */
    switch (h->list[h->i].mode & S_IFMT) {
    case S_IFDIR:
	XtAddCallback(h->list[h->i].icon,XmNactivateCallback,browser_cd,h);
	type = "dir";
	break;
    case S_IFREG:
	XtAddCallback(h->list[h->i].icon,XmNactivateCallback,browser_file,
		      h->list[h->i].filename);
	XtAddCallback(h->list[h->i].icon,XmNconvertCallback,browser_drag,
		      h->list[h->i].filename);
	type = "file";
	break;
    default:
	type = NULL;
    }
    if (type) {
	pix = XmGetPixmap(XtScreen(h->list[h->i].icon),type,0,0);
	XtVaSetValues(h->list[h->i].icon,
		      XmNlabelPixmap,pix,
		      XmNlabelType,XmPIXMAP,
		      NULL);
    }

    h->i++;
    return FALSE;
}

static int browser_readdir(struct browser_handle *h)
{
    struct browser_file *tmp;
    struct dirent *dirent;
    WidgetList children;
    Cardinal nchildren;
    XmString str,elem;
    struct stat st;
    DIR *dir;
    int len,i;

    /* status line */
    str  = XmStringGenerate("scanning ", NULL, XmMULTIBYTE_TEXT, NULL);
    elem = XmStringGenerate(h->dirname, NULL, XmMULTIBYTE_TEXT, NULL);
    str  = XmStringConcatAndFree(str,elem);
    elem = XmStringGenerate(" ...", NULL, XmMULTIBYTE_TEXT, NULL);
    str  = XmStringConcatAndFree(str,elem);
    XtVaSetValues(h->status,XmNlabelString,str,NULL);
    XmStringFree(str);

    /* read + sort dir */
    dir = opendir(h->dirname);
    if (NULL == dir) {
	fprintf(stderr,"opendir %s: %s\n",h->dirname,strerror(errno));
	return -1;
    }
    h->dirs = 0;
    h->sfiles = 0;
    h->afiles = 0;
    for (h->files = 0; NULL != (dirent = readdir(dir));) {
	/* get memory */
	if (0 == (h->files % 16)) {
	    tmp = realloc(h->list, sizeof(struct browser_file)*
			  (h->files + 16));
	    if (NULL == tmp)
		break;
	    h->list = tmp;
	    memset(h->list+h->files, 0, sizeof(struct browser_file)*16);
	}

	/* skip dotfiles */
	if (dirent->d_name[0] == '.' && 0 != strcmp(dirent->d_name,".."))
	    continue;

	/* get file type */
	h->list[h->files].dirent = *dirent;
	len = strlen(h->dirname)+strlen(h->list[h->files].dirent.d_name)+4;
	h->list[h->files].filename = malloc(len);
	sprintf(h->list[h->files].filename,"%s/%s",h->dirname,
		h->list[h->files].dirent.d_name);
	if (h->list[h->files].dirent.d_type != DT_UNKNOWN) {
	    h->list[h->files].mode = DTTOIF(h->list[h->files].dirent.d_type);
	} else {
	    if (-1 == lstat(h->list[h->files].filename,&st)) {
		fprintf(stderr,"lstat %s: %s\n",
			h->list[h->files].filename,strerror(errno));
	    } else
		h->list[h->files].mode = st.st_mode;
	}

	/* user-specified filter */
	if (S_ISDIR(h->list[h->files].mode)) {
	    h->dirs++;
	} else {
	    h->afiles++;
	    if (h->filter && 0 != fnmatch(h->filter,dirent->d_name,0))
		continue;
	    else
		h->sfiles++;
	}

	h->files++;
    }
    closedir(dir);
    qsort(h->list,h->files,sizeof(struct browser_file),browser_cmpfile);

    /* create widgets */
    for (i = 0; i < h->files; i++)
	browser_addfile(h,h->list+i);
    XtVaGetValues(h->rowcol,XtNchildren,&children,
                  XtNnumChildren,&nchildren,NULL);
    XtManageChildren(children,nchildren);
    XtVaSetValues(h->shell,XtNtitle,h->dirname,NULL);

    /* start bg processing */
    h->i = 0;
    h->wproc = XtAppAddWorkProc(app_context,browser_statfiles,h);
    return 0;
}

static void browser_bgcancel(struct browser_handle *h)
{
    if (h->wproc)
	XtRemoveWorkProc(h->wproc);
    h->wproc = 0;
    if (h->state == 1  &&  h->wdata)
	h->loader->done(h->wdata);
    if (h->state == 2  &&  h->wdata)
	desc_resize.done(h->wdata);
    if (h->wimg.data)
	free(h->wimg.data);
    h->wimg.data = NULL;
    if (h->simg.data)
	free(h->simg.data);
    h->simg.data = NULL;
}

static void browser_delwidgets(struct browser_handle *h)
{
    WidgetList children,list;
    Cardinal nchildren;
    int i;

    /* delete widgets */
    XtVaGetValues(h->rowcol,XtNchildren,&children,
                  XtNnumChildren,&nchildren,NULL);
    list = malloc(sizeof(Widget*)*nchildren);
    memcpy(list,children,sizeof(Widget*)*nchildren);
    XtUnmanageChildren(list,nchildren);
    for (i = 0; i < nchildren; i++)
        XtDestroyWidget(list[i]);
    free(list);

    /* reset state */
    if (h->list) {
	for (i = 0; i < h->files; i++)
	    if (h->list[i].filename)
		free(h->list[i].filename);
	free(h->list);
    }
    h->list = NULL;
    h->files = 0;
    h->state = 0;
}

/*----------------------------------------------------------------------*/

static void
browser_resize(Widget widget, XtPointer clientdata, XEvent *event, Boolean *d)
{
    struct browser_handle *h = clientdata;
    Widget clip;
    Dimension width;

    XtVaGetValues(h->viewport,XmNclipWindow,&clip,NULL);
    XtVaGetValues(clip,XtNwidth,&width,NULL);
    XtVaSetValues(h->rowcol,XtNwidth,width,NULL);
}

static void
browser_cd(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    struct browser_handle *h = clientdata;
    char *dir, *newpath, *tmp;

    /* build new dir path */
    dir = XtName(XtParent(widget));
    if (0 == strcmp(dir,"..")) {
	tmp = strrchr(h->dirname,'/');
	if (tmp == h->dirname)
	    tmp++;
	*tmp = 0;
    } else {
	newpath = malloc(strlen(h->dirname)+strlen(dir)+4);
	if (0 == strcmp(h->dirname,"/")) {
	    sprintf(newpath,"/%s",dir);
	} else {
	    sprintf(newpath,"%s/%s",h->dirname,dir);
	}
	free(h->dirname);
	h->dirname = newpath;
    }

    /* cleanup old stuff + read dir */
    browser_bgcancel(h);
    browser_delwidgets(h);
    browser_readdir(h);
}

static void
browser_filter_done(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    struct browser_handle *h = clientdata;
    XmSelectionBoxCallbackStruct *cb = call_data;
    char *filter;

    if (cb->reason == XmCR_OK) {
	filter = XmStringUnparse(cb->value,NULL,
				 XmMULTIBYTE_TEXT,XmMULTIBYTE_TEXT,
				 NULL,0,0);
	if (h->filter)
	    free(h->filter);
	h->filter = NULL;
	if (strlen(filter) > 0)
	    h->filter = strdup(filter);
	XtFree(filter);
	
	if (debug)
	    fprintf(stderr,"filter: %s\n", h->filter ? h->filter : "[none]");
	browser_bgcancel(h);
	browser_delwidgets(h);
	browser_readdir(h);
    }
    XtDestroyWidget(widget);
}

static void
browser_filter(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    struct browser_handle *h = clientdata;
    Widget shell;

    shell = XmCreatePromptDialog(h->shell,"filter",NULL,0);
    XtUnmanageChild(XmSelectionBoxGetChild(shell,XmDIALOG_HELP_BUTTON));
    XtAddCallback(shell,XmNokCallback,browser_filter_done,h);
    XtAddCallback(shell,XmNcancelCallback,browser_filter_done,h);
    XtManageChild(shell);
}

static void
browser_nofilter(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    struct browser_handle *h = clientdata;

    if (!h->filter)
	return;
    if (debug)
	fprintf(stderr,"filter: reset\n");
    free(h->filter);
    h->filter = NULL;
    
    browser_bgcancel(h);
    browser_delwidgets(h);
    browser_readdir(h);
}

static void
browser_file(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    new_file(clientdata,1);
}

static void
browser_drag(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    XmConvertCallbackStruct *ccs = call_data;
    char *name,*file = NULL;
    Atom *targs;
    int n;

    if ((ccs->target == TARGETS)                  ||
	(ccs->target == _MOTIF_CLIPBOARD_TARGETS) ||
	(ccs->target == _MOTIF_EXPORT_TARGETS)) {
	targs = (Atom*)XtMalloc(sizeof(Atom)*8);
	n = 0;
	targs[n++] = TARGETS;
	targs[n++] = XA_FILE_NAME;
	targs[n++] = XA_FILE;
	targs[n++] = _NETSCAPE_URL;
	ccs->value  = targs;
	ccs->length = n;
	ccs->type   = XA_ATOM;
	ccs->format = 32;
	ccs->status = XmCONVERT_MERGE;
    }
    if (ccs->target == XA_FILE_NAME ||
	ccs->target == XA_FILE      ||
	ccs->target == XA_STRING) {
	file = XtMalloc(strlen(clientdata)+1);
	strcpy(file,clientdata);
	ccs->value  = file;
	ccs->length = strlen(file);
	ccs->type   = XA_STRING;
	ccs->format = 8;
	ccs->status = XmCONVERT_DONE;
    }
    if (ccs->target == _NETSCAPE_URL) {
	file = XtMalloc(strlen(clientdata)+8);
	sprintf(file,"file:%s",(char*)clientdata);
	ccs->value  = file;
	ccs->length = strlen(file);
	ccs->type   = _NETSCAPE_URL;
	ccs->format = 8;
	ccs->status = XmCONVERT_DONE;
    }
    if (debug) {
	name = XGetAtomName(dpy,ccs->target);
	fprintf(stderr,"drag: %s [%s]\n",name,file ? file : "-");
	XFree(name);
    }
}

static void
browser_destroy(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    struct browser_handle *h = clientdata;
    int i;

    browser_bgcancel(h);
    ptr_unregister(h->shell);
    if (h->list) {
	for (i = 0; i < h->files; i++)
	    if (h->list[i].filename)
		free(h->list[i].filename);
	free(h->list);
    }
    free(h);
}

/*----------------------------------------------------------------------*/

static char*
browser_fixpath(char *dir)
{
    char path[1024];
    char *s,*d;

    memset(path,0,sizeof(path));
    if (dir[0] == '/') {
	/* absolute */
	strncpy(path,dir,sizeof(path)-1);
    } else {
	/* relative */
	getcwd(path,sizeof(path)-1);
	if (strlen(path)+strlen(dir)+4 < sizeof(path)) {
	    strcat(path,"/");
	    strcat(path,dir);
	}
    }

    for (s = d = path; *s != 0;) {
	if (0 == strncmp(s,"//",2)) {
	    s++;
	    continue;
	}
	if (0 == strncmp(s,"/./",3)) {
	    s+=2;
	    continue;
	}
	if (0 == strcmp(s,"/"))
	    s++;
	if (0 == strcmp(s,"/."))
	    s+=2;
	*d = *s;
	s++, d++;
    }
    return strdup(path);
}

void
browser_window(char *dirname)
{
    Widget form,clip,menubar,menu,push;
    struct browser_handle *h;

    h = malloc(sizeof(*h));
    if (NULL == h) {
	fprintf(stderr,"out of memory");
	return;
    }
    memset(h,0,sizeof(*h));
    h->dirname = browser_fixpath(dirname);

    h->shell = XtVaAppCreateShell("browser","Iv",
				  topLevelShellWidgetClass,
				  dpy,
				  XtNclientLeader,app_shell,
				  XmNdeleteResponse,XmDESTROY,
				  NULL);
    XmdRegisterEditres(h->shell);
    XtAddCallback(h->shell,XtNdestroyCallback,browser_destroy,h);

    /* widgets */
    form = XtVaCreateManagedWidget("form", xmFormWidgetClass, h->shell,
				   NULL);
    menubar = XmCreateMenuBar(form,"bar",NULL,0);
    XtManageChild(menubar);
    h->status = XtVaCreateManagedWidget("status",xmLabelWidgetClass, form,
					NULL);
    h->viewport = XmCreateScrolledWindow(form,"view",NULL,0);
    h->rowcol = XtVaCreateManagedWidget("box", xmRowColumnWidgetClass,
					h->viewport, NULL);
    XtManageChild(h->viewport);
    XtVaGetValues(h->viewport,XmNclipWindow,&clip,NULL);
    if (NULL == clip)
	fprintf(stderr,"FIXME: clip is NULL\n");
    else
	XtAddEventHandler(clip,StructureNotifyMask,True,
			  browser_resize,h);

    /* menu - file */
    menu = XmCreatePulldownMenu(menubar,"fileM",NULL,0);
    XtVaCreateManagedWidget("file",xmCascadeButtonWidgetClass,menubar,
			    XmNsubMenuId,menu,NULL);
    push = XtVaCreateManagedWidget("close",xmPushButtonWidgetClass,menu,NULL);
    XtAddCallback(push,XmNactivateCallback,destroy_cb,h->shell);
    
    /* menu - view */
    menu = XmCreatePulldownMenu(menubar,"viewM",NULL,0);
    XtVaCreateManagedWidget("view",xmCascadeButtonWidgetClass,menubar,
			    XmNsubMenuId,menu,NULL);
    push = XtVaCreateManagedWidget("filter",xmPushButtonWidgetClass,menu,NULL);
    XtAddCallback(push,XmNactivateCallback,browser_filter,h);
    push = XtVaCreateManagedWidget("freset",xmPushButtonWidgetClass,menu,NULL);
    XtAddCallback(push,XmNactivateCallback,browser_nofilter,h);
    
    /* read dir and show window */
    browser_readdir(h);
    XtPopup(h->shell,XtGrabNone);
    ptr_register(h->shell);
}

void
browser_ac(Widget widget, XEvent *event,
	   String *params, Cardinal *num_params)
{
    if (*num_params > 0)
	browser_window(params[0]);
    else
	browser_window(".");
}
