/*
** Copyright (C) 2001-2005 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

#include <string.h>
#include "buffer.h"

Buffer*
buffer_new ()
{
	Buffer *new_buffer;

	new_buffer = (Buffer*) malloc (sizeof(Buffer));
	if (!new_buffer)
		return NULL;

	new_buffer->_data  = NULL;
	new_buffer->_len   = 0;

	return new_buffer;
}


void 
buffer_destroy (Buffer *buffer)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	
	buffer->_len = -1;

	free (buffer->_data);
	free (buffer);
}



int
buffer_set_data (Buffer* buffer, const char* data, int len)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0);
	assert (data || len == 0);
	
	buffer->_data = (char*) realloc (buffer->_data, len);
	if (!buffer->_data)
		return -1;
		
	buffer->_data = memmove (buffer->_data, data, len);
	buffer->_len  = len;

	return len;
}



int
buffer_manage_data (Buffer* buffer, char* data, int len)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0);
	assert (data || len == 0);

	free (buffer->_data);

	buffer->_data = data;
	buffer->_len  = len;

	return len;
}



char        
buffer_at (const Buffer* buffer, BufferIterator pos)
{       
	assert (buffer);
	assert (buffer->_len > 0);
	assert (buffer->_data);
	assert (pos >= 0 && pos < buffer_end(buffer));
	
	return buffer->_data[pos];
}
                   

	
const char* 
buffer_data (const Buffer* buffer)
{
	assert (buffer);
	assert (buffer->_len >= 0);

	return buffer->_data;
}
	


const char* 
buffer_data_pos (const Buffer* buffer, BufferIterator pos)
{
	assert (buffer);
	assert (buffer->_data);
	assert (buffer->_len >= 0);
	assert (pos >= 0 && pos < buffer_end(buffer));
	
	return buffer->_data + pos;
}



int 
buffer_length (const Buffer *buffer)
{
	assert (buffer);
	assert (buffer->_len >= 0);

	return buffer->_len;
}

	
int
buffer_append (Buffer* buffer, const char* data, int len)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0);
	assert (data || len == 0);
	
	buffer->_data = (char*)realloc (buffer->_data, buffer->_len + len);
	if (!buffer->_data)
		return -1;

	memcpy (buffer->_data + buffer->_len, data, len);

	buffer->_len += len;

	return buffer->_len;
}

int 
buffer_prepend (Buffer *buffer, const char *data, int len)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0);
	assert (data || len == 0);
	
	return buffer_insert (buffer, data, len, 0);
}


int
buffer_insert (Buffer *buffer, const char *data, int len, int pos)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0 && pos >= 0);
	assert (data || len == 0);
	assert (buffer->_len > pos);
	
	buffer->_data = (char*)realloc (buffer->_data, buffer->_len + len);
	if (!buffer->_data)
		return -1;

	memmove (buffer->_data + pos + len, 
		buffer->_data + pos, 
		buffer->_len - pos);
	memmove (buffer->_data + pos, data, len);

	buffer->_len += len;

	return buffer->_len;
}


int
buffer_erase (Buffer *buffer, int pos, int len)
{
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0);
	assert (buffer->_len >= pos + len);
	
	memmove (buffer->_data + pos,
		 buffer->_data + pos + len,
		 buffer->_len  - (pos + len));
	
	buffer->_len  -= len;
	buffer->_data = realloc (buffer->_data, buffer->_len);
	
	return buffer->_len;
}




BufferIterator 
buffer_find_offset (const Buffer *buffer, int offset, const char* pattern, int len)
{	
	int i, j;
	int match = 1;
	
	assert (buffer);
	assert (buffer->_len >= 0);
	assert (len >= 0);
	assert (pattern || len == 0);
	assert (offset >= 0);
	assert (offset < buffer->_len);
	
	for (i = offset; i < buffer->_len - len; ++i) {
		for (j = 0; j < len; ++j) {
			match = (buffer->_data[i+j] == pattern[j]);
			if (!match) 
				break;
		}
		if (match && (j == len)) 
			break;
	}
	return i;
}


void
buffer_clear (Buffer *buffer)
{
	assert (buffer);
	assert (buffer->_len >= 0);

	free (buffer->_data);

	buffer->_data = NULL;
	buffer->_len  = 0;
}
