/*
 * $Header: /home/cvspublic/jakarta-commons/httpclient/src/test/org/apache/commons/httpclient/TestHttpConnectionManager.java,v 1.1 2002/04/12 21:17:06 marcsaeg Exp $
 * $Revision: 1.1 $
 * $Date: 2002/04/12 21:17:06 $
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import junit.framework.*;
import java.net.MalformedURLException;
import java.lang.reflect.*;

import org.apache.commons.httpclient.methods.*;

/**
 *
 * Unit tests for {@link HttpConnectionManager}.  These tests
 * do not require any network connection or web app.
 *
 * @author Marc A. Saegesser
 * @version $Id: TestHttpConnectionManager.java,v 1.1 2002/04/12 21:17:06 marcsaeg Exp $
 */
public class TestHttpConnectionManager extends TestCase {
    // ----------------------------------------------------- Instance Variables

    // ------------------------------------------------------------ Constructor
    public TestHttpConnectionManager(String testName) {
        super(testName);
    }

    // ------------------------------------------------------------------- Main
    public static void main(String args[]) {
        String[] testCaseName = { TestHttpConnectionManager.class.getName() };
        junit.textui.TestRunner.main(testCaseName);
    }

    // ------------------------------------------------------- TestCase Methods

    public static Test suite() {
        return new TestSuite(TestHttpConnectionManager.class);
    }


    // ----------------------------------------------------------- Test Methods

    // Test the accessor methods
    public void testProxyHostAccessors() {
        HttpConnectionManager mgr = new HttpConnectionManager();

        mgr.setProxyHost("proxyhost");
        assertEquals("Proxy Host", "proxyhost", mgr.getProxyHost());
    }

    public void testProxyPortAccessors() {
        HttpConnectionManager mgr = new HttpConnectionManager();

        mgr.setProxyPort(8888);
        assertEquals("Proxy Port", 8888, mgr.getProxyPort());
    }

    public void testMaxConnectionsAccessors() {
        HttpConnectionManager mgr = new HttpConnectionManager();

        // First test the default value
        assertEquals("Default MaxConnections", 2, mgr.getMaxConnectionsPerHost());

        mgr.setMaxConnectionsPerHost(10);
        assertEquals("MaxConnections", 10, mgr.getMaxConnectionsPerHost());
    }

    public void testGetConnection() {
        HttpConnectionManager mgr = new HttpConnectionManager();

        try{
            // Create a new connection
            HttpConnection conn = mgr.getConnection("http://www.nosuchserver.com/path/path?query=string");
            // Validate the connection properties
            assertEquals("Host", "www.nosuchserver.com", conn.getHost());
            assertEquals("Port", 80, conn.getPort());
            // Release the connection
            mgr.releaseConnection(conn);

            // Create a new connection
            conn = mgr.getConnection("https://www.nosuchserver.com/path/path?query=string");
            // Validate the connection properties
            assertEquals("Host", "www.nosuchserver.com", conn.getHost());
            assertEquals("Port", 443, conn.getPort());
            // Release the connection
            mgr.releaseConnection(conn);

            // Create a new connection
            conn = mgr.getConnection("http://www.nowhere.org:8080/path/path?query=string");
            // Validate the connection properties
            assertEquals("Host", "www.nowhere.org", conn.getHost());
            assertEquals("Port", 8080, conn.getPort());
            // Release the connection
            mgr.releaseConnection(conn);

        }catch(MalformedURLException e){
            fail("Caught unexpected MalformedURLException (" + e.toString() + ")");
        }catch(HttpException e){
            fail("Caught unexpected HttpException (" + e.toString() + ")");
        }


    }

    public void testGetMultipleConnections() {
        HttpConnectionManager mgr = new HttpConnectionManager();

        try{
            // Create a new connection
            HttpConnection conn1 = mgr.getConnection("http://www.nosuchserver.com/path/path?query=string");
            // Release the connection
            mgr.releaseConnection(conn1);

            // Get the same connection again
            HttpConnection conn2 = mgr.getConnection("http://www.nosuchserver.com/");
            assertEquals("Same connection", conn1, conn2);
            // don't release yet

            // Get another new connection
            HttpConnection conn3 = mgr.getConnection("http://www.nosuchserver.com/");
            assertTrue(conn2 != conn3);

        }catch(MalformedURLException e){
            fail("Caught unexpected MalformedURLException (" + e.toString() + ")");
        }catch(HttpException e){
            fail("Caught unexpected HttpException (" + e.toString() + ")");
        }
    }

    public void testTimeout()
    {
        HttpConnectionManager mgr = new HttpConnectionManager();
        
        try{
            HttpConnection conn1 = mgr.getConnection("http://www.nosuchserver.com/path/path?query=string");
            HttpConnection conn2 = mgr.getConnection("http://www.nosuchserver.com/path/path?query=string");
            HttpConnection conn3 = mgr.getConnection("http://www.nosuchserver.com/path/path?query=string", 5000);
            fail("Expected an HttpException.");
        }catch(MalformedURLException e){
            fail("Caught unexpected MalformedURLException (" + e.toString() + ")");
        }catch(HttpException e){
            //Expected result
        }
    }
}

