
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.component;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.table.*;

/**
 * A button that is used as a table cell.
 * This component is actually implemented both as a cell renderer and a
 * cell editor.  Because the
 * user must interact directly with this component, it must be
 * registered <b>both as a renderer and an editor</b> for the
 * appropriate table cells and the table model must declare that
 * <b>such cells be editable</b>, regardless of whether the table
 * contents will ever be changed.
 * <p>
 * To listen for button presses, you can simply register an action
 * listener with this object; there is no need to access the underlying
 * <tt>JButton</tt>s.  An action event will be fired when the
 * table cell button is pressed, and the corresponding routine that is
 * called (at the listener's end) should end with a call to
 * either <tt>stopCellEditing()</tt> (to save any changes to the table)
 * or to <tt>cancelCellEditing()</tt> (to make no changes to the table).
 * <p>
 * If changes are to be made to the table, routine
 * <tt>getCellEditorValue()</tt> must be overridden to return a new
 * value for the table cell.
 */
public class ButtonRenderer extends AbstractCellEditor
        implements TableCellRenderer, TableCellEditor, ActionListener {
    /**
     * The rendering button.
     */
    private JButton renderer;
    /**
     * The editing button.
     */
    private JButton editor;

    /**
     * Row of the cell being edited.
     */
    private int edRow;
    /**
     * Column of the cell being edited.
     */
    private int edCol;
    /**
     * Value of the cell being edited.
     */
    private Object edValue;

    /**
     * A list of event listeners registered with this object.
     */
    private EventListenerList listenerList = new EventListenerList();

    /**
     * Creates a new button renderer/editor displaying the given text.
     *
     * @param text the text to place on the button.
     */
    public ButtonRenderer(String text) {
        renderer = new JButton(text);
        renderer.setBorder(BorderFactory.createRaisedBevelBorder());
        editor = new JButton(text);
        editor.setBorder(BorderFactory.createRaisedBevelBorder());
        editor.addActionListener(this);
    }

    /**
     * Returns the component that does the rendering.
     * This implementation ignores the value of the table cell and
     * simply returns this button.
     *
     * @param table the table in which a cell is being rendered.
     * @param value the value of the table cell being rendered.
     * @param isSelected whether or not the table cell is currently
     * selected.
     * @param hasFocus whether or not the table cell currently has the
     * focus.
     * @param row the row of the table cell being rendered.
     * @param column the column of the table cell being rendered.
     * @return the component that does the rendering.
     */
    public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus,
            int row, int column) {
        return renderer;
    }

    /**
     * Returns the component that does the editing.
     * This implementation ignores the value of the table cell and
     * simply returns the editor button.
     *
     * @param table the table in which a cell is being edited.
     * @param value the value of the table cell being edited.
     * @param isSelected whether or not the table cell is currently
     * selected.
     * @param row the row of the table cell being edited.
     * @param column the column of the table cell being edited.
     * @return the component that does the editing.
     */
    public Component getTableCellEditorComponent(JTable table, Object value,
            boolean isSelected, int row, int column) {
        edRow = row;
        edCol = column;
        edValue = value;
        return editor;
    }

    /**
     * Returns the row containing the cell editor.
     *
     * @return the row containing the cell editor.
     */
    public int getCellEditorRow() {
        return edRow;
    }
    /**
     * Returns the column containing the cell editor.
     *
     * @return the column containing the cell editor.
     */
    public int getCellEditorColumn() {
        return edCol;
    }
    /**
     * Returns the value contained in the cell editor.
     * This implementation simply returns the value of the table cell
     * being edited.
     *
     * @return the value contained in the cell editor.
     */
    public Object getCellEditorValue() {
        return edValue;
    }

    /**
     * Returns the button used for cell rendering.
     * Note that this is not the button used for cell editing.
     * Generally you should not need to call this function.
     *
     * @return the button used for cell rendering.
     * @see #getEditorButton
     */
    public JButton getRendererButton() {
        return renderer;
    }

    /**
     * Returns the button used for cell editing.
     * Note that this is not the button used for cell rendering.
     * Generally you should not need to call this function.
     *
     * @return the button used for cell editing.
     * @see #getRendererButton
     */
    public JButton getEditorButton() {
        return editor;
    }

    /**
     * Sets the text to display on both the rendering and editing buttons.
     * <p>
     * Note that there are two buttons used by this class; one that renders
     * table cells and one that the user interacts with when editing table
     * cells.  This routine affects both of these buttons.
     *
     * @param text the text to display on both buttons.
     */
    public void setText(String text) {
        renderer.setText(text);
        editor.setText(text);
    }
    /**
     * Sets the border to use for both the rendering and editing buttons.
     * <p>
     * Note that there are two buttons used by this class; one that renders
     * table cells and one that the user interacts with when editing table
     * cells.  This routine affects both of these buttons.
     *
     * @param border the border to use for both buttons.
     */
    public void setBorder(Border border) {
        renderer.setBorder(border);
        editor.setBorder(border);
    }

    /**
     * Registers the given listener to listen for action events coming from
     * this object.  This object fires an action event when the button
     * in the table cell is pressed.
     *
     * @param l the listener to add.
     */
    public void addActionListener(ActionListener l) {
        listenerList.add(ActionListener.class, l);
    }
    /**
     * Unregisters the given listener so it no longer listens for action
     * events coming from this object.  This object fires an action event
     * when the button in the table cell is pressed.
     *
     * @param l the listener to remove.
     */
    public void removeActionListener(ActionListener l) {
        listenerList.remove(ActionListener.class, l);
    }
    /**
     * For internal use only.
     * Called when the table cell button is pressed; this routine fires
     * a new action event to all listeners registered with this object.
     *
     * @param e the details of the button press event.
     */
    public void actionPerformed(ActionEvent e) {
        ActionEvent event = null;
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length - 2; i >= 0; i -= 2)
            if (listeners[i] == ActionListener.class) {
                if (event == null)
                    event = new ActionEvent(this, e.getID(),
                        e.getActionCommand(), e.getModifiers());
                ((ActionListener)listeners[i + 1]).actionPerformed(event);
            }
    }

    /**
     * Prepares the component for editing as requested by the given
     * event.
     *
     * @param e the event that caused editing to begin.
     * @return <tt>true</tt> if and only if the table cell being edited 
     * should be selected.
     */
    public boolean shouldSelectCell(EventObject e) {
        editor.requestFocus();
        return false;
    }
}

