/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.pool;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.excalibur.collections.Buffer;
import org.apache.avalon.excalibur.collections.FixedSizeBuffer;
import org.apache.avalon.excalibur.concurrent.Mutex;

/**
 * This is an <code>Pool</code> that caches Poolable objects for reuse.
 * Please note that this pool offers no resource limiting whatsoever.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.2 $ $Date: 2001/12/26 16:15:23 $
 * @since 4.1
 */
public final class FixedSizePool
    implements Pool, Disposable
{
    private boolean m_disposed = false;
    private final Buffer m_buffer;
    private final ObjectFactory m_factory;
    private final Mutex m_mutex = new Mutex();

    public FixedSizePool( ObjectFactory factory, int size )
        throws Exception
    {
        m_buffer = new FixedSizeBuffer( size );
        m_factory = factory;

        for ( int i = 0; i < size; i++ )
        {
            m_buffer.add( m_factory.newInstance() );
        }
    }

    public Poolable get()
    {
        if ( m_disposed )
        {
            throw new IllegalStateException( "Cannot get an object from a disposed pool" );
        }

        Poolable object = null;

        try
        {
            m_mutex.acquire();
            object = (Poolable) m_buffer.remove();
        }
        catch ( InterruptedException ie )
        {
            // NOTE: HANDLE THIS
        }
        finally
        {
            m_mutex.release();
        }

        return object;
    }

    public void put( Poolable object )
    {
        if ( m_disposed )
        {
            try
            {
                m_factory.decommission( object );
            }
            catch ( Exception e )
            {
                // We should never get here, but ignore the exception if it happens
            }
        }
        else
        {
            try
            {
                m_mutex.acquire();
                m_buffer.add( object );
            }
            catch ( InterruptedException e )
            {
              // NOTE: HANDLE THIS
            }
            finally
            {
                m_mutex.release();
            }
        }
    }

    public void dispose()
    {
        m_disposed = true;

        try
        {
            m_mutex.acquire();

            while ( ! m_buffer.isEmpty() )
            {
                try
                {
                    m_factory.decommission( m_buffer.remove() );
                }
                catch ( Exception e )
                {
                    // We should never get here, but ignore the exception if it happens
                }
            }
        }
        catch ( InterruptedException ie )
        {
            // NOTE: HANDLE THIS
        }
        finally
        {
            m_mutex.release();
        }
    }
}

