/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.proxy;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * This makes a dynamic proxy for an object.  The object can be represented
 * by one, some of all of it's interfaces.
 *
 * Amongst other things, it's an anti hackinge measure.  Suitable armed code
 * could have case an interface for a thing back to it's impl and used methods
 * and properties that were not it's authors intention.  Reflection too allows
 * some powerful introspection things and some traversal even more things
 * including private member vars by a serialisation trick... hence the transient.
 *
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @author <a href="mailto:Paul_Hammant@yahoo.com">Paul Hammant</a>
 * @version CVS $Revision: 1.5 $ $Date: 2001/12/11 09:53:32 $
 * @since 4.0b5
 */
public final class DynamicProxy
    implements InvocationHandler
{
    private transient Object m_object;

    /**
     * Private constructor that blocks instantiation outside this class.
     *
     * @param object the underlying object
     */
    private DynamicProxy( final Object object )
    {
        m_object = object;
    }

    /**
     * Create a proxy object that has all of it's underlying
     * interfaces implemented by proxy.
     *
     * @param object the underling object to proxy
     * @return the proxied object
     */
    public static Object newInstance( final Object object )
    {
        return newInstance( object, object.getClass().getInterfaces() );
    }

    /**
     * Create a proxy object that has specified interfaces implemented by proxy.
     *
     * @param object the underling object to proxy
     * @return the proxied object
     */
    public static Object newInstance( final Object object, final Class[] interfaces )
    {
        final ClassLoader classLoader = object.getClass().getClassLoader();
        final DynamicProxy proxy = new DynamicProxy( object );

        return Proxy.newProxyInstance( classLoader, interfaces, proxy );
    }

    /**
     * Invoke the specified method on underlying object.
     * This is called by proxy object.
     *
     * @param proxy the proxy object
     * @param method the method invoked on proxy object
     * @param args the arguments supplied to method
     * @return the return value of method
     * @exception Throwable if an error occurs
     */
    public Object invoke( final Object proxy,
                          final Method method,
                          final Object[] args )
        throws Throwable
    {
        try
        {
            return method.invoke( m_object, args );
        }
        catch( final InvocationTargetException ite )
        {
            throw ite.getTargetException();
        }
    }
}
