/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.extension;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * Basic Implementation Of PackageManager Interface used to manage 
 * "Optional Packages" (formerly known as "Standard Extensions"). 
 * The "Optional Packages" are stored on file system in a number of
 * directories.
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @version $Revision: 1.7 $ $Date: 2001/12/11 09:53:34 $
 * @see PackageRepository
 */
public class PackageManager
{
    ///Ordered list of repositories to search in
    private PackageRepository   m_repository;

    /**
     * Construct a PackageManager for a repositories.
     *
     * @param repository the repository to use in PackageManager
     */
    public PackageManager( final PackageRepository repository )
    {
        if( null == repository )
        {
            throw new NullPointerException( "repository property is null" );
        }

        m_repository = repository;
    }

    /**
     * Return the <code>OptionalPackage</code> that provides specified
     * <code>Extension</code>. If the specified <code>Extension</code> 
     * can not be found then <code>null</code> is returned. If there is
     * multiple implementations that satisfy <code>Extension</code>, 
     * then an <code>OptionalPackage</code> returned is based on the 
     * following heristic;
     *
     * <p>Return the first Optional Package. (This heuristic will 
     * be replaced in time).</p>
     *
     * @param extension Description of the extension that needs to be provided by 
     *                  optional package
     * @see OptionalPackage
     * @see Extension
     */
    public OptionalPackage getOptionalPackage( final Extension extension )
    {
        final OptionalPackage[] packages = m_repository.getOptionalPackages( extension );

        if( null == packages || 0 == packages.length ) return null;

        //TODO: Use heurisitic to find which is best package

        return packages[ 0 ];        
    }


    /**
     * Build a list of dependencies based on specified <code>Extension</code>s.
     * Each specified <code>Extension</code> is expected to be a required extension
     * of another "Optional Package". 
     *
     * <p>If the required <code>Extension</code> can not be found locally then
     * an UnsatisfiedPackageException is thrown. if an <code>OptionalPackage</code>
     * is found locally that satisfies specified required <code>Extension</code>
     * then it is returned in the array of OptionalPackages. scanDependencies() is then recursively
     * called on all of the candidates required extensions.</p>
     *
     * @param required the array of required Extensions.
     * @param available the array of Extensions already available to caller.
     * @return the list of OptionalPackages that satisfy required Extensions
     * @exception UnsatisfiedPackageException if unable to satisfy all extensions
     * @see #scanDependencies
     */
    public OptionalPackage[] scanDependencies( final Extension required, 
                                               final Extension[] available )
        throws UnsatisfiedExtensionException
    {
        final Extension[] extensions = new Extension[] { required };
        return scanDependencies( required, available );
    }

    /**
     * Build a list of dependencies based on specified <code>Extension</code>.
     * The specified <code>Extension</code> is expected to be a required extension
     * of another "Optional Package". 
     *
     * <p>If the required <code>Extension</code> can not be found locally then
     * an UnsatisfiedPackageException is thrown. if an <code>OptionalPackage</code>
     * is found locally that satisfies specified required <code>Extension</code>
     * then it is returned in the array of OptionalPackages. scanDependencies() is then recursively
     * called on all of the candidates required extensions.</p>
     *
     * @param required the array of required Extensions.
     * @param available the array of Extensions already available to caller.
     * @return the list of OptionalPackages that satisfy required Extensions
     * @exception UnsatisfiedPackageException if unable to satisfy all extensions
     * @see #scanDependencies
     */
    public OptionalPackage[] scanDependencies( final Extension[] required, 
                                               final Extension[] available )
        throws UnsatisfiedExtensionException
    {
        final ArrayList dependencies = new ArrayList();
        final ArrayList unsatisfied = new ArrayList();

        scanDependencies( required, available, dependencies, unsatisfied );

        if( 0 != unsatisfied.size() )
        {
            final Extension extension = (Extension)unsatisfied.get( 0 );
            throw new UnsatisfiedExtensionException( extension );
        }

        return (OptionalPackage[])dependencies.toArray( new OptionalPackage[ 0 ] );
    }

    /**
     * Build a list of dependencies based on specified <code>Extension</code>s.
     * Each specified <code>Extension</code> is expected to be a required extension
     * of another "Optional Package". 
     *
     * <p>If the required <code>Extension</code> can not be found locally then
     * it is placed in list of unsatisfied Extensions. If a candidate <code>Extension</code>
     * is found locally that satisfies specified required <code>Extension</code>
     * then it is added to list of dependencies. scanDependencies() is then recursively
     * called on all of the candidates required extensions.</p>
     *
     * @param required the array of required Extensions.
     * @param available the array of Extensions already available to caller.
     * @param dependencies the list of dependencies.
     * @param unsatisfied the list of unsatisfied (ie non-local) dependencies.
     * @see #scanDependencies
     */
    public void scanDependencies( final Extension[] required, 
                                  final Extension[] available, 
                                  final List dependencies,
                                  final List unsatisfied )
    {
        for( int i = 0; i < required.length; i++ )
        {
            scanDependencies( required[ i ], available, dependencies, unsatisfied );
        }
    }

    /**
     * Build a list of dependencies based on specified <code>Extension</code>.
     * The specified <code>Extension</code> is expected to be a required extension
     * of another "Optional Package". 
     *
     * <p>If the required <code>Extension</code> can not be found locally then
     * it is placed in list of unsatisfied Extensions. If a candidate <code>OptionalPackage</code>
     * is found locally that satisfies specified required <code>Extension</code>
     * then it is added to list of dependencies. scanDependencies() is then recursively
     * called on all of the candidates required extensions.</p>
     *
     * @param required the required Extension.
     * @param available the array of Extensions already available to caller.
     * @param dependencies the list of OptionalPackages required to satisfy extension.
     * @param unsatisfied the list of unsatisfied (ie non-local) dependencies.
     * @see #scanDependencies
     */
    public void scanDependencies( final Extension required, 
                                  final Extension[] available,
                                  final List dependencies,
                                  final List unsatisfied )
    {
        //Check to see if extension is satisifed by the
        //list of available extensions passed in
        for( int i = 0; i < available.length; i++ )
        {
            final Extension other = available[ i ];
            if( other.isCompatibleWith( required ) )
            {
                return;
            }
        }

        //Check to see if extension is satisifed by one 
        //of the extensions already found
        final int size = dependencies.size();
        for( int i = 0; i < size; i++ )
        {
            final OptionalPackage optionalPackage = (OptionalPackage)dependencies.get( i );
            if( optionalPackage.isCompatible( required ) )
            {
                return;
            }
        }

        final OptionalPackage optionalPackage = getOptionalPackage( required );
        if( null == optionalPackage )
        {
            if( !unsatisfied.contains( required ) )
            {
                unsatisfied.add( required );
            }
        }
        else
        {
            if( !dependencies.contains( optionalPackage ) )
            {
                dependencies.add( optionalPackage );
            }
            
            scanDependencies( optionalPackage.getRequiredExtensions(), 
                              available,
                              dependencies, 
                              unsatisfied );
        }
    }
}
