/* ControlManager.m - Menu suppor class to 
   the Control menu functions for the 
   Macintosh OS X SDL port of Atari800
   Mark Grebe <atarimac@cox.net>
   
   Based on the Preferences pane of the
   TextEdit application.

*/
#import "ControlManager.h"
#import "SDL.h"

/* Definition of Mac native keycodes for characters used as menu shortcuts the bring up a windo. */
#define QZ_a			0x00
#define QZ_l			0x25
#define QZ_m			0x2E
#define QZ_s			0x01
#define QZ_h			0x04
#define QZ_SLASH		0x2C
#define QZ_F8			0x64

extern void PauseAudio(int pause);
extern void monitorEnter(void);
extern int monitorCmd(char *input);

extern int pauseEmulator;
extern int requestPauseEmulator;
extern int requestColdReset;
extern int requestWarmReset;
extern int requestSaveState;
extern int requestLoadState;
extern int requestLimitChange;
extern int requestQuit;
extern int speed_limit;
extern char saveFilename[FILENAME_MAX];
extern char loadFilename[FILENAME_MAX];
extern char atari_state_dir[FILENAME_MAX];
extern int assemblerMode;

/* Functions which provide an interface for C code to call this object's shared Instance functions */
void SetControlManagerLimit(int limit) {
    [[ControlManager sharedInstance] setLimitMenu:(limit)];
    }
    
int ControlManagerFatalError() {
    return([[ControlManager sharedInstance] fatalError]);
    }

void ControlManagerSaveState() {
    [[ControlManager sharedInstance] saveState:nil];
}

void ControlManagerLoadState() {
    [[ControlManager sharedInstance] loadState:nil];
}

void ControlManagerPauseEmulator() {
    [[ControlManager sharedInstance] pause:nil];
}

void ControlManagerHideApp() {
    [NSApp hide:[ControlManager sharedInstance]];
    [[ControlManager sharedInstance] releaseCmdKeys:@"h":QZ_h];
}

void ControlManagerAboutApp() {
    [NSApp orderFrontStandardAboutPanel:[ControlManager sharedInstance]];
    [[ControlManager sharedInstance] releaseCmdKeys:@"a":QZ_a];
}

void ControlManagerShowHelp() {
    [NSApp showHelp:[ControlManager sharedInstance]];
    [[ControlManager sharedInstance] releaseCmdKeys:@"?":QZ_SLASH];
}

void ControlManagerMiniturize() {
    [[NSApp keyWindow] performMiniaturize:[ControlManager sharedInstance]];
    [[ControlManager sharedInstance] releaseCmdKeys:@"m":QZ_m];
}

void ControlManagerMessagePrint(char *string)
{
    [[ControlManager sharedInstance] messagePrint:string];
}

void ControlManagerMonitorPrint(char *string)
{
    [[ControlManager sharedInstance] monitorPrint:string];
}

int ControlManagerMonitorRun()
{
    return([[ControlManager sharedInstance] monitorRun]);
}

@implementation ControlManager
static ControlManager *sharedInstance = nil;
#define MAX_MONITOR_OUTPUT 4096
static char monitorOutput[MAX_MONITOR_OUTPUT];
static int monitorCharCount = 0;

+ (ControlManager *)sharedInstance {
    return sharedInstance ? sharedInstance : [[self alloc] init];
}

- (id)init {
    NSDictionary *attribs;

    if (sharedInstance) {
	[self dealloc];
    } else {
        [super init];
        sharedInstance = self;
        if (!errorTextField) {
            if (![NSBundle loadNibNamed:@"ControlManager" owner:self])  {
                NSLog(@"Failed to load ControlManager.nib");
                NSBeep();
                return nil;
                }
            }
	[[errorTextField window] setExcludedFromWindowsMenu:YES];
	[[errorTextField window] setMenu:nil];
	[[monitorInputField window] setExcludedFromWindowsMenu:YES];
	[[monitorInputField window] setMenu:nil];
	[[messageOutputView window] setExcludedFromWindowsMenu:YES];
	[[messageOutputView window] setMenu:nil];
        attribs = [[NSDictionary alloc] initWithObjectsAndKeys:
                   [NSFont fontWithName:@"Monaco" size:10.0], NSFontAttributeName,
                   nil]; 
        [monitorOutputView setTypingAttributes:attribs];
        [messageOutputView setTypingAttributes:attribs];
        [monitorOutputView setString:@"Atari800MacX Monitor\nType '?' for help, 'CONT' to exit\n> "];
        [attribs release];
    }
    return sharedInstance;
}

/*------------------------------------------------------------------------------
*  setDoublesizeMenu - This method is used to set the menu check state for the 
*     limit emulator speed menu item.
*-----------------------------------------------------------------------------*/
- (void)setLimitMenu:(int)limit
{
    if (limit)
        [limitItem setState:NSOnState];
    else
        [limitItem setState:NSOffState];
}

/*------------------------------------------------------------------------------
*  browseFileInDirectory - This allows the user to chose a file to read in from
*     the specified directory.
*-----------------------------------------------------------------------------*/
- (NSString *) browseFileInDirectory:(NSString *)directory {
    NSOpenPanel *openPanel = nil;
    
    openPanel = [NSOpenPanel openPanel];
    [openPanel setCanChooseDirectories:NO];
    [openPanel setCanChooseFiles:YES];
    
    if ([openPanel runModalForDirectory:directory file:nil types:nil] == NSOKButton)
        return([[openPanel filenames] objectAtIndex:0]);
    else
        return nil;
    }

/*------------------------------------------------------------------------------
*  saveFileInDirectory - This allows the user to chose a filename to save in from
*     the specified directory.
*-----------------------------------------------------------------------------*/
- (NSString *) saveFileInDirectory:(NSString *)directory:(NSString *)type {
    NSSavePanel *savePanel = nil;
    
    savePanel = [NSSavePanel savePanel];
    
    [savePanel setRequiredFileType:type];
    
    if ([savePanel runModalForDirectory:directory file:nil] == NSOKButton)
        return([savePanel filename]);
    else
        return nil;
    }

/*------------------------------------------------------------------------------
*  coldReset - This method handles the cold reset menu selection.
*-----------------------------------------------------------------------------*/
- (IBAction)coldReset:(id)sender
{
    requestColdReset = 1;
}

/*------------------------------------------------------------------------------
*  limit - This method handles the limit emulator speed menu selection.
*-----------------------------------------------------------------------------*/
- (IBAction)limit:(id)sender
{
    requestLimitChange = 1;
}

/*------------------------------------------------------------------------------
*  loadState - This method handles the load state file menu selection.
*-----------------------------------------------------------------------------*/
- (IBAction)loadState:(id)sender
{
    NSString *filename;
    
    PauseAudio(1);
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_state_dir]];
    if (filename != nil) {
        [filename getCString:loadFilename];
        requestLoadState = 1;
    }
    [self releaseCmdKeys:@"l":QZ_l];
    PauseAudio(0);
}

/*------------------------------------------------------------------------------
*  loadStateFile - This method handles the load state file menu selection.
*-----------------------------------------------------------------------------*/
- (void)loadStateFile:(NSString *)filename
{
    if (filename != nil) {
        [filename getCString:loadFilename];
        requestLoadState = 1;
    }
}

/*------------------------------------------------------------------------------
*  pause - This method handles the pause emulator menu selection.
*-----------------------------------------------------------------------------*/
- (IBAction)pause:(id)sender
{
    if (1 - pauseEmulator)
        [pauseItem setState:NSOnState];
    else
        [pauseItem setState:NSOffState];
    requestPauseEmulator = 1;
}

/*------------------------------------------------------------------------------
*  saveState - This method handles the save state file menu selection.
*-----------------------------------------------------------------------------*/
- (IBAction)saveState:(id)sender
{
    NSString *filename;
    
    PauseAudio(1);
    filename = [self saveFileInDirectory:[NSString stringWithCString:atari_state_dir]:@"a8s"];
    if (filename != nil) {
        [filename getCString:saveFilename];
        requestSaveState = 1;
    }
    [self releaseCmdKeys:@"s":QZ_s];
    PauseAudio(0);
}

/*------------------------------------------------------------------------------
*  warmReset - This method handles the warm reset menu selection.
*-----------------------------------------------------------------------------*/
- (IBAction)warmReset:(id)sender
{
    requestWarmReset = 1;
}

/*------------------------------------------------------------------------------
*  fatalError - This method displays the fatal error dialogue box.
*-----------------------------------------------------------------------------*/
- (int)fatalError
{
    return([NSApp runModalForWindow:[errorTextField window]]);
}

/*------------------------------------------------------------------------------
*  coldStartSelected - This method handles the warm start button from the fatal 
*     error dialogue.
*-----------------------------------------------------------------------------*/
- (IBAction)coldStartSelected:(id)sender
{
    [NSApp stopModalWithCode:2];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  warmStartSelected - This method handles the warm start button from the fatal 
*     error dialogue.
*-----------------------------------------------------------------------------*/
- (IBAction)warmStartSelected:(id)sender
{
    [NSApp stopModalWithCode:1];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  monitorSelected - This method handles the monitor button from the fatal 
*     error dialogue.
*-----------------------------------------------------------------------------*/
- (IBAction)monitorSelected:(id)sender
{
    [NSApp stopModalWithCode:3];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  ejectCartridgeSelected - This method handles the eject cartridge button from 
*     the fatal error dialogue.
*-----------------------------------------------------------------------------*/
- (IBAction)ejectCartridgeSelected:(id)sender
{
    [NSApp stopModalWithCode:4];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  ejectDiskSelected - This method handles the eject disk 1 button from 
*     the fatal error dialogue.
*-----------------------------------------------------------------------------*/
- (IBAction)ejectDiskSelected:(id)sender
{
    [NSApp stopModalWithCode:5];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  quitSelected - This method handles the quit button from the fatal error
*     dialogue.
*-----------------------------------------------------------------------------*/
- (IBAction)quitSelected:(id)sender
{
    [NSApp stopModalWithCode:0];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  releaseCmdKeys - This method fixes an issue when modal windows are used with
*     the Mac OSX version of the SDL library.
*     As the SDL normally captures all keystrokes, but we need to type in some 
*     Mac windows, all of the control menu windows run in modal mode.  However, 
*     when this happens, the release of the command key and the shortcut key 
*     are not sent to SDL.  We have to manually cause these events to happen 
*     to keep the SDL library in a sane state, otherwise only everyother shortcut
*     keypress will work.
*-----------------------------------------------------------------------------*/
- (void) releaseCmdKeys:(NSString *)character:(int)keyCode
{
    NSEvent *event1, *event2;
    NSPoint point;
    
    event1 = [NSEvent keyEventWithType:NSKeyUp location:point modifierFlags:0
                    timestamp:nil windowNumber:0 context:nil characters:character
                    charactersIgnoringModifiers:character isARepeat:NO keyCode:keyCode];
    [NSApp postEvent:event1 atStart:NO];
    
    event2 = [NSEvent keyEventWithType:NSFlagsChanged location:point modifierFlags:0
                    timestamp:nil windowNumber:0 context:nil characters:nil
                    charactersIgnoringModifiers:nil isARepeat:NO keyCode:0];
    [NSApp postEvent:event2 atStart:NO];
}

/*------------------------------------------------------------------------------
*  releaseKey - This method fixes an issue when modal windows are used with
*     the Mac OSX version of the SDL library.
*     As the SDL normally captures all keystrokes, but we need to type in some 
*     Mac windows, all of the control menu windows run in modal mode.  However, 
*     when this happens, the release of function key which started the process
*     is not sent to SDL.  We have to manually cause these events to happen 
*     to keep the SDL library in a sane state, otherwise only everyother shortcut
*     keypress will work.
*-----------------------------------------------------------------------------*/
- (void) releaseKey:(int)keyCode
{
    NSEvent *event1;
    NSPoint point;
    
    event1 = [NSEvent keyEventWithType:NSKeyUp location:point modifierFlags:0
                    timestamp:nil windowNumber:0 context:nil characters:@" "
                    charactersIgnoringModifiers:@" " isARepeat:NO keyCode:keyCode];
    [NSApp postEvent:event1 atStart:NO];
}

/*------------------------------------------------------------------------------
*  monitorExecute - This method handles the execute button (or return key) from
*     the monitor window.
*-----------------------------------------------------------------------------*/
- (IBAction)monitorExecute:(id)sender
{
    char input[256];
    int retValue = 0;
   
    [[monitorInputField stringValue] getCString:input];
    [monitorInputField setStringValue:@""];
    
    monitorCharCount = 0;
    [self monitorPrint:input];
    [self monitorPrint:"\n"];
    retValue = monitorCmd(input);
    if (!assemblerMode)
        [self monitorPrint:"> "];

    [NSApp stopModalWithCode:retValue];
}

/*------------------------------------------------------------------------------
*  messageWindowShow - This method makes the emulator message window visable
*-----------------------------------------------------------------------------*/
- (void)messageWindowShow:(id)sender
{
    [[messageOutputView window] makeKeyAndOrderFront:self];
}

/*------------------------------------------------------------------------------
*  messagePrint - This method handles the printing of information to the
*     message window.  It replaces printf.
*-----------------------------------------------------------------------------*/
- (void)messagePrint:(char *)printString
{
    NSRange theEnd;
    NSString *stringObj;

    theEnd=NSMakeRange([[messageOutputView string] length],0);
    stringObj = [[NSString alloc] initWithCString:printString];
    [messageOutputView replaceCharactersInRange:theEnd withString:stringObj]; // append new string to the end
    theEnd.location += strlen(printString); // the end has moved
    [messageOutputView scrollRangeToVisible:theEnd];
}

/*------------------------------------------------------------------------------
*  monitorPrint - This method handles the printing of information from the
*     monitor routines.  It replaces printf.
*-----------------------------------------------------------------------------*/
- (void)monitorPrint:(char *)string
{
    strncpy(&monitorOutput[monitorCharCount], string, MAX_MONITOR_OUTPUT - monitorCharCount);
    monitorCharCount += strlen(string);
}

/*------------------------------------------------------------------------------
*  monitorMenuRun - This method runs the monitor from the menu, and calls the
*     normal key based monitor run.
*-----------------------------------------------------------------------------*/
- (IBAction)monitorMenuRun:(id)sender
{
    int retValue;
    
    retValue = [self monitorRun];
    if (retValue == 0) 
        requestQuit = 1;
}

/*------------------------------------------------------------------------------
*  monitorRun - This method runs the monitor, and contains the main loop, one
*     loop for each command entered.
*-----------------------------------------------------------------------------*/
-(int)monitorRun
{
    int retValue = 0;
    NSRange theEnd;
    NSString *stringObj;

    PauseAudio(1);
            
    monitorEnter();
    theEnd=NSMakeRange([[monitorOutputView string] length],0);
    stringObj = [[NSString alloc] initWithCString:monitorOutput];
    [monitorOutputView replaceCharactersInRange:theEnd withString:stringObj]; // append new string to the end
    theEnd.location += monitorCharCount; // the end has moved
    [monitorOutputView scrollRangeToVisible:theEnd];
    
    while (retValue == 0) {
        retValue = [NSApp runModalForWindow:[monitorInputField window]];
        theEnd=NSMakeRange([[monitorOutputView string] length],0);
        stringObj = [[NSString alloc] initWithCString:monitorOutput];
        [monitorOutputView replaceCharactersInRange:theEnd withString:stringObj]; // append new string to the end
        theEnd.location += monitorCharCount; // the end has moved
        [monitorOutputView scrollRangeToVisible:theEnd];
        }
    
    monitorCharCount = 0;
    [[monitorInputField window] close];
    [self releaseKey:QZ_F8];

    PauseAudio(0);
    
    if (retValue == -1)
        return(1);
    else
        return(0);
}

@end
