/*
   gif.c
         - Dave Raggett, 16th December 1994
         - revised 9th Feb 1994 to speed decoding & add gif89 support

   Derived from the pbmplus package, so we include David's copyright.

   The colours are preallocated by image.c with 4:8:4 R:G:B colours and
   16 grey scales. These are rendered using an ordered dither based on
   16x16 dither matrices for monochrome, greyscale and colour images.

   This approach ensures that we won't run out of colours regardless of
   how many images are on screen or how many independent copies of www
   are running at any time. The scheme is compatible with HP's MPOWER
   tools. The code will be extended to take advantage of 24 bit direct
   colour or secondary true colour hardware colourmaps as soon as practical.
*/

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, David Koblas.                                     | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */


#include <X11/Intrinsic.h>
#include <stdio.h>
#include <stdlib.h>

#include "arena.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "dither.h"
#include "gif_arena.h"
#include "image.h"
#include "main.h"
#include "status.h"
#include "util.h"


#define MAX_LWZ_BITS    12


static int GetDataBlock(Block* bp, unsigned char *buf);


struct
{
 unsigned int Width;
 unsigned int Height;
        Colour colours[MAXCOLOURMAPSIZE];
 unsigned int BitPixel;
 unsigned int ColourResolution;
 unsigned int Background;
 unsigned int AspectRatio;
          int xGreyScale;
} GifScreen;

struct
{
 int transparent;
 int delayTime;
 int inputFlag;
 int disposal;
} Gif89 = { -1, -1, -1, 0 };

int ZeroDataBlock = False;


size_t ReadOK(Block *bp, unsigned char *buffer, int len)
{
 int x = bp->size - bp->next;
 if (x > 0)
   {
    if (len > x) len = x;

    memcpy(buffer, bp->buffer + bp->next, len);
    bp->next += len;
    return len;
   }
  else
   return 0;
 /* End if */
}


/*
 *  Pulled out of nextCode
 */
static  int             curbit, lastbit, get_done, last_byte;
static  int             return_clear;

/*
 *  Out of nextLWZ
 */
#define InternalStackSize (1<<(MAX_LWZ_BITS))

static int      stack[InternalStackSize*2], *sp;
static int      code_size, set_code_size;
static int      max_code, max_code_size;
static int      clear_code, end_code;


static void initLWZ(int input_code_size)
{
 set_code_size = input_code_size;
 code_size     = set_code_size + 1;
 clear_code    = 1 << set_code_size ;
 end_code      = clear_code + 1;
 max_code_size = 2 * clear_code;
 max_code      = clear_code + 2;

 curbit = lastbit = 0;
 last_byte = 2;
 get_done = False;

 return_clear = True;

 sp = stack;
}


static int nextCode(Block *bp, int code_size)
{
 static unsigned char buf[280];
 static int maskTbl[16] = {
   0x0000, 0x0001, 0x0003, 0x0007,
   0x000f, 0x001f, 0x003f, 0x007f,
   0x00ff, 0x01ff, 0x03ff, 0x07ff,
   0x0fff, 0x1fff, 0x3fff, 0x7fff,
 };
 int i, j, ret, end;
#ifdef ARENA_DEBUG
 char Iam[] = "nextCode";
#endif


 if (return_clear)
   {
    return_clear = False;
    return clear_code;
   }

 end = curbit + code_size;

 if (end >= lastbit)
   {
    int     count;

    if (get_done)
      {
       if (curbit >= lastbit)
	 {
#ifdef ARENA_DEBUG
	  Arena_TracePrint(Iam, " ran off the end of my bits.\n");
# else
	  Arena_PrintError(_("(GIF) ran off the end of my bits.\n"));
#endif
	 }

       return -1;
      }

    buf[0] = buf[last_byte - 2];
    buf[1] = buf[last_byte - 1];

    if ((count = GetDataBlock(bp, &buf[2])) == 0) get_done = True;

    last_byte = 2 + count;
    curbit = (curbit - lastbit) + 16;
    lastbit = (2 + count) * 8 ;

    end = curbit + code_size;
   }

 j = end >> 3 ;
 i = curbit >> 3;

 if (i == j)
   ret = (int)buf[i];
  else
   if (i + 1 == j)
     ret = (int)buf[i] | ((int)buf[i + 1] << 8);
    else
     ret = (int)buf[i] | ((int)buf[i + 1] << 8) | ((int)buf[i + 2] << 16);

 ret = (ret >> (curbit & 7)) & maskTbl[code_size];

 curbit += code_size;

 return ret;
}


#define readLWZ(bp) ((sp > stack) ? *--sp : nextLWZ(bp))


static int nextLWZ(Block *bp)
{
   static int table[2][(1 << MAX_LWZ_BITS)];
   static int firstcode, oldcode;
          int code, incode;
 register int i;
#ifdef ARENA_DEBUG
 char Iam[] = "nextLWZ";
#endif


 while ((code = nextCode(bp, code_size)) >= 0)
   {
    if (code == clear_code)
      {
      /* corrupt GIFs can make this happen */
       if (clear_code >= (1 << MAX_LWZ_BITS))
         {
#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE) Arena_TracePrint(Iam, " corrupt GIF.\n");
#endif
          return -2;
         }

       for (i = 0; i < clear_code; ++i)
         {
          table[0][i] = 0;
          table[1][i] = i;
         }

       for (; i < (1 << MAX_LWZ_BITS); ++i) table[0][i] = table[1][i] = 0;

       code_size = set_code_size + 1;
       max_code_size = clear_code * 2;
       max_code = clear_code + 2;
       sp = stack;

       do
         {
          firstcode = oldcode = nextCode(bp, code_size);
         } while (firstcode == clear_code);

       return firstcode;
      }

    if (code == end_code)
      {
       int           count;
       unsigned char buf[260];

       if (ZeroDataBlock) return -2;

       while ((count = GetDataBlock(bp, buf)) > 0) ;

       if (count != 0)
         {
#if 0
          INFO_MSG(("missing EOD in data stream (common occurence)"));
#endif
         }
       return -2;
      }

    incode = code;

    if (code >= max_code)
      {
       *sp++ = firstcode;
       code = oldcode;
      }

    while (code >= clear_code)
      {

#ifdef ARENA_DEBUG	/* QingLong.03-11-96 */
       if ((int)sp < 0x1000)
	 {
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam,
			     " Suspicious sp = "POINTER_FORMAT", exiting.\n",
			     sp);
	  exit(-1);
	 }
#endif
       *sp++ = table[1][code];
       if (code == table[0][code])
         {
#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam, " circular table entry BIG ERROR.\n");
# else
          Arena_PrintError(_("Circular table entry BIG ERROR.\n"));
#endif
          return(code);
         }
       code = table[0][code];
      }

    *sp++ = firstcode = table[1][code];

    if ((code = max_code) < (1 << MAX_LWZ_BITS))
      {
       table[0][code] = oldcode;
       table[1][code] = firstcode;
       ++max_code;
       if ((max_code >= max_code_size) &&
	   (max_code_size < (1 << MAX_LWZ_BITS)))
	 {
	  max_code_size *= 2;
	  ++code_size;
	 }
      }

    oldcode = incode;

    if (sp > stack) return *--sp;
   }
 return code;
}


static int GetDataBlock(Block *bp, unsigned char *buf)
{
 unsigned char count;
#ifdef ARENA_DEBUG
 char Iam[] = "GetDataBlock";
#endif


 count = 0;

 if (!ReadOK(bp, &count, (int)1))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " error in getting DataBlock size.\n");
# else
    Arena_PrintError(_("Error in getting DataBlock size.\n"));
#endif
    return -1;
   }

 ZeroDataBlock = count == 0;

 if ((count != 0) && (!ReadOK(bp, buf, (int)count)))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " error in reading DataBlock.\n");
#endif
    return -1;
   }

 return ((int)count);
}


static ImageData* ReadImage1_2_4(Block* bp,
				 int len, int height,
				 Colour colours[MAXCOLOURMAPSIZE],
				 int grey, int interlace, int ignore,
				 int depth)
{
 unsigned char c;
 int           mlen, v, transparentColour;
 int           xpos = 0, ypos = 0;
 ImageData*    theImageData;
 unsigned char mShift = 0x01;
 unsigned char *dp = NULL, *mp = NULL;
 int           cr, cg, cb, r, g, b, row, col;
 Colour         colour;
 int           ppb, bpl, shift = 0;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadImage1_2_4";
#endif


/* howcome added support for 1,2,4 depths */

 if ((depth != 1) && (depth != 2) && (depth != 4))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " sorry, images are not supported for depth %d.\n",
		       depth);
# else
    Arena_PrintError(_("Sorry, images are not supported for depth %d.\n"),
		     depth);
#endif
    return NULL;
   }


 /*
  * Initialize the Compression routines
  */

 if (!ReadOK(bp, &c, 1))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " EOF/read error on image data.\n");
# else
    Arena_PrintError(_("EOF/read error on image data.\n"));
#endif
    return NULL;
   }

 initLWZ(c);


 /*
  * If this is an "uninteresting picture" ignore it.
  */

 if (ignore)
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " skipping image...\n");
#endif
    while (readLWZ(bp) >= 0);

    return NULL;
   }


 transparentColour = Gif89.transparent;

 if (transparentColour >= 0)
   mlen = len/8 + ((len%8) ? 1 : 0);	/* mask bitmap width (in bytes) */
  else
   mlen = 0;

 ppb = 8/depth; /* pixels per byte */
 bpl = len/ppb + ((len&(ppb - 1)) ? 1 : 0); /* bytes per line */

 if ((theImageData = NewImageData((bpl * height), mlen ? (mlen * height) : 0))
     == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE) Arena_TracePrint(Iam, " no space left for the image.\n");
#endif
    return NULL;
   }

#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (IMAGE_TRACE)
   Arena_TracePrint(Iam,
		    "reading %d by %d%s%s GIF image.\n",
		    len, height,
		    interlace ? " interlaced" : "",
		    mlen ? " transparent" : "");
#endif

 if (interlace)
   {
    int i;
    int pass = 0, step = 8;
    int newbyte = 1;

    for (i = 0; i < height; ++i)
      {
       newbyte = 1;
       dp = &(theImageData->image)[bpl*ypos]; /* howcome 7/2/95: len -> bpl */
       col = ypos & 0x0F;
       if (mlen)
	 {
	  mp = &((theImageData->mask)[mlen * ypos]);
	  mShift = 0x01;
	 }

       for (xpos = 0; xpos < len; ++xpos)
	 {
	  if ((v = readLWZ(bp)) < 0) goto fini;

	  if (mlen)
	    {
	     if (v == transparentColour)
	       *mp &= ~mShift;
	      else
	       *mp |= mShift;

	     if (mShift == 0x80)
	       {
		mShift = 0x01;
		mp++;
	       }
	      else
	       mShift <<= 1;
	    }

	  if (newbyte)
	    {
	     *dp = 0;
	     newbyte = 0;
	    }

	  colour = colours[v];
	  row = xpos & 0x0F;

	  if (!grey && imaging == COLOUR232)
	    {
	     if (colour.grey > 0) goto grey_colour1;

	     cr = colour.red;
	     cg = colour.green;
	     cb = colour.blue;

	     r = cr & 0xC0;
	     g = cg & 0xE0;
	     b = cb & 0xC0;

	     v = (row << 4) + col;

	     if (cr - r > Magic64[v]) r += 64;
	     if (cg - g > Magic32[v]) g += 32;
	     if (cb - b > Magic64[v]) b += 64;

	     /* clamp error to keep colour in range 0 to 255 */

	     r = min(r, 255) & 0xC0;
	     g = min(g, 255) & 0xE0;
	     b = min(b, 255) & 0xC0;

/*
 * Some quick notes to remeber how shift is calculated:
 *
 *   (%8 = &7) --- 12-Mar-96 <herman@htbrug.hobby.nl>
 *
 *  0 1 2 3 4 5 6 7          xpos
 *  0 1 2 3 4 5 6 7          xpos % 8
 *  7 6 5 4 3 2 1 0     7 - (xpos % 8)   (%8 = &7)
 *
 *  1 0 1 0 1 0 1 0    (7 - (xpos % 8)) % ppb           pixels per byte = 2
 *  4 0 4 0 4 0 4 0   ((7 - (xpos % 8)) % ppb) * depth  depth = 4
 *
 *  3 2 1 0 3 2 1                                       pixels per byte = 4
 *  6 4 2 0 6 4 2                                       depth = 2
 */

	     shift = (((xpos & 7) % ppb) * depth);
	     *dp |= (stdcmap[(r >> 6) | (g >> 3) | (b >> 1)]) << shift;
	     if ((shift + depth) == 8)
	       {
		dp++;
		newbyte = 1;
	       }

	     continue;
	    }

	  if (imaging == MONO)
	    {
	     shift = (((xpos & 7) % ppb) * depth);

	     if (colour.grey < Magic256[(row << 4) + col])
	       *dp |= greymap[0] << shift;
	      else
	       *dp |= greymap[15] << shift;

	     if ((shift + depth) == 8)
	       {
		dp++;
		newbyte = 1;
	       }

	     continue;
	    }

	 grey_colour1:

	  cg  = colour.grey;
	  g = cg & 0xF0;

	  if (cg - g > Magic16[(row << 4) + col]) g += 16;

	  g = min(g, 0xF0);

	  shift = (((xpos & 7) % ppb) * depth);

	  *dp |= greymap[g >> 4] << shift;

	  if ((shift + depth) == 8)
	    {
	     dp++;
	     newbyte = 1;
	    }
	 }

       if ((ypos += step) >= height)
	 {
	  do
	    {
	     if (pass++ > 0) step >>= 1;
	     ypos = step >> 1;
	    } while (ypos >= height); /* DTRT when height <= 4 */
	 }
      }
   }
  else
   {
    dp = theImageData->image;
    if (mlen)
      {
       mp = theImageData->mask;
       mShift = 0x01;
      }

    for (ypos = 0; ypos < height; ++ypos)
      {
       int newbyte = 1;

       col = ypos & 0x0F;
       if (mShift != 0x01)	/* Force aligning to byte border */
	 {
	  mShift = 0x01;
	  mp++;
	 }

       for (xpos = 0; xpos < len; ++xpos)
	 {
	  if ((v = readLWZ(bp)) < 0) goto fini;

	  if (mlen)
	    {
	     if (v == transparentColour)
	       *mp &= ~mShift;
	      else
	       *mp |= mShift;

	     if (mShift == 0x80)
	       {
		mShift = 0x01;
		mp++;
	       }
	      else
	       mShift <<= 1;
	    }

	  if(newbyte)
	    {
	     *dp = 0;
	     newbyte = 0;
	    }

	  colour = colours[v];
	  row = xpos & 0x0F;

	  if (!grey && imaging == COLOUR232)
	    {
	     if (colour.grey > 0) goto grey_colour2;

	     cr = colour.red;
	     cg = colour.green;
	     cb = colour.blue;

	     r = cr & 0xC0;
	     g = cg & 0xE0;
	     b = cb & 0xC0;

	     v = (row << 4) + col;

	     if (cr - r > Magic64[v]) r += 64;
	     if (cg - g > Magic32[v]) g += 32;
	     if (cb - b > Magic64[v]) b += 64;

	     /* clamp error to keep colour in range 0 to 255 */

	     r = min(r, 255) & 0xC0;
	     g = min(g, 255) & 0xE0;
	     b = min(b, 255) & 0xC0;

	     shift = (((xpos & 7) % ppb) * depth);

	     *dp |= stdcmap[(r >> 6) | (g >> 3) | (b >> 1)] << shift;

	     if ((shift + depth) == 8)
	       {
		dp++;
		newbyte = 1;
	       }

	     continue;
	    }

	  if (imaging == MONO)
	    {
	     shift = (((xpos & 7) % ppb) * depth);

	     if (colour.grey < Magic256[(row << 4) + col])
	       *dp |= greymap[0] << shift;
	      else
	       *dp |= greymap[15] << shift;

	     if ((shift + depth) == 8)
	       {
		dp++;
		newbyte = 1;
	       }

	     continue;
	    }

	 grey_colour2:

	  cg  = colour.grey;
	  g = cg & 0xF0;

	  if (cg - g > Magic16[(row << 4) + col]) g += 16;

	  g = min(g, 0xF0);

	  shift = (((xpos & 7) % ppb) * depth);

	  *dp |= greymap[g >> 4] << shift;

	  if ((shift + depth) == 8)
	    {
	     dp++;
	     newbyte = 1;
	    }
	 }
       if (shift)
	 {
	  dp++;   /* make sure we start on a new byte for the next line */
	  newbyte = 1;
	 }
      }
   }

 fini:

#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (readLWZ(bp) >= 0)
   if (IMAGE_TRACE && VERBOSE_TRACE)
     Arena_TracePrint(Iam, " too much input data, ignoring extra...\n");
#endif
 return theImageData;
}


static ImageData* ReadImage(Block *bp,
			    int len, int height,
			    Colour colours[MAXCOLOURMAPSIZE],
			    int grey, int interlace, int ignore)
{
 unsigned char c;      
 int           mlen, v, transparentColour;
 int           xpos = 0, ypos = 0;
 ImageData*    theImageData;
 unsigned char mShift = 0x01;
 unsigned char *dp = NULL, *mp = NULL;
 int           cr, cg, cb, r, g, b, row, col;
 Colour         colour;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadImage";
#endif


 /*
  *  Initialize the Compression routines
  */

 if (!ReadOK(bp, &c, 1))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " EOF/read error on image data.\n");
# else
    Arena_PrintError(_("EOF/read error on image data.\n"));
#endif
    return(NULL);
   }

 initLWZ(c);

 /*
  *  If this is an "uninteresting picture" ignore it.
  */

 if (ignore)
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " skipping image...\n" );
#endif
    while (readLWZ(bp) >= 0);

    return NULL;
   }


 transparentColour = Gif89.transparent;

 if (transparentColour >= 0)
   mlen = len/8 + ((len%8) ? 1 : 0);	/* mask bitmap width (in bytes) */
  else
   mlen = 0;


 if ((theImageData = NewImageData((len * height), mlen ? (mlen * height) : 0))
     == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE) Arena_TracePrint(Iam, " no space left for the image.\n");
#endif
    return NULL;
   }


#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (IMAGE_TRACE)
   Arena_TracePrint(Iam,
		    " reading %d by %d%s%s GIF image.\n",
		    len, height,
		    interlace ? " interlaced" : "",
		    mlen ? " transparent" : "");
#endif

 if (interlace)
   {
    int i;
    int pass = 0, step = 8;

    for (i = 0; i < height; ++i)
      {
       dp = &((theImageData->image)[len * ypos]);
       col = ypos & 0xF;
       if (mlen)
	 {
	  mp = &((theImageData->mask)[mlen * ypos]);
	  mShift = 0x01;
	 }

       for (xpos = 0; xpos < len; ++xpos)
	 {
	  if ((v = readLWZ(bp)) < 0) goto fini;

	  if (mlen)
	    {
	     if (v == transparentColour)
	       *mp &= ~mShift;
	      else
	       *mp |= mShift;

	     if (mShift == 0x80)
	       {
		mShift = 0x01;
		mp++;
	       }
	      else
	       mShift <<= 1;
	    }

	  colour = colours[v];
	  row = xpos & 0x0F;

	  if (!grey && imaging == COLOUR232)
	    {
	     if (colour.grey > 0) goto grey_colour1;

	     cr = colour.red;
	     cg = colour.green;
	     cb = colour.blue;

	     r = cr & 0xC0;
	     g = cg & 0xE0;
	     b = cb & 0xC0;

	     v = (row << 4) + col;

	     if (cr - r > Magic64[v]) r += 64;
	     if (cg - g > Magic32[v]) g += 32;
	     if (cb - b > Magic64[v]) b += 64;

	     /* clamp error to keep colour in range 0 to 255 */

	     r = min(r, 255) & 0xC0;
	     g = min(g, 255) & 0xE0;
	     b = min(b, 255) & 0xC0;

	     *dp++ = stdcmap[(r >> 6) | (g >> 3) | (b >> 1)];

	     continue;
	    }

	  if (imaging == MONO)
	    {
	     if (colour.grey < Magic256[(row << 4) + col])
	       *dp++ = greymap[0];
	      else
	       *dp++ = greymap[15];

	     continue;
	    }

	 grey_colour1:

	  cg  = colour.grey;
	  g = cg & 0xF0;

	  if (cg - g > Magic16[(row << 4) + col]) g += 16;

	  g = min(g, 0xF0);
	  *dp++ = greymap[g >> 4];
	 }

       if ((ypos += step) >= height)
	 {
	  do
	    {
	     if (pass++ > 0) step >>= 1;
	     ypos = step >> 1;
	    } while (ypos >= height); /* DTRT when height <= 4 */
	 }
      }
   }
  else
   {
    dp = theImageData->image;
    if (mlen)
      {
       mp = theImageData->mask;
       mShift = 0x01;
      }

    for (ypos = 0; ypos < height; ++ypos)
      {
       col = ypos & 0xF;
       if (mShift != 0x01)	/* Force aligning to byte border */
	 {
	  mShift = 0x01;
	  mp++;
	 }

       for (xpos = 0; xpos < len; ++xpos)
	 {
	  if ((v = readLWZ(bp)) < 0) goto fini;

	  if (mlen)
	    {
	     if (v == transparentColour)
	       *mp &= ~mShift;
	      else
	       *mp |= mShift;

	     if (mShift == 0x80)
	       {
		mShift = 0x01;
		mp++;
	       }
	      else
	       mShift <<= 1;
	    }

	  colour = colours[v];
	  row = xpos & 0x0F;

	  if (!grey && imaging == COLOUR232)
	    {
	     if (colour.grey > 0) goto grey_colour2;

	     cr = colour.red;
	     cg = colour.green;
	     cb = colour.blue;

	     r = cr & 0xC0;
	     g = cg & 0xE0;
	     b = cb & 0xC0;

	     v = (row << 4) + col;

	     if (cr - r > Magic64[v]) r += 64;
	     if (cg - g > Magic32[v]) g += 32;
	     if (cb - b > Magic64[v]) b += 64;

	     /* clamp error to keep colour in range 0 to 255 */

	     r = min(r, 255) & 0xC0;
	     g = min(g, 255) & 0xE0;
	     b = min(b, 255) & 0xC0;

	     *dp++ = stdcmap[(r >> 6) | (g >> 3) | (b >> 1)];

	     continue;
	    }

	  if (imaging == MONO)
	    {
	     if (colour.grey < Magic256[(row << 4) + col])
	       *dp++ = greymap[0];
	      else
	       *dp++ = greymap[15];

	     continue;
	    }

	 grey_colour2:

	  cg  = colour.grey;
	  g = cg & 0xF0;

	  if (cg - g > Magic16[(row << 4) + col]) g += 16;

	  g = min(g, 0xF0);
	  *dp++ = greymap[g >> 4];
	 }
      }
   }

fini:

#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (readLWZ(bp) >= 0)
   if (IMAGE_TRACE && VERBOSE_TRACE)
     Arena_TracePrint(Iam, " too much input data, ignoring extra...\n");
#endif

 return theImageData;
}


static ImageData* ReadImage12_16_24(Block* bp,
				    int len, int height,
				    Colour colours[MAXCOLOURMAPSIZE],
				    int nColours, int interlace, int ignore,
				    int depth)
{
 char x;
 unsigned char c;
 int           bpp;	/* bytes per pixel */
 int           mlen, v, transparentColour;
 int           xpos = 0, ypos = 0;
 ImageData*    theImageData;
 unsigned char mShift = 0x01;
 unsigned char *dp = NULL, *mp = NULL;
 unsigned char *pixels = NULL, *pp;
 unsigned long int ulp;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadImage12_16_24";
#endif

 switch (depth)
   {
    case 12:
      bpp = 4;
      break;

    case 16:
      bpp = 2;
      break;

    case 24:
      bpp = 4;
      break;

    default:
#ifdef ARENA_DEBUG
      if (IMAGE_TRACE)
	Arena_TracePrint(Iam, " depth %d isn't supported!\n", depth);
#endif
      return NULL;
      break;
   }
 /* End ``switch (depth)'' */


 if ((pixels = (unsigned char *)Arena_CAlloc((MAXCOLOURMAPSIZE * bpp),
					     sizeof(unsigned char),
					     False))
     == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " no space left for image data (pixels).\n");
#endif
    return NULL;
   }

 /* setup pixel table for faster rendering */

 pp = (unsigned char *)&(pixels[0]);

 transparentColour = Gif89.transparent;

 if (transparentColour >= 0)
   mlen = len/8 + ((len%8) ? 1 : 0);	/* mask bitmap width (in bytes) */
  else
   mlen = 0;


 for (v = 0; v < nColours; ++v)
   {
    if (mlen)
      {
       if (v == transparentColour)
	 {
	  *pp++ = (windowColour     ) & 0xFF;   /* LSB first! */
	  *pp++ = (windowColour >> 8) & 0xFF;

	  if (bpp == 4)
	    {
	     *pp++ = (windowColour >> 16) & 0xFF;
	     *pp++ = '\0';
	    }

	  continue;
	 }
      }

    /* From Scott Nelson <snelson@canopus.llnl.gov> 1/12/94  24bit colour */

    GetColour(colours[v].red, colours[v].green, colours[v].blue, &ulp);

    if (bpp == 2)
      {
       *pp++ = ((char*)&ulp)[0];   /* LSB first! */
       *pp++ = ((char*)&ulp)[1]; 
      }
     else
      {
       *pp++ = (ulp      ) & 0xff;   /* LSB first! */
       *pp++ = (ulp >>  8) & 0xff; 
       *pp++ = (ulp >> 16) & 0xff; 
       *pp++ = '\0';
      }
    }


 /*
  *  Initialize the Compression routines
  */

 if (!ReadOK(bp, &c, 1))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " EOF/read error on image data.\n");
# else
    Arena_PrintError(_("EOF/read error on image data.\n"));
#endif
    Free(pixels);
    return NULL;
   }

 initLWZ(c);


 /*
  *  If this is an "uninteresting picture" ignore it.
  */

 if (ignore)
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " skipping image...\n" );
#endif
    while (readLWZ(bp) >= 0);
    Free(pixels);
    return NULL;
   }


 if ((theImageData = NewImageData((len * height * bpp),
				  mlen ? (mlen * height) : 0))
     == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE) Arena_TracePrint(Iam, " no space left for the image.\n");
#endif
    Free(pixels);
    return NULL;
   }


#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (IMAGE_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    " reading %d by %d%s%s GIF image.\n",
		    len, height,
		    interlace ? " interlaced" : "",
		    mlen ? " transparent" : "");
#endif

 if (interlace)
   {
    int i;
    int pass = 0, step = 8;

    for (i = 0; i < height; ++i)
      {
       dp = &(theImageData->image)[len * ypos * bpp];

       if (mlen)
	 {
	  mp = &((theImageData->mask)[mlen * ypos]);
	  mShift = 0x01;
	 }

       for (xpos = 0; xpos < len; ++xpos)
	 {
	  if ((v = readLWZ(bp)) < 0) goto fini;

	  if (mlen)
	    {
	     if (v == transparentColour)
	       *mp &= ~mShift;
	      else
	       *mp |= mShift;

	     if (mShift == 0x80)
	       {
		mShift = 0x01;
		mp++;
	       }
	      else
	       mShift <<= 1;
	    }

	  for (x = bpp; x; x--) *dp++ = pixels[(v + 1) * bpp - x];
	 }

       if ((ypos += step) >= height)
	 {
	  do
	    {
	     if (pass++ > 0) step >>= 1;
	     ypos = step >> 1;
	    } while (ypos >= height); /* DTRT when height <= 4 */
	 }
      }
   }
  else
   {
    dp = theImageData->image;
    if (mlen)
      {
       mp = theImageData->mask;
       mShift = 0x01;
      }

    for (ypos = 0; ypos < height; ++ypos)
      {
       if (mShift != 0x01)	/* Force aligning to byte border */
	 {
	  mShift = 0x01;
	  mp++;
	 }

       for (xpos = 0; xpos < len; ++xpos)
	 {
	  if ((v = readLWZ(bp)) < 0) goto fini;

	  if (mlen)
	    {
	     if (v == transparentColour)
	       *mp &= ~mShift;
	      else
	       *mp |= mShift;

	     if (mShift == 0x80)
	       {
		mShift = 0x01;
		mp++;
	       }
	      else
	       mShift <<= 1;
	    }

	  {
	   unsigned char* xpixels = &(pixels[v * bpp]);

	   for (x = bpp; x; x--) *dp++ = *xpixels++;
	  }
	 }
      }
   }

 fini:

#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (readLWZ(bp) >= 0)
   if (IMAGE_TRACE && VERBOSE_TRACE)
     Arena_TracePrint(Iam, " too much input data, ignoring extra...\n");
#endif
 Free(pixels);
 return theImageData;
}



static int DoExtension(Block *bp, int label)
{
 static char buf[256];
 char *str;
#ifdef ARENA_DEBUG
 char Iam[] = "DoExtension";
#endif


 switch (label)
   {
    case 0x01:              /* Plain Text Extension */
      str = "Plain Text Extension";
      break;

    case 0xff:              /* Application Extension */
      str = "Application Extension";
      break;

    case 0xfe:              /* Comment Extension */
      str = "Comment Extension";
      while (GetDataBlock(bp, (unsigned char*) buf) != 0)
	{
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
	 if (IMAGE_TRACE && VERBOSE_TRACE)
	   Arena_TracePrint(Iam, " gif comment: \"%s\".\n", buf);
#endif
	}
      return False;

    case 0xf9:              /* Graphic Control Extension */
      str = "Graphic Control Extension";
      (void) GetDataBlock(bp, (unsigned char*) buf);
      Gif89.disposal  = (buf[0] >> 2) & 0x7;
      Gif89.inputFlag = (buf[0] >> 1) & 0x1;
      Gif89.delayTime = LM_to_uint(buf[1], buf[2]);

      if ((buf[0] & 0x1) != 0)
	Gif89.transparent = (int)((unsigned char)buf[3]);

      while (GetDataBlock(bp, (unsigned char*) buf) != 0);
      return False;

    default:
      str = buf;
      sprintf(buf, "UNKNOWN (0x%02x)", label);
      break;
   }

#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
 if (IMAGE_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " got a \"%s\" extension.\n", str);
#endif

 while (GetDataBlock(bp, (unsigned char*) buf) != 0);

 return False;
}


static int ReadColourMap(Block *bp, int ncolours, Colour *colours, int *grey)
{
 int i, flag, drb, dgb, dbr;
 unsigned char rgb[3];
#ifdef ARENA_DEBUG
 char Iam[] = "ReadColourMap";
#endif


 flag = 1;

 if (imaging == GREY4 || imaging == MONO) *grey = flag = 1;

 for (i = 0; i < ncolours; ++i)
   {
    if (!ReadOK(bp, rgb, sizeof(rgb)))
      {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
       if (IMAGE_TRACE) Arena_TracePrint(Iam, " bad colourmap.\n");
# else
       Arena_PrintError(_("Bad colourmap.\n"));
#endif
       return 0;
      }

    colours->red   = rgb[0];
    colours->green = rgb[1];
    colours->blue  = rgb[2];


    /* apply gamma correction to map voltages to brightness values */

    if (imaging != COLOUR888)
      {
       colours->red   = Voltage2Brightness(colours->red);
       colours->green = Voltage2Brightness(colours->green);
       colours->blue  = Voltage2Brightness(colours->blue);
      }
#if 0
    /* Always gamma correct palette (if needed) */
    /* if(gamma_table) {*/
    colours->red   = gamma_table[colours->red];
    colours->green = gamma_table[colours->green];
    colours->blue  = gamma_table[colours->blue];
    /*	}*/
#endif
    /* set grey value iff colour is very close to grey (or GREY/MONO imaging) */

    drb = abs(rgb[1] - rgb[0]);
    dgb = abs(rgb[2] - rgb[1]);
    dbr = abs(rgb[0] - rgb[2]);

    if (*grey || (drb < 20 && dgb < 20 && dbr < 20))
      {   
       flag &= 1;
       colours->grey = (30 * colours->red +
			59 * colours->green +
			11 * colours->blue)/100;
      }
     else
      {
       if (!(*grey)) flag = 0;
       colours->grey = 0;
      }

    ++colours;
   }

 *grey = flag;
 return 1;
}


ImageXyoke* LoadGIFimage(Block *bp, unsigned int depth)
{
 unsigned char buf[16];
 ImageData*    theImageData = NULL;
 ImageXyoke*   theImageXcouple = NULL;
 unsigned int  imageWidth = 0, imageHeight = 0;
 unsigned char c;
 Colour         *cmap, colours[MAXCOLOURMAPSIZE];
 int           useGlobalColourmap;
 int           bitPixel;
 int           imageCount = 0;
 char          version[4];
 int           imageNumber = 1;
 int           greyScale = 0;
#ifdef ARENA_DEBUG
 char Iam[] = "LoadGIFimage";
#endif


 /* initialize GIF89 extensions */

 Gif89.transparent = -1;
 Gif89.delayTime   = -1;
 Gif89.inputFlag   = -1;
 Gif89.disposal    = 0;

 if (!ReadOK(bp, buf, (int)6))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " error reading magic number.\n");
# else
    Arena_PrintError(_("Error reading magic number.\n"));
#endif
    return NULL;
   }

 if (strncmp((char *)buf, "GIF", 3) != 0)
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE) Arena_TracePrint(Iam, " not a GIF file.\n");
#endif
    return NULL;
   }

 strncpy(version, (char *)buf + 3, 3);
 version[3] = '\0';

 if ((strcmp(version, "87a") != 0) && (strcmp(version, "89a") != 0))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " bad version number, not \"87a\" or \"89a\".\n");
# else
    Arena_PrintError(_("Bad version number, not \"87a\" or \"89a\".\n"));
#endif
    return NULL;
   }

 if (!ReadOK(bp, buf, 7))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " failed to read screen descriptor.\n");
# else
    Arena_PrintError(_("Failed to read screen descriptor.\n"));
#endif
    return NULL;
   }

 GifScreen.Width           = LM_to_uint(buf[0], buf[1]);
 GifScreen.Height          = LM_to_uint(buf[2], buf[3]);
 GifScreen.BitPixel        = 2 << (buf[4] & 0x07);
 GifScreen.ColourResolution =    (((buf[4] & 0x70) >> 3) + 1);
 GifScreen.Background      = buf[5];
 GifScreen.AspectRatio     = buf[6];
 GifScreen.xGreyScale      = 0;

 if (BitSet(buf[4], LOCALCOLOURMAP))
   {    /* Global Colourmap */
    if (!ReadColourMap(bp,
		       GifScreen.BitPixel,
		       GifScreen.colours,
		       &GifScreen.xGreyScale))
      {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " error reading global colourmap.\n");
# else
       Arena_PrintError(_("Error reading global colourmap.\n"));
#endif
       return NULL;
      }
   }

 if (GifScreen.AspectRatio != 0 && GifScreen.AspectRatio != 49)
   {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, "\n\tWarning:  non-square pixels!\n");
# else
    Arena_PrintError(_("Warning: non-square pixels!\n"));
#endif
   }

 while (theImageData == NULL)
   {
    if (!ReadOK(bp, &c, 1))
      {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " EOF/read error on image data.\n");
# else
       Arena_PrintError(_("EOF/read error on image data.\n"));
#endif
       return NULL;
      }

    if (c == ';')
      {         /* GIF terminator */
       if (imageCount < imageNumber)
	 {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam, " No images found in file.\n");
# else
	  Arena_PrintError(_("No images found in file.\n"));
#endif
	  return NULL;
	 }
       break;
      }

    if (c == '!')
      {         /* Extension */
       if (!ReadOK(bp, &c, 1))
	 {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam,
			     " EOF/read error on extention function code.\n");
# else
	  Arena_PrintError(_("EOF/read error on extention function code.\n"));
#endif
	  return NULL;
	 }

       DoExtension(bp, c);
       continue;
      }

    if (c != ',')
      {         /* Not a valid start character */
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " bogus character 0x%02x, ignoring.\n", (int)c);
# else
       Arena_PrintError(_("Bogus character 0x%02x, ignoring.\n"), (int)c);
#endif
       continue;
      }

    ++imageCount;

    if (!ReadOK(bp, buf, 9))
      {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " couldn't read left/top/width/height.\n");
# else
       Arena_PrintError(_("Couldn't read left/top/width/height.\n"));
#endif
       return NULL;
      }

    useGlobalColourmap = ! BitSet(buf[8], LOCALCOLOURMAP);

    bitPixel = 1 << ((buf[8] & 0x07) + 1);

    /* Just set width/height for the imageNumber we are requesting */

    if (imageCount == imageNumber)
      {
       imageWidth  = LM_to_uint(buf[4], buf[5]);
       imageHeight = LM_to_uint(buf[6], buf[7]);
      }

    if (!useGlobalColourmap)
      {
       if (!ReadColourMap(bp, bitPixel, colours, &greyScale))
	 {
#ifdef ARENA_DEBUG	/* QingLong.26-01-97 */
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam, " error reading local colourmap.\n");
# else
	  Arena_PrintError(_("Error reading local colourmap.\n"));
#endif
	  return NULL;
	 }

       cmap = colours;
      }
     else
      {
       cmap      = GifScreen.colours;
       bitPixel  = GifScreen.BitPixel;
       greyScale = GifScreen.xGreyScale;
      }

    if ((depth == 12) || (depth == 16) || (depth == 24))
      {
       theImageData = ReadImage12_16_24(bp,
					LM_to_uint(buf[4], buf[5]),
					LM_to_uint(buf[6], buf[7]),
					cmap, bitPixel,
					BitSet(buf[8], INTERLACE),
					imageCount != imageNumber,
					depth);
      }
     else
      if (depth == 1 || depth == 2 || depth == 4)
	{
	 theImageData = ReadImage1_2_4(bp,
				       LM_to_uint(buf[4], buf[5]),
				       LM_to_uint(buf[6], buf[7]),
				       cmap, greyScale,
				       BitSet(buf[8], INTERLACE),
				       imageCount != imageNumber,
				       depth);
	}
       else
	{
	 theImageData = ReadImage(bp,
				  LM_to_uint(buf[4], buf[5]),
				  LM_to_uint(buf[6], buf[7]),
				  cmap, greyScale,
				  BitSet(buf[8], INTERLACE),
				  imageCount != imageNumber);
	}

    if ((imageCount != imageNumber) && (theImageData != NULL))
      {
       FreeImageData(theImageData);
       theImageData = NULL;
      }
   }

 if ((imageWidth == 0) || (imageHeight == 0))
   {
#ifdef ARENA_DEBUG
    Arena_TracePrint(Iam, " ERROR! No image found in the file.\n");
#endif
    Warn(_("Error! No image found in the file."));
    FreeImageData(theImageData);
    return NULL;
   }

 theImageXcouple = processImage_data2image(theImageData,
					   imageWidth, imageHeight, depth);
 theImageData->image = theImageData->mask = NULL;
 FreeImageData(theImageData);

 return theImageXcouple;
}
