/* File: blinkd.c
   (C) 1998 W. Borgert debacle@debian.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static char *rcsid="@(#) $Id: blinkd.c,v 1.1.1.1 1998/10/14 18:33:23 debacle Exp $";

#include <stdio.h>
#include <getopt.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
#include <paths.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/kd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <errno.h>

/* macros */
#define VERSION         "0.1"
#define PROGRAM         "blinkd"
#define KEYBOARDDEVICE	"/dev/console"
#define NUMLOCKLED	2
#define PWLENGTH        8       /* length of password */
#define SCROLLLOCKLED	3       /* default */
#define SLEEPFACTOR     100000	/* tenth of a second in micro seconds */

/* type definitions */
typedef enum {CLEAR, SET, TOGGLE} LedMode;

/* function prototypes */
static void clear_led_on_exit (void);
static void cleanup_on_signal (int);
static void control_led       (LedMode);
static int  create_socket     (void);
static void daemon_start      (void);
static void loop              (int);
static void process_opts      (int , char **);
static void usage             (char *);
static void wrong_use         (char *);

/* global variables */
static int  keyboardDevice         = 0;
static int  led                    = SCROLLLOCKLED;
static char passWord[PWLENGTH + 1] = "";
static int  serv_tcp_port          = SERV_TCP_PORT;
static int  off_time               = 2;
static int  pause_time             = 6;
static int  rate                   = 0;
static int  on_time                = 2;
static int  children               = 0;
static int  childpid               = 0;

/* main - does not return */
int
main (int argc,
      char **argv)
{
  int sockfd = 0;

  process_opts (argc, argv);
  daemon_start ();              /* start one or two daemons */
  sockfd = create_socket ();
  if (atexit (&clear_led_on_exit))
  {
    syslog (LOG_ERR, PROGRAM ": atexit() error\n");
  }
  signal (SIGTERM, cleanup_on_signal);
  loop (sockfd);
  return 0;                     /* never */
}

/* clear_led_on_exit - no hanging LED after exiting blinkd, please */
static void
clear_led_on_exit (void)
{
  if (keyboardDevice)
  {
    control_led (CLEAR);
  }
}

/* cleanup_on_signal - no hanging LED after killing blinkd, please */
static void
cleanup_on_signal (int x)
{
  if (childpid)
  {
    if (kill (childpid, SIGTERM) == -1)
    {
      syslog (LOG_ERR, PROGRAM ": kill() %m\n");
    }
  }
  clear_led_on_exit ();
  exit (x);
}

/* control_led - switch LED on or off

   This is more or less stolen from the tleds progam, written by
   Jouni.Lohikoski@iki.fi, any bugs in this routine are added by me.
*/
static void
control_led (LedMode mode)
{
  char	ledVal;
  struct {
    int	led_mode;
    int	led;
  } values;
  switch (mode)
  {
    case SET:
      values.led_mode = 1;
      break;
    case CLEAR:
      values.led_mode = 0;
      break;
    case TOGGLE:
      values.led_mode = values.led_mode? 0: 1;
  }
  values.led = led;
  if (ioctl (keyboardDevice, KDGETLED, &ledVal))
  {
    syslog (LOG_ERR, PROGRAM ": ioctl() %m\n");
  }
  switch (led)
  {
    case SCROLLLOCKLED:
      if (mode == SET)
      {
        ledVal |= LED_SCR;
      }
      else
      {
        ledVal &= ~LED_SCR;
      }
      break;
    case NUMLOCKLED:
      if (mode == SET)
      {
        ledVal |= LED_NUM;
      }
      else
      {
        ledVal &= ~LED_NUM;
      }
      break;
    default:
      syslog (LOG_ERR, PROGRAM ": internal error\n");
      break;
  }
  if (ioctl (keyboardDevice, KDSETLED, ledVal))
  {
    syslog (LOG_ERR, PROGRAM ": ioctl() %m\n");
  }
}

/* create_socket - create network socket for incoming tcp connection */
static int
create_socket (void)
{
  int                sockfd;
  struct sockaddr_in serv_addr;

  if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
  {
    syslog (LOG_ERR, PROGRAM ": socket() %m\n");
    exit (1);
  }
  bzero ((char *) &serv_addr, sizeof (serv_addr));
  serv_addr.sin_family      = AF_INET;
  serv_addr.sin_addr.s_addr = htonl (INADDR_ANY);
  serv_addr.sin_port        = htons (serv_tcp_port);
  if (bind (sockfd, &serv_addr, sizeof (serv_addr)) == -1)
  {
    syslog (LOG_ERR, PROGRAM ": bind() %m\n");
    exit (1);
  }
  if (fcntl (sockfd, F_SETFL, O_NONBLOCK) == -1)
  {
    syslog (LOG_ERR, PROGRAM ": fcntl() %m\n");
    exit (1);
  }
  if (listen (sockfd, 5) == -1)
  {
    syslog (LOG_ERR, PROGRAM ": listen() %m\n");
    exit (1);
  }
  return sockfd;
}

/* daemon_start - become process group leader, disconnect from tty etc. */
static void
daemon_start (void)
{
  int fd;

  if (getppid () != 1)
  {
    signal (SIGTTOU, SIG_IGN);
    signal (SIGTTIN, SIG_IGN);
    signal (SIGTSTP, SIG_IGN); 
    if ((childpid = fork ()) < 0)
    {
      perror ("fork");
      exit (1);
    }
    else if (childpid > 0)
    {
      exit (0);			/* parent */
    }
    if (setpgrp () == -1)
    {
      perror ("setpgrp");
      exit (1);
    }
    signal (SIGHUP, SIG_IGN);
    if ((childpid = fork ()) < 0)
    {
      perror ("fork");
      exit (1);
    }
    else if (childpid > 0)
    {
      exit (0);			/* parent */
    }

    if (children)               /* we have to serve both LEDs */
    {
      if ((childpid = fork ()) < 0)
      {
        perror ("fork");
        exit (1);
      }
      else if (childpid > 0)    /* parent takes care of the Num-Lock LED
                                   and uses the higher TCP port */
      {
        serv_tcp_port++;
        led = NUMLOCKLED;
      }
    }

  }
  else
  {
    for (fd = 0; fd < _POSIX_OPEN_MAX; fd++)
    {
      if (close (fd) == -1)
      {
        syslog (LOG_ERR, PROGRAM ": close() %m\n");
      }
    }
    if (chdir ("/tmp") == -1)
    {
      syslog (LOG_ERR, PROGRAM ": chdir() %m\n");
    }
    umask (0);
  }
}

/* loop - endless loop, wait for tcp connections and blink LED */
static void
loop (int sockfd)
{
  char               c = 'x';
  unsigned int       clilen, i, rr;
  struct sockaddr_in cli_addr;
  int                newsockfd;

  clilen = sizeof (cli_addr);
  /* The main loop */
  while (1)
  {
    newsockfd = accept (sockfd, &cli_addr, &clilen);
    if (newsockfd == -1)
    {
      if (errno != EAGAIN)
      {
        syslog (LOG_ERR, PROGRAM ": accept() %m\n");
      }
    }
    else
    {
      if ((rr = read (newsockfd, &c, 1)) == 1)
      {
        rate = c;
      }
      else if (rr == -1)
      {
        syslog (LOG_ERR, PROGRAM ": read() %m\n");
      }
      else if (rr)
      {
        syslog (LOG_ERR, PROGRAM ": read() returned %d\n", rr);
      }
      if (close (newsockfd) == -1)
      {
        syslog (LOG_ERR, PROGRAM ": close() %m\n");
      }
    }
    if (rate)                   /* only if there is something to do */
    {
                                /* we have to open/close permanently,
                                   to follow the current virtual tty */
      if ((keyboardDevice = open (KEYBOARDDEVICE, O_RDONLY)) == -1)
      {
        syslog (LOG_ERR, PROGRAM ": open() %m\n");
      }
      for (i = 0; i < rate; i++)
      {
        control_led (SET);
        usleep (on_time * SLEEPFACTOR);
        control_led (CLEAR);
        usleep (off_time * SLEEPFACTOR);
      }
      usleep (pause_time * SLEEPFACTOR);
      if (close (keyboardDevice) == -1)
      {
        syslog (LOG_ERR, PROGRAM ": close() %m\n");
      }
      keyboardDevice = 0;       /* reset for clear_led_on_exit */
    }
    else
    {
      sleep (1);
    }
  }
}

/* process_opts - process command line, see function usage() for options */
static void
process_opts (int argc,
	      char **argv)
{
  int c           = 0,
    i             = 0,
    off_time_flag = 0,
    led_flag      = 0,
    on_time_flag  = 0,
    pause_flag    = 0,
    rate_flag     = 0,
    tcp_flag      = 0;

  while (1)
  {
    int option_index                    = 0;
    static struct option long_options[] =
    {
      {"both-leds",     0, 0, 'b'},
      {"off-time",      1, 0, 'f'},
      {"help",          0, 0, 'h'},
      {"numlockled",    0, 0, 'n'},
      {"on-time",       1, 0, 'o'},
      {"pause",         1, 0, 'p'},
      {"rate",          1, 0, 'r'},
      {"scrolllockled", 0, 0, 's'},
      {"tcp-port",      1, 0, 't'},
      {"version",       0, 0, 'v'},
      {"password",      1, 0, 'w'},
      {0,               0, 0, 0}
    };
    c = getopt_long (argc, argv, "bf:hno:p:r:st:vw:",
                     long_options, &option_index);
    if (c == -1)
    {
      break;
    }
    switch (c)
    {
      case 'b':
        if (led_flag)
        {
          wrong_use (argv[0]);
        }
        led_flag = 1;
        children = 1;
        break;
      case 'f':
        if (off_time_flag)
        {
          wrong_use (argv[0]);
        }
        off_time_flag = 1;
        off_time      = atoi (optarg);
        break;
      case 'h':
        usage (argv[0]);
        exit (0);
      case 'n':
        if (led_flag)
        {
          wrong_use (argv[0]);
        }
        led_flag  = 1;
        led       = NUMLOCKLED;
        break;
      case 'o':
        if (on_time_flag)
        {
          wrong_use (argv[0]);
        }
        on_time_flag = 1;
        on_time      = atoi (optarg);
        break;
      case 'r':
        if (rate_flag)
        {
          wrong_use (argv[0]);
        }
        rate_flag = 1;
        rate      = atoi (optarg);
        break;
      case 'p':
        if (pause_flag)
        {
          wrong_use (argv[0]);
        }
        pause_flag = 1;
        pause_time = atoi (optarg);
        break;
      case 's':
        if (led_flag)
        {
          wrong_use (argv[0]);
        }
        led_flag = 1;
        led      = SCROLLLOCKLED;
        break;
      case 't':
        if (tcp_flag)
        {
          wrong_use (argv[0]);
        }
        tcp_flag      = 1;
        serv_tcp_port = atoi (optarg);
        break;
      case 'v':
        puts (PROGRAM " " VERSION);
        exit (0);
        break;
      case 'w':
        strncpy (passWord, optarg, PWLENGTH - 1);
        for (i = 0; i < strlen (optarg); i++)
        {
          optarg[i] = '*';
        }
        puts ("Password not yet supported, sorry.  I'll proceed anyway.");
        break;
      default:
        wrong_use (argv[0]);
    }
  }
  if (optind < argc)
  {
    wrong_use (argv[0]);
  }
}

/* usage - help on options */
static void
usage (char* name)
{
  printf ("Usage: %s [options]\n"
          "Options are\n"
          "  -b,   --both-leds     use both the Num-Lock and the "
                                  "Scroll-Lock LEDs\n"
          "  -f t, --off-time=t    set off blink time to t\n"
          "  -h,   --help          display this help and exit\n"
          "  -n,   --numlockled    use Num-Lock LED, not Scroll-Lock LED\n"
          "  -o t, --on-time=t     set on blink time to t\n"
          "  -p t, --pause=t       set pause time to t\n"
          "  -r n, --rate=n        set initial blink rate to n\n"
          "  -s,   --scrolllockled use Scroll-Lock LED, not Num-Lock LED\n"
          "  -t n, --tcp-port=n    use tcp port n\n"
          "  -v,   --version       output version information and exit\n"
#if 0
          "  -w s, --password=s    set password to string s\n"
#endif
          "Unit for all time values t is tenth of a second.\n",
          name);
}

/* wrong_use - output for the user, if options cannot be interpreted */
static void
wrong_use (char *name)
{
  fprintf (stderr, "%s: Error in arguments.\nTry %s --help\n", name, name);
  exit (1);
}
