/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for IRQ handling from GUS (DB16,MAX,PnP) card
 */

#include "driver.h"
#ifdef GUSCFG_MIDI
#include "midi.h"
#endif

int gus_intr_count = 0;
static int gus_int_count = 0;

static inline void gus_int_disable__0( void )
{
  int loop;

  for ( loop = 0; loop < gus_cards_count; loop++ )
    disable_irq( gus_cards[ loop ] -> gf1.irq );  
}

static inline void gus_int_enable__0( void )
{
  int loop;

  for ( loop = 0; loop < gus_cards_count; loop++ )
    enable_irq( gus_cards[ loop ] -> gf1.irq );  
}

void gus_int_disable__1( void )
{
  unsigned long flags;

  CLI( &flags );
  if ( gus_int_count++ ) goto __end;
  gus_int_disable__0();
  __end:
  STI( &flags );
}

void gus_int_enable__1( void )
{
  unsigned long flags;

  CLI( &flags );
#ifdef GUSCFG_DEBUG
  if ( !gus_int_count )
    {
      PRINTK( "gus_int_enable: underrun???\n" );
      goto __end;
    }
#endif
  if ( --gus_int_count ) goto __end;
  gus_int_enable__0();
  __end:
  STI( &flags );
}


#ifdef GUSCFG_INTERRUPTS_PROFILE
#define STAT_ADD( what ) (what)++
#else
#define STAT_ADD( what ) { ; }
#endif

void gus_interrupt( int irq, void *dev_id, struct pt_regs *regs )
{
  gus_card_t *card;
  register unsigned char status;
  int loop;

  cli();
  if ( gus_cards_count > 1 )
    gus_int_disable__0();
  if ( ( card = gus_cards_irq_index[ irq ] ) == NULL )
    {
      sti();
      return;
    }
  gus_intr_count++;
  sti();


#ifdef GUSCFG_INTERRUPTS_PROFILE
  if ( gus_intr_count > 1 )
    printk( "Eiee - interrupt reentered %i times!!!\n", gus_intr_count );
#endif    

  card -> interrupt_stat++;
  
#if 0
  PRINTK( "gus_interrupt: monitor\n" );
#endif
  do {
    loop = 0;
    status = INB( card -> gf1.reg_irqstat );
    if ( !status ) goto __codec;	/* grrr.. I don't like goto command, but speedup */
    loop++;
#if 0
    printk( "status = 0x%x\n", status );
#endif
#ifdef GUSCFG_MIDI_DEVICES
    if ( status & 0x01 )
      {
        STAT_ADD( card -> gf1.interrupt_stat_midi_out );
        card -> gf1.interrupt_handler_midi_out( card );
      }
    if ( status & 0x02 )
      {
        STAT_ADD( card -> gf1.interrupt_stat_midi_in );
        card -> gf1.interrupt_handler_midi_in( card );
      }
#endif
    if ( status & ( 0x20 | 0x40 ) )
      {
        unsigned int i, already, _current_;
        unsigned char voice_status, voice;
        struct GF1_STRU_VOICE_RANGE *range;
          
        already = 0;
        while ( ( ( voice_status = gus_i_read8( card, GF1_GB_VOICES_IRQ ) ) & 0xc0 ) != 0xc0 )
          {
            voice = voice_status & 0x1f;
            _current_ = 1 << voice;
            if ( already & _current_ ) continue;	/* multi request */
            already |= _current_;			/* mark request */
            gf1_select_voice( card, voice );
#if 0
            printk( "voice = %i, voice_status = 0x%x, voice_verify = %i\n", voice, voice_status, INB( GUSP( card, GF1PAGE ) ) );
#endif
	    i = 0;
	    range = card -> gf1.voice_ranges;
            while ( i < GF1_VOICE_RANGES )
              {
                if ( voice >= range -> min && voice <= range -> max )
                  {
                    if ( !(voice_status & 0x80) )	/* voice position IRQ */
                      {
                        STAT_ADD( range -> interrupt_stat_wave );
                        range -> interrupt_handler_wave( card, voice );
                      }
                    if ( !(voice_status & 0x40) )	/* volume ramp IRQ */
                      {
                        STAT_ADD( range -> interrupt_stat_volume );
                        range -> interrupt_handler_volume( card, voice );
                      }
                    goto __speedup1;
                  }
                i++;
                range++;
              }
            STAT_ADD( card -> gf1.interrupt_stat_voice_lost );
            gus_ctrl_stop( card, GF1_VB_ADDRESS_CONTROL );
            gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
            __speedup1:
          }
      }
    if ( status & 0x04 )
      {
        STAT_ADD( card -> gf1.interrupt_stat_timer1 );
        card -> gf1.interrupt_handler_timer1( card );
      }
    if ( status & 0x08 )
      {
        STAT_ADD( card -> gf1.interrupt_stat_timer2 );
        card -> gf1.interrupt_handler_timer2( card );
      }
    if ( status & 0x80 )
      {
        if ( gus_i_look8( card, GF1_GB_DRAM_DMA_CONTROL ) & 0x40 )
          {
            STAT_ADD( card -> gf1.interrupt_stat_dma_write );
            if ( card -> gf1.mode & GF1_MODE_PCM_PLAY )
              card -> gf1.interrupt_handler_pcm_dma_write( card );
             else
              card -> gf1.interrupt_handler_synth_dma_write( card );
          }
        if ( gus_i_look8( card, GF1_GB_REC_DMA_CONTROL ) & 0x40 )
          {
            STAT_ADD( card -> gf1.interrupt_stat_dma_read );
            card -> gf1.interrupt_handler_pcm_dma_read( card );
          }
      }

  __codec:
        
#if defined( GUSCFG_GUSMAX ) || defined( GUSCFG_PNP )
  if ( card -> use_codec && card -> max_flag &&
       ( ( status = codec_in( card, CODEC_IRQ_STATUS ) ) & CODEC_ALL_IRQS ) )
      {
#ifdef GUSCFG_INTERRUPTS_PROFILE
        if ( status & CODEC_PLAYBACK_IRQ )
          card -> codec.interrupt_stat_playback++;
        if ( status & CODEC_RECORD_IRQ )
          card -> codec.interrupt_stat_record++;
        if ( status & CODEC_TIMER_IRQ )
          card -> codec.interrupt_stat_timer++;
#endif
        if ( card -> codec.mode & CODEC_MODE_OPEN )
          codec_interrupt( card, status );
        OUTB( 0, CODECP( card, STATUS ) );	/* clear global interrupt bit */
        codec_out( card, CODEC_IRQ_STATUS, CODEC_ALL_IRQS & ~status );
        loop++;
      }
#endif
  } while ( loop );

#if 0
  PRINTK( "IRQ END\n" );
#endif

  gus_intr_count--;
  cli();
  if ( gus_cards_count > 1 )
    gus_int_enable__0();
  sti();
}

#ifdef GUSCFG_GUSDB16

void gus_interrupt_db16( int irq, void *dev_id, struct pt_regs *regs )
{
  gus_card_t *card;
  register unsigned char status;
 
  if ( ( card = gus_cards_irq_index[ irq ] ) == NULL )
    {
#if 0
      PRINTK( "gus_interrupt_db16: wrong IRQ? No such card..." );
#endif
      return;
    }

  sti();

  card -> interrupt_stat++;
  
  status = codec_in( card, CODEC_IRQ_STATUS );
#if 0
  PRINTK( "gus_interrupt_sb16: codec irq status = 0x%x\n", status );
#endif
  if ( status & CODEC_ALL_IRQS )
    {
#ifdef GUSCFG_INTERRUPTS_PROFILE
      if ( status & CODEC_PLAYBACK_IRQ )
        card -> codec.interrupt_stat_playback++;
      if ( status & CODEC_RECORD_IRQ )
        card -> codec.interrupt_stat_record++;
      if ( status & CODEC_TIMER_IRQ )
        card -> codec.interrupt_stat_timer++;
#endif
      if ( card -> codec.mode & CODEC_MODE_OPEN )
        codec_interrupt( card, status );
      OUTB( 0, CODECP( card, STATUS ) );	/* clear global interrupt bit */
      codec_out( card, CODEC_IRQ_STATUS, CODEC_ALL_IRQS & ~status );
    }
}

#endif /* GUSCFG_GUSDB16 */

#ifdef GUSCFG_ESS

void gus_interrupt_ess( int irq, void *dev_id, struct pt_regs *regs )
{
  gus_card_t *card;

  if ( ( card = gus_cards_irq_index[ irq ] ) == NULL )
    {
#if 0
      PRINTK( "gus_interrupt_ess: wrong IRQ? No such card..." );
#endif
      return;
    }

  sti();

  card -> interrupt_stat++;
  if ( card -> ess.mode & ESS_MODE_OPEN )
    ess_interrupt( card );

  INB( ESSP( card, DATA_AVAIL ) );      /* ack interrupt */
}

#endif
