/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for effects (effect processor [voices] control)
 */

#include "driver.h"

#ifdef GUSCFG_INTERWAVE

static void gus_interrupt_iw_effect_wave( gus_card_t *card, int voice )
{
  gus_gf1_smart_stop_voice( card, voice );
}

static void gus_interrupt_iw_effect_volume( gus_card_t *card, int voice )
{
  gus_gf1_smart_stop_voice( card, voice );
}

static int gus_iw_effects_reset( gus_card_t *card )
{
  int i;
  struct GUS_STRU_EFFECT_INTERWAVE *peffect;
  
  gus_gf1_stop_voices( card, 24, 31 );
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].rvoices = 0;
  gus_reselect_active_voices( card );
  for ( peffect = card -> gf1.effects -> chip.interwave.voices, i = 0; i < 8; i++, peffect++ )
    {
      if ( peffect -> buffer_addr != -1 )
        gus_memory_manager_mfree( card, &card -> gf1.mem_alloc, peffect -> buffer_addr );
    }
  gus_gf1_close( card, GF1_MODE_EFFECT );
  return 0;
}

static int gus_iw_effects_setup( gus_card_t *card )
{
  int i, fail, ok;
  unsigned long flags;
  unsigned int address;
  struct GUS_STRU_EFFECT_INTERWAVE *peffect;
  struct GUS_STRU_IW_LFO_PROGRAM lfo_program;

  if ( !card -> gf1.enh_mode ) return -ENXIO;
  fail = ok = 0;
  for ( peffect = card -> gf1.effects -> chip.interwave.voices, i = 0; i < 8; i++, peffect++ )
    {
      if ( peffect -> flags & GUS_EFFECT_F_IW_USED )
        {
          if ( peffect -> buffer_size )
            address = gus_memory_manager_malloc( card, &card -> gf1.mem_alloc, "Effect Processor", peffect -> buffer_size, 1, 2 );
           else
            address = -1;
          if ( address == -1 ) fail++;
          peffect -> buffer_addr = address;
          if ( address != -1 )
            {
              ok++;
              gus_dram_setmem( card, address, 0x0000, peffect -> buffer_size );
            }
        }
    }
  if ( fail )
    {
      for ( peffect = card -> gf1.effects -> chip.interwave.voices, i = 0; i < 8; i++, peffect++ )
        {
          if ( (peffect -> flags & GUS_EFFECT_F_IW_USED) && peffect -> buffer_addr != -1 )
            gus_memory_manager_mfree( card, &card -> gf1.mem_alloc, peffect -> buffer_addr );
        }
      return -ENOMEM;
    }
  if ( !ok ) return -EINVAL;		/* no voice is used? */
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].interrupt_handler_wave = gus_interrupt_iw_effect_wave;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].interrupt_handler_volume = gus_interrupt_iw_effect_volume;
  gus_gf1_open( card, GF1_MODE_EFFECT );
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].rvoices = 8;
  gus_reselect_active_voices( card );
  for ( peffect = card -> gf1.effects -> chip.interwave.voices, i = 0; i < 8; i++, peffect++ )
    {
      if ( !(peffect -> flags & GUS_EFFECT_F_IW_USED) ) continue;
      CLI( &flags );
      gf1_select_voice( card, i + 24 );
      gus_write8( card, GF1_VB_MODE, 0x21 | ( peffect -> flags & GUS_EFFECT_F_IW_ALTERNATE ? 0x10 : 0x00 ) );
      gus_write_addr( card, GF1_VA_START, peffect -> buffer_addr << 4, 1 );
      gus_write_addr( card, GF1_VA_END, ( peffect -> buffer_addr + peffect -> buffer_size - 2 ) << 4, 1 );
      gus_write_addr( card, GF1_VA_EFFECT, ( peffect -> buffer_addr + peffect -> buffer_pos ) << 4, 1 );
      gus_write_addr( card, GF1_VA_CURRENT, peffect -> buffer_addr << 4, 1 );
      gus_write16( card, GF1_VW_OFFSET_LEFT_FINAL, 0xfff0 );
      gus_write16( card, GF1_VW_OFFSET_LEFT, 0xfff0 );
      gus_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, 0xfff0 );
      gus_write16( card, GF1_VW_OFFSET_RIGHT, 0xfff0 );
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, 0x07 );
      gus_write8( card, GF1_VB_VOLUME_CONTROL, 0x07 );
      gus_write16( card, GF1_VW_VOLUME, peffect -> volume << 4 );
      if ( peffect -> flags & GUS_EFFECT_F_IW_ALTERNATE )
        {
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, peffect -> effect_volume << 4 );
          gus_write16( card, GF1_VW_EFFECT_VOLUME, peffect -> effect_volume << 4 );
        }
       else
        {
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, 0 );
          gus_write16( card, GF1_VW_EFFECT_VOLUME, 0 );
        }
      gus_write16( card, GF1_VW_OFFSET_LEFT_FINAL, peffect -> l_offset << 4 );
      gus_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, peffect -> r_offset << 4 );
      gus_write16( card, GF1_VW_FREQUENCY, 0x400 );	/* 44100Hz */
      gus_write8( card, GF1_VB_FREQUENCY_LFO, 0 );
      gus_write8( card, GF1_VB_VOLUME_LFO, 0 );
      gus_write8( card, GF1_VB_ACCUMULATOR, peffect -> write_mask );
      STI( &flags );
      lfo_program.freq_and_control = peffect -> vibrato_control;
      lfo_program.depth_final = lfo_program.depth = peffect -> vibrato_depth;
      lfo_program.depth_inc = 0;
      lfo_program.twave = 0;
      gus_lfo_program( card, i + 24, GUS_LFO_VIBRATO, &lfo_program );
      lfo_program.freq_and_control = peffect -> tremolo_control;
      lfo_program.depth_final = lfo_program.depth = peffect -> tremolo_depth;
      lfo_program.depth_inc = 0;
      lfo_program.twave = 0;
      gus_lfo_program( card, i + 24, GUS_LFO_TREMOLO, &lfo_program );
    }
  CLI( &flags );
  for ( peffect = card -> gf1.effects -> chip.interwave.voices, i = 0; i < 8; i++, peffect++ )
    {
      if ( !(peffect -> flags & GUS_EFFECT_F_IW_USED) ) continue;
      gf1_select_voice( card, i + 24 );
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, 0x08 | 0x04 );		/* loop enable */
    }
  STI( &flags );
  return 0;
}

#endif /* GUSCFG_INTERWAVE */

int gus_effects_reset( gus_card_t *card )
{
  int error = 0;

  if ( card -> gf1.effects )
    {
      switch ( card -> gf1.effects -> chip_type ) {
#ifdef GUSCFG_INTERWAVE
        case GUS_EFFECT_CHIP_INTERWAVE:
          error = gus_iw_effects_reset( card );
          break;
#endif
        default:
          error = -EINVAL;
      }
      gus_free( card -> gf1.effects, sizeof( struct GUS_STRU_EFFECT ) );
      card -> gf1.effects = NULL;
    }
  return error;
}

int gus_effects_setup( gus_card_t *card, struct GUS_STRU_EFFECT *effects, int space )
{
  int error;
  struct GUS_STRU_EFFECT seffects, *peffects;

  if ( VERIFY_AREA( VERIFY_READ, effects, sizeof( *effects ) ) ) return -EIO;
  MEMCPY_FROMFS( &seffects, effects, sizeof( seffects ) );
  effects = &seffects;

  if ( card -> gf1.effects )
    if ( ( error = gus_effects_reset( card ) ) < 0 ) return error;
  peffects = gus_malloc( sizeof( struct GUS_STRU_EFFECT ) );
  if ( !peffects ) return -ENOMEM;
  MEMCPY( peffects, effects, sizeof( struct GUS_STRU_EFFECT ) );
  card -> gf1.effects = peffects;
  error = 0;
  switch ( effects -> chip_type ) {
#ifdef GUSCFG_INTERWAVE
    case GUS_EFFECT_CHIP_INTERWAVE:
      error = gus_iw_effects_setup( card );
      break;
#endif
    default:
      return -EINVAL;
  }
  if ( error < 0 )
    {
      gus_free( peffects, sizeof( struct GUS_STRU_EFFECT ) );
      card -> gf1.effects = NULL;
    }
  return error;
}
