/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  SIMPLE INTERFACE
 */

#include "driver.h"
#include "tables.h"

static void voice_interrupt_wave( gus_card_t *card, gus_voice_t *voice );
static void voice_interrupt_volume( gus_card_t *card, gus_voice_t *voice );
static void voice_interrupt_effect( gus_card_t *card, gus_voice_t *voice );
static void voice_start( gus_card_t *card, gus_voice_t *voice );
static void voice_go( gus_card_t *card, gus_voice_t *voice );
static void voice_stop( gus_card_t *card, gus_voice_t *voice, unsigned char mode );
static void voice_control( gus_card_t *card, gus_voice_t *voice, unsigned char control );
static void voice_freq( gus_card_t *card, gus_voice_t *voice, unsigned int freq );
static void voice_volume( gus_card_t *card, gus_voice_t *voice, unsigned short volume );
static void voice_loop( gus_card_t *card, gus_voice_t *voice, unsigned int start, unsigned int end );
static void voice_ramp( gus_card_t *card, gus_voice_t *voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control );
static void voice_pos( gus_card_t *card, gus_voice_t *voice, unsigned int position );
static void voice_pan( gus_card_t *card, gus_voice_t *voice, unsigned short pan );
static void voice_lfo( gus_card_t *card, gus_voice_t *voice, unsigned char *data );
static void voice_private1( gus_card_t *card, gus_voice_t *voice, unsigned char *data );

static struct GUS_STRU_INSTRUMENT_VOICE_COMMANDS voice_commands = {
  voice_interrupt_wave,
  voice_interrupt_volume,
  voice_interrupt_effect,
  voice_start,
  voice_go,
  voice_stop,
  voice_control,
  voice_freq,
  voice_volume,
  voice_loop,
  voice_ramp,
  voice_pos,
  voice_pan,
  voice_lfo,
  voice_private1
};

static void note_stop( gus_card_t *card, gus_voice_t *voice, int wait );
static void note_wait( gus_card_t *card, gus_voice_t *voice );
static void note_off( gus_card_t *card, gus_voice_t *voice );
static void note_volume( gus_card_t *card, gus_voice_t *voice );
static void note_pitchbend( gus_card_t *card, gus_voice_t *voice );
static void note_vibrato( gus_card_t *card, gus_voice_t *voice );
static void note_tremolo( gus_card_t *card, gus_voice_t *voice );

static struct GUS_STRU_INSTRUMENT_NOTE_COMMANDS note_commands = {
  note_stop,
  note_wait,
  note_off,
  note_volume,
  note_pitchbend,
  note_vibrato,
  note_tremolo
};

static void chn_trigger_down( gus_card_t *card, gus_channel_t *channel, gus_instrument_t *instrument, unsigned char note, unsigned char velocity, unsigned char priority );
static void chn_trigger_up( gus_card_t *card, gus_note_t *note );
static void chn_control( gus_card_t *card, gus_channel_t *channel, unsigned short p1, unsigned short p2 );

static struct GUS_STRU_INSTRUMENT_CHANNEL_COMMANDS channel_commands = {
  chn_trigger_down,
  chn_trigger_up,
  chn_control
};

static void do_volume_envelope( gus_card_t *card, gus_voice_t *voice );
static void do_pan_envelope( gus_card_t *card, gus_voice_t *voice );

/*
 *
 */

static void voice_interrupt_wave( gus_card_t *card, gus_voice_t *voice )
{
  gus_gf1_stop_voice( card, voice -> number );
  gus_i_write16( card, GF1_VW_VOLUME, 0 );
  voice -> flags &= ~VFLG_RUNNING;
  if ( voice -> flags & VFLG_SHUTDOWN )
    {
      voice -> flags &= ~VFLG_SHUTDOWN;
      voice_go( card, voice );
    }
}

static void voice_interrupt_volume( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> flags & (VFLG_RUNNING|VFLG_SHUTDOWN) )
    do_volume_envelope( card, voice );
   else
    gus_gf1_stop_voice( card, voice -> number );
}

static void voice_interrupt_effect( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> flags & (VFLG_RUNNING|VFLG_SHUTDOWN) )
    if ( voice -> flags & VFLG_EFFECT_TIMER1 )
      do_pan_envelope( card, voice );
}

/*
 *
 */

static void select_wave( gus_card_t *card, gus_voice_t *voice )
{
#if 0
  printk( " -%i- select wave\n", voice -> number );
#endif
  voice -> flags &= ~VFLG_SELECT_WAVE;
  if ( voice -> instrument )
    {
      unsigned char control, mode, format;
      gus_wave_t *wave;
    
      voice -> layer = voice -> instrument -> info.layer;
      voice -> wave = wave = voice -> layer -> wave;
      format = wave -> format;
      voice -> data.simple.loop_start = wave -> loop_start;
      if ( format & GUS_WAVE_LOOP )
        voice -> data.simple.loop_end = wave -> loop_end;
       else
        voice -> data.simple.loop_end = ( wave -> size << 4 ) - ( format & GUS_WAVE_16BIT ? 40 : 24 );
      voice -> data.simple.position =
        (format & GUS_WAVE_BACKWARD) && !wave -> start ?
          wave -> loop_end :
          wave -> start;
      control = 0x00;
      mode = 0x20;		/* enable offset registers */
      if ( format & GUS_WAVE_16BIT ) control |= 0x04;
      if ( format & GUS_WAVE_BACKWARD ) control |= 0x40;
      if ( format & GUS_WAVE_LOOP ) control |= 0x08; else control |= 0x20;
      if ( format & GUS_WAVE_BIDIR ) control |= 0x10;
      if ( format & GUS_WAVE_ULAW ) mode |= 0x40;
      if ( format & GUS_WAVE_ROM ) mode |= 0x80;
      voice -> data.simple.control = control;
      voice -> data.simple.mode = mode;
    }
   else
    {
#ifdef GUSCFG_DEBUG_INSTRUMENTS
      printk( " -%i- instrument not found..\n", voice -> number );
#endif
      voice -> wave = NULL;
      voice -> layer = NULL;
    }
}

static void do_volume_envelope( gus_card_t *card, gus_voice_t *voice )
{
  unsigned short next = 0, rate = 0, old_volume;
  unsigned char program_next_ramp;
  gus_wave_t *wave;
  unsigned long flags;
  
  if ( !card -> gf1.volume_ramp )
    {
      gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
      return;
    }
  wave = voice -> wave;
  while ( 1 ) {
    program_next_ramp = 0;
    switch ( voice -> data.simple.venv_state ) {
      case SIMPLE_BEFORE:
        voice -> data.simple.venv_next = 0;
        voice -> data.simple.venv_state++;
        CLI( &flags );
        gf1_select_voice( card, voice -> number );
        gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
        gus_i_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
        STI( &flags );
        break;
      case SIMPLE_ATTACK:
        program_next_ramp++;
        next = 255;
        rate = card -> gf1.volume_ramp;
        voice -> data.simple.venv_state++;
        break;
      case SIMPLE_SUSTAIN:
        CLI( &flags );
        gf1_select_voice( card, voice -> number );
        gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
        gus_i_write16( card, GF1_VW_VOLUME, ( (int)voice -> data.simple.gf1_volume * (int)voice -> data.simple.venv_next ) / 255 );
        STI( &flags );
        return;
      case SIMPLE_RELEASE:
        program_next_ramp++;
        next = 0;
        rate = card -> gf1.volume_ramp;
        voice -> data.simple.venv_state++;
        break;
      case SIMPLE_DONE:
        gus_gf1_stop_voice( card, voice -> number );
 	voice -> flags &= ~VFLG_RUNNING;
        if ( voice -> flags & VFLG_SHUTDOWN )
          {
            voice -> flags &= ~VFLG_SHUTDOWN;
            voice_go( card, voice );
          }
        return;
      case SIMPLE_VOLUME:
        program_next_ramp++;
        next = voice -> data.simple.venv_next;
        rate = card -> gf1.volume_ramp;
        voice -> data.simple.venv_state = voice -> data.simple.venv_state_prev;
        break;
    }
    voice -> data.simple.venv_next = next;
    if ( !program_next_ramp ) continue;
    CLI( &flags );
    gf1_select_voice( card, voice -> number );
    gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
    STI( &flags );
    old_volume = gus_i_read16( card, GF1_VW_VOLUME ) >> 8;
    if ( !rate ) continue;
    next = ( ( (int)voice -> data.simple.gf1_volume * (int)next ) / 255 ) >> 8;
    if ( old_volume < MIN_OFFSET ) old_volume = MIN_OFFSET;
    if ( next < MIN_OFFSET ) next = MIN_OFFSET;
    if ( next > MAX_OFFSET ) next = MAX_OFFSET;
    if ( old_volume == next ) continue;
    voice -> data.simple.volume_control &= ~0xc3;
    voice -> data.simple.volume_control |= 0x20;
    CLI( &flags );
    gf1_select_voice( card, voice -> number );
    if ( old_volume > next )
      {
        gus_write8( card, GF1_VB_VOLUME_START, next );
        gus_write8( card, GF1_VB_VOLUME_END, old_volume );
        voice -> data.simple.volume_control |= 0x40;
      }
     else
      {
        gus_write8( card, GF1_VB_VOLUME_START, old_volume );
        gus_write8( card, GF1_VB_VOLUME_END, next );
      }
    gus_write8( card, GF1_VB_VOLUME_RATE, rate );
    gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.simple.volume_control );
    if ( !card -> gf1.enh_mode )
      {
        gus_delay( card );
        gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.simple.volume_control );
      }
    STI( &flags );
    return;
  }
}

static void do_pan_envelope( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned char old_pan;

#if 0
  gf1_select_voice( card, voice -> number );
  printk( " -%i- do_pan_envelope - flags = 0x%x (0x%x -> 0x%x)\n", voice -> number, voice -> flags, voice -> data.simple.gf1_pan, gus_i_read8( card, GF1_VB_PAN ) & 0x0f );
#endif
  if ( card -> gf1.enh_mode )
    {
      voice -> flags &= ~(VFLG_EFFECT_TIMER1|VFLG_PAN);
      return;
    }
  if ( !(voice -> flags & VFLG_PAN) )		/* before */
    voice -> flags |= VFLG_EFFECT_TIMER1|VFLG_PAN;
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  old_pan = gus_i_read8( card, GF1_VB_PAN ) & 0x0f;
  if ( old_pan > voice -> data.simple.gf1_pan ) old_pan--;
  if ( old_pan < voice -> data.simple.gf1_pan ) old_pan++;
  gus_write8( card, GF1_VB_PAN, old_pan );
  STI( &flags );
  if ( old_pan == voice -> data.simple.gf1_pan )	/* goal reached */
    voice -> flags &= ~(VFLG_EFFECT_TIMER1|VFLG_PAN);
#if 0
  gf1_select_voice( card, voice -> number );
  printk( " -%i- (1) do_pan_envelope - flags = 0x%x (0x%x -> 0x%x)\n", voice -> number, voice -> flags, voice -> data.simple.gf1_pan, gus_i_read8( card, GF1_VB_PAN ) & 0x0f );
#endif
}

static void set_enhanced_pan( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned short vlo, vro, pan;
  
  pan = voice -> data.simple.pan >> 7;
  vlo = ATTEN( (ATTEN_TABLE_SIZE-1) - pan );
  vro = ATTEN( pan );
  if ( pan != ATTEN_TABLE_SIZE - 1 && pan != 0 )
    {
      vlo >>= 1;
      vro >>= 1;
    }
  vlo <<= 4;
  vro <<= 4;
#if 0
  printk( "vlo = 0x%x (0x%x), vro = 0x%x (0x%x)\n", vlo, gus_i_read16( card, GF1_VW_OFFSET_LEFT ), vro, gus_i_read16( card, GF1_VW_OFFSET_RIGHT ) );
#endif
  if ( !(voice -> flags & VFLG_WAIT_FOR_START) && (voice -> flags & VFLG_RUNNING) )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      if ( voice -> data.simple.vlo != vlo )
        gus_i_write16( card, GF1_VW_OFFSET_LEFT_FINAL, vlo );
      if ( voice -> data.simple.vro != vro )
        gus_i_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, vro );
      STI( &flags );
    }
  voice -> data.simple.vlo = vlo;
  voice -> data.simple.vro = vro;
}

/*
 *
 */

static void voice_start( gus_card_t *card, gus_voice_t *voice )
{
#if 0
  printk( " -%i- start\n", voice -> number );
#endif
  if ( voice -> flags & VFLG_RUNNING )
    voice_stop( card, voice, 0x00 );
  voice -> flags &= VFLG_DYNAMIC | VFLG_SHUTDOWN | VFLG_WAIT_FOR_START1;
  voice -> flags |= VFLG_USED | VFLG_WAIT_FOR_START | VFLG_SELECT_WAVE;
}

static void voice_go( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned int begin;
  int w_16;

  if ( !(voice -> flags & VFLG_WAIT_FOR_START) ) return;
  if ( voice -> flags & VFLG_SHUTDOWN ) return;	/* wait for shutdown of old wave */
  if ( voice -> flags & VFLG_SELECT_WAVE ) select_wave( card, voice );
  if ( !voice -> wave ) return;
  begin = voice -> wave -> begin.memory;
  w_16 = voice -> data.simple.control & 0x04;
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  gus_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
  voice -> data.simple.venv_state = SIMPLE_BEFORE;
  voice -> data.simple.volume_control = 0x03;
  gus_write_addr( card, GF1_VA_START, begin + voice -> data.simple.loop_start, w_16 );
  gus_write_addr( card, GF1_VA_END, begin + voice -> data.simple.loop_end, w_16 );
  gus_write_addr( card, GF1_VA_CURRENT, begin + voice -> data.simple.position, w_16 );
  if ( !card -> gf1.enh_mode )
    gus_write8( card, GF1_VB_PAN, voice -> data.simple.gf1_pan );
   else
    {
      gus_write16( card, GF1_VW_OFFSET_LEFT, voice -> data.simple.vlo );
      gus_write16( card, GF1_VW_OFFSET_LEFT_FINAL, voice -> data.simple.vlo );
      gus_write16( card, GF1_VW_OFFSET_RIGHT, voice -> data.simple.vro );
      gus_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, voice -> data.simple.vro );
      gus_write8( card, GF1_VB_ACCUMULATOR, voice -> data.simple.effect_accumulator );
      gus_write16( card, GF1_VW_EFFECT_VOLUME, voice -> data.simple.effect_volume );
      gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, voice -> data.simple.effect_volume );
    }
  STI( &flags );
  if ( card -> gf1.volume_ramp > 0 )
    do_volume_envelope( card, voice );
   else
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_write16( card, GF1_VW_VOLUME, voice -> data.simple.gf1_volume );
      STI( &flags );
    }
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  if ( card -> gf1.enh_mode )
    gus_write8( card, GF1_VB_MODE, voice -> data.simple.mode );
  gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice -> data.simple.control );
  if ( !card -> gf1.enh_mode )
    {
      gus_delay( card );
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice -> data.simple.control );
    }
  STI( &flags );
#if 0
  gus_print_voice_registers( card );
#endif
  voice -> flags &= ~VFLG_WAIT_FOR_START;
  voice -> flags |= VFLG_RUNNING;
}

static void voice_stop( gus_card_t *card, gus_voice_t *voice, unsigned char mode )
{
  unsigned char control;
  unsigned long flags;

  if ( !(voice -> flags & VFLG_RUNNING) ) return;
  switch ( mode ) {
    case GUS_STOP_NOW:
    case GUS_STOP_ENVELOPE:
      if ( card -> gf1.volume_ramp > 0 &&
           (voice -> flags & VFLG_RUNNING) && !(voice -> flags & VFLG_SHUTDOWN) )
        {
          if ( voice -> data.simple.venv_state < SIMPLE_RELEASE )
            {
              voice -> data.simple.venv_state = SIMPLE_RELEASE;
              do_volume_envelope( card, voice );
            }
          if ( voice -> flags & VFLG_RUNNING )
            voice -> flags |= VFLG_SHUTDOWN;
        }
      if ( !(voice -> flags & VFLG_SHUTDOWN) )
        {
          CLI( &flags );
          gus_gf1_stop_voice( card, voice -> number );
          gus_i_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
          STI( &flags );
        }
      voice -> flags &= ~VFLG_RUNNING;
      break;
    case GUS_STOP_LOOP:		/* disable loop only */
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      control = gus_read8( card, GF1_VB_ADDRESS_CONTROL );
      control &= ~(0x83 | 0x04);
      control |= 0x20;
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, control );
      STI( &flags );
      break;
  }
}

static void voice_control( gus_card_t *card, gus_voice_t *voice, unsigned char control )
{
  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
  voice -> data.simple.control = control;
}

static void voice_freq( gus_card_t *card, gus_voice_t *voice, unsigned int freq )
{
  unsigned long flags;

  if ( (freq & GUS_FREQ_MASK) != GUS_FREQ_HZ ) return;
  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
  voice -> data.simple.frequency = freq;
  voice -> fc_register = gus_translate_freq( card, freq << 1 );
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( voice -> flags & VFLG_RUNNING )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_i_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
      STI( &flags );
    }
}

static void voice_volume( gus_card_t *card, gus_voice_t *voice, unsigned short volume )
{
  unsigned long flags;
  unsigned short new_volume;

  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
  voice -> data.simple.volume = volume;
  new_volume = gus_translate_voice_volume( card, volume ) << 4;
  if ( new_volume == voice -> data.simple.gf1_volume ) return;	/* speedup */
  voice -> data.simple.gf1_volume = new_volume;
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( voice -> flags & VFLG_RUNNING )
    {
      if ( card -> gf1.volume_ramp > 0 )
        {
          voice -> data.simple.venv_state_prev = voice -> data.simple.venv_state;
          voice -> data.simple.venv_state = SIMPLE_VOLUME;
          do_volume_envelope( card, voice );
        }
       else
        {
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write16( card, GF1_VW_VOLUME, new_volume );
          STI( &flags );
        }
    }
}

static void voice_loop( gus_card_t *card, gus_voice_t *voice, unsigned int start, unsigned int end )
{
  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
#if 0
  printk( "voice_loop: start = 0x%x, end = 0x%x\n", start, end );
#endif
  voice -> data.simple.loop_start = start;
  voice -> data.simple.loop_end = end;
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( voice -> flags & VFLG_RUNNING )
    {
      unsigned long flags;
      int w_16 = voice -> data.simple.control & 0x04;
      unsigned int begin = voice -> wave -> begin.memory;
    
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_write_addr( card, GF1_VA_START, begin + voice -> data.simple.loop_start, w_16 ); 
      gus_write_addr( card, GF1_VA_END, begin + voice -> data.simple.loop_end, w_16 );
      STI( &flags );
    }
}

static void voice_ramp( gus_card_t *card, gus_voice_t *voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control )
{
  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
  voice -> data.simple.ramp_start = start;
  voice -> data.simple.ramp_end = end;
  voice -> data.simple.ramp_rate = rate;
  voice -> data.simple.ramp_control = control;
}

static void voice_pos( gus_card_t *card, gus_voice_t *voice, unsigned int position )
{
  unsigned long flags;

  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
  voice -> data.simple.position = position;
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( voice -> flags & VFLG_RUNNING )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_write_addr( card, GF1_VA_CURRENT, voice -> wave -> begin.memory + voice -> data.simple.position, voice -> data.simple.control & 0x04 );
      STI( &flags );
    }
}

static void voice_pan( gus_card_t *card, gus_voice_t *voice, unsigned short pan )
{
  unsigned long flags;
  unsigned char old_pan;

  if ( ( pan & GUS_PAN_MASK ) != GUS_PAN_LINEAR ) return;
  if ( voice -> flags & VFLG_SELECT_WAVE )
    select_wave( card, voice );
  pan &= GUS_PAN_DATA;
  voice -> data.simple.pan = pan;
  if ( !card -> gf1.enh_mode )
    {
      old_pan = voice -> data.simple.gf1_pan;
      voice -> data.simple.gf1_pan = pan >> 10;
      if ( !card -> gf1.full_range_pan )
        {
          if ( voice -> data.simple.gf1_pan == 0 ) voice -> data.simple.gf1_pan++;
          if ( voice -> data.simple.gf1_pan == 15 ) voice -> data.simple.gf1_pan--;
        }
      if ( voice -> data.simple.gf1_pan == old_pan ) return;	/* speedup */
      if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
      if ( voice -> flags & VFLG_RUNNING )
        {
          if ( card -> gf1.smooth_pan )
            {
              voice -> flags &= ~VFLG_PAN;			/* before */
              do_pan_envelope( card, voice );
            }
           else
            {
              CLI( &flags );
              gf1_select_voice( card, voice -> number );
              gus_write8( card, GF1_VB_PAN, voice -> data.simple.gf1_pan );
              STI( &flags );
            }
        }
    }
   else
    set_enhanced_pan( card, voice );
}

static void voice_lfo( gus_card_t *card, gus_voice_t *voice, unsigned char *data )
{
  gus_lfo_command( card, voice -> number, data );
}

static unsigned char get_effects_mask( gus_card_t *card, int value )
{
  if ( value > 7 ) return 0;
  if ( card -> gf1.effects && card -> gf1.effects -> chip_type == GUS_EFFECT_CHIP_INTERWAVE )
    return card -> gf1.effects -> chip.interwave.voice_output[ value ];
  return 0;
}

static void voice_private1( gus_card_t *card, gus_voice_t *voice, unsigned char *data )
{
  unsigned long flags;
  unsigned char uc;

  switch ( *data ) {
    case GUS_PRIV1_IW_EFFECT:
      uc = get_effects_mask( card, gus_get_byte( data, 4 ) );
      uc |= get_effects_mask( card, gus_get_byte( data, 4 ) >> 4 );
      uc |= get_effects_mask( card, gus_get_byte( data, 5 ) );
      uc |= get_effects_mask( card, gus_get_byte( data, 5 ) >> 4 );
      voice -> data.simple.effect_accumulator = uc;
      voice -> data.simple.effect_volume = gus_translate_voice_volume( card, gus_get_word( data, 2 ) ) << 4;
      if ( !card -> gf1.enh_mode ) return;
      if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
      if ( voice -> flags & VFLG_RUNNING )
        {
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_ACCUMULATOR, voice -> data.simple.effect_accumulator );
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, voice -> data.simple.effect_volume );
          STI( &flags );
        }
      break;
  }
}

/*
 *
 */

static void note_stop( gus_card_t *card, gus_voice_t *voice, int wait )
{
}

static void note_wait( gus_card_t *card, gus_voice_t *voice )
{
}

static void note_off( gus_card_t *card, gus_voice_t *voice )
{
}

static void note_volume( gus_card_t *card, gus_voice_t *voice )
{
}

static void note_pitchbend( gus_card_t *card, gus_voice_t *voice )
{
}

static void note_vibrato( gus_card_t *card, gus_voice_t *voice )
{
}

static void note_tremolo( gus_card_t *card, gus_voice_t *voice )
{
}

/*
 *
 */
 
static void chn_trigger_down( gus_card_t *card, gus_channel_t *channel, gus_instrument_t *instrument, unsigned char note, unsigned char velocity, unsigned char priority )
{
}

static void chn_trigger_up( gus_card_t *card, gus_note_t *note )
{
}

static void chn_control( gus_card_t *card, gus_channel_t *channel, unsigned short p1, unsigned short p2 )
{
}

/*
 *
 */
 
void gus_gf1_simple_init( gus_card_t *card )
{
  gus_engine_instrument_register( GUS_INSTR_SIMPLE, &voice_commands, &note_commands, &channel_commands );
}
