/*  Station  Abstract superclass of ReferenceStation and SubordinateStation.
    Last modified 1998-04-10

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

class Station {
public:
  Station (TideContext *in_context);
  virtual ~Station();
  virtual int is_reference_station() = 0;

  // This preserves settings like marklevel and units.
  Station *clone();

  Dstr name;
  Dstr timeZone;
  Coordinates coordinates;
  TideContext *context;
  ConstantSetWrapper *constants;
  StationRef *stationRef;
  PredictionValue::Unit myUnits;  // Never KnotsSquared
  void setUnits (PredictionValue::Unit in_units);

  int isCurrent;
  int isHydraulicCurrent;

  // The implementations given in Station are usable as-is for a
  // Reference Station but are overridden by SubordinateStation.
  virtual PredictionValue minLevel() const;
  virtual PredictionValue maxLevel() const;

  // Stettings that don't necessarily relate directly to the station,
  // but which must transfer with it anyway.  NOTE:  Graph and possibly
  // other code assumes that markLevel will be in the correct units
  // for the station.
  PredictionValue *markLevel; // This may be null.
  double aspect;              // Aspect for graphing.

  enum EventType {max, min, slackrise, slackfall, markrise, markfall,
    sunrise, sunset, newmoon, firstquarter, fullmoon, lastquarter};
  enum Direction {forward, backward};

  // predictApproximate is used for drawing tide graphs and estimating
  // the heights in between tide events.  Its results do not
  // necessarily adhere exactly to the offsets specified for
  // subordinate stations, but it makes a good attempt.
  // The implementation given in Station is usable as-is for a
  // Reference Station but is overridden by SubordinateStation.

  virtual PredictionValue predictApproximate (Timestamp tm);

  // predictExactTideEvent is used to enumerate the tide events.
  // Offsets are reliably applied to the heights and timestamps
  // returned.  For subordinate stations, it is possible for the
  // t_outs to come back out of order.

  // tm (in-out) is "uncorrected" or "internal" timestamp
  // t_out is "corrected" timestamp (with offsets applied)
  // For reference station or simple offsets, they are identical.
  // pv_out is undefined if the event is a sun or moon event.
  virtual void predictExactTideEvent (Timestamp &tm, Direction d,
    Timestamp &t_out, EventType &etype_out, Dstr &etype_desc,
    PredictionValue &pv_out) = 0;

protected:

  // Sum long-term constituents only (per reference station).
  PredictionValue movingMean (Timestamp tm);

  // This contains hairy common code between reference and offset
  // implementations of predictExactTideEvent.
  // tm (in-out) is "uncorrected" or "internal" timestamp
  void _predictExactTideEvent (Timestamp &tm, Direction d,
    EventType &etype_out);
  // Use this after pv has been corrected as needed.
  void etypedesc (EventType etype, PredictionValue pv, Dstr &etype_desc);

  // G. Dairiki code, revised to use new data types and to reverse
  // direction.
  PredictionValue marklev;
  PredictionValue f_hiorlo (Timestamp t, unsigned deriv);
  PredictionValue f_mark (Timestamp t, unsigned deriv);
  Timestamp find_zero (Timestamp tl, Timestamp tr,
    PredictionValue (Station::*f)(Timestamp, unsigned deriv),
    Direction d);
  Timestamp find_mark_crossing (Timestamp t1, Timestamp t2, int &risingflag,
    Direction d);
  Timestamp next_zero (Timestamp t,
    PredictionValue (Station::*f)(Timestamp, unsigned deriv),
    int &risingflag, Amplitude max_fp, Amplitude max_fpp,
    Direction d);
  Timestamp next_high_or_low_tide (Timestamp t, int &hiflag, Direction d);

  // Cached values for predictExactTideEvent.  This needs to be
  // invalidated if offsets change.  Units don't matter.
  Timestamp last_tm, t_hilo, t_slack, t_mark, t_moon, t_sun;
  EventType hilo_t, slack_t, mark_t, moon_t, sun_t;
  Direction cache_d;
};

int isSunMoonEvent (Station::EventType a);
int isMaxMinEvent (Station::EventType a);

ostream &operator<< (ostream &out, const Station &s);
// ostream &operator<< (ostream &out, Station::EventType etype);
