/*  Calendar  Manage construction, organization, and printing of calendars.
    Last modified 1998-05-01

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

Calendar::Calendar (Station *station, Timestamp start_tm, Timestamp end_tm) {
  for (unsigned a=0; a<calhashsize; a++)
    hash[a] = NULL;

  timezone = station->timeZone;
  settings = station->context->settings;

  start_day_start = start_tm;
  start_day_start.prev_day (timezone, settings);
  end_day_start = end_tm;
  end_day_start.prev_day (timezone, settings);

  Timestamp tm = start_tm;
  while (tm <= end_tm) {
    struct blurb *b = new blurb;
    Station::EventType etype;
    station->predictExactTideEvent (tm, Station::forward, b->t_out,
      etype, b->etype_desc, b->pv_out);
    if (tm > end_tm) {
      delete b;
      break;
    }
    b->no_pv = isSunMoonEvent (etype);
    add_blurb (b);
  }
}

Calendar::~Calendar () {
  for (unsigned a=0; a<calhashsize; a++) {
    struct day *d = hash[a];
    while (d) {
      struct blurb *b = d->blurbs;
      while (b) {
        struct blurb *tb = b;
        b = b->next;
        delete tb;
      }
      struct day *td = d;
      d = d->next;
      delete td;
    }
  }
}

void Calendar::add_blurb (struct blurb *b) {
  Timestamp start = b->t_out;
  start.prev_day (timezone, settings);
  struct day *d = lookup_day (start);
  if (!d)
    d = add_day (start);
  add_blurb (b, d);
}

void Calendar::add_blurb (struct blurb *b, struct day *destination) {
  assert (b->t_out >= destination->start);
  if (!(destination->blurbs)) {
    destination->blurbs = b;
    b->next = NULL;
  } else if (b->t_out < destination->blurbs->t_out) {
    b->next = destination->blurbs;
    destination->blurbs = b;
  } else {
    struct blurb *prev_b = destination->blurbs;
    while (prev_b->next) {
      if (b->t_out < prev_b->next->t_out)
        break;
      prev_b = prev_b->next;
    }
    b->next = prev_b->next;
    prev_b->next = b;
  }
}

struct Calendar::day *Calendar::lookup_day (Timestamp start) {
  struct day *d = hash[start.timet() % calhashsize];
  while (d) {
    if (d->start == start)
      return d;
    d = d->next;
  }
  return NULL;
}

struct Calendar::day *Calendar::add_day (Timestamp start) {
  struct day *d = new day;
  d->start = start;
  d->blurbs = NULL;
  d->next = hash[start.timet() % calhashsize];
  hash[start.timet() % calhashsize] = d;
  return d;
}

void Calendar::add_month_banner (Dstr &text_out, Timestamp t, int html_form) {
  Dstr heading;
  t.printcalheading (heading, timezone, settings);
  if (html_form) {
    text_out += "<TABLE BORDER><TR><TH COLSPAN=7 ALIGN=CENTER>";
    text_out += heading;
    text_out += "</TH></TR>\n";
  } else {
    int numspaces = (settings->tw - heading.length()) / 2;
    for (int a=0; a<numspaces; a++)
      text_out += ' ';
    text_out += heading;
    text_out += "\n\n";
  }
}

void Calendar::flush_weekbuf (Dstr &text_out, int html_form, int endmonth) {
  int a, isdone=1;

  // If it's totally null, don't write a blank line.
  // </TABLE> is suppressed as well (should be the first call).
  for (a=0; a<7; a++)
    if (weekbuf[a].length()) {
      isdone = 0;
      break;
    }
  if (isdone)
    return;

  if (html_form) {
    // Strip out the day headers into a separate row
    text_out += "<TR>";
    for (a=0; a<7; a++) {
      Dstr strip;
      text_out += "<TH ALIGN=CENTER>";
      weekbuf[a].getline (strip);
      text_out += strip;
      text_out += "</TH>";
    }
    text_out += "</TR>";

    // Do the rest
    text_out += "<TR>";
    for (a=0; a<7; a++) {
      text_out += "<TD><SMALL>";
      text_out += weekbuf[a];
      weekbuf[a] = (char *)NULL;
      text_out += "</SMALL></TD>";
    }
    text_out += "</TR>";
    if (endmonth)
      text_out += "</TABLE>\n";
  } else {
    char fmt[80];
    int colwid = settings->tw / 7;
    if (colwid < 2)
      return;
    char *tbuf = (char *) malloc (colwid+1);
    sprintf (fmt, "%%-%d.%ds ", colwid-1, colwid-1);
    isdone = 0;
    while (!isdone) {
      isdone = 1;
      for (a=0; a<7; a++) {
	if (weekbuf[a].length())
	  isdone = 0;
	Dstr strip;
	weekbuf[a].getline (strip);
	sprintf (tbuf, fmt, strip.aschar());
	text_out += tbuf;
      }
      text_out += '\n';
    }
    free (tbuf);
  }
}

void Calendar::print (Dstr &text_out, int html_form) {
  int a;
  for (a=0; a<7; a++)
    weekbuf[a] = (char *)NULL;
  int month = -1;
  text_out = (char *)NULL;
  Timestamp tm = start_day_start;
  while (tm <= end_day_start) {
    struct tm *tmtm = tm.get_tm (timezone, settings);
    int wday = tmtm->tm_wday;
    if (tmtm->tm_mon != month) {
      month = tmtm->tm_mon;
      flush_weekbuf (text_out, html_form, 1);
      add_month_banner (text_out, tm, html_form);
    } else if (wday == 0) {
      flush_weekbuf (text_out, html_form);
    }

    Dstr temp;
    tm.printdayheading (temp, timezone, settings);
    weekbuf[wday] += temp;
    weekbuf[wday] += '\n';

    struct day *d = lookup_day (tm);
    if (d) {
      struct blurb *b = d->blurbs;
      while (b) {
        weekbuf[wday] += b->etype_desc;
        if (!(b->no_pv)) {
          b->pv_out.printnp (temp);
          weekbuf[wday] += ' ';
          weekbuf[wday] += temp;
        }
        if (html_form)
          weekbuf[wday] += "<BR>";
        weekbuf[wday] += '\n';
        b->t_out.printtime (temp, timezone, settings);
        weekbuf[wday] += temp;
        if (html_form)
          weekbuf[wday] += "<BR>";
        weekbuf[wday] += '\n';
        b = b->next;
      }
    }
    tm.inc_day (timezone, settings);
  }
  flush_weekbuf (text_out, html_form, 1);
}
