/*  slice.c */

/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Functions for selecting & moving 2-D slices. */


#include <stdio.h>
#include <math.h>
#include "api.h"
#include "gui.h"


/* useful macros: */

#define CROSS( c, a, b )  { c[0] =  a[1]*b[2]-a[2]*b[1]; \
                            c[1] = -a[0]*b[2]+a[2]*b[0]; \
                            c[2] =  a[0]*b[1]-a[1]*b[0]; \
                          }

#define MAGNITUDE( a )    sqrt( a[0]*a[0] + a[1]*a[1] + a[2]*a[2] )

#define CLAMP(VAL,MIN,MAX)   ( (VAL<MIN) ? MIN : ((VAL>MAX) ? MAX : VAL) )


#define DISTANCE( x1, y1, x2, y2 )   sqrt( (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) )



/* for horizontal slices:
     corner 0 = North East
     corner 1 = South East
     corner 2 = South West
     corner 3 = North West
*/





/*
 * Calculate the distance from the cursor to the nearest corner of
 * the given hslice.
 * Input:  curx, cury - cursor coords in pixels
 *         time, var - which timestep and variable
 *         level - position of slice in grid coords [0..Nl-1].
 * Output:  corner - number of nearest corner in [0..3]
 * Returned:  distance from cursor to corner.
 */
static float distance_to_hslice( int index, int curx, int cury,
                                 int time, int var, float level, int *corner )
{
   float cx, cy, px, py;
   float dist, neardist, p[3];
   float r, c;
   float flevel = (float) level;
   int i;
   int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
  
   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

   neardist = 1000000.0;  /* any large value */

   cx = (float) curx;
   cy = (float) cury;

   /* Find distance to each corner of the slice */
   for (i=0;i<4;i++) {
      switch (i) {
         case 0:  r = 0.0;  c = (float) (Nc-1);  break;
         case 1:  r = (float) (Nr-1);  c = (float) (Nc-1);  break;
         case 2:  r = (float) (Nr-1);  c = 0.0;  break;
         case 3:  r = 0.0;  c = 0.0;  break;
      }
      vis5d_grid_to_xyz(index, time, var, r, c, flevel, &p[0], &p[1], &p[2] );

      vis5d_project(index, p, &px, &py );

      dist = (px-cx)*(px-cx) + (py-cy)*(py-cy);
      if (dist<=neardist) {
         neardist = dist;
         *corner = i;
      }
   }

   return neardist;
}



/*
 * Calculate the distance from the cursor to the nearest corner of
 * the given vslice.
 * Input:  curx, cuy - cursor coords in pixels
 *         time, var - which timestep and variable
 *         r1,c1,r2,c2 - position of slice in grid coords
 * Output:  corner - number of nearest corner or edge in 0..5
 * Returned:  distance from cursor to corner.
 */
static float distance_to_vslice( int index, int curx, int cury,
                                 int time, int var,
                                 float r1, float c1, float r2, float c2,
                                 int *corner )
{
   float cx, cy;
   float px, py;
   float r, c, l;
   float dist, neardist, p[3];
   int i;
   int curved;
   int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
  
   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

   neardist = 1000000.0;  /* any large value */

   cx = (float) curx;
   cy = (float) cury;

   /* find nearest corner */
   for (i=0;i<4;i++) {
      switch (i) {
          case 0:  r = r1;  c = c1;  l = (float) (Nl[var]-1+LowLev[var]);  break;
          case 1:  r = r2;  c = c2;  l = (float) (Nl[var]-1+LowLev[var]);  break;
          case 2:  r = r1;  c = c1;  l = (float) LowLev[var];  break;
          case 3:  r = r2;  c = c2;  l = (float) LowLev[var];  break;
      }
      /* convert r,c,l to graphics coord p */
      vis5d_grid_to_xyz(index, time, var, r, c, l, &p[0], &p[1], &p[2] );
      vis5d_project(index, p, &px, &py );

      dist = (px-cx)*(px-cx) + (py-cy)*(py-cy);
      if (dist<neardist) {
         neardist = dist;
         *corner = i;
      }
   }

   vis5d_get_curved(index, &curved);
   if (curved==0) {
      /* try to find a closer top or bottom edge midpoint */
      for (i=4;i<6;i++) {
         c = ( c1 + c2 ) / 2.0;
         r = ( r1 + r2 ) / 2.0;
         if (i==4) {
            l = (float) (Nl[var]-1+LowLev[var]);
         }
         else {
            l = (float) LowLev[var];
         }
         vis5d_grid_to_xyz(index, time, var, r, c, l, &p[0], &p[1], &p[2] );
         vis5d_project(index, p, &px, &py );
         dist = (px-cx)*(px-cx) + (py-cy)*(py-cy);
         if (dist<neardist) {
            neardist = dist;
            *corner = i;
         }
      }
   }

   return neardist;
}




/*
 * Given a cursor position, find the slice to be selected for moving.
 * Input:  curx,cury - the cursor position in pixels (0,0) = upper-left
 *         time - which timestep
 *         contype   - pointer to int
 *         contour   - pointer to int
 *         corner    - pointer to int
 * Output:  contype - either HSLICE, VSLICE, CHSLICE, or CVSLICE.
 *          contour - the number of the nearest contour (which variable)
 *          corner - the number of the nearest corner in [0..3] or
 *                   if a vslice edge midpoint has been selected 4, or 5
 * Returned:  1 - a contour has been identified.
 *            0 - a contour was not identified.
 */
static int find_nearest_slice( int index, int curx, int cury, int time,
                               int *contype, int *contour, int *corner )
{
   float dist, neardist;
   int var, corn, w;

  float interval, low, high, level;
  float row0, col0, row1, col1;
  float density, scale;
  int Uvar, Vvar, Wvar, Uvar2, Vvar2, Wvar2, TrajU, TrajV, TrajW;
  int NumVars;
  int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
  
   vis5d_get_numvars( index, &NumVars );
   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

   neardist = 15.0 * 15.0;  /* 15 pixel radius */

   *contype = *contour = *corner = -1;

   /* try horizontal contour slices */
   for (var=0;var<NumVars;var++) {
      if (vis5d_enable_graphics(index, VIS5D_HSLICE, var, VIS5D_GET)) {
         vis5d_get_hslice(index, var, &interval, &low, &high, &level);
         dist = distance_to_hslice(index, curx, cury, time, var, level, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_HSLICE;
            *contour = var;
            *corner = corn;
         }
      }
   }

   /* try horizontal color slices */
   for (var=0;var<NumVars;var++) {
      if (vis5d_enable_graphics(index, VIS5D_CHSLICE, var, VIS5D_GET)) {
         vis5d_get_chslice(index, var, &level);
         dist = distance_to_hslice(index, curx, cury, time, var, level, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_CHSLICE;
            *contour = var;
            *corner = corn;
         }
      }
   }

   /* try vertical contour slices */
   for (var=0;var<NumVars;var++) {
      if (vis5d_enable_graphics(index, VIS5D_VSLICE, var, VIS5D_GET)) {
         vis5d_get_vslice(index, var, &interval, &low, &high, &row0, &col0, &row1, &col1);
         dist = distance_to_vslice(index, curx, cury,  time, var,
                                   row0, col0, row1, col1, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_VSLICE;
            *contour = var;
            *corner = corn;
         }
      }
   }

   /* try vertical colored slices */
   for (var=0;var<NumVars;var++) {
      if (vis5d_enable_graphics(index, VIS5D_CVSLICE, var, VIS5D_GET)) {
         vis5d_get_cvslice(index, var, &row0, &col0, &row1, &col1);
         dist = distance_to_vslice(index, curx, cury,  time, var,
                                   row0, col0, row1, col1, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_CVSLICE;
            *contour = var;
            *corner = corn;
         }
      }
   }

   vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                       &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);

   /* try horizontal wind slices */
   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      if (vis5d_enable_graphics(index, VIS5D_HWIND, w, VIS5D_GET)) {
         int uvar = w == 0 ? Uvar : Uvar2;
         vis5d_get_hwindslice(index, w, &density, &scale, &level);
         dist = distance_to_hslice(index, curx, cury, time, uvar, level, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_HWIND;
            *contour = w;
            *corner = corn;
         }
      }
   }

   /* try vertical wind slice */
   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      if (vis5d_enable_graphics(index, VIS5D_VWIND, w, VIS5D_GET)) {
         int uvar = w == 0 ? Uvar : Uvar2;
         vis5d_get_vwindslice(index, w, &density, &scale, &row0, &col0, &row1, &col1);
         dist = distance_to_vslice(index, curx, cury, time, uvar,
                                   row0, col0, row1, col1, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_VWIND;
            *contour = w;
            *corner = corn;
         }
      }
   }

   /* try horizontal stream slices */
   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      if (vis5d_enable_graphics(index, VIS5D_HSTREAM, w, VIS5D_GET)) {
         int uvar = w == 0 ? Uvar : Uvar2;
         vis5d_get_hstreamslice(index, w, &density, &level);
         dist = distance_to_hslice(index, curx, cury, time, uvar, level, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_HSTREAM;
            *contour = w;
            *corner = corn;
         }
      }
   }

   /* try vertical stream slice */
   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      if (vis5d_enable_graphics(index, VIS5D_VSTREAM, w, VIS5D_GET)) {
         int uvar = w == 0 ? Uvar : Uvar2;
         vis5d_get_vstreamslice(index, w, &density, &row0, &col0, &row1, &col1);
         dist = distance_to_vslice(index, curx, cury, time, uvar,
                                   row0, col0, row1, col1, &corn );
         if (dist<neardist) {
            neardist = dist;
            *contype = VIS5D_VSTREAM;
            *contour = w;
            *corner = corn;
         }
      }
   }

   /** ALL DONE!  contype, contour, corner have the results */
   if (*contype!=-1) {
      /*printf("grab dist=%f type %d, var %d, corner %d\n",
             neardist, *contype, *contour, *corner );*/
      return 1;
   }
   else
      return 0;
}



/*** point_nearest_line ***********************************************
   Given two lines 'a' and 'b' find the point along 'a' which is
   closest to 'b'.  The result is in the form of a parameter 't' along
   line 'a'.
   Input:  a, da - point on 'a' and direction of 'a'
           b, db - point on 'b' and direction of 'b'
   Return:  t, such that a+t*da = the nearest point.
**********************************************************************/
static float point_nearest_line( float a[3], float da[3],
                                 float b[3], float db[3] )
{
   float dc[3], mc;
   float m[3][3], det, t;

   /* find point on 'a' closest to 'b' */
   CROSS( dc, da, db );   /* dc = da X db */
   mc = MAGNITUDE( dc );
   if (mc<=0.0001) {
      /* a is nearly parallel to b */
      return 0.0;
   }
   else {
      m[0][0] = b[0]-a[0];  m[0][1] = b[1]-a[1];  m[0][2] = b[2]-a[2];
      m[1][0] = db[0];      m[1][1] = db[1];      m[1][2] = db[2];
      m[2][0] = dc[0];      m[2][1] = dc[1];      m[2][2] = dc[2];

      det = m[0][0] * (m[1][1]*m[2][2]-m[1][2]*m[2][1])
          - m[0][1] * (m[1][0]*m[2][2]-m[1][2]*m[2][0])
           + m[0][2] * (m[1][0]*m[2][1]-m[1][1]*m[2][0]);

      t = det / (mc*mc);
      return t;
   }
}



/*** move_hslice ******************************************************
   Move a horizontal slice according to the cursor position.
   Input:  curx, cury - cursor position in pixels
           level - pointer to current slice level
           corner - which corner is 'grabbed'
   Output:  level - modified to change position.
   Returned:  0 = no movement
              1 = slice was moved.
**********************************************************************/
static int move_hslice( int index, int curx, int cury,
                        int time, int var,
                        float *level, int corner )
{
   float cx,cy;
   float a[3], da[3];
   float b[3], db[3];
   float t;
   float topl, botl, aa[3];
   float r, c;
   float maxlevel, minlevel;
   int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
  
   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

   if (Nl[var]==1) {
      /* Special case: hslice of a 2-D grid can be at any level*/
      maxlevel = (float) (MaxNl-1);
      minlevel = 0.0;
   }
   else {
      maxlevel = (float) (Nl[var]-1+LowLev[var]);
      minlevel = (float) LowLev[var];
   }

   /* "unproject" cursor position to a line 'b' in 3-D graphics coordinates */
   cx = (float) curx;
   cy = (float) cury;
   vis5d_unproject( index, cx, cy, b, db );

   /* let 'a'+'da' be the line corresponding to the edge of the box
    * along which the slice is being dragged.
    */
   switch (corner) {
      case 0:  r = 0.0;  c = (float) (Nc-1);  break;
      case 1:  r = (float) (Nr-1);  c = (float) (Nc-1);  break;
      case 2:  r = (float) (Nr-1);  c = 0.0;  break;
      case 3:  r = 0.0;  c = 0.0;  break;
   }
   topl = maxlevel;
   botl = minlevel;
   vis5d_grid_to_xyz(index, time, var, r, c, botl, &a[0], &a[1], &a[2] );
   vis5d_grid_to_xyz(index, time, var, r, c, topl, &aa[0], &aa[1], &aa[2] );
   da[0] = aa[0] - a[0];
   da[1] = aa[1] - a[1];
   da[2] = aa[2] - a[2];


   /* find point on 'a' closest to 'b' */
   t = point_nearest_line( a, da, b, db );
   if (t==0.0) {
      return 0;
   }
   else {
      float x, y, z, row, col;
      /* convert "closest" z value back to a grid level */
      x = a[0] + t * da[0];
      y = a[1] + t * da[1];
      z = a[2] + t * da[2];
      vis5d_xyz_to_grid(index, time, var, x, y, z, &row, &col, level );
      return 1;
   }

}



/*
 * Move a vertical slice according to the cursor position.
 * Input:  curx, cury - cursor position in pixels
 *         time, var - which timestep and variable.
 *         r1,c1,r2,c2 - pointers to current slice position values
 *         corner - which corner is 'grabbed'
 * Output:  r1,c1,r2,c2 - possibly modified slice position.
 * Returned:  0 = no movement
 *            1 = slice was moved.
 */
static int move_vslice( int index, int curx, int cury,
                        int time, int var,
                        float *r1, float *c1, float *r2, float *c2,
                        int corner )
{
   float cx, cy;
   float b[3], db[3];
   float p[3], px, py;
   float nr, nc;
   int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
  
   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

   nr = (float) (Nr-1);
   nc = (float) (Nc-1);

   cx = (float) curx;
   cy = (float) cury;
   vis5d_unproject( index, cx, cy, b, db );

   if (corner<4) {
      /** dragging slice by a corner **/
      float dist, neardist, neart, t, l;
      float newcol, newrow;
      int edge, nearedge;

      if (corner<2) {
         l = (float) (Nl[var]-1+LowLev[var]);
      }
      else {
         l = (float) LowLev[var];
      }

      /* find the point on one of the top or bottom box edges which */
      /* is closest to the cursor */
      neardist = 10000.0;
      for (edge=0;edge<4;edge++) {
         float r0, c0, r1, c1, a[3], da[3], aa[3];

         switch (edge) {
            case 0:
               /* north */
               r0 = 0.0;  c0 = 0.0;
               r1 = 0.0;  c1 = (float) (Nc-1);
               break;
            case 1:
               /* east */
               r0 = 0.0;  c0 = (float) (Nc-1);
               r1 = (float) (Nr-1);  c1 = c0;
               break;
            case 2:
               /* south */
               r0 = (float) (Nr-1);  c0 = 0.0;
               r1 = (float) (Nr-1);  c1 = (float) (Nc-1);
               break;
            case 3:
               /* west */
               r0 = 0.0;  c0 = 0.0;
               r1 = (float) (Nr-1);  c1 = 0.0;
               break;
         }
         vis5d_grid_to_xyz( index, time, var, r0, c0, l, &a[0], &a[1], &a[2] );
         vis5d_grid_to_xyz( index, time, var, r1, c1, l, &aa[0], &aa[1], &aa[2] );

         da[0] = aa[0] - a[0];
         da[1] = aa[1] - a[1];
         da[2] = aa[2] - a[2];
         t = point_nearest_line( a, da, b, db );
         p[0] = a[0] + da[0]*t;
         p[1] = a[1] + da[1]*t;
         p[2] = a[2] + da[2]*t;
         vis5d_project( index, p, &px, &py );
         dist = sqrt( (px-cx)*(px-cx) + (py-cy)*(py-cy) );
         if (dist==0.0)  return 0;
         if (dist<neardist) {
            neardist = dist;
            nearedge = edge;
            neart = CLAMP( t, 0.0, 1.0 );
         }
      }

      /* compute new row and column for corner of the slice */
      switch (nearedge) {
         case 0:
            /* north */
            newcol = nc*neart;
            newrow = 0.0;
            break;
         case 1:
            /* east */
            newcol = nc;
            newrow = nr*neart;
            break;
         case 2:
            /* south */
            newcol = nc*neart;
            newrow = nr;
            break;
         case 3:
            /* west */
            newcol = 0.0;
            newrow = nr*neart;
            break;
      }

      if (corner%2==0) {
         /* make sure the corners won't be too close */
         if (DISTANCE( *r2, *c2, newrow, newcol) > 2.0) {
            *r1 = newrow;
            *c1 = newcol;
         }
      }
      else {
         /* make sure the corners won't be too close */
         if (DISTANCE( *r1, *c1, newrow, newcol) > 2.0) {
            *r2 = newrow;
            *c2 = newcol;
         }
      }
      return 1;

   }
   else {
      /** dragging slice by midpoint of edge **/

      float prow, pcol, plev, drow, dcol;
      float newr1, newc1, newr2, newc2;
      float x, y, t;


      if (corner==4) {
         /* find intersection 'p', of 'b' with top plane of box */
         vis5d_grid_to_xyz(index, time, var, 1.0, 1.0, (float) (Nl[var]-1+LowLev[var]),
                           &x, &y, &t);
         t = (t - b[2]) / db[2];
      }
      else {
         /* find intersection 'p', of 'b' with bottom plane of box */
         vis5d_grid_to_xyz(index, time, var, 1.0, 1.0, (float) LowLev[var], &x, &y, &t);
         t = (t - b[2]) / db[2];
      }
      p[0] = b[0] + t*db[0];
      p[1] = b[1] + t*db[1];
      p[2] = b[2] + t*db[2];

      /* convert p to row and columns */
      vis5d_xyz_to_grid( index, time, var, p[0], p[1], p[2], &prow, &pcol, &plev );

      /* let d = 2-D vector parallel to current slice */
      drow = *r2 - *r1;
      dcol = *c2 - *c1;

      if (drow==0.0) {
         /* east/west slice */
         newc1 = *c1;
         newc2 = *c2;
         newr1 = newr2 = prow;
      }
      else if (dcol==0.0) {
         /* north/south slice */
         newr1 = *r1;
         newr2 = *r2;
         newc1 = newc2 = pcol;
      }
      else {
         /* diagonal slice */
         /* recompute r1,c1, r2,c2 such that they fall on the line */
         /* defined by point (prow,pcol) and direction (drow,dcol) */
         t = -prow / drow;
         newr1 = 0.0;
         newc1 = pcol + dcol*t;

         t = (nr-prow) / drow;
         newr2 = Nr-1.0;
         newc2 = pcol + dcol*t;

         /* trim */
         if (newc1<0.0) {
            t = -pcol / dcol;
            newr1 = prow + drow*t;
            newc1 = 0.0;
         }
         if (newc1>(float)(Nc-1)) {
            t = (nc-pcol) / dcol;
            newr1 = prow + drow*t;
            newc1 = (float) (Nc-1);
         }
         if (newc2<0.0) {
            t = -pcol / dcol;
            newr2 = prow + drow*t;
            newc2 = 0.0;
         }
         if (newc2>(float)(Nc-1)) {
            t = (nc-pcol) / dcol;
            newr2 = prow + drow*t;
            newc2 = (float) (Nc-1);
         }
      }

      newc1 = CLAMP( newc1, 0.0, nc );
      newc2 = CLAMP( newc2, 0.0, nc );
      newr1 = CLAMP( newr1, 0.0, nr );
      newr2 = CLAMP( newr2, 0.0, nr );
#ifdef LEAVEOUT
      if ((newc1==0.0 && newc2==0.0) || (newc1==nc && newc2==nc)) {
         if (newr1<nr/2.0)
           newr1 = 0.0;
         else
           newr1 = nr;
         newr2 = nr - newr1;
      }
      if ((newr1==0.0 && newr2==0.0) || (newr1==nc && newr2==nc)) {
         if (newc1<nc/2.0)
           newc1 = 0.0;
         else
           newc1 = nc;
         newc2 = nc - newc1;
      }
#endif
      /* make sure the corners won't be too close */
      if (DISTANCE( newr1, newc1, newr2, newc2 ) > 2.0) {
         *r1 = newr1;
         *c1 = newc1;
         *r2 = newr2;
         *c2 = newc2;
      }
      return 1;
   }
}



/*** move_slice ****************************************************
   Move the position of a contour.  This function is called to
   respond to a user action.
   Input:  curx, cury - the cursor position in pixels (0,0) = upper-left.
           contype    - the type of contour to move (HSLICE, CVSLICE,etc)
           time, var  - which timestep and variable 
           corner     - the corner of the contour to use as reference.
           recalc     - non-zero if slice should be recalculated now.
**********************************************************************/
static void move_slice( int index, int curx, int cury,
                        int contype, int time, int var, int corner,
                        int recalc )
{
  float interval, low, high, level;
  float row0, col0, row1, col1;
  float density, scale;
  int Uvar, Vvar, Wvar, Uvar2, Vvar2, Wvar2, TrajU, TrajV, TrajW;
  int it, numtimes, curtime;

  vis5d_get_numtimes( index, &numtimes );
  vis5d_get_timestep( index, &curtime );
   if (contype==VIS5D_HSLICE) {
      vis5d_get_hslice(index, var, &interval, &low, &high, &level);
      if (move_hslice( index, curx, cury, time, var, &level, corner ) && recalc) {
         vis5d_set_hslice(index, var, interval, low, high, level);
         for (it=0;it<numtimes;it++) {
            vis5d_make_hslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_CHSLICE) {
      vis5d_get_chslice(index, var, &level);
      if (move_hslice( index, curx, cury, time, var, &level, corner ) && recalc) {
         vis5d_set_chslice(index, var, level);
         for (it=0;it<numtimes;it++) {
            vis5d_make_chslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_VSLICE) {
      vis5d_get_vslice(index, var, &interval, &low, &high, &row0, &col0, &row1, &col1);
      if (move_vslice( index, curx, cury, time, var, &row0, &col0, &row1, &col1, corner)
          && recalc) {
         vis5d_set_vslice(index, var, interval, low, high, row0, col0, row1, col1);
         for (it=0;it<numtimes;it++) {
            vis5d_make_vslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_CVSLICE) {
      vis5d_get_cvslice(index, var, &row0, &col0, &row1, &col1);
      if (move_vslice( index, curx, cury, time, var, &row0, &col0, &row1, &col1, corner)
          && recalc) {
         vis5d_set_cvslice(index, var, row0, col0, row1, col1);
         for (it=0;it<numtimes;it++) {
            vis5d_make_cvslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_HWIND) {
      int uvar;
      vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                          &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);
      uvar = var == 0 ? Uvar : Uvar2;
      vis5d_get_hwindslice(index, var, &density, &scale, &level);
      if (move_hslice( index, curx, cury, time, uvar, &level, corner) && recalc) {
         vis5d_set_hwindslice(index, var, density, scale, level);
         for (it=0;it<numtimes;it++) {
            vis5d_make_hwindslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_VWIND) {
      int uvar;
      vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                          &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);
      uvar = var == 0 ? Uvar : Uvar2;
      vis5d_get_vwindslice(index, var, &density, &scale, &row0, &col0, &row1, &col1);
      if (move_vslice( index, curx, cury, time, uvar,
                       &row0, &col0, &row1, &col1, corner) && recalc) {
         vis5d_set_vwindslice(index, var, density, scale, row0, col0, row1, col1);
         for (it=0;it<numtimes;it++) {
            vis5d_make_vwindslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_HSTREAM) {
      int uvar;
      vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                          &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);
      uvar = var == 0 ? Uvar : Uvar2;
      vis5d_get_hstreamslice(index, var, &density, &level);
      if (move_hslice( index, curx, cury, time, uvar, &level, corner) && recalc) {
         vis5d_set_hstreamslice(index, var, density, level);
         for (it=0;it<numtimes;it++) {
            vis5d_make_hstreamslice(index, it, var, it==curtime);
         }
      }
   }
   else if (contype==VIS5D_VSTREAM) {
      int uvar;
      vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                          &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);
      uvar = var == 0 ? Uvar : Uvar2;
      vis5d_get_vstreamslice(index, var, &density, &row0, &col0, &row1, &col1);
      if (move_vslice( index, curx, cury, time, uvar,
                       &row0, &col0, &row1, &col1, corner) && recalc) {
         vis5d_set_vstreamslice(index, var, density, row0, col0, row1, col1);
         for (it=0;it<numtimes;it++) {
            vis5d_make_vstreamslice(index, it, var, it==curtime);
         }
      }
   }
   vis5d_invalidate_frames(index);
}




/*
 * This function is called whenever an input event occurs in the
 * main viewing window while in 'slice moving mode'.
 * Input:  event - the X event
 *         time - current timestep
 * Return:  1 = event was used to move a slice
 *          0 = event was not used
 */
int slice_event( int index, XEvent event, int time )
{
   int result;           /* value returned */
   static int flag;      /* has a contour been selected? */
   static int contype;    /* the type of contour selected */
   static int contour;    /* the contour selectd */
   static int corner;     /* the contour corner selected */
   int recalc;

   GuiContext gtx = get_gui_gtx(index);

   result = 0;

   if (event.type==ButtonPress && event.xbutton.button==Button3) {
      /* The right mouse button has been pressed. */
      /* Determine which slice is closest to the current cursor position. */
      flag = find_nearest_slice(index, event.xbutton.x, event.xbutton.y, time,
                                 &contype, &contour, &corner );
      result = 1;
   }

   if ( (event.type==MotionNotify || event.type==ButtonPress)
       && flag==1) {
      /* The cursor has been moved while the right mouse button */
      /* is pressed.  Update the position of the slice. */
#ifdef SINGLE_TASK
      /* If we don't have parallelism, recompute only if not animating */
      recalc = (gtx->GoTime==0) ? 1 : 0;
#else
      /* If we have parallelism, recompute whether animating or not */
      recalc = 1;
#endif
      move_slice( index, event.xbutton.x, event.xbutton.y,
                  contype, time, contour, corner, recalc );
      result = 1;
   }

   if (event.type==ButtonRelease && event.xbutton.button==Button3) {
      /* The right mouse button has been released. */
      move_slice( index, event.xbutton.x, event.xbutton.y,
                  contype, time, contour, corner, 1 );
      flag = 0;
      result = 1;
   }

   return result;
}


