/* proj.c */

/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Map projection and vertical coordinate system arithmetic.
 * This file contains functions to convert coordinates between the
 * grid, geographic and grapic coordinate systems.
 *
 * It should be straight forward to add new projection types in this file.
 */



#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "globals.h"
#include "proj.h"
#include "box.h"



#define ABS(X)  ( (X) < 0 ? -(X) : (X) )

#ifndef M_PI
#  define M_PI 3.14159265
#endif


#define DEG2RAD    (M_PI/180.0)
#define RAD2DEG    (180.0/M_PI)
#define RADIUS     6371.23


/* for PROJ_SPHERICAL: */
#define SPHERE_SIZE    0.5
#define SPHERE_SCALE   0.125


/* Convert Height to Pressure: */
#define HGT_TO_P( H )   ( ctx->LogScale * exp( H / ctx->LogExp ) )
/* Convert Pressure to Height: */
#define P_TO_HGT( P )   ( ctx->LogExp * log( P / ctx->LogScale ) )
/* Convert Pressure to graphics Z: */
#define P_TO_Z( P )     (ctx->Zmin + (ctx->Zmax-ctx->Zmin) * \
                        (P - ctx->Pbot) / (ctx->Ptop-ctx->Pbot))
/* Convert graphics Z to Pressure: */
#define Z_TO_P( Z )     (ctx->Pbot + (z-ctx->Zmin) * \
                        (ctx->Ptop-ctx->Pbot) / (ctx->Zmax-ctx->Zmin))


/*
 * IF Projection==PROJ_GENERIC THEN
 *        NorthBound = physical location of grid row 0, no units
 *        WestBound = physical location of grid column 0, no units
 *        RowInc = phys. location increment between rows, no units
 *        ColInc = phys. location increment between columns, no units
 *        [Units increase to the left and upward]
 * ELSE IF Projection==PROJ_LINEAR THEN
 *        NorthBound = Latitude of grid row 0, in degrees
 *        WestBound = Longitude of grid column 0, in degrees
 *        RowInc = degrees of latitude between grid rows
 *        ColInc = degrees of longitude between grid columns
 *        [Degrees increase to the left and upward]
 * ELSE IF Projection==PROJ_LAMBERT THEN
 *        Lat1, Lat2 = standard latitudes of conical projection
 *        Note:  Lat1 must be >= Lat2
 *               if (Lat1>0 and Lat2>0) then
 *                  northern hemisphere
 *               else if (Lat1<0 and Lat2<0) then
 *                  southern hemisphere
 *               else
 *                  error
 *               endif
 *               if Lat1==Lat2 then
 *                  Polar Stereographic projection
 *               endif
 *        PoleRow, PoleCol = row and column of north/south pole
 *        CentralLon = which longitude is parallel to the columns
 *        ColInc = increment between grid columns in kilometers
 *        Cone = Cone constant
 *        Hemisphere = +1=northern, -1=southern
 *        ConeFactor = A useful quantity
 * ELSE IF Projection==PROJ_STEREO THEN
 *        CentralLat - latitude of center in degrees
 *        CentralLon - longitude of center in degrees
 *        CentralRow - row # of center
 *        CentralCol - column # of center
 *        ColInc - spacing between columns at center in kilometers
 *        CosCentralLat, SinCentralLat;
 *        StereoScale, InvScale;
 * ELSE IF Projection==PROJ_ROTATED THEN
 *        NorthBound = Latitude on rotated globe of grid row 0, in degrees
 *        WestBound = Longitude on rotated globe of grid column 0, in degrees
 *        RowInc = degrees of latitude on rotated globe between grid rows
 *        ColInc = degrees of longitude on rotated globe between grid columns
 *        CentralLat - Earth latitude of (0, 0) on rotated globe, in radians
 *        CentralLon - Earth longitude of (0, 0) on rotated globe, in radians
 *        Rotation = Clockwise rotation of rotated globe in radians
 *        [Degrees increase to the left and upward]
 * ELSE IF Projection==PROJ_CYLINDRICAL THEN
 *        Use same paramenters as for PROJ_LINEAR, plus:
 *        CylinderScale =  A useful value
 * ELSE IF Projection==PROJ_SPHERICAL THEN
 *        Use same parameters as for PROJ_LINEAR, plus:
 * ENDIF
 */




/* Return the sign of x */
static float sign( float x )
{
   if (x<0.0) {
      return -1.0;
   }
   else if (x>0.0) {
      return 1.0;
   }
   else {
      return 0.0;
   }
}




/*
 * Initialize all map projection stuff.
 * Input:  ctx - the vis5d context
 * Return:  1 = success, 0 = failure
 */
int setup_projection( Context ctx )
{
   float lat1, lat2;
   float *projargs;

   /*
    * Usually, we use the projection info from the v5d file but if
    * ctx->UserProjection>0 then we use the projection parameters from
    * vis5d_init_projection().
    */

   if (ctx->UserProjection>=0) {
      projargs = ctx->UserProjArgs;
      ctx->Projection = ctx->UserProjection;
   }
   else {
      projargs = ctx->G.ProjArgs;
      ctx->Projection = ctx->G.Projection;
   }

   switch (ctx->Projection) {
      case PROJ_GENERIC:
         /* FALL-THROUGH */
      case PROJ_LINEAR:
      case PROJ_CYLINDRICAL:
      case PROJ_SPHERICAL:
         ctx->NorthBound = projargs[0];
         ctx->WestBound  = projargs[1];
         ctx->RowInc     = projargs[2];
         ctx->ColInc     = projargs[3];
         break;
      case PROJ_ROTATED:
         ctx->NorthBound = projargs[0];
         ctx->WestBound  = projargs[1];
         ctx->RowInc     = projargs[2];
         ctx->ColInc     = projargs[3];
         ctx->CentralLat = DEG2RAD * projargs[4];
         ctx->CentralLon = DEG2RAD * projargs[5];
         ctx->Rotation   = DEG2RAD * projargs[6];
         break;
      case PROJ_LAMBERT:
         ctx->Lat1       = projargs[0];
         ctx->Lat2       = projargs[1];
         ctx->PoleRow    = projargs[2];
         ctx->PoleCol    = projargs[3];
         ctx->CentralLon = projargs[4];
         ctx->ColInc     = projargs[5];
         break;
      case PROJ_STEREO:
         ctx->CentralLat = projargs[0];
         ctx->CentralLon = projargs[1];
         ctx->CentralRow = projargs[2];
         ctx->CentralCol = projargs[3];
         ctx->ColInc     = projargs[4];
         break;
      default:
         printf("Error: unknown projection type in grid.c\n");
         return 0;
   }


   /*
    * Precompute useful values for coordinate transformations.
    */
   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
         ctx->SouthBound = ctx->NorthBound - ctx->RowInc * (ctx->Nr-1);
         ctx->EastBound = ctx->WestBound - ctx->ColInc * (ctx->Nc-1);
         break;
      case PROJ_LAMBERT:
         if (ctx->Lat1==ctx->Lat2) {
            /* polar stereographic? */
            if (ctx->Lat1>0.0) {
               lat1 = (90.0 - ctx->Lat1) * DEG2RAD;
            }
            else {
               lat1 = (90.0 + ctx->Lat1) * DEG2RAD;
            }
            ctx->Cone = cos( lat1 );
            ctx->Hemisphere = 1.0;
         }
         else {
            /* general Lambert conformal */
            float a, b;
            if (sign(ctx->Lat1) != sign(ctx->Lat2)) {
               printf("Error: standard latitudes must have the same sign.\n");
               return 0;
            }
            if (ctx->Lat1<ctx->Lat2) {
               printf("Error: Lat1 must be >= ctx->Lat2\n");
               return 0;
            }
            ctx->Hemisphere = 1.0;
            lat1 = (90.0 - ctx->Lat1) * DEG2RAD;
            lat2 = (90.0 - ctx->Lat2) * DEG2RAD;
            a = log(sin(lat1)) - log(sin(lat2));
            b = log( tan(lat1/2.0) ) - log( tan(lat2/2.0) );
            ctx->Cone = a / b;
         }

         /* Cone is in [-1,1] */
         ctx->ConeFactor = RADIUS * sin(lat1)
                          / (ctx->ColInc * ctx->Cone
                             * pow(tan(lat1/2.0), ctx->Cone) );
/*
         printf("Hemisphere: %f\n", ctx->Hemisphere );
         printf("Cone: %f\n", ctx->Cone );
         printf("ConeFactor: %f\n", ctx->ConeFactor );
*/
         break;
      case PROJ_STEREO:
         ctx->CosCentralLat = cos( ctx->CentralLat * DEG2RAD );
         ctx->SinCentralLat = sin( ctx->CentralLat * DEG2RAD );
         ctx->StereoScale = (2.0 * RADIUS / ctx->ColInc);
         ctx->InvScale = 1.0 / ctx->StereoScale;
         break;
      case PROJ_ROTATED:
         ctx->SouthBound = ctx->NorthBound - ctx->RowInc * (ctx->Nr-1);
         ctx->EastBound = ctx->WestBound - ctx->ColInc * (ctx->Nc-1);
         break;
      case PROJ_CYLINDRICAL:
         ctx->CylinderScale = 1.0 / (90.0-ctx->SouthBound);
         ctx->SouthBound = ctx->NorthBound - ctx->RowInc * (ctx->Nr-1);
         ctx->EastBound = ctx->WestBound - ctx->ColInc * (ctx->Nc-1);
         break;
      case PROJ_SPHERICAL:
         ctx->SouthBound = ctx->NorthBound - ctx->RowInc * (ctx->Nr-1);
         ctx->EastBound = ctx->WestBound - ctx->ColInc * (ctx->Nc-1);
         break;
      default:
         printf("Error in set_projection\n");
         return 0;
   }
   if (ctx->Projection != PROJ_GENERIC) {
     if (ctx->SouthBound < -90.0) {
       printf("SouthBound less than -90.0\n");
       return 0;
     }
     if (ctx->NorthBound < ctx->SouthBound) {
       printf("NorthBound less than SouthBound\n");
       return 0;
     }
     if (90.0 < ctx->NorthBound) {
       printf("NorthBound greater than 90.0\n");
       return 0;
     }
   }
   return 1;
}





/*
 * Initialize the vertical coordinate system.
 * Return:  1 = success, 0 = error
 */
int setup_vertical_system( Context ctx )
{
   int i;
   float pressure;
   float *vertargs;

   /*
    * Usually, we use the VCS (Vertical Coordinate System) info from the
    * v5d file.  However, if the user VCS variable is set we use the VCS
    * info from vis5d_init_vertical() function.
    */

   if (ctx->UserVerticalSystem>=0) {
      /* use user-provided parameters */
      vertargs = ctx->UserVertArgs;
      ctx->VerticalSystem = ctx->UserVerticalSystem;
   }
   else {
      /* use parameters from v5d file */
      vertargs = ctx->G.VertArgs;
      ctx->VerticalSystem = ctx->G.VerticalSystem;
   }

   switch (ctx->VerticalSystem) {
      case VERT_GENERIC:
         /* FALL-THROUGH */
      case VERT_EQUAL_KM:
         ctx->BottomBound = vertargs[0];
         ctx->LevInc      = vertargs[1];
         ctx->TopBound = ctx->BottomBound + ctx->LevInc * (ctx->MaxNl-1);
         for (i=0;i<ctx->MaxNl;i++) {
            ctx->Height[i] = ctx->BottomBound + i * ctx->LevInc;
         }
         break;
      case VERT_NONEQUAL_MB:
         /* FALL-THROUGH */
      case VERT_NONEQUAL_KM:
         for (i=0;i<ctx->MaxNl;i++) {
            ctx->Height[i] = vertargs[i];
         }
         ctx->BottomBound = ctx->Height[0];
         ctx->TopBound = ctx->Height[ctx->MaxNl-1];
         break;
      default:
         printf("Error in grid.c, unknown vertical coord system\n");
         return 0;
   }

   /* Precompute useful values */
   switch (ctx->VerticalSystem) {
      case VERT_GENERIC:
      case VERT_EQUAL_KM:
         ctx->TopBound = ctx->BottomBound + ctx->LevInc * (ctx->MaxNl-1);
         for (i=0;i<ctx->MaxNl;i++) {
            ctx->Height[i] = ctx->BottomBound + i * ctx->LevInc;
         }
         if (ctx->LogFlag) {
           ctx->Ptop = ctx->LogScale * exp( ctx->TopBound / ctx->LogExp );
           ctx->Pbot = ctx->LogScale * exp( ctx->BottomBound / ctx->LogExp );
         }
         break;
      case VERT_NONEQUAL_KM:
         if (ctx->LogFlag) {
           ctx->Ptop = ctx->LogScale * exp( ctx->Height[ctx->MaxNl-1] / ctx->LogExp );
           ctx->Pbot = ctx->LogScale * exp( ctx->Height[0] / ctx->LogExp );
         }
         break;
      case VERT_NONEQUAL_MB:
         ctx->Ptop = height_to_pressure(ctx->Height[ctx->MaxNl-1]);
         ctx->Pbot = height_to_pressure(ctx->Height[0]);
         /* The Height[] array should be OK at this point */
         break;
      default:
         return 0;
   }
   if (ctx->TopBound < ctx->BottomBound) {
       printf("TopBound less than BottomBound\n");
       return 0;
   }
   return 1;
}



/*
 * Return the parameters of the current vertical coordinate system.
 */
void get_projection( Context ctx, int *projection, float *projargs )
{
   if (ctx->Projection<0 || ctx->UserProjection<0) {
      /* Haven't called setup_projection() yet, return v5d file's projection */
      *projection = ctx->G.Projection;
      memcpy( projargs, ctx->G.ProjArgs, MAXPROJARGS*sizeof(float) );
   }
   else {
      /* Return user projection args */
      *projection = ctx->UserProjection;
      memcpy( projargs, ctx->UserProjArgs, MAXPROJARGS*sizeof(float) );
   }
}



/*
 * Return the parameters of the current vertical coordinate system.
 */
void get_vertical_system( Context ctx, int *vertical, float *vertargs )
{
   int numargs;

   /* determine number of arguments to return */
   if (ctx->MaxNl<2) {
      numargs = 2;
   }
   else {
      numargs = ctx->MaxNl;
   }

   if (ctx->VerticalSystem<0 || ctx->UserVerticalSystem<0) {
      /* Haven't called setup_vertical_system() yet, return the v5d files'
       * projection.
       */
      *vertical = ctx->G.VerticalSystem;
      memcpy( vertargs, ctx->G.VertArgs, numargs * sizeof(float) );
   }
   else {
      *vertical = ctx->UserVerticalSystem;
      memcpy( vertargs, ctx->UserVertArgs, numargs * sizeof(float) );
   }
}



/*
 * Perform a binary search of the array for the value and return
 * the index as a float.
 */
static float binary_search( float value, float array[], int size )
{
   int low, high, mid;
   float x;

   if (value<array[0] || size <= 1) {
      return 0.0;
   }
   else if (value>array[size-1]) {
      return (float)(size-1);
   }
   else {
      /* do a binary search of array[] for value */
      low = 0;
      high = size-1;

      while (low<=high) {
         mid = (low+high)/2;
         if (value<array[mid])
           high = mid - 1;
         else if (value>array[mid])
           low = mid + 1;
         else
           return (float) mid;  /* TODO: check this */
      }

      /* interpolate a value between high and low */
      x = (value-array[high]) / (array[low]-array[high]);
      return high * (1.0-x) + low * x;
   }
}



/**********************************************************************/
/*****                Vertical Coordinate Conversion              *****/
/**********************************************************************/


/*
 * Convert a level from grid coordinates to a zvalue in [Zmin,Zmax].
 * Input:  ctx - the vis5d context
 *         time, var - which timestep and variable.
 *         level - the level.
 */
float gridlevel_to_z( Context ctx, int time, int var, float level )
{
   int ilevel;
   float rlevel, p, hgt;

/*
   assert( var>=0 );
   assert( time>=0 );
*/

   if (level<=0.0) {
      return ctx->Zmin;
   }
   else if (level>=ctx->MaxNl-1 || ctx->MaxNl == 1) {
      return ctx->Zmax;
   }
   else {
      switch (ctx->VerticalSystem) {
         case VERT_GENERIC:
         case VERT_EQUAL_KM:
            if (ctx->LogFlag) {
              hgt = ctx->BottomBound + (ctx->TopBound - ctx->BottomBound) *
                    level / (float) (ctx->MaxNl-1);
              p = HGT_TO_P( hgt );
              return P_TO_Z( p );
            }
            else {
              return ctx->Zmin + (ctx->Zmax-ctx->Zmin) * level
                     / (float) (ctx->MaxNl-1);
            }
         case VERT_NONEQUAL_KM:
            ilevel = (int) level;
            rlevel = level - ilevel;
            hgt = ctx->Height[ilevel] * (1.0-rlevel) + ctx->Height[ilevel+1] * rlevel;
            if (ctx->LogFlag) {
              p = HGT_TO_P( hgt );
              return P_TO_Z( p );
            }
            else {
              return ctx->Zmin + (hgt-ctx->BottomBound) /
                (ctx->TopBound-ctx->BottomBound) * (ctx->Zmax-ctx->Zmin);
            }
         case VERT_NONEQUAL_MB:
            ilevel = (int) level;
            rlevel = level - ilevel;
            hgt = ctx->Height[ilevel] * (1.0-rlevel) + ctx->Height[ilevel+1] * rlevel;
            p = height_to_pressure( hgt );
            return P_TO_Z( p );
         default:
            printf("Error in gridlevel_to_z\n");
      }
   }
   return 0.0;
}



/*
 * Convert a z graphics coordinate from [ctx->Zmax,ctx->Zmin] to a grid level in
 * [0,ctx->Nl-1].
 */
static float z_to_gridlev( Context ctx, float z )
{
   float p, hgt;

   if (z>=ctx->Zmax) {
      return (float) (ctx->MaxNl-1);
   }
   else if (z<=ctx->Zmin) {
      return 0.0;
   }
   else {
      switch (ctx->VerticalSystem) {
         case VERT_GENERIC:
         case VERT_EQUAL_KM:
            if (ctx->LogFlag) {
              p = Z_TO_P( z );
              hgt = P_TO_HGT( p );
              return (float) (ctx->MaxNl-1) * (ctx->BottomBound + z) /
                     (ctx->TopBound-ctx->BottomBound);
            }
            else {
              return (float) (ctx->MaxNl-1) * (z-ctx->Zmin)
                     / (ctx->Zmax-ctx->Zmin);
            }
         case VERT_NONEQUAL_KM:
            if (ctx->LogFlag) {
              p = Z_TO_P( z );
              hgt = P_TO_HGT( p );
            }
            else {
              hgt = ctx->BottomBound + (ctx->TopBound-ctx->BottomBound) *
                    (z-ctx->Zmin)/(ctx->Zmax-ctx->Zmin);
            }
            /* do a binary search of ctx->Height[] for hgt */
            return binary_search( hgt, ctx->Height, ctx->MaxNl );
         case VERT_NONEQUAL_MB:
            p = Z_TO_P( z );
            hgt = pressure_to_height(p);
            return binary_search( hgt, ctx->Height, ctx->MaxNl );
         default:
            printf("Error in z_to_gridlev\n");
      }
   }
   return 0.0;
}



/*
 * Convert a grid level to a geographic height.
 * Input:  ctx - the vis5d context
 *         level - grid level
 * Return:  height
 *
 **** This same code is used in sounding.c lines 1834 -1864 
 *
*/
float gridlevel_to_height( Context ctx, float level )
{
   int ilevel;
   float rlevel;

   if (level<=0) {
      return ctx->BottomBound;
   }
   else if (level>=ctx->MaxNl-1 || ctx->MaxNl == 1) {
      return ctx->TopBound;
   }
   else {
      switch (ctx->VerticalSystem) {
         case VERT_GENERIC:
         case VERT_EQUAL_KM:
            return ctx->BottomBound + level * ctx->LevInc;
         case VERT_NONEQUAL_MB:
         case VERT_NONEQUAL_KM:
            ilevel = (int) level;
            rlevel = level - ilevel;
            return ctx->Height[ilevel] * (1.0-rlevel) + ctx->Height[ilevel+1] * rlevel;
         default:
            printf("Error in gridlevel_to_height\n");
      }
   }
   return 0.0;
}



/*
 * Convert a height from [ctx->BottomBound,ctx->TopBound] to a grid level
 * in [0,ctx->MaxNl-1].
 */
static float height_to_gridlev( Context ctx, float hgt )
{
   if (hgt<=ctx->BottomBound) {
      return 0.0;
   }
   else if (hgt>=ctx->TopBound) {
      return (float) (ctx->MaxNl-1);
   }
   else {
      switch (ctx->VerticalSystem) {
         case VERT_GENERIC:
         case VERT_EQUAL_KM:
            return (hgt-ctx->BottomBound) / ctx->LevInc;
         case VERT_NONEQUAL_MB:
         case VERT_NONEQUAL_KM:
            /* do a binary search of ctx->Height[] for hgt */
            return binary_search( hgt, ctx->Height, ctx->MaxNl );
         default:
            printf("Error in height_to_gridlev\n");
      }
   }
   return 0.0;
}




/*
 * Convert a height in [ctx->BottomBound,ctx->TopBound] to a z coordinate in [ctx->Zmax,ctx->Zmin].
 */
float height_to_z( Context ctx, float hgt )
{
   float p;

   if (hgt>=ctx->TopBound) {
      return ctx->Zmax;
   }
   else if (hgt<=ctx->BottomBound) {
      return ctx->Zmin;
   }
   else {
      switch (ctx->VerticalSystem) {
         case VERT_GENERIC:
         case VERT_EQUAL_KM:
         case VERT_NONEQUAL_KM:
            if (ctx->LogFlag) {
              p = HGT_TO_P( hgt );
              return P_TO_Z( p );
            }
            else {
              return ctx->Zmin + (hgt-ctx->BottomBound)
                     / (ctx->TopBound-ctx->BottomBound)
                     * (ctx->Zmax-ctx->Zmin);
            }
         case VERT_NONEQUAL_MB:
            p = height_to_pressure( hgt );
            return P_TO_Z( p );
         default:
            printf("Error in height_to_z\n");
      }
   }
   return 0.0;
}




/*
 * Convert a z value to a height coordinate.
 */
static float z_to_height( Context ctx, float z )
{
   float p;

   switch (ctx->VerticalSystem) {
      case VERT_GENERIC:
      case VERT_EQUAL_KM:
      case VERT_NONEQUAL_KM:
         if (ctx->LogFlag) {
           p = Z_TO_P( z );
           return P_TO_HGT( p );
         }
         else {
           return ctx->BottomBound + (z-ctx->Zmin) *
                  (ctx->TopBound-ctx->BottomBound) / (ctx->Zmax-ctx->Zmin);
         }
      case VERT_NONEQUAL_MB:
         p = Z_TO_P( z );
         return pressure_to_height( p );
      default:
         printf("Error in z_to_height\n");
   }
   return 0.0;
}



/**********************************************************************/
/*****       (x,y,z), (row,col,lev), (lat,lon,hgt) conversion     *****/
/**********************************************************************/



/*
 * Transform an array of (r,c,l) grid coordinates to (x,y,z) graphics
 * coordinates.  (r,c,l) should be in [0,Nr-1][0,Nc-1][0,Nl-1] and
 * (x,y,z) will be returned in [Xmin,Xmax][Ymin,Ymax][Zmax,Zmin].
 *
 * Input:  ctx - the vis5d context
 *         time - which timestep
 *         var - which variable
 *         n - number of coordinates to transform
 *         r, c, l - array of grid coords.
 * Output:  x, y, z - array of graphics coords.
 */
void grid_to_xyz( Context ctx, int time, int var, int n,
                  float r[], float c[], float l[],
                  float x[], float y[], float z[] )
{
   int i;

   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
      case PROJ_LAMBERT:
      case PROJ_STEREO:
      case PROJ_ROTATED:
         switch (ctx->VerticalSystem) {
            case VERT_GENERIC:
            case VERT_EQUAL_KM:
               /* simplest, fast case */
               {
                  float xs, ys, zs;
                  xs = (ctx->Xmax-ctx->Xmin) / (float) (ctx->Nc-1);
                  ys = (ctx->Ymax-ctx->Ymin) / (float) (ctx->Nr-1);
                  if (ctx->MaxNl > 1) zs = (ctx->Zmax-ctx->Zmin) / (float) (ctx->MaxNl-1);
                  else zs = 0.0;
                  for (i=0;i<n;i++) {
                     x[i] = ctx->Xmin + c[i] * xs;
                     y[i] = ctx->Ymax - r[i] * ys;
                     z[i] = ctx->Zmin + l[i] * zs;
                  }
               }
               break;
            case VERT_NONEQUAL_MB:
            case VERT_NONEQUAL_KM:
               {
                  float xs, ys;
                  xs = (ctx->Xmax-ctx->Xmin) / (float) (ctx->Nc-1);
                  ys = (ctx->Ymax-ctx->Ymin) / (float) (ctx->Nr-1);
                  for (i=0;i<n;i++) {
                     x[i] = ctx->Xmin + c[i] * xs;
                     y[i] = ctx->Ymax - r[i] * ys;
                     z[i] = gridlevel_to_z( ctx, time, var, l[i] );
                  }
               }
               break;
         }
         break;
      case PROJ_CYLINDRICAL:
         {
            for (i=0;i<n;i++) {
               float lat, lon, radius;
               lat = ctx->NorthBound - r[i]*(ctx->NorthBound-ctx->SouthBound)/(float) (ctx->Nr-1);
               radius = (90.0 - lat) * ctx->CylinderScale;
               lon = ctx->WestBound - c[i] * (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
               lon = lon * DEG2RAD;
               x[i] = radius * cos(lon);
               y[i] = -radius * sin(lon);
               z[i] = gridlevel_to_z( ctx, time, var, l[i] );
            }
         }
         break;
      case PROJ_SPHERICAL:
         for (i=0;i<n;i++) {
            float lat, lon, hgt;
            float clat, slat, clon, slon, d;

            /* convert (r,c,l) to (lat,lon,hgt) */
            lat = ctx->NorthBound - r[i] * (ctx->NorthBound-ctx->SouthBound) / (float) (ctx->Nr-1);
            lon = ctx->WestBound - c[i] * (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
            hgt = gridlevel_to_height( ctx, l[i] );

            /* convert (lat,lon,hgt) to (x,y,z) */
            clat = cos(lat * DEG2RAD);
            clon = cos(lon * DEG2RAD);
            slat = sin(lat * DEG2RAD);
            slon = sin(lon * DEG2RAD);
            d = (hgt-ctx->BottomBound) / (ctx->TopBound-ctx->BottomBound) * SPHERE_SCALE
                + SPHERE_SIZE;
            x[i] = d * clat * clon;
            y[i] = -d * clat * slon;
            z[i] = d * slat;
         }
         break;
      default:
         printf("Error in grid_to_xyz\n");
   }

}



/*
 * Transform an array of (r,c,l) grid coordinates to compressed (x,y,z)
 * graphics coordinates.  (r,c,l) should be in [0,Nr-1][0,Nc-1][0,Nl-1]
 * and (x,y,z) will be returned in [+/-VERTEX_SCALE]^3.
 * NOTE: This function must be FAST because it's used whenever an
 * isosurface, slice, or trajectory is made to transform all the coords!!!
 *
 * Input:  ctx - the vis5d context
 *         time - which timestep
 *         var - which variable
 *         n - number of coordinates
 *         r, c, l - array of grid coords.
 * Output:  xyz - array of compressed graphics coords.
 */
void grid_to_compXYZ( Context ctx, int time, int var, int n,
                      float r[], float c[], float l[],
                      int_2 xyz[][3] )
{
   int i;

   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
      case PROJ_LAMBERT:
      case PROJ_STEREO:
      case PROJ_ROTATED:
         switch (ctx->VerticalSystem) {
            case VERT_GENERIC:
            case VERT_EQUAL_KM:
               /* simplest, fast case */
               {
                  float xs, ys, zs, xt, yt, zt;
                  xs = (ctx->Xmax-ctx->Xmin) / (float) (ctx->Nc-1) * VERTEX_SCALE;
                  ys = (ctx->Ymax-ctx->Ymin) / (float) (ctx->Nr-1) * VERTEX_SCALE;
                  if (ctx->MaxNl > 1) {
                     zs = (ctx->Zmax-ctx->Zmin) / (ctx->MaxNl-1) * VERTEX_SCALE;
                  }
                  else {
                     zs = 0.0;
                  }
                  xt = ctx->Xmin * VERTEX_SCALE;
                  yt = ctx->Ymax * VERTEX_SCALE;
                  zt = ctx->Zmin * VERTEX_SCALE;
                  for (i=0;i<n;i++) {
                     xyz[i][0] = (int_2) (xt + c[i] * xs);
                     xyz[i][1] = (int_2) (yt - r[i] * ys);
                     xyz[i][2] = (int_2) (zt + l[i] * zs);
                  }
               }
               break;
            case VERT_NONEQUAL_MB:
            case VERT_NONEQUAL_KM:
               {
                  float xs, ys, zs, xt, yt;
                  xs = (ctx->Xmax-ctx->Xmin) / (float) (ctx->Nc-1) * VERTEX_SCALE;
                  ys = (ctx->Ymax-ctx->Ymin) / (float) (ctx->Nr-1) * VERTEX_SCALE;
                  zs = VERTEX_SCALE;
                  xt = ctx->Xmin * VERTEX_SCALE;
                  yt = ctx->Ymax * VERTEX_SCALE;
                  for (i=0;i<n;i++) {
                     xyz[i][0] = (int_2) (xt + c[i] * xs);
                     xyz[i][1] = (int_2) (yt - r[i] * ys);
                     xyz[i][2] = (int_2)
                                (gridlevel_to_z( ctx, time, var, l[i] ) * zs);
                  }
               }
               break;
         }
         break;
      case PROJ_CYLINDRICAL:
         {
            for (i=0;i<n;i++) {
               float lat, lon, radius;
               float cylx, cyly, cylz;
               lat = ctx->NorthBound - r[i]
                       *(ctx->NorthBound-ctx->SouthBound)/(float) (ctx->Nr-1);
               radius = (90.0 - lat) * ctx->CylinderScale;
               lon = ctx->WestBound - c[i]
                       * (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
               lon = lon * DEG2RAD;
               cylx = radius * cos(lon);
               cyly = -radius * sin(lon);
               cylz = gridlevel_to_z( ctx, time, var, l[i] );
               xyz[i][0] = (int_2) (cylx * VERTEX_SCALE);
               xyz[i][1] = (int_2) (cyly * VERTEX_SCALE);
               xyz[i][2] = (int_2) (cylz * VERTEX_SCALE);
            }
         }
         break;
      case PROJ_SPHERICAL:
         for (i=0;i<n;i++) {
            float lat, lon, hgt;
            float clat, clon, slat, slon, d;

            /* convert (r,c,l) to (lat,lon,hgt) */
            lat = ctx->NorthBound - r[i]
                     * (ctx->NorthBound-ctx->SouthBound) / (float) (ctx->Nr-1);
            lon = ctx->WestBound - c[i]
                     * (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
            hgt = gridlevel_to_height( ctx, l[i] );

            /* convert (lat,lon,hgt) to (x,y,z) */
            clat = cos(lat * DEG2RAD);
            clon = cos(lon * DEG2RAD);
            slat = sin(lat * DEG2RAD);
            slon = sin(lon * DEG2RAD);
            d = (hgt-ctx->BottomBound)
               / (ctx->TopBound-ctx->BottomBound) * SPHERE_SCALE + SPHERE_SIZE;
            d *= VERTEX_SCALE;
            xyz[i][0] = (int_2) (d * clat * clon);
            xyz[i][1] = (int_2) (-d * clat * slon);
            xyz[i][2] = (int_2) (d * slat);
         }
         break;
      default:
         printf("Error in grid_to_compXYZ\n");
   }
}



/*
 * Transform an array of (lat,lon,hgt) coordinates to (x,y,z) graphics
 * coodinates.
 * Input:  ctx - the vis5d context
 *         time - which timestep
 *         var - which variable
 *         n - number of coordinates
 *         lat - latitude in degrees
 *         lon - longitude in degrees
 *         hgt - height in current vertical units
 * Output:  x, y, z - graphics coordinates.
 */
void geo_to_xyz( Context ctx, int time, int var, int n,
                 float lat[], float lon[], float hgt[],
                 float x[], float y[], float z[] )
{
   float xscale, yscale;
   int i;

   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
         xscale = (ctx->Xmax-ctx->Xmin) / (ctx->EastBound-ctx->WestBound);
         yscale = (ctx->Ymax-ctx->Ymin) / (ctx->NorthBound-ctx->SouthBound);
         for (i=0;i<n;i++) {
            x[i] = ctx->Xmin + (lon[i]-ctx->WestBound) * xscale;
            y[i] = ctx->Ymin + (lat[i]-ctx->SouthBound) * yscale;
            z[i] = height_to_z( ctx, hgt[i] );
         }
         break;
      case PROJ_LAMBERT:
         xscale = (ctx->Xmax-ctx->Xmin) / (float) (ctx->Nc-1);
         yscale = (ctx->Ymax-ctx->Ymin) / (float) (ctx->Nr-1);
         for (i=0;i<n;i++) {
            float rlon, rlat, r, row, col;
            rlon = lon[i] - ctx->CentralLon;
/*            if (rlon > 180.0)  rlon -= 360.0;*/
            rlon = rlon * ctx->Cone * DEG2RAD;

            if (lat[i]<-85.0) {
               /* infinity */
               r = 10000.0;
            }
            else {
               rlat = (90.0 - ctx->Hemisphere * lat[i]) * DEG2RAD * 0.5;
               r = ctx->ConeFactor * pow( tan(rlat), ctx->Cone );
            }
            row = ctx->PoleRow + r * cos(rlon);
            col = ctx->PoleCol - r * sin(rlon);
            x[i] = ctx->Xmin + col * xscale;
            y[i] = ctx->Ymax - row * yscale;
            z[i] = height_to_z( ctx, hgt[i] );
         }
         break;
      case PROJ_STEREO:
         xscale = (ctx->Xmax-ctx->Xmin) / (float) (ctx->Nc-1);
         yscale = (ctx->Ymax-ctx->Ymin) / (float) (ctx->Nr-1);
         for (i=0;i<n;i++) {
            float rlat, rlon, clon, clat, k, col, row;
            rlat = DEG2RAD * lat[i];
            rlon = DEG2RAD * (ctx->CentralLon - lon[i]);
            clon = cos(rlon);
            clat = cos(rlat);
            k = ctx->StereoScale
                / (1.0 + ctx->SinCentralLat*sin(rlat)
                       + ctx->CosCentralLat*clat*clon);
            col = (ctx->CentralCol-1) + k * clat * sin(rlon);
/* pre-friday:
            row = ctx->Nr - ctx->CentralRow
               - k * (ctx->CosCentralLat * sin(rlat)
                      - ctx->SinCentralLat * clat * clon);
*/
            row = (ctx->CentralRow-1)
               - k * (ctx->CosCentralLat * sin(rlat)
                      - ctx->SinCentralLat * clat * clon);


/*
            if (col<0.0)  col = 0.0;
            if (col>(ctx->Nc-1))  col = ctx->Nc-1;
            if (row<0.0)  row = 0.0;
            if (row>(ctx->Nr-1))  row = ctx->Nr-1;
*/
            x[i] = ctx->Xmin + col * xscale;
            y[i] = ctx->Ymax - row * yscale;
            z[i] = height_to_z( ctx, hgt[i] );
         }
         break;
      case PROJ_ROTATED:
         xscale = (ctx->Xmax-ctx->Xmin) / (ctx->EastBound-ctx->WestBound);
         yscale = (ctx->Ymax-ctx->Ymin) / (ctx->NorthBound-ctx->SouthBound);
         for (i=0;i<n;i++) {
            float lat0, lon0;
            lat0 = lat[i];
            lon0 = lon[i];
            pandg_for(&lat0, &lon0, ctx->CentralLat, ctx->CentralLon,
                      ctx->Rotation);
            x[i] = ctx->Xmin + (lon0-ctx->WestBound) * xscale;
            y[i] = ctx->Ymin + (lat0-ctx->SouthBound) * yscale;
            z[i] = height_to_z( ctx, hgt[i] );
         }
         break;
      case PROJ_CYLINDRICAL:
         for (i=0;i<n;i++) {
            float longitude, radius;
            radius = (90.0 - lat[i]) * ctx->CylinderScale;
            longitude = lon[i] * DEG2RAD;
            x[i] = radius * cos(longitude);
            y[i] = -radius * sin(longitude);
            z[i] = height_to_z( ctx, hgt[i] );
         }
         break;
      case PROJ_SPHERICAL:
         for (i=0;i<n;i++) {
            float clat, clon, slat, slon, d;
            clat = cos(lat[i] * DEG2RAD);
            clon = cos(lon[i] * DEG2RAD);
            slat = sin(lat[i] * DEG2RAD);
            slon = sin(lon[i] * DEG2RAD);
            d = (hgt[i]-ctx->BottomBound)
               / (ctx->TopBound-ctx->BottomBound) * SPHERE_SCALE + SPHERE_SIZE;
            x[i] = d * clat * clon;
            y[i] = -d * clat * slon;
            z[i] = d * slat;
         }
         break;
      default:
         printf("Error in geo_to_xyz\n");
   }
}




/*
 * Transform a (row,column) grid coordinate to (lat,lon) geographic coord.
 * Input:  ctx - the vis5d context
 *         time, var - which timestep and variable
 *         row, col - the row and column
 * Output:  lat, lon - latitude and longitude
 */
void rowcol_to_latlon( Context ctx, int time, int var, float row, float col,
                       float *lat, float *lon )
{

   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
      case PROJ_CYLINDRICAL:
      case PROJ_SPHERICAL:
         *lat = ctx->NorthBound - row * (ctx->NorthBound-ctx->SouthBound)
                / (float) (ctx->Nr-1);
         *lon = ctx->WestBound - col * (ctx->WestBound-ctx->EastBound)
                / (float) (ctx->Nc-1);
         break;
      case PROJ_LAMBERT:
         {
            float xldif, xedif, xrlon, radius;

            xldif = ctx->Hemisphere * (row-ctx->PoleRow) / ctx->ConeFactor;
            xedif = (ctx->PoleCol-col) / ctx->ConeFactor;
            if (xldif==0.0 && xedif==0.0)
               xrlon = 0.0;
            else
               xrlon = atan2( xedif, xldif );
            *lon = xrlon / ctx->Cone * RAD2DEG + ctx->CentralLon;
            if (*lon > 180.0)
               *lon -= 360.0;

            radius = sqrt( xldif*xldif + xedif*xedif );
            if (radius < 0.0001)
               *lat = 90.0 * ctx->Hemisphere;   /* +/-90 */
            else
               *lat = ctx->Hemisphere
                      * (90.0 - 2.0*atan(exp(log(radius)/ctx->Cone))*RAD2DEG);
         }
         break;
      case PROJ_STEREO:
         {
            float xrow, xcol, rho, c, cc, sc;
            xrow = ctx->CentralRow - row;
            xcol = ctx->CentralCol - col;
            rho = xrow*xrow + xcol*xcol;
            if (rho<1.0e-20) {
               *lat = ctx->CentralLat;
               *lon = ctx->CentralLon;
            }
            else {
               rho = sqrt( rho );
               c = 2.0 * atan( rho * ctx->InvScale);
               cc = cos(c);
               sc = sin(c);
               *lat = RAD2DEG
                    * asin( cc*ctx->SinCentralLat
                            + xrow*sc*ctx->CosCentralLat / rho );
               *lon = ctx->CentralLon + RAD2DEG * atan2( xcol * sc,
                         (rho * ctx->CosCentralLat * cc
                      - xrow * ctx->SinCentralLat * sc) );
               if (*lon < -180.0)  *lon += 360.0;
               else if (*lon > 180.0)  *lon -= 360.0;
            }
         }
         break;
      case PROJ_ROTATED:
         *lat = ctx->NorthBound - row
                     * (ctx->NorthBound-ctx->SouthBound) / (float) (ctx->Nr-1);
         *lon = ctx->WestBound - col
                     * (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
         pandg_back(lat, lon, ctx->CentralLat, ctx->CentralLon, ctx->Rotation);
         break;
      default:
         printf("Error in rowcol_to_latlon\n");
   }
}



/*
 * Convert an (x,y,z) graphics coordinate to an (r,c,l) grid coordinate.
 * Input:  ctx - the vis5d context
 *         time, var - which timestep, variable
 *         x, y, z - the graphics coordinate
 * Output:  row, col, lev - the corresponding grid coordinate.
 */
void xyz_to_grid( Context ctx, int time, int var,
                  float x, float y, float z,
                  float *row, float *col, float *lev )
{
   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
      case PROJ_LAMBERT:
      case PROJ_STEREO:
      case PROJ_ROTATED:
         *col = (x-ctx->Xmin) / (ctx->Xmax-ctx->Xmin) * (float) (ctx->Nc-1);
         *row = (ctx->Ymax-y) / (ctx->Ymax-ctx->Ymin) * (float) (ctx->Nr-1);
         *lev = z_to_gridlev( ctx, z );
         break;
      case PROJ_CYLINDRICAL:
         {
            float lat, lon, r;
            r = sqrt( x*x + y*y );
            if (r<0.001) {
               /* pole */
               lat = 90.0;
               lon = 0.0;
            }
            else {
               lat = 90.0 - r / ctx->CylinderScale;
               lon = atan2( -y, x ) * RAD2DEG;
               while (lon<ctx->EastBound)  lon += 360.0;
               while (lon>ctx->WestBound)  lon -= 360.0;
            }
            *col = (lon-ctx->WestBound) / (ctx->EastBound-ctx->WestBound) * (float) (ctx->Nc-1);
            *row = (lat-ctx->NorthBound) / (ctx->SouthBound-ctx->NorthBound) * (float) (ctx->Nr-1);
            *lev = z_to_gridlev( ctx, z );
         }
         break;
      case PROJ_SPHERICAL:
         {
            float r;
            r = sqrt( x*x + y*y + z*z );
            if (r<0.001) {
               /* degenerate case */
               *col = 0.0;
               *row = 0.0;
               *lev = 0.0;
            }
            else {
               float lat, lon, hgt, rxy;
               lon = atan2( -y, x ) * RAD2DEG;
               rxy = sqrt( x*x + y*y );
               if (rxy<0.001) {
                  /* north or south pole */
                  if (z<0.0) {
                     lat = -90.0;
                  }
                  else {
                     lat = 90.0;
                  }
                  lon = 0.0;
               }
               else {
                  lat = atan( z / rxy ) * RAD2DEG;
               }
               hgt = (r-SPHERE_SIZE) / SPHERE_SCALE * (ctx->TopBound-ctx->BottomBound)
                    + ctx->BottomBound;
               *col = (lon-ctx->WestBound) / (ctx->EastBound-ctx->WestBound) * (float) (ctx->Nc-1);
               *row = (lat-ctx->NorthBound) / (ctx->SouthBound-ctx->NorthBound)
                       * (float) (ctx->Nr-1);
               *lev = height_to_gridlev( ctx, hgt );
            }
         }
         break;
      default:
         printf("Error in xyz_to_grid\n");
   }
}



/*
 * Convert an (x,y,z) graphics coordinate to a (lat, lon, hgt) geographic
 * coordinate.
 * Input:  ctx - the vis5d context
 *         time, var - which timestep, variable.
 *         x, y, z - graphics coordinate
 * Output:  lat, lon, hgt - latitude, longitude, and height
 */
void xyz_to_geo( Context ctx, int time, int var,
                 float x, float y, float z,
                 float *lat, float *lon, float *hgt )
{
   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
         *lon = ctx->WestBound - (x-ctx->Xmin)
              * (ctx->WestBound-ctx->EastBound) / (ctx->Xmax-ctx->Xmin);
         *lat = ctx->SouthBound + (y-ctx->Ymin)
              * (ctx->NorthBound-ctx->SouthBound) / (ctx->Ymax-ctx->Ymin);
         *hgt = z_to_height( ctx, z );
         break;
      case PROJ_LAMBERT:
         {
            float col, row, xldif, xedif, xrlon, radius;

            /* convert x,y to row,col */
            col = (x-ctx->Xmin) / (ctx->Xmax-ctx->Xmin) * (float) (ctx->Nc-1);
            row = (ctx->Ymax-y) / (ctx->Ymax-ctx->Ymin) * (float) (ctx->Nr-1);
            /* convert row,col to lat,lon */
            xldif = ctx->Hemisphere * (row-ctx->PoleRow) / ctx->ConeFactor;
            xedif = (ctx->PoleCol-col) / ctx->ConeFactor;
            if (xldif==0.0 && xedif==0.0)
               xrlon = 0.0;
            else
               xrlon = atan2( xedif, xldif );
            *lon = xrlon / ctx->Cone * RAD2DEG + ctx->CentralLon;
#ifdef LEAVEOUT
            /* check if lon is in the undefined wedge-shaped region */
            if (ctx->Lat1==ctx->Lat2) {
               while (*lon > 180.0) {
                  *lon -= 360.0;
               }
               while (*lon <-180.0) {
                  *lon += 360.0;
               }
            }
            else {
               if (*lon > 180.0) {
                  *lon = 180.0;
               }
               if (*lon < -180.0) {
                  *lon = -180.0;
               }
            }
#endif

            radius = sqrt( xldif*xldif + xedif*xedif );
            if (radius < 0.0001)
               *lat = 90.0 * ctx->Hemisphere;   /* +/-90 */
            else
               *lat = ctx->Hemisphere
                      * (90.0 - 2.0*atan(exp(log(radius)/ctx->Cone))*RAD2DEG);

            *hgt = z_to_height( ctx, z );
         }
         break;
      case PROJ_STEREO:
         {
            float row, col, xrow, xcol, rho, c, cc, sc;
            /* convert x,y to row,col */
            col = (x-ctx->Xmin) / (ctx->Xmax-ctx->Xmin) * (float) (ctx->Nc-1);
            row = (ctx->Ymax-y) / (ctx->Ymax-ctx->Ymin) * (float) (ctx->Nr-1);
            /* convert row,col to lat,lon */
            xrow = ctx->CentralRow - row - 1;
            xcol = ctx->CentralCol - col - 1;
            rho = xrow*xrow + xcol*xcol;
            if (rho<1.0e-5) {
               *lat = ctx->CentralLat;
               *lon = ctx->CentralLon;
            }
            else {
               rho = sqrt( rho );
               c = 2.0 * atan( rho * ctx->InvScale);
               cc = cos(c);
               sc = sin(c);
               *lat = RAD2DEG
                   * asin( cc*ctx->SinCentralLat + xrow*sc*ctx->CosCentralLat / rho );
               *lon = ctx->CentralLon + RAD2DEG * atan2( xcol * sc,
                         (rho * ctx->CosCentralLat * cc - xrow * ctx->SinCentralLat * sc) );
               if (*lon < -180.0) {
                  *lon += 360.0;
               }
               else if (*lon > 180.0) {
                  *lon -= 360.0;
               }
            }
            *hgt = z_to_height( ctx, z );
         }
         break;
      case PROJ_ROTATED:
         *lon = ctx->WestBound - (x-ctx->Xmin)
              * (ctx->WestBound-ctx->EastBound) / (ctx->Xmax-ctx->Xmin);
         *lat = ctx->SouthBound + (y-ctx->Ymin)
              * (ctx->NorthBound-ctx->SouthBound) / (ctx->Ymax-ctx->Ymin);
         *hgt = z_to_height( ctx, z );
         pandg_back(lat, lon, ctx->CentralLat, ctx->CentralLon, ctx->Rotation);
         break;
      case PROJ_CYLINDRICAL:
         {
            float r;
            r = sqrt( x*x + y*y );
            if (r<0.001) {
               /* pole */
               *lat = 90.0;
               *lon = 0.0;
            }
            else {
               *lat = 90.0 - r / ctx->CylinderScale;
               *lon = atan2( -y, x ) * RAD2DEG;
               if (ctx->WestBound>180.0)
                  while (*lon<ctx->EastBound)  *lon += 360.0;
               if (ctx->EastBound<-180.0)
                  while (*lon>ctx->WestBound)  *lon -= 360.0;
            }
            *hgt = z_to_height( ctx, z );
         }
         break;
      case PROJ_SPHERICAL:
         {
            float r;
            r = sqrt( x*x + y*y + z*z );
            if (r<0.001) {
               /* degenerate case */
               *lat = 0.0;
               *lon = 0.0;
               *hgt = 0.0;
            }
            else {
               *lon = atan2( -y, x ) * RAD2DEG;
               *lat = atan( z / sqrt(x*x+y*y) ) * RAD2DEG;
               /* TODO: will this work with ctx->VertFlag? */
               *hgt = (r-SPHERE_SIZE) / SPHERE_SCALE
                    * (ctx->TopBound-ctx->BottomBound)
                    + ctx->BottomBound;
            }
         }
         break;
      default:
         printf("Error in xyz_to_geo\n");
   }
}



/*
 * Use the current projection type to transform an array of normal
 * vectors.  Then compress the normals to signed bytes.
 * Input:  ctx - the vis5d context
 *         n - number of normals in the array
 *         vr, vc, vl - the grid coordinates corresponding to
 *                      each normal vector.
 *         nx, ny, nz - array of normal vectors to transform.
 *         cnorms - array to put transformed, compressed normals into.
 * Output:  nx, ny, nz - transformed normals.
 */
void project_normals( Context ctx, int n, float vr[], float vc[], float vl[],
                      float nx[], float ny[], float nz[], int_1 cnorms[][3] )
{
   int i;
   float deltalon, deltalat;

   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
      case PROJ_LAMBERT:
      case PROJ_STEREO:
      case PROJ_ROTATED:
         /* don't need vr, vc, vl location information, just compress */
         for (i=0;i<n;i++) {   /* fully vectorized */
            cnorms[i][0] = (int_1) (-nx[i] * NORMAL_SCALE);
            cnorms[i][1] = (int_1) ( ny[i] * NORMAL_SCALE);
            cnorms[i][2] = (int_1) (-nz[i] * NORMAL_SCALE);
         }
         break;
      case PROJ_CYLINDRICAL:
         /* Rotate the x and y components of the normal by an angle */
         /* theta which is computed from the normal's column position. */

         deltalon = (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
         for (i=0;i<n;i++) {
            float theta, longitude, nnx, nny;

            longitude = ctx->WestBound-vc[i]*deltalon;
            theta = (90.0-longitude) * DEG2RAD;
            nnx = -nx[i] * cos(theta) - ny[i] * sin(theta);
            nny = -nx[i] * sin(theta) + ny[i] * cos(theta);
            cnorms[i][0] = (int_1) (nnx * NORMAL_SCALE);
            cnorms[i][1] = (int_1) (nny * NORMAL_SCALE);
            cnorms[i][2] = (int_1) (-nz[i] * NORMAL_SCALE);
         }
         break;
      case PROJ_SPHERICAL:
         deltalon = (ctx->WestBound-ctx->EastBound) / (float) (ctx->Nc-1);
         deltalat = (ctx->NorthBound-ctx->SouthBound) / (float) (ctx->Nr-1);
/*
         printf("enter sx, sy, sz\n");
         scanf("%f %f %f", &sx, &sy, &sz );
*/
         for (i=0;i<n;i++) {
            float rho, theta, longitude, latitude;
            float nx2, ny2, nz2, nx3, ny3, nz3, nx4, ny4, nz4;
            longitude = ctx->WestBound-vc[i]*deltalon;
            latitude = ctx->NorthBound-vr[i]*deltalat;

            /* flip axes */
            nx2 = -nz[i];
            ny2 =  nx[i];
            nz2 = -ny[i];
/*
            nx2 = sx*nz[i];
            ny2 = sy*nx[i];
            nz2 = sz*ny[i];
*/

            /* rotate about Y axis by latitude */
            rho = -latitude * DEG2RAD;
            nx3 = nx2 * cos(rho) - nz2 * sin(rho);
            ny3 = ny2;
            nz3 = nx2 * sin(rho) + nz2 * cos(rho);

            /* rotate about Z axis by longitude */
            theta = -longitude * DEG2RAD;
            nx4 = nx3 * cos(theta) - ny3 * sin(theta);
            ny4 = nx3 * sin(theta) + ny3 * cos(theta);
            nz4 = -nz3;

            cnorms[i][0] = (int_1) (nx4 * NORMAL_SCALE);
            cnorms[i][1] = (int_1) (ny4 * NORMAL_SCALE);
            cnorms[i][2] = (int_1) (nz4 * NORMAL_SCALE);
         }
         break;

      default:
         printf("Error in project_normals\n");
   }

}





/**********************************************************************/
/*****                      Miscellaneous                         *****/
/**********************************************************************/


#define EARTH_RADIUS 6371230.0


/*
 * Compute the distance (in meters) between two points on the earth surface.
 * Input:  lat1, lon1 - first location
 *         lat2, lon2 - second location
 * Return:  distance in meters
 */
float earth_distance( float lat1, float lon1, float lat2, float lon2 )
{
   float xd, yd, zd, d;

   lat1 *= DEG2RAD;
   lon1 *= DEG2RAD;
   lat2 *= DEG2RAD;
   lon2 *= DEG2RAD;

   xd = EARTH_RADIUS * ( cos(lat2)*cos(lon2) - cos(lat1)*cos(lon1) );
   yd = EARTH_RADIUS * ( cos(lat2)*sin(lon2) - cos(lat1)*sin(lon1) );
   zd = EARTH_RADIUS * ( sin(lat2) - sin(lat1) );

   d = sqrt( xd*xd + yd*yd + zd*zd );
   if (d/(2.0*EARTH_RADIUS) < 0.001) {
      /* d << 2R */
      return d;
   }
   else {
      /* return arc length */
      return 2.0 * EARTH_RADIUS * asin( d / (2.0*EARTH_RADIUS) );
   }
}


/*
 * Compute rough lat lon bounds
 * Input:  ctx - the vis5d context
 * Output: lats, latn, lonw, lone - bounds
 */
void latlon_bounds( Context ctx,
                    float *lats, float *latn, float *lonw, float *lone )
{
  float t, n;

  rowcol_to_latlon( ctx, 0, 0, 0.0, 0.0, &t, &n );
  *latn = *lats = t;
  *lonw = *lone = n;
  rowcol_to_latlon( ctx, 0, 0, (float) ctx->Nr - 1.0, 0.0, &t, &n );
  if (t > *latn) *latn = t;
  if (t < *lats) *lats = t;
  if (n > *lonw) *lonw = n;
  if (n < *lone) *lone = n;
  rowcol_to_latlon( ctx, 0, 0, 0.0, (float) ctx->Nc - 1.0, &t, &n );
  if (t > *latn) *latn = t;
  if (t < *lats) *lats = t;
  if (n > *lonw) *lonw = n;
  if (n < *lone) *lone = n;
  rowcol_to_latlon( ctx, 0, 0, (float) ctx->Nr - 1.0, (float) ctx->Nc - 1.0, &t, &n );
  if (t > *latn) *latn = t;
  if (t < *lats) *lats = t;
  if (n > *lonw) *lonw = n;
  if (n < *lone) *lone = n;
  return;
}

/*
Pete and Greg parameters:
     Pete rotated sphere lat 0, lon 0 -> Earth lat a, lon b
     r = East angle between North half of lon = 0 line on Pete rotated
         sphere and lon = b on Earth

coordinates:
    lat p1, lon g1 on Earth
    lat pr, lon gr on Pete rotated sphere
*/

/* Pete rotated sphere to Earth */
void pandg_back( float *lat, float *lon, float a, float b, float r )
{
  float pr, gr, pm, gm;

  /* NOTE - longitude sign switches - b too! */

  pr = DEG2RAD * *lat;
  gr = -DEG2RAD * *lon;
  pm = asin( cos(pr) * cos (gr) );
  gm = atan2(cos(pr) * sin (gr), -sin(pr) );

  *lat = RAD2DEG * asin( sin(a) * sin(pm) - cos(a) * cos(pm) * cos (gm - r) );
  *lon = -RAD2DEG * (-b + atan2(cos(pm) * sin (gm - r),
                   sin(a) * cos(pm) * cos (gm - r) + cos(a) * sin(pm)));

  return;
}

/* Earth to Pete rotated sphere */
void pandg_for( float *lat, float *lon, float a, float b, float r )
{
  float p1, g1, p, g;

  /* NOTE - longitude sign switches - b too! */

  p1 = DEG2RAD * *lat;
  g1 = -DEG2RAD * *lon;
  p = asin( sin(a) * sin(p1) + cos(a) * cos(p1) * cos (g1 + b) );
  g = r + atan2(cos(p1) * sin (g1 + b),
                sin(a) * cos(p1) * cos (g1 + b) - cos(a) * sin(p1) );

  *lat = RAD2DEG * asin( -cos(p) * cos (g) );
  *lon = -RAD2DEG * atan2(cos(p) * sin (g), sin(p) );

  return;
}

