### $Id: nlme.q,v 1.7 1998/06/13 13:18:43 pinheiro Exp $
###
###            Fit a general nonlinear mixed effects model
###
### Copyright 1997, 1998 Jose C. Pinheiro <jcp@research.bell-labs.com>,
###                      Douglas M. Bates <bates@stat.wisc.edu>
###
### This file is part of the nlme library for S and related languages.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
### 
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
### 
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA

nlme <- 
  function(object,
	   fixed, 
	   data = sys.parent(),
	   random = fixed,
	   groups, 
	   start, 
           correlation = NULL,
           weights = NULL,
	   subset,
	   REML = FALSE,
	   na.action = na.fail,
	   naPattern, 
	   control = list(),
	   verbose= FALSE)
{
  UseMethod("nlme")
}

nlme.nlsList <- 
  function(object,
	   fixed, 
	   data = sys.parent(),
	   random = fixed,
	   groups, 
	   start, 
           correlation = NULL,
           weights = NULL,
	   subset,
	   REML = FALSE,
	   na.action = na.fail,
	   naPattern, 
	   control = list(),
	   verbose= FALSE)
{
  this.call <- as.list(match.call())[-1]
  ## checking the use of arguments defined within the function
  if (any(!is.na(match(names(this.call),
		       c("fixed", "groups", "data", "start"))))) {
    warning(paste("nlme.nlsList will redefine \"fixed\", \"groups\",",
		  "\"data\", and \"start\""))
  }
  ## add object, data, and optionally groups from the call that created object
  last.call <- as.list(attr(object, "call"))[-1]
  last.call$control <- NULL
  last.call$pool <- NULL
  ## was
  ## last.call <- last.call[is.na(match(names(last.call), "control"))]
  this.call[names(last.call)] <- last.call
  ## Got to fix up inconsistency with name of "model" and "object"
  ## Better to do this in the original functions
  this.call[["object"]] <- last.call[["model"]]
  this.call[["model"]] <- NULL
  ## create "fixed" and "start" 
  start <- list(fixed = fixed.effects(object))
  pnames <- names(start$fixed)
  this.call[["fixed"]] <- lapply(as.list(pnames), function(el)
                                 eval(parse(text = paste(el, 1, sep = "~"))))
  if(missing(random)) {
    random <- this.call[["fixed"]]
  }
  reSt <- reStruct(random, data = NULL)
  ranForm <- formula(reSt)[[1]]
  if (!is.list(ranForm)) {
    ranForm <- list(ranForm)
  }
  mData <- this.call[["data"]]
  if (is.null(mData)) {			# will try to construct
    allV <- unique(unlist(lapply(ranForm, function(el) all.vars(el[[3]]))))
    if (length(allV) > 0) {
      alist <- lapply(as.list(allV), as.name)
      names(alist) <- allV
      alist <- c(as.list(as.name("data.frame")), alist)
      mode(alist) <- "call"
      mData <- eval(alist, sys.parent(1))
    }
  } else {
    if (is.name(mData)) {
      mData <- eval(mData)
    } 
  }
  reSt <- reStruct(random, REML = REML, data = mData)
  rnames <- unlist(lapply(ranForm, "[[", 2))
  ## if the random effects are a subset of the coefficients,
  ## construct "random" initial estimates 
  if(all(match(rnames, pnames, 0))) {
    re <- random.effects(object)
    re[is.na(re)] <- 0
    start[["random"]] <- re[, rnames, drop = F]
    sSquared <- pooledSD(object)^2
    if(isInitialized(reSt)) {
      warning("Initial value for reStruct overwritten in nlme.nlsList")
    }
    matrix(reSt) <- 
      var(na.omit(coef(object))[, rnames, drop = F])/sSquared
  }
  this.call[["start"]] <- start
  this.call[["random"]] <- reSt
  do.call("nlme.formula", this.call)
}


nlme.formula <- 
  function(object,
	   fixed, 
	   data = sys.parent(),
	   random = fixed,
	   groups, 
	   start, 
           correlation = NULL,
           weights = NULL,
	   subset,
	   REML = FALSE,
	   na.action = na.fail,
	   naPattern, 
	   control = list(),
	   verbose= FALSE)
{
  ## This is the method that actually does the fitting
  assign("finiteDiffGrad",
	 function(model, data, pars)
	 {
	   dframe <- data.frame(data, pars)
	   base <- eval(model, dframe)
	   nm <- dimnames(pars)[[2]]
	   grad <- array(base, c(length(base), length(nm)), list(NULL, nm))
	   ssize <- sqrt(.Machine$double.eps)
	   for (i in nm) {
	     diff <- pp <- pars[ , i]
	     diff[pp == 0] <- ssize
	     diff[pp != 0] <- pp[pp != 0] * ssize
	     dframe[[i]] <- pp + diff
	     grad[ , i] <- (base - eval(model, dframe))/diff
	     dframe[[i]] <- pp
	   }
	   grad
	 },
	 frame = 1)
  ## keeping the call
  Call <- match.call()
  ## assigning a new name to the "object" argument
  form <- object

  ## control parameters
  controlvals <- nlmeControl()
  if(!missing(control)) {
    controlvals[names(control)] <- control
  }
  ##
  ## checking arguments
  ##
  if (!inherits(form, "formula"))
    stop("\"object\" must be a formula")
  if (length(form)!=3)
    stop("object formula must be of the form \"resp ~ pred\"")

  if(missing(groups)) {
    if (inherits(data, "groupedData")) {
      Call[["groups"]] <- groups <- 
        getGroupsFormula(data)
    } else {
      stop (paste("Data must inherit from \"groupedData\" class ",
		  "if \"groups\" is missing"))
    }
  }

  ##
  ## checking if self-starting formula is given
  ##
  if(missing(start) && !is.null(attr(eval(form[[3]][[1]]), "initial"))) {
    nlmeCall <- Call
    nlsLCall <- nlmeCall[c("","object","data","groups")]
    nlsLCall[[1]] <- as.name("nlsList")
    names(nlsLCall)[2] <- "model"
    for(i in c("data", "groups", "start")) {
      nlmeCall[[i]] <- NULL
    }
    nlmeCall[[1]] <- as.name("nlme.nlsList")
    ## checking if "data" is not equal to sys.parent()
    if(is.null(dim(data))) {
      stop("\"data\" must be given explicitly to use \"nlsList()\"")
    }
    nlsLObj <- eval(nlsLCall)
    nlmeCall[["object"]] <- as.name("nlsLObj")
    nlmeCall <- as.call(nlmeCall)
    return(eval(nlmeCall))
  }
  nlmeModel <- call("-", form[[2]], form[[3]])
  ##
  ## save writing list(...) when only one element
  ##

  if (!is.list(fixed)) {
    fixed <- list(fixed)
  }
  val <- NULL
  for(i in seq(along = fixed)) {
    if (is.name(fixed[[i]][[2]])) {
      val <- c(val, list(fixed[[i]]))
    } else {
      ## multiple parameters on left hand side
      val <- c(val, eval(parse(text = paste("list(",
           paste(paste(all.vars(fixed[[i]][[2]]), fixed[[i]][[3]], sep = "~"),
                 collapse=","),")"))))
    }
  }
  fixed <- val
  fnames <- character(length(fixed))
  for (i in seq(along = fixed)) {
    this <- eval(fixed[[i]])
    if (!inherits(this, "formula"))
      stop ("fixed must be a formula or list of formulae")
    if (length(this) != 3)
      stop ("formulae in fixed must be of the form \"parameter ~ expr\".")
    if (!is.name(this[[2]]))
      stop ("formulae in fixed must be of the form \"parameter ~ expr\".")
    fnames[i] <- as.character(this[[2]])
  }
  names(fixed) <- fnames

  reSt <- reStruct(random, REML = REML, data = NULL)
  ranForm <- formula(reSt)[[1]]         # for now
  if (!is.list(ranForm)) {
    ranForm <- list(ranForm)
  }
  rnames <- character(length(ranForm))
  for (i in seq(along = ranForm)) {
    this <- eval(ranForm[[i]])
    if (!inherits(this, "formula"))
      stop ("random formula must be a formula or list of formulae")
    if (length(this) != 3)
      stop ("formulae in random must be of the form \"parameter ~ expr\".")
    if (!is.name(this[[2]]))
      stop ("formulae in random must be of the form \"parameter ~ expr\".")
    rnames[i] <- as.character(this[[2]])
  }
  names(ranForm) <- rnames
  ## all parameter names
  pnames <- unique(c(fnames, rnames))
  ##
  ##  If data is a pframe, copy the parameters in the frame to frame 1
  ##
  if (inherits(data, "pframe")) {
    pp <- parameters(data)
    for (i in names(pp)) {
      assign(i, pp[[i]], frame = 1)
    }
    attr(data,"parameters") <- NULL
    class(data) <- "data.frame"
  }

  ## check if correlation is present and assign groups to its formula
  if (!is.null(correlation)) {
    ## will assign innermost group
    aux <- getGroupsFormula(eval(parse(text = paste("~1", 
			   deparse(groups[[2]]), sep ="|"))), asList = TRUE)
    aux <- aux[[length(aux)]]
    attr(correlation, "formula") <- 
      eval(parse(text = paste("~", 
		     deparse(getCovariateFormula(formula(correlation))[[2]]),
		     "|", deparse(aux[[2]]))))
  }
  ## create an nlme structure containing the random effects model and plug-ins
  nlmeSt <- nlmeStruct(reStruct = reSt, corStruct = correlation, 
                       varStruct = varFunc(weights))

  ## extract a data frame with enough information to evaluate
  ## form, fixed, random, groups, correlation, and weights
  mfArgs <- list(formula = asOneFormula(formula(nlmeSt), form, fixed,
                   groups, omit = c(pnames, "pi")),
		 data = data, na.action = na.action)
  if (!missing(subset)) {
    mfArgs[["subset"]] <- asOneSidedFormula(Call[["subset"]])[[2]]
  }
  dataMix <- do.call("model.frame", mfArgs)

  origOrder <- row.names(dataMix)	# preserve the original order
  ##
  ## Evaluating the groups expression
  ##  
  grps <- getGroups(dataMix, 
	     eval(parse(text = paste("~1", deparse(groups[[2]]), sep = "|"))))
  N <- dim(dataMix)[1]			# number of observations
  ##
  ## evaluating the naPattern expression, if any
  ##
  if(missing(naPattern)) naPat <- rep(TRUE, N)
  else naPat <- as.logical(eval(asOneSidedFormula(naPattern)[[2]], dataMix))
  origOrderShrunk <- origOrder[naPat]

  ## ordering data by groups
  ord <- order(grps)	#"order" treats a single named argument peculiarly
  grps <- grps[ord]            # ordered group
  ugrp <- unique(grps)
  dataMix <- dataMix[ord,, drop = FALSE]       # ordered data frame
  grps <- data.frame(grps)
  row.names(grps) <- row.names(dataMix)
  names(nlmeSt$reStruct) <- names(grps) <- as.character(deparse((groups[[2]])))
  naPat <- naPat[ord]			# ordered naPat
  dataMixShrunk <- dataMix[naPat, , drop=FALSE]
  ordShrunk <- ord[naPat]
  grpShrunk <- grps[naPat,, drop = FALSE]
  revOrder <- match(origOrder, row.names(dataMix))
  revOrderShrunk <- match(origOrderShrunk, row.names(dataMixShrunk))
  yShrunk <- eval(form[[2]], dataMixShrunk)
  ##
  ## defining list with parameter information
  ##
  plist <- vector("list", length(pnames))
  names(plist) <- pnames
  for (nm in pnames) {
    this <- list(fixed = !is.null(fixed[[nm]]),
                 random = !is.null(ranForm[[nm]]))
    if (this[["fixed"]]) {
      if (as.name(fixed[[nm]][[3]]) != "1") {
	this[["fixed"]] <-
          model.matrix(asOneSidedFormula(fixed[[nm]][[3]]), dataMix)
      }
    }
    if (this[["random"]]) {
      wch <- (1:length(rnames))[regexpr(nm, rnames) != -1]
      if (length(wch) == 1) {           # only one formula for nm
        if (as.name(ranForm[[nm]][[3]]) != "1"){
          this[["random"]] <-
            model.matrix(asOneSidedFormula(ranForm[[nm]][[3]]), dataMix)
        }
      } else {
        this[["random"]] <-
          lapply(ranForm[wch], function(el, data) {
            if (as.name(el[[3]]) == "1") TRUE
            else model.matrix(asOneSidedFormula(el[[3]]), data)
          }, data = dataMix)
      }
    }
    plist[[nm]] <- this
  }
  if (is.null(sfix <- start$fixed))
    stop ("start must have a component called \"fixed\"")
  ##
  ## Fixed effects names
  ##
  fn <- character(0)
  for(nm in fnames) {
    if(is.logical(f <- plist[[nm]]$fixed)) {
      if(is.logical(plist[[nm]]$random)) {
        fn <- c(fn, nm) 
      } else {
        fn <- c(fn, paste(nm, "(Intercept)", sep="."))
      }
    } else {
      fn <- c(fn, paste(nm, dimnames(f)[[2]], sep = "."))
    }
  }
  flength <- length(fn)
  if (length(sfix) != flength) 
    stop ("starting values for the fixed component are not the correct length")
  names(sfix) <- fn
  ##
  ## Random effects names
  ##
  rn <- character(0)
  uRnames <- unique(rnames)
  wchRnames <- integer(length(uRnames))
  names(wchRnames) <- uRnames
  for(i in seq(along = rnames)) {
    nm <- rnames[i]
    wchRnames[nm] <- wchRnames[nm] + 1
    r <- plist[[nm]]$random
    if (data.class(r) == "list") r <- r[[wchRnames[nm]]]
    if(is.logical(r)) {
      if(r) { 
        if(is.logical(plist[[nm]]$fixed)) {
          rn <- c(rn, nm) 
        } else {
          rn <- c(rn, paste(nm,"(Intercept)",sep="."))
        }
      }
    } else {
      rn <- c(rn, paste(nm, dimnames(r)[[2]], sep = ".")) 
    }
  }
  rlength <- length(rn)
  if (is.null(start$random)) {
    sran <- array(0, c(rlength, length(ugrp)),
                  list(rn, as.character(ugrp)))
  } else {
    sran <- start$random
    if (!is.matrix(sran))
      stop ("starting values for the random component should be a matrix")
    dimsran <- dim(sran)
    if (dimsran[1] != length(ugrp))
      stop (paste("number of rows in starting values for random component",
                  "should be", length(ugrp)))
    if (dimsran[2] != rlength)
      stop (paste("number of columns in starting values for",
                  "random component should be", rlength))
    dnamesran <- dimnames(sran)[[2]]
    if ((!is.null(dnamesran)) && (!all(sort(dnamesran) == sort(rn)))) {
      stop ("names mismatch in random and starting values for random")
    }
    if(any(dnamesran != rn)) {
      sran <- sran[, rn, drop = F]
    }
    sran <- t(sran)
  }
  Names(nlmeSt$reStruct[[1]]) <- rn
  ##
  ##   defining values of constants used in calculations
  ##
  p <- flength; q <- rlength; M <- length(ugrp)
  NReal <- sum(naPat)
  Dims  <- c(p, q, NReal, M, REML, N)
  ##
  ## Creating the fixed and random effects maps
  ##
  fmap <- list()
  n1 <- 1
  for(nm in fnames) {
    if(is.logical(f <- plist[[nm]]$fixed)) {
      fmap[[nm]] <- n1
      n1 <- n1 + 1
    } else {
      fmap[[nm]] <- n1:(n1+ncol(f) - 1)
      n1 <- n1 + ncol(f) 
    }
  }
  rmap <- list()
  n1 <- 1
  wchRnames[] <- 0
  for(nm in rnames) {
    wchRnames[nm] <- wchRnames[nm] + 1
    r <- plist[[nm]]$random
    if (data.class(r) == "list") {
      r <- r[[wchRnames[nm]]]
    }
    if(is.logical(r)) {
      val <- n1
      n1 <- n1 + 1
    } else {
      val <- n1:(n1+ncol(r) - 1)
      n1 <- n1 + ncol(r) 
    }
    if (is.null(rmap[[nm]])) {
      rmap[[nm]] <- val
    } else {
      rmap[[nm]] <- c(rmap[[nm]], list(val))
    }
  }
  Q <- ncol(grps)

  ##
  ## defining the nlFrame
  ##
  nlFrame <- new.frame(list(model = nlmeModel,
			    data = dataMix,
			    groups = as.character(grps[, 1]),
			    plist = plist,
			    beta = sfix,
			    b = sran,
			    pars = array(0, c(N, length(pnames)),
			      list(NULL, pnames)),
			    X = array(0, c(N, flength),
			      list(NULL, fn)),
			    Z = array(0, c(N, rlength), list(NULL, rn)),
			    N = N,
			    fmap = fmap,
			    rmap = rmap,
                            level = Q,
			    .parameters = c("beta", "b")),
		       FALSE)
  ##
  ## defining the model expression
  ##
  modelExpression <- ~
  {
    pars <- getPars(plist, fmap, rmap, groups, beta, b, level)
    res <- eval(model, data.frame(data, pars))
    if(!length(grad <- attr(res, "gradient")))
      grad <- finiteDiffGrad(model, data, pars)
    for (nm in names(plist)) {
      gradnm <- grad[, nm]
      if (is.logical(f <- plist[[nm]]$fixed)) {
        if(f) {
	  X[, fmap[[nm]]] <- gradnm
        }
      } else {
        X[, fmap[[nm]]] <- gradnm * f
      }
      if (is.logical(r <- plist[[nm]]$random)) {
        if(r) {
          Z[, rmap[[nm]]] <- gradnm
        }
      } else {
        if (data.class(rmap[[nm]]) != "list") {
          Z[, rmap[[nm]]] <- gradnm * r
        } else {
          for(i in seq(along = rmap[[nm]])) {
            if (is.logical(rr <- r[[i]])) {
              Z[, rmap[[nm]][[i]]] <- gradnm
            } else {
              Z[, rmap[[nm]][[i]]] <- gradnm * rr
            }
          }
        }
      }
    }
    result <- c(res, Z, X)
    result[is.na(result)] <- 0
    result
  }

  ww <- eval(modelExpression[[2]], local = nlFrame)
  w <- ww[1:N][naPat]
  ZX <- array(ww[-(1:N)], c(N, p + q), list(row.names(dataMix),
                                            c(rn, fn)))[naPat,]
  w <- w + ZX[, q + (1:p), drop = F] %*% sfix
  if(!is.null(start$random)) {
    w <- w + (ZX[, 1:q, drop = F] * 
              t(sran)[as.character(grpShrunk[, 1]),,drop = F]) %*% rep(1,q)
  }
  ncols <- c(rlength, flength, 1)
  ## creating the condensed linear model
  attr(nlmeSt, "conLin") <-
    list(Xy = array(c(ZX, w), c(NReal, sum(ncols)), 
	     list(row.names(dataMixShrunk), c(dimnames(ZX)[[2]],
					deparse(form[[2]])))),
	 dims = MEdims(grpShrunk, ncols), logLik = 0)
  ## some groups dimensions
  aGlen <- list(glen = attr(nlmeSt, "conLin")$dims[["ZXlen"]][[1]],
                gstart = attr(nlmeSt, "conLin")$dims$ZXoff[[1]])
  aGlen$maxglen <- max(aGlen$glen)
                
  ## additional attributes of nlmeSt
  attr(nlmeSt, "resp") <- yShrunk
  attr(nlmeSt, "model") <- modelExpression
  attr(nlmeSt, "local") <- nlFrame
  attr(nlmeSt, "N") <- N
  attr(nlmeSt, "NReal") <- NReal
  attr(nlmeSt, "naPat") <- naPat
  ## initialization
  nlmeSt <- initialize(nlmeSt, dataMixShrunk, grpShrunk,
                       control = controlvals)
  parMap <- attr(nlmeSt, "pmap")

  if (length(coef(nlmeSt)) == length(coef(nlmeSt$reStruct)) &&
      !needUpdate(nlmeSt))  {	# can do one decomposition
    ## need to save conLin for calculating updating between steps
    oldConLin <- attr(nlmeSt, "conLin")
    decomp <- T
  } else decomp <- F

  numIter <- 0				# number of iterations
  attach(controlvals)
  pnlsSettings <- c(pnlsMaxIter, minScale, pnlsTol, 0, 0, 0)
  repeat {
  ## alternating algorithm
    numIter <- numIter + 1
    ## LME step
    if (needUpdate(nlmeSt)) {             # updating varying weights
      nlmeSt <- update(nlmeSt, dataMixShrunk)
    }
    if (decomp) {
      attr(nlmeSt, "conLin") <- MEdecomp(oldConLin)
    }
    oldPars <- coef(nlmeSt)
    aMs <- ms(~-logLik(nlmeSt, nlmePars),
              start = list(nlmePars = c(coef(nlmeSt))),
              control = list(rel.tolerance = msTol, maxiter = msMaxIter,
                scale = msScale), trace = msVerbose)
    aConv <- coef(nlmeSt) <- aMs$parameters
    convIter <- aMs$numIter <- aMs$flags[31]
    aFit <- attr(nlmeSt, "lmeFit") <- MEestimate(nlmeSt, grpShrunk)
    if (verbose) {
      cat("\n**Iteration", numIter)
      cat("\n")
      cat("LME step: Loglik:", format(aFit$logLik),
          ", ms iterations:", aMs$numIter, "\n")
      print(nlmeSt)
    }

    ## PNLS step
    dims <- .C("setup_nonlin",
	       n = integer(3),
	       list(modelExpression),
	       as.integer(nlFrame),
	       NAOK = T)$n
    Factor <- pdMatrix(solve(nlmeSt$reStruct[[1]]), factor = TRUE)
    if (is.null(correlation)) {
      cF <- 1
    } else {
      cF <- corFactor(nlmeSt$corStruct)
    }
    if (is.null(weights)) {
      vW <- 1
    } else {
      vW <- varWeights(nlmeSt$varStruct)
    }
    work <- .C("do_nlme", 
	       thetaPNLS = c(sfix, sran), 
	       as.double(cF),
	       as.double(vW),
	       as.integer(Dims),
	       settings = as.double(pnlsSettings),
	       additional = double(NReal * ( 1 + p + q)),
	       as.double(Factor),
	       as.double(t(solve(Factor))),
	       as.integer(unlist(grps)),
	       as.integer(aGlen$glen),
	       as.integer(aGlen$gstart),
	       as.integer(aGlen$maxglen),
	       as.integer(!is.null(correlation)),
	       as.integer(!is.null(weights)),
	       as.integer(rep(naPat, p + q + 1)),
	       NAOK = T);
    if (verbose) {
      cat("\nPNLS step: RSS = ", format(work$set[6]), "\n fixed effects:")
      for (i in 1:p) cat(format(signif(work$thetaPNLS[i]))," ")
      cat("\n iterations:",work$set[5],"\n")
    }
    oldPars <- c(sfix, oldPars)
    convIter <- max(c(convIter, work$settings[5]))
    aConv <- c(work$thetaPNLS[1:p], aConv)
    sfix[] <- work$thetaPNLS[1:p]
    w <- work$additional[1:NReal]
    ZX[] <- work$additional[-(1:NReal)]
    sran[] <- work$thetaPNLS[-(1:p)]
    w <- w + as.vector((ZX[, 1:q, drop = F] * 
                        t(sran)[as.character(grpShrunk[, 1]),,drop=F]) %*%
                       rep(1,q) + ZX[,-(1:q), drop = F] %*% sfix)
    if (decomp) {
      oldConLin$Xy[] <- c(ZX, w)
      oldConLin$logLik <- 0
    } else {
      attr(nlmeSt, "conLin")$Xy[] <- c(ZX, w)
      attr(nlmeSt, "conLin")$logLik <- 0
    }

    conv <- abs((oldPars - aConv)/ifelse(aConv == 0, 1, aConv))
    aConv <- c(max(conv[1:p]))
    names(aConv) <- "fixed"
    conv <- conv[-(1:p)]
    for(i in names(nlmeSt)) {
      if (any(parMap[,i])) {
	aConv <- c(aConv, max(conv[parMap[,i]]))
	names(aConv)[length(aConv)] <- i
      }
    }

    if(verbose) {
      cat("\nConvergence:\n")
      print(aConv)
    }

    if((max(aConv) <= tolerance) || (convIter == 1)) {
      convResult <- 0
      break
    }
    if(numIter >= maxIter) {
      convResult <- 1
      if(returnObject) {
	warning("Maximum number of iterations reached without convergence")
	break
      } else {
	stop("Maximum number of iterations reached without convergence")
      }
    }
  }
  detach()

  ## wraping up
  if (decomp) {
    aFit <- MEestimate(nlmeSt, grpShrunk, oldConLin)
  } else {
    aFit <- MEestimate(nlmeSt, grpShrunk)
  }

  varFix <- crossprod(aFit$sigma * aFit$varFix)
  dimnames(varFix) <- list(fn, fn)
  ##
  ## fitted.values and residuals (in original order)
  ##
#  Resid <- vector("list", Q + 1)
#  names(Resid) <- c("fixed", names(grps))
#  Resid[[2]] <- resid(nlmeSt)
#  assign("level", 0, frame=nlFrame)
#  Resid[["fixed"]] <- resid(nlmeSt)
#  Resid <- as.data.frame(Resid)
#  row.names(Resid) <- row.names(dataMixShrunk)
#  Fitted <- yShrunk - Resid
  if (decomp) {
    Resid <- resid(nlmeSt, level = 0:Q, oldConLin)[revOrderShrunk, ]
  } else {
    Resid <- resid(nlmeSt, level = 0:Q)[revOrderShrunk, ]
  }
  Fitted <- yShrunk[revOrderShrunk] - Resid
  grpShrunk <- grpShrunk[revOrderShrunk, , drop = FALSE]
  attr(Resid, "std") <- aFit$sigma/(varWeights(nlmeSt)[revOrderShrunk])
  ## inverting back reStruct 
  nlmeSt$reStruct <- solve(nlmeSt$reStruct)
  ## saving part of dims
  dims <- attr(nlmeSt, "conLin")$dims[c("N", "Q", "qvec", "ngrps", "ncol")]
  ## getting the approximate var-cov of the parameters 
  if (controlvals$apVar) {
    apVar <- lmeApVar(nlmeSt, aFit$sigma, 
		      .relStep = controlvals[[".relStep"]],
		      natural = controlvals[["natural"]])
  } else {
    apVar <- "Approximate variance-covariance matrix not available"
  }
  ## putting sran in the right format (for now - later multiple levels)
  sran <- list(t(sran))
  names(sran) <- names(grps)
  ## getting rid of condensed linear model and fit
  attr(nlmeSt, "conLin") <- NULL
  attr(nlmeSt, "lmeFit") <- NULL
  ##
  ## creating the  nlme object
  ##
  estOut <- list(modelStruct = nlmeSt,
		 dims = dims,
		 coefficients = list(fixed = sfix, random = t(sran)),
		 varFix = varFix,
		 sigma = aFit$sigma,
		 apVar = apVar,
		 logLik = aFit$logLik,
		 numIter = numIter,
		 groups = grpShrunk,
		 call = Call,
		 estMethod = c("ML", "REML")[REML + 1],
		 fitted = Fitted,
		 residuals = Resid,
		 plist = plist,
                 map = list(fixed = fmap, random = rmap))
  if (inherits(data, "groupedData")) {
    ## saving labels and units for plots
    attr(estOut, "units") <- attr(data, "units")
    attr(estOut, "labels") <- attr(data, "labels")
  }
  class(estOut) <- c("nlme","lme")
  estOut
}
	      
###
### function used to calculate the parameters from 
### the fixed and random effects
###

getPars <- function(plist, fmap, rmap, groups, beta, b, level) {
  pars <- array(0, c(length(groups), length(plist)), list(NULL, names(plist)))
  for (nm in names(plist)) {
    if (is.logical(f <- plist[[nm]]$fixed)) {
      if(f) {
       pars[, nm] <- beta[fmap[[nm]]]
      }
    } else {
      pars[, nm] <- f %*% beta[fmap[[nm]]]
    }
    if (level > 0) {
      if (is.logical(r <- plist[[nm]]$random)) {
        if(r) {
          pars[, nm] <- pars[, nm] + b[rmap[[nm]], groups]
        }
      } else {
        if (data.class(r) != "list") {
          pars[,nm] <- pars[,nm] + (r * t(b)[groups, rmap[[nm]], drop = F])%*% 
            rep(1, ncol(r))
        } else {
          for(i in seq(along = rmap[[nm]])) {
            if (is.logical(rr <- r[[i]])) {
              pars[, nm] <- pars[, nm] + b[rmap[[nm]][[i]], groups]
            } else {
              pars[,nm] <- pars[,nm] +
                (rr * t(b)[groups, rmap[[nm]][[i]], drop = F]) %*%
                  rep(1, ncol(rr))
            }
          }
        }
      }
    }
  }
  pars  
}

###
###  Methods for standard generics
###

formula.nlme <- 
  function(object) 
{
  eval(object$call[["object"]])
}


predict.nlme <- 
  function(object, newdata, level = Q, asList = FALSE, na.action = na.fail,
	   naPattern = NULL)  
{
  ##
  ## method for predict() designed for objects inheriting from class nlme
  ##
  Q <- object$dims$Q
  if (missing(newdata)) {		# will return fitted values
    val <- fitted(object, level, asList)
    if (length(level) == 1) return(val)
    return(data.frame(object[["groups"]][,level[level != 0], drop = FALSE],
		      predict = val))
  }
  maxQ <- max(level)
  newdata <- as.data.frame(newdata)
  mfArgs <- list(formula = asOneFormula(formula(object),
                   formula(object$call$fixed),
                   formula(object$modelStruct), naPattern, 
                   omit = c(names(object$plist),".", "pi",
                     deparse(getResponseFormula(object)[[2]]))),
                 data = newdata, na.action = na.action)
  dataMix <- do.call("model.frame", mfArgs)
  newdata <- newdata[row.names(dataMix), , drop = FALSE]
  if (maxQ > 0) {                       # predictions with random effects
    groups <- getGroupsFormula(object)
    if(!all(match(all.vars(groups), names(newdata), 0))) {
      ## groups cannot be evaluated in newdata
      stop("Cannot evaluate groups for desired levels on \"newdata\"")
    }
    grps <- eval(groups[[2]], newdata)
    naGrp <- is.na(match(grps, levels(object$groups[,1])))
    if(all(naGrp)) {
      stop("Cannot calculate group predictions if all groups are NA.")
    } else {
      if(any(naGrp)) {
        oldGrpsNA <- as.character(grps)[naGrp]
	grps[naGrp] <- grps[!naGrp][1]
      }
#      if(any(is.na(match(levels(grps),
#                         levels((object$groups)[,1]))))) {
#	stop("Groups not used in the fit included in \"newdata\".")
#      }	
    }
  } else {
    grps <- rep(1, dim(newdata)[1])
    ran <- NULL
  }
  N <- dim(newdata)[1]
  ##
  ## evaluating the naPattern expression, if any
  ##
  if(is.null(naPattern)) naPat <- rep(T, N)
  else naPat <- as.logical(eval(asOneSidedFormula(naPattern)[[2]], newdata))

  ##
  ## Getting  the plist for the new data frame
  ##
  ##
  plist <- object$plist
  fixed <- eval(object$call$fixed)
  if (!is.list(fixed)) {
    fixed <- list(fixed)
  }
  val <- NULL
  for(i in seq(along = fixed)) {
    if (is.name(fixed[[i]][[2]])) {
      val <- c(val, list(fixed[[i]]))
    } else {
      ## multiple parameters on left hand side
      val <- c(val, eval(parse(text = paste("list(",
           paste(paste(all.vars(fixed[[i]][[2]]), fixed[[i]][[3]], sep = "~"),
                 collapse=","),")"))))
    }
  }
  fixed <- val
  fnames <- unlist(lapply(fixed, function(el) deparse(el[[2]])))
  names(fixed) <- fnames
  fix <- fixed.effects(object)
  fn <- names(fix)
  for(nm in fnames) {
    if (!is.logical(plist[[nm]]$fixed)) {
      plist[[nm]]$fixed <- model.matrix(asOneSidedFormula(fixed[[nm]][[3]]),
                                        dataMix)
    }
  }

  if (maxQ > 0) {
    ranForm <- formula(object$modelStruct$reStruct)[[1]]
    rnames <- unlist(lapply(ranForm, function(el) deparse(el[[2]])))
    ran <- random.effects(object)
    rn <- dimnames(ran)[[2]]
    ran <- t(ran)
    for(nm in names(plist)) {
      if (!is.logical(plist[[nm]]$random)) {
        wch <- (1:length(rnames))[regexpr(nm, rnames) != -1]
        if (length(wch) == 1) {         # only one formula for nm
          plist[[nm]]$random <-
            model.matrix(asOneSidedFormula(ranForm[[nm]][[3]]), dataMix)
        } else {                        # multiple formulae
          plist[[nm]]$random <- lapply(ranForm[wch],
                           function(el, data) {
                             model.matrix(asOneSidedFormula(el[[3]]), data)
                           }, data = dataMix)
        }
      }
    }
  }
  nlev <- length(level)
  val <- vector("list", nlev)
  namGrp <- names(object$modelStruct$reStruct)
  names(val) <- c("fixed", namGrp)[level + 1]
  grps <- as.character(grps)
  for(i in 1:nlev) {
    val[[i]] <- eval(formula(object)[[3]], data.frame(dataMix,
                  getPars(plist, object$map$fixed, object$map$random, grps,
                          fix, ran, level[i])))
  }

  if (maxQ > 0 && any(naGrp)) {
    val[[namGrp]][naGrp] <- NA
    grps[naGrp] <- oldGrpsNA
  }
  if (nlev == 1) {
    val <- unlist(val)
    if (maxQ > 0) {                     # only group predictions
      if (asList) {
        val <- split(val, ordered(grps, levels = unique(grps)))
      } else {
        names(val) <- grps
      }
    }
    return(val)
  }
  grps <- data.frame(grps)
  names(grps) <- namGrp
  data.frame(grps, predict = as.data.frame(val))
}

update.nlme <- 
  function(object,
	   fixed, 
	   random = fixed,
	   groups,
	   data = sys.parent(),
	   start, 
           correlation = NULL,
           weights = NULL,
	   subset,
	   REML = FALSE,
	   na.action = na.fail,
	   naPattern, 
	   control = list(),
	   verbose = FALSE)
{
  thisCall <- as.list(match.call())[-(1:2)]
  nextCall <- as.list(object$call)[-1]
  nextCall[names(thisCall)] <- thisCall
  do.call("nlme", nextCall)
}

###*### nlmeStruct - a model structure for nlme fits

nlmeStruct <-
  ## constructor for nlmeStruct objects
  function(reStruct, corStruct = NULL, varStruct = NULL, resp = NULL,
           model = NULL, local = NULL, N = NULL, naPat = NULL)
{

  val <- list(reStruct = reStruct, corStruct = corStruct,
              varStruct = varStruct)
  val <- val[!sapply(val, is.null)]	# removing NULL components
  attr(val, "settings") <- attr(val$reStruct, "settings")
  attr(val, "resp") <- resp
  attr(val, "model") <- model
  attr(val, "local") <- local
  attr(val, "N") <- N
  attr(val, "naPat") <- naPat
  class(val) <- c("nlmeStruct", "lmeStruct", "modelStruct")
  val
}

##*## nlmeStruct methods for standard generics

fitted.nlmeStruct <-
  function(object, level = Q,  conLin = attr(object, "conLin"), ...)
{
  Q <- attr(object, "conLin")$dims[["Q"]]
  attr(object, "resp") - resid(object, level, conLin)
}

residuals.nlmeStruct <-
  function(object, level = Q, conLin = attr(object, "conLin"), ...)
{
  Q <- conLin$dims[["Q"]]
  loc <- attr(object, "local")
  oLev <- get("level", frame = loc)
  on.exit(assign("level", oLev, frame = loc))
  dn <- c("fixed", rev(names(object$reStruct)))[level + 1]
  val <- array(0, c(attr(object, "NReal"), length(level)), 
       list(dimnames(conLin$Xy)[[1]], dn))
  for(i in 1:length(level)) {
    assign("level", level[i], frame = loc, immediate = TRUE)
    val[, i] <- c(eval(attr(object, "model")[[2]],
      local=loc))[1:attr(object, "N")][attr(object, "naPat")]
  }
  val
}

nlmeControl <-
  ## Set control values for iterations within nlme
  function(maxIter = 50, pnlsMaxIter = 7, msMaxIter = 50,
	   minScale = 0.001, tolerance = 0.000001, niterEM = 25,
           pnlsTol = 0.001, msTol = 0.000001, msScale = lmeScale,
           returnObject = F, verbose = F, msVerbose = F, gradHess = TRUE,
           apVar = TRUE, .relStep = (.Machine$double.eps)^(1/3),
           natural = TRUE)
{
  list(maxIter = maxIter, pnlsMaxIter = pnlsMaxIter, msMaxIter = msMaxIter,
       minScale = minScale, tolerance = tolerance, niterEM = niterEM,
       pnlsTol = pnlsTol, msTol = msTol, msScale = msScale,
       returnObject = returnObject, verbose = verbose,
       msVerbose = msVerbose, gradHess = gradHess,
       apVar = apVar, .relStep = .relStep, natural = natural)
}

### Local Variables:
### mode:S
### S-keep-dump-files: t
### End:
 
