### $Id: gls.q,v 1.12 1998/06/13 13:18:38 pinheiro Exp $
###
###  Fit a linear model with serial correlation or heteroscedasticity
###
### Copyright 1997, 1998 Jose C. Pinheiro <jcp@research.bell-labs.com>,
###                      Douglas M. Bates <bates@stat.wisc.edu>
###
### This file is part of the nlme library for S and related languages.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
### 
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
### 
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA

gls <- 
  ## fits linear model with serial correlation and variance functions,
  ## by maximum likelihood using a Newton-Raphson algorithm.
  function(model,
	   data = sys.parent(),
	   correlation = NULL,
	   weights = NULL,
	   subset,
	   REML = TRUE,
	   na.action = na.fail, 
	   control = list(),
	   verbose = F)
{
  Call <- match.call()

  ## control parameters
  controlvals <- glsControl()
  controlvals[names(control)] <- control

  ##
  ## checking arguments
  ##
  if (!inherits(model, "formula") || length(model) != 3) {
    stop("\nModel must be a formula of the form \"resp ~ pred\"")
  }
  ## check if correlation is present and has groups
  if (!is.null(correlation)) {
    groups <- getGroupsFormula(correlation, asList = TRUE)
    if (!is.null(groups)) {
      if (length(groups) > 1) {
	stop("Only single level of grouping allowed")
      }
      groups <- groups[[1]]
    } else {
      if (inherits(data, "groupedData")) { # will use as groups
	groups <- getGroupsFormula(data, asList = TRUE)
	if (length(groups) > 1) {	# ignore it
	  groups <- NULL
	}
	groups <- groups[[1]]
	attr(correlation, "formula") <- 
	  eval(parse(text = paste("~", 
		      deparse(getCovariateFormula(formula(correlation))[[2]]),
			 "|", deparse(groups[[2]]))))
      }
    }
  } else groups <- NULL
  ## create a gls structure containing the plug-ins
  glsSt <- 
    glsStruct(corStruct = correlation, varStruct = varFunc(weights))

  ## extract a data frame with enough information to evaluate
  ## formula, groups, corStruct, and varStruct
  mfArgs <- list(formula = asOneFormula(formula(glsSt), model, groups),
		 data = data, na.action = na.action)
  if (!missing(subset)) {
    mfArgs[["subset"]] <- asOneSidedFormula(Call[["subset"]])[[2]]
  }
  dataMod <- do.call("model.frame", mfArgs)
  origOrder <- row.names(dataMod)	# preserve the original order
  if (!is.null(groups)) {
    ## sort the model.frame by groups and get the matrices and parameters
    ## used in the estimation procedures
    grps <- getGroups(dataMod, 
	      eval(parse(text = paste("~1", deparse(groups[[2]]), sep = "|"))))

    ## ordering data by groups
    ord <- order(grps)	
    grps <- grps[ord]
    dataMod <- dataMod[ord, ,drop = F]
    revOrder <- match(origOrder, row.names(dataMod)) # putting in orig. order
  } else grps <- NULL
  
  ## obtaing basic model matrices
  X <- model.frame(model, dataMod)
  ## keeping the contrasts for later use in predict
  contr <- lapply(X, function(el) 
		  if (inherits(el, "factor")) contrasts(el))
  contr <- contr[!unlist(lapply(contr, is.null))]
  X <- model.matrix(model, X)
  y <- eval(model[[2]], dataMod)
  N <- nrow(X)
  p <- ncol(X)				# number of coefficients
    
  ## creating the condensed linear model
  attr(glsSt, "conLin") <-
    list(Xy = array(c(X, y), c(N, ncol(X) + 1), list(row.names(dataMod), 
	     c(dimnames(X)[[2]], deparse(model[[2]])))), 
	 dims = list(N = N, p = p, REML = as.integer(REML)), logLik = 0)

  ## initialization
  glsEstControl <- controlvals[c("singular.ok","qrTol")]
  glsSt <- initialize(glsSt, dataMod, glsEstControl)
  parMap <- attr(glsSt, "pmap")

  ##
  ## getting the fitted object, possibly iterating for variance functions
  ##
  numIter <- numIter0 <- 0
  attach(controlvals)
  repeat {
    oldPars <- c(attr(glsSt, "glsFit")[["beta"]], coef(glsSt))
    if (length(coef(glsSt))) {		# needs ms()
      if( exists( "is.R" ) && is.function( is.R ) && is.R() ) {
	aNlm <- nlm(f = function(glsPars) -logLik(glsSt, glsPars),
		    p = c(coef(glsSt)),
		    hessian = TRUE,
		    print = ifelse(msVerbose, 0, 2))
	numIter0 <- NULL
	coef(glsSt) <- aNlm$estimate
      } else {
	aMs <- ms(~-logLik(glsSt, glsPars),
		  start = list(glsPars = c(coef(glsSt))),
		  control = list(rel.tolerance = msTol, maxiter = msMaxIter,
		    scale = msScale), trace = msVerbose)
	coef(glsSt) <- aMs$parameters
	numIter0 <- aMs$numIter <- aMs$flags[31]
      }
    }
    attr(glsSt, "glsFit") <- glsEstimate(glsSt, control = glsEstControl)
    ## checking if any updating is needed
    if (!needUpdate(glsSt)) break
    ## updating the fit information
    numIter <- numIter + 1
    glsSt <- update(glsSt, dataMod)
    ## calculating the convergence criterion
    aConv <- c(attr(glsSt, "glsFit")[["beta"]], coef(glsSt))
    conv <- abs((oldPars - aConv)/ifelse(aConv == 0, 1, aConv))
    aConv <- c("beta" = max(conv[1:p]))
    conv <- conv[-(1:p)]
    for(i in names(glsSt)) {
      if (any(parMap[,i])) {
	aConv <- c(aConv, max(conv[parMap[,i]]))
	names(aConv)[length(aConv)] <- i
      }
    }
    if (verbose) {
      cat("\nIteration:",numIter)
      if( exists( "is.R" ) && is.function( is.R ) && is.R() ) {
        cat("\nObjective:", format(aNlm$minimum), "\n")
      } else {
        cat("\nObjective:",format(aMs$value),", ms iterations:",
            aMs$numIter, "\n")
      }
      print(glsSt)
      cat("\nConvergence:\n")
      print(aConv)
    }
    if (max(aConv) <= tolerance) {
      break
    }
    if (numIter > maxIter) {
      stop("Maximum number of iterations reached without convergence.")
    }
  }
  detach()
  ## wrapping up
  glsFit <- attr(glsSt, "glsFit")
  namBeta <- names(glsFit$beta)
  p <- length(namBeta)
  varBeta <- crossprod(glsFit$sigma * glsFit$varBeta)*(N - REML * p)/(N - p)
  dimnames(varBeta) <- list(namBeta, namBeta)
  ##
  ## fitted.values and residuals (in original order)
  ##
  Fitted <- fitted(glsSt)
  ## putting groups back in original order, if present
  if (!is.null(grps)) {
    grps <- grps[revOrder]
    Fitted <- Fitted[revOrder]
    Resid <- y[revOrder] - Fitted
    attr(Resid, "std") <- glsFit$sigma/(varWeights(glsSt)[revOrder])
  } else {
    Resid <- y - Fitted
    attr(Resid, "std") <- glsFit$sigma/(varWeights(glsSt))
  }
    
  ## getting the approximate var-cov of the parameters 
  if (controlvals$apVar) {
    apVar <- glsApVar(glsSt, glsFit$sigma, 
		      .relStep = controlvals[[".relStep"]])
  } else {
    apVar <- "Approximate variance-covariance matrix not available"
  }
  ## getting rid of condensed linear model and fit
  dims <- attr(glsSt, "conLin")[["dims"]]
  dims[["p"]] <- p
  attr(glsSt, "conLin") <- NULL
  attr(glsSt, "glsFit") <- NULL
  ##
  ## creating the  gls object
  ##
  estOut <- list(glsStruct = glsSt,
		 dims = dims,
		 contrasts = contr,
		 coefficients = glsFit[["beta"]],
		 varBeta = varBeta,
		 sigma = glsFit$sigma,
		 apVar = apVar,
		 logLik = glsFit$logLik,
		 numIter = if (needUpdate(glsSt)) numIter
		   else numIter0, 
		 groups = grps,
		 call = Call,
		 estMethod = c("ML", "REML")[REML + 1],
		 fitted = Fitted,
		 residuals = Resid)
  if (inherits(data, "groupedData")) {
    ## saving labels and units for plots
    attr(estOut, "units") <- attr(data, "units")
    attr(estOut, "labels") <- attr(data, "labels")
  }
  attr(estOut, "namBetaFull") <- dimnames(X)[[2]]
  class(estOut) <- "gls"
  estOut
}

### Auxiliary functions used internally in gls and its methods

glsApVar <-
  function(glsSt, sigma, conLin = attr(glsSt, "conLin"),
           .relStep = (.Machine$double.eps)^(1/3))
{
  ## calculate approximate variance-covariance matrix of all parameters
  ## except the coefficients
  fullGlsLogLik <-
    function(Pars, object, conLin, dims, N) {
      ## logLik as a function of sigma and coef(glsSt)
      npar <- length(Pars)
      lsigma <- Pars[npar]              # within-group std. dev.
      Pars <- Pars[-npar]
      coef(object) <- Pars
      conLin <- recalc(object, conLin)
      val <- .C("gls_loglik",
		as.double(conLin$Xy),
		as.integer(unlist(dims)),
		logLik = double(1),
		lRSS = double(1))[c("logLik", "lRSS")]
      aux <- 2 * (val[["lRSS"]] - lsigma)
      conLin[["logLik"]] + val[["logLik"]] + (N * aux - exp(aux))/2
    }
  if (length(glsCoef <- coef(glsSt)) > 0) {
    dims <- conLin$dims
    N <- dims$N - dims$REML * dims$p
    conLin[["logLik"]] <- 0               # making sure
    Pars <- c(glsCoef, lSigma = log(sigma))
    val <- fdHess(Pars, fullGlsLogLik, glsSt, conLin, dims, N,
		  .relStep = .relStep)[["Hessian"]]
    if (all(eigen(val)$values < 0)) {
      ## negative definite - OK
      val <- solve(-val)
      nP <- names(Pars)
      dimnames(val) <- list(nP, nP)
      attr(val, "Pars") <- Pars
      val
    } else {
      ## problem - solution is not a maximum
      "Non-positive definite approximate variance-covariance"
    }
  } else {
    NULL
  }
}

glsEstimate <-
  function(object, conLin = attr(object, "conLin"), 
	   control = list(singular.ok = F, qrTol = .Machine$single.eps))
{
  dd <- conLin$dims
  p <- dd$p
  oXy <- conLin$Xy
  conLin <- recalc(object, conLin)	# updating for corStruct and varFunc
  val <- .C("gls_estimate",
	    as.double(conLin$Xy),
	    as.integer(unlist(dd)),
	    beta = double(p),
	    sigma = double(1),
	    logLik = double(1),
	    varBeta = double(p * p),
	    rank = integer(1),
	    pivot = as.integer(1:(p + 1)))[c("beta","sigma","logLik","varBeta",
		"rank", "pivot")]
  rnk <- val[["rank"]]
  rnkm1 <- rnk - 1
  if (!(control$singular.ok) && (rnkm1 < p )) {
    stop(paste("computed gls fit is singular, rank", rnk))
  }
  N <- dd$N - dd$REML * p
  namCoef <- dimnames(oXy)[[2]][val[["pivot"]][1:rnkm1] + 1]	# coef names
  ll <- conLin$logLik + val[["logLik"]]
  varBeta <- t(array(val[["varBeta"]], c(rnkm1, rnkm1), 
		     list(namCoef, namCoef)))
  beta <- val[["beta"]][1:rnkm1]
  names(beta) <- namCoef
  fitVal <- oXy[, namCoef, drop = F] %*% beta
  list(logLik = N * (log(N) - (1 + log(2 * pi)))/2 + ll, beta = beta,
       sigma = val[["sigma"]], varBeta = varBeta, 
       fitted = fitVal, resid = oXy[, p + 1] - fitVal)
}

### Methods for standard generics

anova.gls <- 
  function(object, ..., test = TRUE, verbose = F)
{
  ## returns the likelihood ratio statistics, the AIC, and the BIC
  dots <- list(...)
  if ((rt <- length(dots) + 1) == 1) {
    if (!inherits(object,"gls")) {
      stop("Object must inherit from class \"gls\" ")
    }
    dims <- object$dims
    N <- dims$N
    p <- dims$p
    REML <- dims$REML
    ##
    ## if just one object returns the t.table for the coefficients
    ##
    stdBeta <- sqrt(diag(object$varBeta))
    ##
    ## coefficients, std. deviations, t-ratios, and p-values
    ##
    beta <- coef(object)
    tratio <- beta/stdBeta
    aod <- data.frame(beta, stdBeta, tratio, 2 * pt(-abs(tratio), 
						    dims$N - dims$p))
    dimnames(aod) <- 
      list(names(beta),c("Value","Std.Error","t-value", "p-value"))
    attr(aod,"rt") <- rt
    class(aod) <- c("anova.lme", "data.frame")
    aod
  }
  ##
  ## Otherwise construct the likelihood ratio and information table
  ## objects in ... may inherit from gls, lm, lmList, and lme (for now)
  ##
  else do.call("anova.lme", as.list(match.call()[-1]))
}

augPred.gls <- 
  function(object, primary = NULL, minimum = min(primary), 
	   maximum = max(primary), length.out = 51, ...)
{
  data <- eval(object$call$data)
  if (!inherits(data, "data.frame")) {
    stop(paste("Data in", substitute(object),
               "call must evaluate to a data frame"))
  }
  if(is.null(primary)) {
    if (!inherits(data, "groupedData")) {
      stop(paste(sys.call()[[1]],
      "without \"primary\" can only be used with fits of groupedData objects"))
    }
    primary <- getCovariate(data)
    prName <- deparse(getCovariateFormula(data)[[2]])
  } else{
    primary <- asOneSidedFormula(primary)[[2]]
    prName <- deparse(primary)
    primary <- eval(primary, data)
  }
  newprimary <- seq(from = minimum, to = maximum, length.out = length.out)
  groups <- getGroups(object)
  grName <- ".groups"
  if (is.null(groups)) {		# no groups used
    noGrp <- T
    groups <- rep("1", length(primary))
    value <- data.frame(newprimary, rep("1", length(newprimary)))
  } else {
    noGrp <- F
    ugroups <- unique(groups)
    value <- data.frame(rep(newprimary, length(ugroups)),
			rep(ugroups, rep(length(newprimary), length(ugroups))))
  }
  names(value) <- c(prName, grName)
  ## recovering other variables in data that may be needed for predictions
  ## varying variables will be replaced by their means
  summData <- gsummary(data, groups = groups)
  if (any(toAdd <- is.na(match(names(summData), names(value))))) {
    summData <- summData[, toAdd, drop = F]
  }
  value[, names(summData)] <- summData[value[, 2], ]
  pred <- predict(object, value)
  newvals <- cbind(value[, 1:2], pred)
  names(newvals)[3] <- respName <-
    deparse(getResponseFormula(object)[[2]])
  orig <- data.frame(primary, groups, getResponse(object))
  names(orig) <- names(newvals)
  value <- rbind(orig, newvals)
  attributes(value[, 2]) <- attributes(groups)
  value[, ".type"] <- ordered(c(rep("original", nrow(data)),
				rep("predicted", nrow(newvals))),
			      levels = c("predicted", "original"))
  class(value) <- c("augPred", class(value))
  labs <- list(x = prName, y = respName)
  unts <- list(x = "", y = "")
  if(inherits(data, "groupedData")) {
    labs[names(attr(data, "labels"))] <- attr(data, "labels")
    unts[names(attr(data, "units"))] <- attr(data, "units")
    attr(value, "units") <- attr(data, "units")
  } 
  attr(value, "labels") <- labs
  attr(value, "units") <- unts
  if (noGrp) {
    attr(value, "formula") <- 
      eval(parse(text = paste(respName, prName, sep = "~")))
  } else {
    attr(value, "formula") <- 
      eval(parse(text = paste(respName, "~", prName, "|", grName)))
  }
  value
}

coef.gls <-
  function(object, allCoef = F)
{
  val <- object$coefficients
  if (allCoef) {
    namFull <- attr(object, "namBetaFull")
    if (length(val) != (lF <- length(namFull))) {
      aux <- rep(NA, lF)
      names(aux) <- namFull
      aux[names(val)] <- val
      val <- aux
    }
  }
  val
}

comparePred.gls <-
  function(object1, object2, primary = NULL, 
	   minimum = min(primary), maximum = max(primary),
	   level = NULL, length.out = 51, ...) 
{
  if (length(level) > 1) {
    stop("Only one level allowed for predictions")
  }
  args <- list(object = object1, 
	       primary = primary,
	       level = level,
	       length.out = length.out)
  if (!is.null(primary)) {
    args[["minimum"]] <- minimum
    args[["maximum"]] <- maximum
  }
  val1 <- do.call("augPred", args)
  dm1 <- dim(val1)
  c1 <- deparse(substitute(object1))
  levels(val1[,4])[1] <- c1
  args[["object"]] <- object2
  val2 <- do.call("augPred", args)
  dm2 <- dim(val2)
  c2 <- deparse(substitute(object2))
  levels(val2[, 4])[1] <- c2
  val2 <- val2[val2[, 4] != "original", , drop = F]
  names(val2) <- names(val1)

  if (dm1[1] == dm2[1]) {
    lv1 <- sort(levels(val1[, 2]))
    lv2 <- sort(levels(val2[, 2]))
    if ((length(lv1) != length(lv2)) || any(lv1 != lv2)) {
      stop(paste(c1, "and", c2, "must have the same group levels"))
    }
    val <- rbind(val1[, -4], val2[, -4])
    val[, ".type"] <- 
      ordered(c(as.character(val1[,4]), as.character(val2[,4])),
		levels = c(c1, c2, "original"))
    attr(val, "formula") <- attr(val1, "formula")
  } else {				# one may have just "fixed"
    if (dm1[1] > dm2[1]) {
      mult <- dm1[1] %/% dm2[1]
      if ((length(levels(val2[, 2])) != 1) ||
	  (length(levels(val1[, 2])) != mult))
	{
	  stop("Wrong group levels")
	}
      val <- 
	data.frame(c(val1[,1], rep(val2[,1], mult)), rep(val1[,1], 2),
	   c(val1[,3], rep(val2[,3], mult)),
	   ordered(c(as.character(val1[,4]), 
		     rep(as.character(val2[,4]), mult)), 
		   levels = c(c1, c2, "original")))
      attr(val, "formula") <- attr(val1, "formula")
    } else {
      mult <- dm2[1] %/% dm1[1]
      if ((length(levels(val1[, 2])) != 1) ||
	  (length(levels(val2[, 2])) != mult))
	{
	  stop("Wrong group levels")
	}
      val <- 
	data.frame(c(rep(val1[,1], mult), val2[,1]), rep(val2[,1], 2),
	   c(rep(val1[,3], mult), val2[,3]),
	   ordered(c(rep(as.character(val1[,4]), mult), 
		     as.character(val1[,4])), levels = c(c1, c2, "original")))
      attr(val, "formula") <- attr(val2, "formula")
    }
  }
  class(val) <- c("comparePred", "augPred", class(val))
  attr(val, "labels") <- attr(val1, "labels")
  attr(val, "units") <- attr(val1, "units")
  val
}

fitted.gls <-
  function(object)
{
  val <- object$fitted
  lab <- "Fitted values"
  if (!is.null(aux <- attr(object, "units")$y)) {
    lab <- paste(lab, aux)
  }
  attr(val, "label") <- lab
  val
}


formula.gls <- function(object) eval(object$call$model)

getGroups.gls <- function(object) object$groups

getGroupsFormula.gls <-
  function(object, asList = FALSE)
{
  if (!is.null(cSt <- object$glsStruct$corStruct)) {
    getGroupsFormula(cSt, asList)
  } else {
    NULL
  }
}

getResponse.gls <-
  function(object, form)
{
  val <- resid(object) + fitted(object)
  if (is.null(lab <- attr(object, "labels")$y)) {
    lab <- deparse(getResponseFormula(object)[[2]])
  }
  if (!is.null(aux <- attr(object, "units")$y)) {
    lab <- paste(lab, aux)
  }
  attr(val, "label") <- lab
  val
}

intervals.gls <-
  function(object, level = 0.95, which = c("all", "var-cov", "coef"))
{
  which <- match.arg(which)
  val <- list()
  dims <- object$dims
  if (which != "var-cov") {		# coefficients included
    mult <- -qt((1-level)/2, dims$N - dims$p)
    est <- coef(object)
    std <- sqrt(diag(object$varBeta))
    val[["coef"]] <- 
      array(c(est - mult * std, est, est + mult * std),
	    c(length(est), 3), list(names(est), c("lower", "est.", "upper")))
    attr(val[["coef"]], "label") <- "Coefficients:"
  }

  if (which != "coef") {		# variance-covariance included
    if (is.null(aV <- object$apVar)) {	# only sigma
      Nr <- dims$N - dims$REML * dims*p
      est <- object$sigma * sqrt(Nr)
      val[["sigma"]] <- 
	array(c(est/qchisq((1+level)/2, Nr), object$sigma, 
		est/qchisq((1-level)/2, Nr)), c(1, 3),
	      list("sigma", c("lower", "est.", "upper")))
      attr(val[["sigma"]], "label") <- "Residual standard error:"
    } else {
      if (is.character(aV)) {
	stop(paste("Cannot get confidence intervals on var-cov components:",
		   aV))
      }
      mult <- -qnorm((1-level)/2)
      est <- attr(aV, "Pars")
      nP <- length(est)
      std <- sqrt(diag(aV))
      glsSt <- object[["glsStruct"]]
      namG <- names(glsSt)
      auxVal <- vector("list", length(namG) + 1)
      names(auxVal) <- c(namG, "sigma")
      aux <-
	array(c(est - mult * std, est, est + mult * std),
	      c(nP, 3), list(NULL, c("lower", "est.", "upper")))
      auxVal[["sigma"]] <- exp(aux[nP,])
      attr(auxVal[["sigma"]], "label") <- "Residual standard error:"
      aux <- aux[-nP,, drop = F]
      dimnames(aux)[[1]] <- namP <- names(coef(glsSt, F))
      for(i in 1:3) {
	coef(glsSt) <- aux[,i]
	aux[,i] <- coef(glsSt, unconstrained = F)
      }
      for(i in namG) {
	auxVal[[i]] <- aux[regexpr(i, namP)!=-1, , drop = F]
	dimnames(auxVal[[i]])[[1]] <- 
	  substring(dimnames(auxVal[[i]])[[1]], nchar(i) + 2)
	attr(auxVal[[i]], "label") <-
	  switch(i,
		 corStruct = "Correlation structure:",
		 varStruct = "Variance function:",
		 paste(i,":",sep=""))
      }
      val <- c(val, auxVal)
    }
  }
  attr(val, "level") <- level
  class(val) <- "intervals.gls"
  val
}

logLik.gls <-
  function(object, REML)
{
  p <- object$dims$p
  N <- object$dims$N
  Np <- N - p
  estM <- object$estMethod
  if (missing(REML)) REML <- estM == "REML"
  val <- object[["logLik"]]
  if (REML && (estM == "ML")) {			# have to correct logLik
    val <- val + (p * (log(2 * pi) + 1) + Np * log(1 - p/N) +
		  sum(log(abs(svd(object$varBeta)$d)))) / 2
  }
  if (!REML && (estM == "REML")) {	# have to correct logLik
    val <- val - (p * (log(2*pi) + 1) + N * log(1 - p/N) +
		  sum(log(abs(svd(object$varBeta)$d)))) / 2
  }
  attr(val, "nobs") <- N - REML * p
  attr(val, "df") <- p + length(coef(object[["glsStruct"]])) + 1
  class(val) <- "logLik"
  val
}

plot.gls <- 
  function(object, form = resid(., type = "pearson") ~ fitted(.), abline, 
	   id = NULL, idLabels = NULL,  grid, ...)  
  ## Diagnostic plots based on residuals and/or fitted values
{
  do.call("plot.lme", as.list(match.call()[-1]))
}

predict.gls <- 
  function(object, newdata, na.action = na.fail)  
{
  ##
  ## method for predict() designed for objects inheriting from class gls
  ##
  if (missing(newdata)) {		# will return fitted values
    return(fitted(object))
  }
  form <- getCovariateFormula(object)
  mfArgs <- list(formula = form, data = newdata, na.action = na.action)
  dataMod <- do.call("model.frame", mfArgs)
  ## making sure factor levels are the same as in contrasts
  contr <- object$contrasts
  for(i in names(dataMod)) {
    if (inherits(dataMod[,i], "factor") && !is.null(contr[[i]])) {
      levs <- levels(dataMod[,i])
      levsC <- dimnames(contr[[i]])[[1]]
      if (any(wch <- is.na(match(levs, levsC)))) {
        stop(paste("Levels", paste(levs[wch], collapse = ","),
                   "not allowed for", i))
      }
      if (length(levs) < length(levsC)) {
        if (inherits(dataMod[,i], "ordered")) {
          dataMod[,i] <- ordered(as.character(dataMod[,i]), levels = levsC)
        } else {
          dataMod[,i] <- factor(as.character(dataMod[,i]), levels = levsC)
        }
      }
    }
  }
  N <- nrow(dataMod)
  if (length(all.vars(form)) > 0) {
    X <- model.matrix(form, dataMod, contr)
  } else {
    X <- array(1, c(N, 1), list(row.names(dataMod), "(Intercept)"))
  }
  cf <- coef(object)
  c(X[, names(cf), drop = F] %*% cf)
}

print.intervals.gls <-
  function(x, ...)
{
  cat(paste("Approximate ", attr(x, "level") * 100,
	    "% confidence intervals\n", sep = ""))
  for(i in names(x)) {
    aux <- x[[i]]
    cat(" ",attr(aux, "label"), "\n", sep = "")
    if (i == "sigma") print(c(aux), ...)
    else print.matrix(aux, ...)
  }
}

print.gls <- 
  ## method for print() used for gls objects
  function(x, ...)
{
  dd <- x$dims
  cat("Call:\n")
  cat(deparse(x$call),"\n")
  cat("\nCoefficients:\n")
  print(coef(x))
  cat("\n")
  print(summary(x$glsStruct))
  cat("Degrees of freedom:", dd[["N"]],"total;",dd[["N"]] - dd[["p"]],
      "residual\n")
  cat("Residual standard Error:", format(x$sigma),"\n")
}

print.summary.gls <-
  function(x, verbose = F, digits = .Options$digits, ...)
{
  dd <- x$dims
  verbose <- verbose || attr(x, "verbose")
  cat("Call:\n")
  cat(deparse(x$call),"\n")
  cat("Standardized Residuals:\n")
  print(x$residuals)

  estMethod <- x$estMethod
  cat("\nEstimation Method:", estMethod,"\n")
  if (verbose) {
    cat("Convergence at iteration:",x$numIter,"\n")
    if (estMethod == "REML") {
      cat("Restricted ")
    }
    cat("Loglikelihood:",format(x$logLik),"\n")
    if (estMethod == "REML") {
      cat("Restricted ")
    }
    cat("AIC:",format(x$AIC),"\n")
    if (estMethod == "REML") {
      cat("Restricted ")
    }
    cat("BIC:",format(x$BIC))
   cat("\n")
  }
  cat("\nCoefficients:\n")
  print(x$tTable)
  if (nrow(x$tTable) > 1) {
    corr <- x$corBeta
    class(corr) <- "correlation"
    print(corr,
	  title = "\n Correlation of Coefficients",
	  ...)
  }
  cat("\n")
  print(summary(x$glsStruct))
  cat("Degrees of freedom:", dd[["N"]],"total;",dd[["N"]] - dd[["p"]],
      "residual\n")
  cat("Residual standard Error:", format(x$sigma),"\n")
}

residuals.gls <- 
  function(object, type = c("response", "pearson"))
{
  type <- match.arg(type)
  val <- object$residuals
  if (type == "pearson") {
    val <- val/attr(val, "std")
    attr(val, "label") <- "Standardized residuals"
  } else {
    lab <- "Residuals"
    if (!is.null(aux <- attr(object, "units")$y)) {
      lab <- paste(lab, " (", aux, ")", sep = "")
    }
    attr(val, "label") <- lab
  }
  val
}

summary.gls <- function(object, verbose = F) {
  ##
  ## generates an object used in the print.summary method for lme
  ##
  ##  variance-covariance estimates for coefficients
  ##
  stdBeta <- sqrt(diag(as.matrix(object$varBeta)))
  corBeta <- t(object$varBeta/stdBeta)/stdBeta
  ##
  ## coefficients, std. deviations and z-ratios
  ##
  beta <- coef(object)
  dims <- object$dims
  dimnames(corBeta) <- list(names(beta),names(beta))
  object$corBeta <- corBeta
  tTable <- data.frame(beta, stdBeta, beta/stdBeta, beta)
  dimnames(tTable)<-
    list(names(beta),c("Value","Std.Error","t-value","p-value"))
  tTable[, "p-value"] <- 2 * pt(-abs(tTable[,"t-value"]), dims$N - dims$p)
  object$tTable <- as.matrix(tTable)
  ##
  ## residuals
  ##
  resd <- resid(object, type = "pearson")
  if (length(resd) > 5) {
    resd <- quantile(resd)
    names(resd) <- c("Min","Q1","Med","Q3","Max")
  }
  object$residuals <- resd
  ##
  ## generating the final object
  ##
  aux <- logLik(object)
  object$BIC <- BIC(aux)
  object$AIC <- AIC(aux)
  class(object) <- c("summary.gls", class(object))
  attr(object, "verbose") <- verbose
  object
}

update.gls <-
  function(object, 
	   model,
	   data,
	   correlation,
	   weights,
	   subset,
	   REML,
	   na.action, 
	   control,
	   verbose)
{
  thisCall <- as.list(match.call())[-(1:2)]
  nextCall <- as.list(object$call)[-1]
  nextCall[names(thisCall)] <- thisCall
  do.call("gls", nextCall)
}

###*### glsStruct - a model structure for gls fits

glsStruct <-
  ## constructor for glsStruct objects
  function(corStruct = NULL, varStruct = NULL)
{
  val <- list(corStruct = corStruct, varStruct = varStruct)
  val <- val[!sapply(val, is.null)]	# removing NULL components
  class(val) <- c("glsStruct", "modelStruct")
  val
}

##*## glsStruct methods for standard generics

fitted.glsStruct <-
  function(object, glsFit = attr(object, "glsFit"))
{
  glsFit[["fitted"]]
}

initialize.glsStruct <-
  function(object, data, control = list(singular.ok = F,
                           qrTol = .Machine$single.eps))
{
  if (length(object)) {
    object[] <- lapply(object, initialize, data)
    theta <- lapply(object, coef)
    len <- unlist(lapply(theta, length))
    num <- seq(along = len)
    if (sum(len) > 0) {
      pmap <- outer(rep(num, len), num, "==")
    } else {
      pmap <- array(F, c(1, length(len)))
    }
    dimnames(pmap) <- list(NULL, names(object))
    attr(object, "pmap") <- pmap
    attr(object, "glsFit") <- 
      glsEstimate(object, control = control)
    if (needUpdate(object)) {
      object <- update(object, data)
    } 
  }
  object
}

logLik.glsStruct <-
  function(object, Pars, conLin = attr(object, "conLin"))
{
  coef(object) <- Pars			# updating parameter values
  conLin <- recalc(object, conLin)	# updating conLin
  val <- .C("gls_loglik",
	    as.double(conLin[["Xy"]]),
	    as.integer(unlist(conLin[["dims"]])),
	    logLik = as.double(conLin[["logLik"]]),
	    double(1))
  val[["logLik"]]
}

residuals.glsStruct <-
  function(object, glsFit = attr(object, "glsFit"))
{
  glsFit[["resid"]]
}

varWeights.glsStruct <-
  function(object)
{
  if (is.null(object$varStruct)) rep(1, attr(object, "conLin")$dims$N)
  else varWeights(object$varStruct)
}

## Auxiliary control functions

glsControl <-
  ## Control parameters for gls
  function(maxIter = 50, msMaxIter = 50, tolerance = 1e-6, msTol = 1e-7, 
	   msScale = lmeScale, msVerbose = F, singular.ok = F, 
	   qrTol = .Machine$single.eps, returnObject = F,
	   apVar = TRUE, .relStep = (.Machine$double.eps)^(1/3))
{
  list(maxIter = maxIter, msMaxIter = msMaxIter, tolerance = tolerance,
       msTol = msTol, msScale = msScale, msVerbose = msVerbose, 
       singular.ok = singular.ok, qrTol = qrTol, 
       returnObject = returnObject, apVar = apVar, 
       .relStep = .relStep)
}

### local generics for objects inheriting from class lme




## Local Variables:
## mode:S
## End:
