/*
 * Grace - Graphics for Exploratory Data Analysis
 * 
 * Home page: http://plasma-gate.weizmann.ac.il/Grace/
 * 
 * Copyright (c) 1991-95 Paul J Turner, Portland, OR
 * Copyright (c) 1996-98 GRACE Development Team
 * 
 * Maintained by Evgeny Stambulchik <fnevgeny@plasma-gate.weizmann.ac.il>
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *
 * strings, lines, boxes and elipses
 *
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <Xm/Xm.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Scale.h>

#include "globals.h"
#include "utils.h"
#include "protos.h"
#include "motifinc.h"

static Widget objects_frame;
static Widget strings_frame;
static Widget lines_frame;
static Widget boxes_frame;
static Widget ellip_frame;

static OptionStructure strings_font_item;
static Widget strings_rot_item;
static Widget strings_size_item;
static Widget *strings_loc_item;
static Widget *strings_pen_item;
static Widget *strings_just_item;
Widget strings_x_item;
Widget strings_y_item;

static Widget *lines_arrow_item;
static Widget lines_asize_item;
static Widget *lines_atype_item;
static Widget *lines_pen_item;
static Widget *lines_style_item;
static Widget *lines_width_item;
static Widget *lines_loc_item;
static Widget *boxes_pen_item;
static Widget *boxes_lines_item;
static Widget *boxes_linew_item;
static Widget *boxes_fillpat_item;
static Widget *boxes_fillcol_item;
static Widget *boxes_loc_item;

static Widget *ellip_pen_item;
static Widget *ellip_lines_item;
static Widget *ellip_linew_item;
static Widget *ellip_fillpat_item;
static Widget *ellip_fillcol_item;
static Widget *ellip_loc_item;
void define_ellip_popup(Widget w, XtPointer client_data, XtPointer call_data);

void ellip_def_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    defellip.color = GetChoice(ellip_pen_item);
    defellip.loctype = GetChoice(ellip_loc_item) ? COORD_VIEW : COORD_WORLD;
    defellip.lines = GetChoice(ellip_lines_item);
    defellip.linew = GetChoice(ellip_linew_item);
    defellip.fillcolor = GetChoice(ellip_fillcol_item);
    defellip.fillpattern = GetChoice(ellip_fillpat_item);
    defellip.active = TRUE;
}

void update_ellip(void)
{
    if (ellip_frame) {
	SetChoice(ellip_pen_item, defellip.color);
	SetChoice(ellip_lines_item, defellip.lines);
	SetChoice(ellip_linew_item, defellip.linew);
	SetChoice(ellip_fillpat_item, defellip.fillpattern);
	SetChoice(ellip_fillcol_item, defellip.fillcolor);
	SetChoice(ellip_loc_item, defellip.loctype == COORD_VIEW ? 1 : 0);
    }
}

void boxes_def_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    box_color = GetChoice(boxes_pen_item);
    box_loctype = GetChoice(boxes_loc_item) ? COORD_VIEW : COORD_WORLD;
    box_lines = GetChoice(boxes_lines_item);
    box_linew = GetChoice(boxes_linew_item);
    box_fillcolor = GetChoice(boxes_fillcol_item);
    box_fillpat = GetChoice(boxes_fillpat_item);
}

void lines_def_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    Arg a;
    int value;
    
    XtSetArg(a, XmNvalue, &value);
    XtGetValues(lines_asize_item, &a, 1);
    line_asize = value / 50.0;
    line_color = GetChoice(lines_pen_item);
    line_arrow = GetChoice(lines_arrow_item);
    line_atype = GetChoice(lines_atype_item);
    line_lines = GetChoice(lines_style_item);
    line_linew = GetChoice(lines_width_item);
    line_loctype = GetChoice(lines_loc_item) ? COORD_VIEW : COORD_WORLD;
}

void updatestrings(void)
{
    if (strings_frame) {
	SetOptionChoice(strings_font_item, string_font);
	SetChoice(strings_pen_item, string_color);
	SetCharSizeChoice(strings_size_item, string_size);
	SetAngleChoice(strings_rot_item, string_rot);
	SetChoice(strings_loc_item, string_loctype == COORD_VIEW ? 1 : 0);
	SetChoice(strings_just_item, string_just);
    }
}

void update_lines(void)
{
    Arg a;
    int iv;
    if (lines_frame) {
	SetChoice(lines_pen_item, line_color);
	SetChoice(lines_style_item, line_lines);
	SetChoice(lines_width_item, line_linew);
	SetChoice(lines_arrow_item, line_arrow);
	SetChoice(lines_atype_item, line_atype);
	iv = (int) (50 * line_asize);
	XtSetArg(a, XmNvalue, iv);
	XtSetValues(lines_asize_item, &a, 1);
	SetChoice(lines_loc_item, line_loctype == COORD_VIEW ? 1 : 0);
    }
}

void update_boxes(void)
{
    if (boxes_frame) {
	SetChoice(boxes_pen_item, box_color);
	SetChoice(boxes_lines_item, box_lines);
	SetChoice(boxes_linew_item, box_linew);
	SetChoice(boxes_fillpat_item, box_fillpat);
	SetChoice(boxes_fillcol_item, box_fillcolor);
	SetChoice(boxes_loc_item, box_loctype == COORD_VIEW ? 1 : 0);
    }
}

void define_string_defaults(Widget w, XtPointer client_data, XtPointer call_data)
{
    if (strings_frame) {
	string_font = GetOptionChoice(strings_font_item);
	string_color = GetChoice(strings_pen_item);
	string_size = GetCharSizeChoice(strings_size_item);
	string_rot = GetAngleChoice(strings_rot_item);
	string_loctype = GetChoice(strings_loc_item) ? COORD_VIEW : COORD_WORLD;
	string_just = GetChoice(strings_just_item);
    }
}

void define_objects_popup(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget wbut;
    Widget panel, rc;
    set_wait_cursor();
    if (objects_frame == NULL) {
	objects_frame = XmCreateDialogShell(app_shell, "Objects", NULL, 0);
	handle_close(objects_frame);
	panel = XmCreateRowColumn(objects_frame, "ticks_rc", NULL, 0);
        XtVaSetValues(panel, XmNorientation, XmHORIZONTAL, NULL);

        rc = XmCreateRowColumn(panel, "rc", NULL, 0);

	wbut = XtVaCreateManagedWidget("Text", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) STR_LOC);

	wbut = XtVaCreateManagedWidget("Text at angle", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) STR_LOC1ST);

	wbut = XtVaCreateManagedWidget("Edit Text", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) STR_EDIT);

	wbut = XtVaCreateManagedWidget("Text props...", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) define_strings_popup, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Line", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) MAKE_LINE_1ST);

	wbut = XtVaCreateManagedWidget("Line props...", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) define_lines_popup, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Box", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) MAKE_BOX_1ST);

	wbut = XtVaCreateManagedWidget("Box props...", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) define_boxes_popup, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Ellipse", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) MAKE_ELLIP_1ST);

	wbut = XtVaCreateManagedWidget("Ellipse props...", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) define_ellip_popup, (XtPointer) NULL);
	XtManageChild(rc);

        rc = XmCreateRowColumn(panel, "rc", NULL, 0);

	wbut = XtVaCreateManagedWidget("Edit object", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) EDIT_OBJECT);

	wbut = XtVaCreateManagedWidget("Move object", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) MOVE_OBJECT_1ST);

	wbut = XtVaCreateManagedWidget("Copy object", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) COPY_OBJECT1ST);

	wbut = XtVaCreateManagedWidget("Delete object", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) set_actioncb, (XtPointer) DEL_OBJECT);

	wbut = XtVaCreateManagedWidget("Clear all text", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) do_clear_text, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Clear all lines", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) do_clear_lines, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Clear all boxes", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) do_clear_boxes, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Clear all ellip", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) do_clear_ellipses, (XtPointer) NULL);

	wbut = XtVaCreateManagedWidget("Close", xmPushButtonWidgetClass, rc,
				       NULL);
	XtAddCallback(wbut, XmNactivateCallback, (XtCallbackProc) destroy_dialog, (XtPointer) objects_frame);
	XtManageChild(rc);
	XtManageChild(panel);
    }
    XtRaise(objects_frame);
    unset_wait_cursor();
}

void define_ellip_popup(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget rc;
    Widget buts[2];
    Widget panel;

    set_wait_cursor();
    if (ellip_frame == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	ellip_frame = XmCreateDialogShell(app_shell, "Ellipses", NULL, 0);
	handle_close(ellip_frame);
	panel = XmCreateRowColumn(ellip_frame, "ellip_rc", NULL, 0);

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	ellip_pen_item = CreateColorChoice(rc, "Color: ");

	ellip_linew_item = CreateLineWidthChoice(rc, "Line width:");

	ellip_lines_item = CreateLineStyleChoice(rc, "Line style:");

	ellip_fillpat_item = CreatePatternChoice(rc, "Fill pattern:");
	ellip_fillcol_item = CreateColorChoice(rc, "Fill color: ");
	XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	ellip_loc_item = CreatePanelChoice(rc, " ",
					   3,
					   "World coordinates",
					   "Viewport coordinates",
					   0,
					   0);
	XtManageChild(rc);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
		      (XtCallbackProc) ellip_def_proc, (XtPointer) 0);
	XtAddCallback(buts[1], XmNactivateCallback,
		  (XtCallbackProc) destroy_dialog, (XtPointer) ellip_frame);

	XtManageChild(panel);
    }
    XtRaise(ellip_frame);
    update_ellip();
    unset_wait_cursor();
}

void define_strings_popup(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget rc;
    Widget buts[2];
    Widget panel;

    set_wait_cursor();
    if (strings_frame == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	strings_frame = XmCreateDialogShell(app_shell, "Strings", NULL, 0);
	handle_close(strings_frame);
	panel = XmCreateRowColumn(strings_frame, "strings_rc", NULL, 0);

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	strings_font_item = CreateFontChoice(rc, "Font:");

	strings_pen_item = CreateColorChoice(rc, "Color: ");

        XtVaCreateManagedWidget("Justification:", xmLabelWidgetClass, rc, NULL);
	strings_just_item = CreatePanelChoice(rc, " ",
					      4,
					      "Left",
					      "Right",
					      "Centered",
					      0,
					      0);

        XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	strings_loc_item = CreatePanelChoice(rc, " ",
					     3,
					     "World coordinates",
					     "Viewport coordinates",
					     0,
					     0);
	XtManageChild(rc);

	strings_rot_item = CreateAngleChoice(panel, "Rotation:");

	strings_size_item = CreateCharSizeChoice(panel, "Size:");

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
		    (XtCallbackProc) define_string_defaults, (XtPointer) 0);
	XtAddCallback(buts[1], XmNactivateCallback,
		(XtCallbackProc) destroy_dialog, (XtPointer) strings_frame);

	XtManageChild(panel);
    }
    XtRaise(strings_frame);
    updatestrings();
    unset_wait_cursor();
}

void define_lines_popup(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget rc;
    Widget buts[2];
    Widget panel;

    set_wait_cursor();
    if (lines_frame == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	lines_frame = XmCreateDialogShell(app_shell, "Lines", NULL, 0);
	handle_close(lines_frame);
	panel = XmCreateRowColumn(lines_frame, "lines_rc", NULL, 0);

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	lines_pen_item = CreateColorChoice(rc, "Color: ");

	XtVaCreateManagedWidget("Width:", xmLabelWidgetClass, rc, NULL);
	lines_width_item = (Widget *) CreatePanelChoice(rc, " ",
							10,
				"1", "2", "3", "4", "5", "6", "7", "8", "9",
							NULL,
							NULL);

	lines_style_item = CreateLineStyleChoice(rc, "Style:");

	XtVaCreateManagedWidget("Arrow:", xmLabelWidgetClass, rc, NULL);
	lines_arrow_item = CreatePanelChoice(rc, " ",
					     5,
					     "None",
					     "At start",
					     "At end",
					     "Both ends",
					     0,
					     0);

	XtVaCreateManagedWidget("Arrow type:", xmLabelWidgetClass, rc, NULL);
	lines_atype_item = CreatePanelChoice(rc, " ",
					     4,
					     "Line",
					     "Filled",
					     "Hollow",
					     0,
					     0);

	XtVaCreateManagedWidget("Arrow head size:", xmLabelWidgetClass, rc, NULL);
	lines_asize_item = XtVaCreateManagedWidget("arrowsize", xmScaleWidgetClass, rc,
						   XmNminimum, 0,
						   XmNmaximum, 400,
						   XmNvalue, 100,
						   XmNshowValue, True,
				     XmNprocessingDirection, XmMAX_ON_RIGHT,
					       XmNorientation, XmHORIZONTAL,
						   NULL);

	XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	lines_loc_item = CreatePanelChoice(rc, " ",
					   3,
					   "World coordinates",
					   "Viewport coordinates",
					   0,
					   0);
	XtManageChild(rc);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
		      (XtCallbackProc) lines_def_proc, (XtPointer) 0);
	XtAddCallback(buts[1], XmNactivateCallback,
		  (XtCallbackProc) destroy_dialog, (XtPointer) lines_frame);

	XtManageChild(panel);
    }
    update_lines();
    XtRaise(lines_frame);
    unset_wait_cursor();
}

void define_boxes_popup(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget rc;
    Widget buts[2];
    Widget panel;

    set_wait_cursor();
    if (boxes_frame == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	boxes_frame = XmCreateDialogShell(app_shell, "Boxes", NULL, 0);
	handle_close(boxes_frame);
	panel = XmCreateRowColumn(boxes_frame, "boxes_rc", NULL, 0);

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	boxes_pen_item = CreateColorChoice(rc, "Color: ");

	boxes_linew_item = CreateLineWidthChoice(rc, "Line width:");

	boxes_lines_item = CreateLineStyleChoice(rc, "Line style:");

	boxes_fillpat_item = CreatePatternChoice(rc, "Fill pattern:");
	boxes_fillcol_item = CreateColorChoice(rc, "Fill color: ");
	XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	boxes_loc_item = CreatePanelChoice(rc, " ",
					   3,
					   "World coordinates",
					   "Viewport coordinates",
					   0,
					   0);
	XtManageChild(rc);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
		      (XtCallbackProc) boxes_def_proc, (XtPointer) 0);
	XtAddCallback(buts[1], XmNactivateCallback,
		  (XtCallbackProc) destroy_dialog, (XtPointer) boxes_frame);

	XtManageChild(panel);
    }
    XtRaise(boxes_frame);
    update_boxes();
    unset_wait_cursor();
}

typedef struct {
    Widget top;
    Widget *color_item;
    Widget *linew_item;
    Widget *lines_item;
    Widget *fill_color_item;
    Widget *fill_pattern_item;
    Widget *loc_item;
    Widget x1_item;
    Widget x2_item;
    Widget y1_item;
    Widget y2_item;
    int boxno;
} EditBoxUI;

static EditBoxUI box_ui,ellip_ui;

void update_box_edit(EditBoxUI *ui)
{
    if (ui->top) {
	int boxno = ui->boxno;
	SetChoice(ui->color_item, boxes[boxno].color);
	SetChoice(ui->lines_item, boxes[boxno].lines);
	SetChoice(ui->linew_item, boxes[boxno].linew);
	SetChoice(ui->fill_pattern_item, boxes[boxno].fillpattern);
	SetChoice(ui->fill_color_item, boxes[boxno].fillcolor);
	SetChoice(ui->loc_item, boxes[boxno].loctype == COORD_VIEW ? 1 : 0);
	sprintf(buf, "%.12f", boxes[boxno].x1);
	xv_setstr(ui->x1_item, buf);
	sprintf(buf, "%.12f", boxes[boxno].x2);
	xv_setstr(ui->x2_item, buf);
	sprintf(buf, "%.12f", boxes[boxno].y1);
	xv_setstr(ui->y1_item, buf);
	sprintf(buf, "%.12f", boxes[boxno].y2);
	xv_setstr(ui->y2_item, buf);
    }
}

void update_ellipse_edit(EditBoxUI *ui)
{
    if (ui->top) {
	int ellipno = ui->boxno;
	SetChoice(ui->color_item, ellip[ellipno].color);
	SetChoice(ui->lines_item, ellip[ellipno].lines);
	SetChoice(ui->linew_item, ellip[ellipno].linew);
	SetChoice(ui->fill_pattern_item, ellip[ellipno].fillpattern);
	SetChoice(ui->fill_color_item, ellip[ellipno].fillcolor);
	SetChoice(ui->loc_item, ellip[ellipno].loctype == COORD_VIEW ? 1 : 0);
	sprintf(buf, "%.12f", 0.5*(ellip[ellipno].x1+ellip[ellipno].x2));
	xv_setstr(ui->x1_item, buf);
	sprintf(buf, "%.12f", 0.5*(ellip[ellipno].y1+ellip[ellipno].y2));
	xv_setstr(ui->y1_item, buf);
	sprintf(buf, "%.12f", fabs(ellip[ellipno].x1-ellip[ellipno].x2) );
	xv_setstr(ui->x2_item, buf);
	sprintf(buf, "%.12f", fabs(ellip[ellipno].y1-ellip[ellipno].y2) );
	xv_setstr(ui->y2_item, buf);
    }
}


void swap_ellipwv_coords(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditBoxUI *ui = (EditBoxUI *)client_data;
    int ellipno = ui->boxno;
    double x1, x2, y1, y2;
	
    if(((int)GetChoice(ui->loc_item)?COORD_VIEW : COORD_WORLD) == ellip[ellipno].loctype) {
        return;
    }
               
    xv_evalexpr(ui->x1_item, &x1);
    xv_evalexpr(ui->x2_item, &x2);
    xv_evalexpr(ui->y1_item, &y1);
    xv_evalexpr(ui->y2_item, &y2);
	
    if( ellip[ellipno].loctype == COORD_VIEW ) {
    ellip[ellipno].gno = cg;
	ellip[ellipno].loctype = COORD_WORLD;
	view2world( x1-x2/2., y1-y2/2., &ellip[ellipno].x1,&ellip[ellipno].y1 );
	view2world( x1+x2/2., y1+y2/2., &ellip[ellipno].x2,&ellip[ellipno].y2 );
    } else {
	ellip[ellipno].loctype = COORD_VIEW;
	world2view( x1-x2/2., y1-y2/2., &ellip[ellipno].x1,&ellip[ellipno].y1 );
	world2view( x1+x2/2., y1+y2/2., &ellip[ellipno].x2,&ellip[ellipno].y2 );
    }
    update_ellipse_edit( ui );
}


void ellipse_edit_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditBoxUI *ui = (EditBoxUI *) client_data;
    int ellipno = ui->boxno;
    double a, b, c, d;

    if (xv_evalexpr( ui->x1_item, &a ) != GRACE_EXIT_SUCCESS ||
        xv_evalexpr( ui->x2_item, &b ) != GRACE_EXIT_SUCCESS ||
        xv_evalexpr( ui->y1_item, &c ) != GRACE_EXIT_SUCCESS ||
        xv_evalexpr( ui->y2_item, &d ) != GRACE_EXIT_SUCCESS ) {
        return;
    }
    ellip[ellipno].color = GetChoice(ui->color_item);
    ellip[ellipno].loctype = GetChoice(ui->loc_item) ? COORD_VIEW : COORD_WORLD;
    ellip[ellipno].lines = GetChoice(ui->lines_item);
    ellip[ellipno].linew = GetChoice(ui->linew_item);
    ellip[ellipno].fillcolor = GetChoice(ui->fill_color_item);
    ellip[ellipno].fillpattern = GetChoice(ui->fill_pattern_item);
    ellip[ellipno].x1 = a - b/2.;
    ellip[ellipno].x2 = a + b/2.;
    ellip[ellipno].y1 = c - d/2.;
    ellip[ellipno].y2 = c + d/2.;
    
    drawgraph();
}


void swap_boxwv_coords(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditBoxUI *ui = (EditBoxUI *)client_data;
    int boxno = ui->boxno;

    if( ((int)GetChoice(ui->loc_item)?COORD_VIEW:COORD_WORLD) == boxes[boxno].loctype ) {
        return;
    }
	
    if( boxes[boxno].loctype == COORD_VIEW ) {
    boxes[boxno].gno = cg;
	boxes[boxno].loctype = COORD_WORLD;
	view2world( atof(xv_getstr(ui->x1_item)), atof(xv_getstr(ui->y1_item)),
				&boxes[boxno].x1,&boxes[boxno].y1 );
	view2world( atof(xv_getstr(ui->x2_item)), atof(xv_getstr(ui->y2_item)),
				&boxes[boxno].x2,&boxes[boxno].y2 );
    } else {
	boxes[boxno].loctype = COORD_VIEW;
	world2view( atof(xv_getstr(ui->x1_item)), atof(xv_getstr(ui->y1_item)),
				&boxes[boxno].x1,&boxes[boxno].y1 );
	world2view( atof(xv_getstr(ui->x2_item)), atof(xv_getstr(ui->y2_item)),
				&boxes[boxno].x2,&boxes[boxno].y2 );
    }
    update_box_edit( ui );
}


void box_edit_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditBoxUI *ui = (EditBoxUI *) client_data;
    int boxno = ui->boxno;

    boxes[boxno].color = GetChoice(ui->color_item);
    boxes[boxno].loctype = GetChoice(ui->loc_item) ? COORD_VIEW : COORD_WORLD;
    boxes[boxno].lines = GetChoice(ui->lines_item);
    boxes[boxno].linew = GetChoice(ui->linew_item);
    boxes[boxno].fillcolor = GetChoice(ui->fill_color_item);
    boxes[boxno].fillpattern = GetChoice(ui->fill_pattern_item);
    xv_evalexpr( ui->x1_item, &boxes[boxno].x1 );
    xv_evalexpr( ui->x2_item, &boxes[boxno].x2 );
    xv_evalexpr( ui->y1_item, &boxes[boxno].y1 );
    xv_evalexpr( ui->y2_item, &boxes[boxno].y2 );
    
    drawgraph();
}

void box_edit_popup(int boxno)
{
    Widget rc;
    Widget panel;
    Widget buts[2];

    set_wait_cursor();
    if (box_ui.top == NULL) {
	char *label1[3];
	label1[0] = "Accept";
	label1[1] = "Close";
	box_ui.top = XmCreateDialogShell(app_shell, "Edit box", NULL, 0);

	handle_close(box_ui.top);
	panel = XmCreateRowColumn(box_ui.top, "boxes_rc", NULL, 0);

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	box_ui.color_item = CreateColorChoice(rc, "Color: ");

	box_ui.linew_item = CreateLineWidthChoice(rc, "Line width:");

	box_ui.lines_item = CreateLineStyleChoice(rc, "Line style:");

	XtVaCreateManagedWidget("Fill:", xmLabelWidgetClass, rc, NULL);

	box_ui.fill_pattern_item = CreatePatternChoice(rc, "Fill pattern:");
	box_ui.fill_color_item = CreateColorChoice(rc, "Fill color: ");
	XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	box_ui.loc_item = CreatePanelChoice(rc, " ",
					   3,
					   "World coordinates",
					   "Viewport coordinates",
					   0,
					   0);
                                          
        XtAddCallback(box_ui.loc_item[2], XmNactivateCallback, 
               (XtCallbackProc) swap_boxwv_coords, (XtPointer) &box_ui);
        XtAddCallback(box_ui.loc_item[3], XmNactivateCallback, 
               (XtCallbackProc) swap_boxwv_coords, (XtPointer) &box_ui);


	box_ui.x1_item = CreateTextItem4(rc, 8, "Xmin = ");
	box_ui.y1_item = CreateTextItem4(rc, 8, "Ymin = ");
	box_ui.x2_item = CreateTextItem4(rc, 8, "Xmax = ");
	box_ui.y2_item = CreateTextItem4(rc, 8, "Ymax = ");
	XtManageChild(rc);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
	  	(XtCallbackProc) box_edit_proc, (XtPointer) &box_ui);
	XtAddCallback(buts[1], XmNactivateCallback,
		(XtCallbackProc) destroy_dialog, (XtPointer) box_ui.top);
	XtManageChild(panel);
    }
    box_ui.boxno = boxno;
    update_box_edit(&box_ui);
   	XtRaise(box_ui.top);
    unset_wait_cursor();
}

void ellipse_edit_popup(int boxno)
{
/*    static EditBoxUI ui;*/
    Widget rc;
    Widget panel;
    Widget buts[2];

    set_wait_cursor();
    if (ellip_ui.top == NULL) {
		char *label1[2];
		label1[0] = "Accept";
		label1[1] = "Close";
		ellip_ui.top = XmCreateDialogShell(app_shell, "Edit ellipse", NULL, 0);
		handle_close(ellip_ui.top);
		panel = XmCreateRowColumn(ellip_ui.top, "ellipses_rc", NULL, 0);

    	 rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	ellip_ui.color_item = CreateColorChoice(rc, "Color: ");

	ellip_ui.linew_item = CreateLineWidthChoice(rc, "Line width:");

	ellip_ui.lines_item = CreateLineStyleChoice(rc, "Line style:");

	ellip_ui.fill_pattern_item = CreatePatternChoice(rc, "Fill pattern:");
	ellip_ui.fill_color_item = CreateColorChoice(rc, "Fill color: ");
	XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	ellip_ui.loc_item = CreatePanelChoice(rc, " ",
					   3,
					   "World coordinates",
					   "Viewport coordinates",
					   0,
					   0);
        XtAddCallback(ellip_ui.loc_item[2], XmNactivateCallback, 
               (XtCallbackProc) swap_ellipwv_coords, (XtPointer) &ellip_ui );
        XtAddCallback(ellip_ui.loc_item[3], XmNactivateCallback, 
               (XtCallbackProc) swap_ellipwv_coords, (XtPointer) &ellip_ui );


	ellip_ui.x1_item = CreateTextItem4(rc, 8, "Xcentre = ");
	ellip_ui.y1_item = CreateTextItem4(rc, 8, "Ycentre = ");
	ellip_ui.x2_item = CreateTextItem4(rc, 8, "Width = ");
	ellip_ui.y2_item = CreateTextItem4(rc, 8, "Height = ");
    	XtManageChild(rc);

    	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

    	ellip_ui.boxno = boxno;
    	CreateCommandButtons(panel, 2, buts, label1);
    	XtAddCallback(buts[0], XmNactivateCallback,
		    	  (XtCallbackProc) ellipse_edit_proc, (XtPointer) &ellip_ui);
    	XtAddCallback(buts[1], XmNactivateCallback,
			  (XtCallbackProc) destroy_dialog, (XtPointer) ellip_ui.top);
    	XtManageChild(panel);
    }
    ellip_ui.boxno = boxno;
    update_ellipse_edit(&ellip_ui);
    XtRaise(ellip_ui.top);
    unset_wait_cursor();
}

typedef struct {
    Widget top;
    Widget *color_item;
    Widget *linew_item;
    Widget *lines_item;
    Widget *loc_item;
    Widget *arrow_item;
    Widget *atype_item;
    Widget asize_item;
    Widget x1_item;
    Widget y1_item;
    Widget x2_item;
    Widget y2_item;
    int lineno;
} EditLineUI;

void update_line_edit(EditLineUI *ui)
{
    int iv;
    if (ui->top) {
	int lineno = ui->lineno;
	SetChoice(ui->color_item, lines[lineno].color);
	SetChoice(ui->lines_item, lines[lineno].lines);
	SetChoice(ui->linew_item, lines[lineno].linew);
	SetChoice(ui->arrow_item, lines[lineno].arrow);
	SetChoice(ui->atype_item, lines[lineno].atype);
	iv = (int) (50 * lines[lineno].asize);
	XtVaSetValues(ui->asize_item, XmNvalue, iv, NULL);
	SetChoice(ui->loc_item, lines[lineno].loctype == COORD_VIEW ? 1 : 0);
	sprintf(buf, "%.12f", lines[lineno].x1);
	xv_setstr(ui->x1_item, buf);
	sprintf(buf, "%.12f", lines[lineno].y1);
	xv_setstr(ui->y1_item, buf);
	sprintf(buf, "%.12f", lines[lineno].x2);
	xv_setstr(ui->x2_item, buf);
	sprintf(buf, "%.12f", lines[lineno].y2);
	xv_setstr(ui->y2_item, buf);
    }
}

void swap_linewv_coords(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditLineUI *ui = (EditLineUI *)client_data;
    int lineno = ui->lineno;
	
    if( lines[lineno].loctype == ((int) GetChoice(ui->loc_item)?COORD_VIEW:COORD_WORLD) ) {
        return;
    }
	   
	if( lines[lineno].loctype == COORD_VIEW ) {
		lines[lineno].gno = cg;
		lines[lineno].loctype = COORD_WORLD;
		view2world( atof(xv_getstr(ui->x1_item)), atof(xv_getstr(ui->y1_item)),
										&lines[lineno].x1,&lines[lineno].y1 );
		view2world( atof(xv_getstr(ui->x2_item)), atof(xv_getstr(ui->y2_item)),
										&lines[lineno].x2,&lines[lineno].y2 );
	} else {
		lines[lineno].loctype = COORD_VIEW;
		world2view( atof(xv_getstr(ui->x1_item)), atof(xv_getstr(ui->y1_item)),
										&lines[lineno].x1,&lines[lineno].y1 );
		world2view( atof(xv_getstr(ui->x2_item)), atof(xv_getstr(ui->y2_item)),
										&lines[lineno].x2,&lines[lineno].y2 );
	}
	update_line_edit( ui );
}


void line_edit_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditLineUI *ui = (EditLineUI *) client_data;
    int value;
    int lineno = ui->lineno;

    lines[lineno].color = GetChoice(ui->color_item);
    lines[lineno].loctype = GetChoice(ui->loc_item) ? COORD_VIEW : COORD_WORLD;
    lines[lineno].lines = GetChoice(ui->lines_item);
    lines[lineno].linew = GetChoice(ui->linew_item);
    xv_evalexpr(ui->x1_item, &lines[lineno].x1);
    xv_evalexpr(ui->y1_item, &lines[lineno].y1);
    xv_evalexpr(ui->x2_item, &lines[lineno].x2);
    xv_evalexpr(ui->y2_item, &lines[lineno].y2);
    XtVaGetValues(ui->asize_item, XmNvalue, &value, NULL);
    lines[lineno].asize = value / 50.0;
    lines[lineno].arrow = GetChoice(ui->arrow_item);
    lines[lineno].atype = GetChoice(ui->atype_item);
    
    drawgraph();
}

static EditLineUI line_ui;

void line_edit_popup(int lineno)
{
    Widget rc;
    Widget panel;
    Widget buts[2];

    set_wait_cursor();
    if (line_ui.top == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	line_ui.top = XmCreateDialogShell(app_shell, "Edit Line", NULL, 0);
	handle_close(line_ui.top);
	panel = XmCreateRowColumn(line_ui.top, "lines_rc", NULL, 0);

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	line_ui.color_item = CreateColorChoice(rc, "Color: ");

	line_ui.linew_item = CreateLineWidthChoice(rc, "Line width:");

	line_ui.lines_item = CreateLineStyleChoice(rc, "Line style:");
	XtVaCreateManagedWidget("Arrow:", xmLabelWidgetClass, rc, NULL);
	line_ui.arrow_item = CreatePanelChoice(rc, " ",
					     5,
					     "None",
					     "At start",
					     "At end",
					     "Both ends",
					     0,
					     0);

	XtVaCreateManagedWidget("Arrow type:", xmLabelWidgetClass, rc, NULL);
	line_ui.atype_item = CreatePanelChoice(rc, " ",
					     4,
					     "Line",
					     "Filled",
					     "Hollow",
					     0,
					     0);

	XtVaCreateManagedWidget("Arrow head size:", xmLabelWidgetClass, rc, NULL);
	line_ui.asize_item = XtVaCreateManagedWidget("arrowsize", xmScaleWidgetClass, rc,
						   XmNminimum, 0,
						   XmNmaximum, 400,
						   XmNvalue, 100,
						   XmNshowValue, True,
				     XmNprocessingDirection, XmMAX_ON_RIGHT,
					       XmNorientation, XmHORIZONTAL,
						   NULL);
	XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
	line_ui.loc_item = CreatePanelChoice(rc, " ",
					   3,
					   "World coordinates",
					   "Viewport coordinates",
					   0,
					   0);
        XtAddCallback(line_ui.loc_item[2], XmNactivateCallback,
                   (XtCallbackProc) swap_linewv_coords, (XtPointer) &line_ui);
        XtAddCallback(line_ui.loc_item[3], XmNactivateCallback,
                   (XtCallbackProc) swap_linewv_coords, (XtPointer) &line_ui);


	line_ui.x1_item = CreateTextItem4(rc, 8, "X1 = ");
	line_ui.y1_item = CreateTextItem4(rc, 8, "Y1 = ");
	line_ui.x2_item = CreateTextItem4(rc, 8, "X2 = ");
	line_ui.y2_item = CreateTextItem4(rc, 8, "Y2 = ");
	XtManageChild(rc);

	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
		(XtCallbackProc) line_edit_proc, (XtPointer) &line_ui);
	XtAddCallback(buts[1], XmNactivateCallback,
		(XtCallbackProc) destroy_dialog, (XtPointer) line_ui.top);

	XtManageChild(panel);
    }
    XtRaise(line_ui.top);
    line_ui.lineno = lineno;
    update_line_edit(&line_ui);
    unset_wait_cursor();
}

typedef struct {
    Widget top;
    Widget string_item;
    Widget *color_item;
    Widget *loc_item;
    OptionStructure font_item;
    Widget size_item;
    Widget rot_item;
    Widget *just_item;
    Widget x1_item;
    Widget y1_item;
    int stringno;
} EditStringUI;


void update_string_edit(EditStringUI *ui)
{
    if (ui->top) {
		plotstr *pstring = &pstr[ui->stringno];
		xv_setstr(ui->string_item, pstring->s);
		SetChoice(ui->color_item, pstring->color);
		SetChoice(ui->just_item, pstring->just);
		SetOptionChoice(ui->font_item, pstring->font );
                SetCharSizeChoice(ui->size_item, pstring->charsize);
                SetAngleChoice(ui->rot_item, pstring->rot);
		sprintf(buf, "%.12f", pstring->x);
		xv_setstr(ui->x1_item, buf);
		sprintf(buf, "%.12f", pstring->y);
		xv_setstr(ui->y1_item, buf);
    }
}

void swap_stringwv_coords(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditStringUI *ui = (EditStringUI *)client_data;
    int stringno = ui->stringno;
	
    if( pstr[stringno].loctype == ((int)GetChoice(ui->loc_item)?COORD_VIEW:COORD_WORLD) ) {
        return;
    }
	   
	if( pstr[stringno].loctype == COORD_VIEW ) {
		pstr[stringno].gno = cg;
		pstr[stringno].loctype = COORD_WORLD;
		view2world( atof(xv_getstr(ui->x1_item)), atof(xv_getstr(ui->y1_item)),
								&pstr[stringno].x,&pstr[stringno].y );
	} else {
		pstr[stringno].loctype = COORD_VIEW;
		world2view( atof(xv_getstr(ui->x1_item)), atof(xv_getstr(ui->y1_item)),
								&pstr[stringno].x,&pstr[stringno].y );
	}
	update_string_edit( ui );
}


void string_edit_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    EditStringUI *ui = (EditStringUI *) client_data;
    int stringno = ui->stringno;

    pstr[stringno].s = copy_string(pstr[stringno].s, xv_getstr(ui->string_item));
    pstr[stringno].color = GetChoice(ui->color_item);
    pstr[stringno].loctype = GetChoice(ui->loc_item) ? COORD_VIEW : COORD_WORLD;
    pstr[stringno].font = GetOptionChoice(ui->font_item);
    pstr[stringno].just = GetChoice(ui->just_item);
    xv_evalexpr(ui->x1_item, &pstr[stringno].x);
    xv_evalexpr(ui->y1_item, &pstr[stringno].y);
    pstr[stringno].charsize = GetCharSizeChoice(ui->size_item);
    pstr[stringno].rot = GetAngleChoice(ui->rot_item);
    
    drawgraph();
}

static EditStringUI string_ui;

void string_edit_popup(int stringno)
{
    Widget rc;
    Widget buts[2];
    Widget panel;

    set_wait_cursor();
    if (string_ui.top == NULL) {
	char *label1[2];
	label1[0] = "Accept";
	label1[1] = "Close";
	string_ui.top = XmCreateDialogShell(app_shell, "Edit String", NULL, 0);
	handle_close(string_ui.top);
	panel = XmCreateRowColumn(string_ui.top, "strings_rc", NULL, 0);

        XtVaCreateManagedWidget("String:", xmLabelWidgetClass, panel, NULL);
	string_ui.string_item = CreateTextItem2(panel, 40, "");

        rc = XtVaCreateWidget("rc", xmRowColumnWidgetClass, panel, NULL);

	string_ui.color_item = CreateColorChoice(rc, "Color: ");
	
	
	string_ui.font_item = CreateFontChoice(rc, "Font:");

        XtVaCreateManagedWidget("Justification:", xmLabelWidgetClass, rc, NULL);
		string_ui.just_item = CreatePanelChoice(rc, " ",
					      4,
					      "Left",
					      "Right",
					      "Centered",
					      0,
					      0);

        XtVaCreateManagedWidget("Position in:", xmLabelWidgetClass, rc, NULL);
		string_ui.loc_item = CreatePanelChoice(rc, " ",
					     3,
					     "World coordinates",
					     "Viewport coordinates",
					     0,
					     0);
        XtAddCallback(string_ui.loc_item[2], XmNactivateCallback,
                  (XtCallbackProc) swap_stringwv_coords, (XtPointer) &string_ui);
        XtAddCallback(string_ui.loc_item[3], XmNactivateCallback,
                  (XtCallbackProc) swap_stringwv_coords, (XtPointer) &string_ui);
					     
	string_ui.x1_item = CreateTextItem4(rc, 8, "X = ");
	string_ui.y1_item = CreateTextItem4(rc, 8, "Y = ");	

	XtManageChild(rc);

	string_ui.rot_item = CreateAngleChoice(panel, "Rotation:");

	string_ui.size_item = CreateCharSizeChoice(panel, "Size:");
						    
	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, panel, NULL);

	CreateCommandButtons(panel, 2, buts, label1);
	XtAddCallback(buts[0], XmNactivateCallback,
		    (XtCallbackProc) string_edit_proc, (XtPointer) &string_ui );
	XtAddCallback(buts[1], XmNactivateCallback,
		(XtCallbackProc) destroy_dialog, (XtPointer) string_ui.top);

	XtManageChild(panel);
    }
    XtRaise(string_ui.top);
    string_ui.stringno = stringno;
    update_string_edit(&string_ui);
    unset_wait_cursor();
}
