/* Repack.c 					*/

/* This is a repack engine to be run separately or called from 
CirclePack; based on "remote_shell.c". Incorporates Chuck Collins 
modifications to packing algorithms. Also used as testbed for
algorithm changes before incorporation in CirclePack.
Compile with 

	cc -o rp Repack.c cp_remote_library.c hyp_math.c eucl_math.c sph_math.c complex_math.c -g -lm 

Call standalone with

	"rp <packname> -a d -c m -r n"

where d = number of digits of accuracy desired, m = limit on total 
repack cycles, n = cycles between status reports; Results are stored
in file <packname>p

Call from CirclePack with 

	"exec rp p -a d -c m -r n", 

where p = pack number. All of d, m, and n are obtional. The routine 
computes only the new radii. */
 
/* ----- user specified variables set before compiling -------- */

#undef _KERNEL
#include <sys/time.h>
#include <sys/resource.h>
#define TOLER .0000000001		/* threshold for repack computations */
#define OKERR .00000001		/* Roundoff for zero */
#include "cp_remote.h"

struct p_data *packdata;

extern struct Pathlist *pathlist;	/* storage for paths */
extern int pathlength;

char msgbuf[4096];	/* utility buffers for strings. */
float toler=TOLER,okerr=OKERR;
float cycles=2000;		/* default number of repack cycles */
int localcycles=1000;		/* default interval between status reports */
int iterates=2;			/* parameter of current packing routines */
struct rusage RU;
FILE *fp;

/* -------- main routine ----------- */

main(argc,argv)
int argc;
char *argv[];
{
	int i,j,p=0,limit_flag=0,stop_flag=0,nextcount,digits,flag,incompat;
	int cmdmode=0,elapse_hr,elapse_min,elapse_sec,aimcount=0;
	float cycount=0.0,accum=0.0,worst=0.0,hold,curv;
	double e_time;
	long timein,elapsetime;
	char filename[256];
	extern void fillcurves();
	extern void h_anglesum_overlap();

		/* ---------- initialization --------------- */

	if ((packdata=(struct p_data *)
		malloc(sizeof(struct p_data)))==NULL)
	 {
		sprintf(msgbuf,"%s exited: insufficient memory.",argv[0]);
		emsg(msgbuf);
		exit(0);
	 }
	packdata->overlap_status=0;
	packdata->faces=(f_data *)NULL;	

		/* -------- interpret arguments ------------- */

	if (argc < 2)  /* not enough arguments */
	 {
		sprintf(msgbuf,"%s died. Need pack number or filename.",argv[0]);
		emsg(msgbuf);
		exit(0); 	
	 }
	if (strlen(argv[1])==1) p=atoi(argv[1]); 	
		/* pack no. of input data from parent */
	else	/* must be filename; read/write data in file */
	 {
		filename[0]='\0';
		strncpy(filename,argv[1],255);
		cmdmode=1;
	 }
	i=2;
	while (i< argc) /* get other command line info */
	 {
		if (!strcmp(argv[i],"-a"))
		 {
			if (sscanf(argv[++i],"%d",&digits) && digits>2)
			 {
				toler=0.1;
				for (j=1;j<digits;j++) toler *=0.1;
				if (okerr>(.01)*toler) okerr=(.01)*toler;
			 }
		 }
		else if (!strcmp(argv[i],"-c"))
		 {
			if (sscanf(argv[++i],"%lf",&cycles) ) 
			 { 
				if (localcycles>cycles) localcycles=cycles;
				limit_flag=1;
			 }
		 }
		else if (!strcmp(argv[i],"-r"))
			{if (sscanf(argv[++i],"%d",&localcycles) ) ;}
		i++;
	 }

   if (!cmdmode) /* running under CirclePack */
    {

		/* --- if called from CirclePack, lock packing there ---- */

	if (!cmdmode && !lock_set(p)) 
	 {
		sprintf(msgbuf,"rp: lock failed.");
		emsg(msgbuf);
	 }

		/* -------- read data phase ---------- */

	if ( (packdata->sizelimit=pack_read(packdata,p,2))<=0 
	   || (packdata->packK_ptr=
		(struct K_data *)malloc((packdata->sizelimit+1)
			*sizeof(struct K_data)))==NULL
	   || (packdata->packR_ptr=
		(struct R_data *)malloc((packdata->sizelimit+1)
			*sizeof(struct R_data)))==NULL
	   || !readpack(packdata,stdin,packdata->sizelimit+1) )
	 {
		sprintf(msgbuf,"%s: read from parent failed.",argv[0]);
		emsg(msgbuf);
		exit(1);
	 }
    }
   else /* read from file */
    {
	if ((fp=fopen(filename,"r"))!=NULL) 
	 {
		packdata->sizelimit=get_pack_size(packdata,fp);
		fclose(fp);
	 }
	if (!packdata->sizelimit || (fp=fopen(filename,"r"))==NULL
	   || (packdata->packK_ptr=
		(struct K_data *)malloc((packdata->sizelimit+1)
			*sizeof(struct K_data)))==NULL
	   || (packdata->packR_ptr=
		(struct R_data *)malloc((packdata->sizelimit+1)
			*sizeof(struct R_data)))==NULL
	   || !readpack(packdata,fp,packdata->sizelimit+1) )
	 {
		sprintf(msgbuf,"%s: read of file %s failed.",argv[0],filename);
		emsg(msgbuf);
		fclose(fp);
		exit(2);
	 }
	fclose(fp);
    }

	fillcurves(packdata);

/* ======================= manipulations =================== */

	for (i=1;i<=packdata->nodecount;i++) /* set bdry flags */
	 {
		if (packdata->packK_ptr[i].flower[0]==
		   packdata->packK_ptr[i].flower[packdata->packK_ptr[i].num])
			packdata->packK_ptr[i].bdry_flag=0;
		else packdata->packK_ptr[i].bdry_flag=1;
	 }
	getrusage(RUSAGE_SELF,&RU);
	timein=RU.ru_utime.tv_sec;
	while ( !stop_flag && (!limit_flag || cycount<cycles) )
	 {
		if (packdata->hes < 0) /* hyperbolic */
		 {
/*			if (cycount<40)
			   nextcount=old_h_riffle(packdata,localcycles);
			else nextcount=h_riffle(packdata,localcycles);
*/
			nextcount=h_riffle(packdata,localcycles);
			if (nextcount==0) stop_flag=1;
		 }
		else if (packdata->hes==0) /* euclidean */
		 {
			nextcount=e_riffle(packdata,localcycles);
			if (nextcount==0) stop_flag=1;
		 }
		else if (packdata->hes>OKERR) /* spherical, no routine */
		 {
			sprintf(msgbuf,"rp: no packing algorithm available in spherical geometry.");
			emsg(msgbuf);
			exit(3);
		 }
		cycount += (float) nextcount;
		if (nextcount < localcycles) break;
		sprintf(msgbuf,"%s status: count = %lf  ",argv[0],cycount);
		msg(msgbuf);
	 }
	getrusage(RUSAGE_SELF,&RU);
	elapsetime=RU.ru_utime.tv_sec-timein;	
	if (limit_flag && cycount >= cycles)
		sprintf(msgbuf,
		   "%s stopped due to specified cycle limits: count = %lf  ",
		   argv[0],cycount);
	else sprintf(msgbuf,"%s completed: count = %lf  ", argv[0],cycount);
	msg(msgbuf);
/* run time feedback */
	e_time=(double)elapsetime;
	elapse_hr=(int)floor((double)(e_time/3600.0));
	elapse_min=(int)floor((double)(e_time-elapse_hr*3660.0)/60.0);
	elapse_sec=(int)(e_time-elapse_hr*3600.0-elapse_min*60.0);
	sprintf(msgbuf,"Elapsed time:  %02d:%02d:%02d (hh:mm:ss). \n",
		elapse_hr,elapse_min,elapse_sec);
	msg(msgbuf);
/* error feedback */
	fillcurves(packdata);
	incompat=0;
	for (i=1;i<=packdata->nodecount;i++)
		if (packdata->packR_ptr[i].aim >= 0)
		 {
		   if (packdata->hes<-okerr)
		     {
		       h_anglesum_overlap(packdata,i,packdata->packR_ptr[i].rad,&curv,&flag);
		       if (flag) incompat++;
		       else
			 {
			   hold=fabs(curv-packdata->packR_ptr[i].aim);
			   accum += hold;
			   worst=(hold>worst) ? hold : worst;
			   aimcount++;
			 }
		     }
		   else
		     {
			hold=fabs(packdata->packR_ptr[i].curv-
				packdata->packR_ptr[i].aim);
			accum += hold;
			worst = (hold > worst) ? hold : worst;
 		 	aimcount++;
		     }
		 }
	sprintf(msgbuf,"Worst curv error, %d vertices: % .15e\n     Accumulated curv error:        % .15e\n",
		aimcount,worst,accum);
	msg(msgbuf);
	if (incompat)
	  {
	    sprintf(msgbuf,"%d vertices had incompatibilities.\n",incompat);
	    msg(msgbuf);
	  }

/* ===================== end of manipulations =============== */

   if (!cmdmode) /* send data to parent. */
    {

		/* -------- write data phase ---------------- */

	if (!pack_write(packdata,p,101)) /* send back only radii */
	 {
		sprintf(msgbuf,"rp: write to parent failed.");
		emsg(msgbuf);
	 }

		/* -------- unlock phase ---------------- */

	if (!unlock(p))
	 {
		sprintf(msgbuf,"rp: unlock failed.");
		emsg(msgbuf);
	 }
    }
   else /* store data in file */
    {
	strcat(filename,"p"); /* add extra p to filename */
	if ((fp=fopen(filename,"w"))==NULL || !writepack(packdata,fp,101))
	 {
		sprintf(msgbuf,"%s: failed to write data to %s.",argv[0],filename);
		emsg(msgbuf);
		fclose(fp);
		exit(4);
	 }
	sprintf(msgbuf,"%s results are stored in file %s.",argv[0],filename);
	msg(msgbuf);
	fclose(fp);
    }

/* -------- exit ------ */

	sprintf(msgbuf,"%s: finished successfully.",argv[0]);
	msg(msgbuf);
	exit(8);

} /* end of main */




/*=================== repacking routines =======================*/

int 
e_riffle(p,passes)   /* CRC - modified 5/28/97 */
/* uses the latest model, with super steps and safety checks */
/* adjust eucl radii to meet curvature targets in 'aim'.
A target less than zero means that vertex is free - no adjustments made to it.
Bdry radii adjusted only if their aim >= 0. */

int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,key,key0,N,flag;
	float r, r2, fbest, faim, del, bet;
        float ttoler, cut=100, fact=-1, cut0, fact0;
        float lmax, rat,rr,*R0,ftol=.05;
	struct R_data *pR_ptr;
        struct K_data *pK_ptr;
        extern void e_anglesum_overlap();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	R0 = (float *)malloc((p->nodecount+1)*sizeof(float));
	pR_ptr=p->packR_ptr;
        pK_ptr=p->packK_ptr;
	for (i=1;i<=p->nodecount;i++)
 		if (pR_ptr[i].aim>0)
		 {
			index[aimnum]=i;
			aimnum++;
		 }
	if (aimnum==0) 
         {free(index); free(R0);return count; }
        ttoler = 3*aimnum*toler; /* adjust tolerance */

while (cut >ttoler && count<passes)
 {
/* save values */
	cut0 = cut;key0 = key;fact0 = fact; 
	for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
/* do update using uniform label model */
	cut = 0;
	for (j=0;j<aimnum;j++) 
	 {
		i = index[j]; 		/* point to active node */
		faim = pR_ptr[i].aim;	/* get target sum */
		r = pR_ptr[i].rad;	/* get present label */
		e_anglesum_overlap(p,i,r,&fbest,&flag);  /* compute sum */
/* use the model to predict the next value */
		N = 2*pK_ptr[i].num;
		del = sin(faim/N);
		bet = sin(fbest/N);
		r2 = r*bet*(1-del)/(del*(1-bet));
/* store as new radius label */
		pR_ptr[i].rad = r2; 
		pR_ptr[i].curv = fbest;	/* store new angle sum */
		fbest -= faim;
		cut += fbest*fbest;	/* accum abs error */
	 }
	cut = sqrt(cut);
/* do super step? */
	key = 1;
	if (key0==1)
	 {
		fact = cut/cut0;
		if (fabs(fact-fact0)<ftol) fact = fact/(1-fact);
		lmax = 1000;
		for (j=0;j<aimnum;j++)	 /* find max step */
		 {
		   i = index[j];
		   r = pR_ptr[i].rad;
		   rat = r - R0[i];
		   if (rat<0)
			lmax = (lmax < (rr=(-r/rat))) ? lmax : rr;
		 }
		fact = (fact < 0.5*lmax) ? fact : 0.5*lmax; 
			/* compute new step */
		for (j=0;j<aimnum;j++)     /* interpolate to new radius */
		 {
		   i = index[j];
		   pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
		 }
		key = 0;
	 }

	count++;
 } /* end of while */

	free(index);
	free(R0);
	return count;
} /* e_riffle */

	/* --------- pre 1996 riffle routine (slow, but dependable) -- */
int
old_e_riffle(p,passes) /* (Could use some reworking.) */
int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,dummy,flag;
	float recip,accum=0,verr,err,cut;
	struct R_data *pR_ptr;
	extern float e_radcalc();
	extern void e_anglesum_overlap();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	pR_ptr=p->packR_ptr;
	for (i=1;i<=p->nodecount;i++)
	if (pR_ptr[i].aim>0)
	 {
		index[aimnum]=i;
		aimnum++;
		err=pR_ptr[i].curv-pR_ptr[i].aim;
		accum += (err<0) ? (-err) : err;
	 }
	if (aimnum==0) {free(index);return count;}
	recip=.333333/aimnum;
	cut=accum*recip;
	while (cut >toler && count<passes)
	 {
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			e_anglesum_overlap(p,i,pR_ptr[i].rad,
				&pR_ptr[i].curv,&flag);
			verr=pR_ptr[i].curv-pR_ptr[i].aim;
			if (fabs(verr)>cut)
			 {
			   pR_ptr[i].rad=e_radcalc(p,i,
				pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
			   count++;
			 }
		 }
		accum=0;
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
		cut=accum*recip;
	
	 } /* end of while */
	free(index);
	return count;
} /* old_e_riffle */

float
e_radcalc(p,i,r,aim,chgflag) /* returns best radius at vert i to get
anglesum aim, via binary search method.
First guess = r. */
int i,*chgflag;
float r,aim;
struct p_data *p;
{
	int n,flag;
	float bestcurv,lower,upper,upcurv,lowcurv,factor=0.5;
	extern void e_anglesum_overlap();

	e_anglesum_overlap(p,i,r,&bestcurv,&flag);
	if (bestcurv>(aim+okerr))
	 {
		upper=r/factor;
		e_anglesum_overlap(p,i,upper,&upcurv,&flag);
		(*chgflag)++;
		if (upcurv>aim) return upper;
	 }
	else if (bestcurv<(aim-okerr))
	 {
		lower=r*factor;
		e_anglesum_overlap(p,i,lower,&lowcurv,&flag);
		(*chgflag)++;
		if (lowcurv<aim) return lower;
	 }
	else return r;
	(*chgflag)++;
	for (n=1;n<=iterates;n++)
	 {
		if (bestcurv>(aim+okerr)) 
		 {
		   lower=r;
		   lowcurv=bestcurv;
	   	   r += (aim-bestcurv)*(upper-r)/(upcurv-bestcurv);
		 }
		else if (bestcurv<(aim-okerr))
		 {
		   upper=r;
		   upcurv=bestcurv;
		   r -= (bestcurv-aim)*(lower-r)/(lowcurv-bestcurv);
		 }
		else return r;
		e_anglesum_overlap(p,i,r,&bestcurv,&flag);
	 }
	return r;
} /* e_radcalc */

int
h_riffle(p,passes)   /* CRC - modified 5/28/97 */
/* uses the best model, with super steps and safety checks 
   also has the anglesum calculation inlined.
   NOTE: overlap routines could use work. */
/* adjust radii to meet curvature targets in 'aim'.
A target less than zero means that vertex is free - no adjustments made.
Bdry radii adjusted only if their aim >= 0. */
int passes;
struct p_data *p;
{
	int i,j,k,j1,j2,aimnum=0,*index,count=0,key=0,key0,N,flag;
	float r, r1,r2,r3,fbest,faim,del,bet;
        float ttoler,cut=100,fact=-1,cut0,fact0;
        float ftol=0.05,lmax,rat,twor,*R0;
        float m2,m3,sr,t1,t2,t3,tr,o1,o2,o3;
	struct R_data *pR_ptr;
        struct K_data *pK_ptr;
        extern float h_cos_overlap();
	extern int nghb();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
        R0 = (float *)malloc((p->nodecount+1)*sizeof(float));
	pR_ptr=p->packR_ptr;
	pK_ptr=p->packK_ptr;
   /* list radii to be adjusted, store squared radii */
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0 
		   && pR_ptr[i].aim<.001)
			pR_ptr[i].rad = (-.2);
		else if (pR_ptr[i].aim>0)
		 {
			if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001)
				pR_ptr[i].rad = .01;
			index[aimnum]=i;
			aimnum++;
		 }
		if (pR_ptr[i].rad>0)
		pR_ptr[i].rad = pR_ptr[i].rad*pR_ptr[i].rad;
	 }
	if (aimnum==0) 
	 {free(index);free(R0);return count;}
	ttoler = 3*sqrt((float)aimnum)*toler;		/* adjust tolerance */
	cut = ttoler+1;
/*	cut =3*aimnum*10;*/
while (cut >ttoler && count<passes)         /* Begin Main Loop */
 {
	cut0 = cut;key0 = key;fact0 = fact;	/* save prev values */
	for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
	cut = 0;				    
/* Begin Iteration Loop */
	for (j=0;j<aimnum;j++) 
	 {
	/* anglesum calc */
		fbest = 0;
		i = index[j];
		if ((r = pR_ptr[i].rad)<0) r = 0;
		sr = sqrt(r);
		N = pK_ptr[i].num;
		if (!p->overlap_status)
		 { 			/* no overlaps */
               		twor = 2*r; 
               		r2 = pR_ptr[pK_ptr[i].flower[0]].rad;
               		m2 = (r2>0) ? (1-r2)/(1-r*r2) : 1;
               		for (k=1;k<=N;k++)	/* loop through petals */
               		 {
               			r3 = pR_ptr[pK_ptr[i].flower[k]].rad;
                   		m3 = (r3>0) ? (1-r3)/(1-r*r3) : 1;
                   		fbest += acos(1-twor*m2*m3); /* angle calc */
                   		m2 = m3;
               		 } 
               	 }
		else /* with overlaps, old routine */
		 {
			j2 = pK_ptr[i].flower[0];
		 	((r2 = pR_ptr[j2].rad) > 0) ? r2=sqrt(r2) : 0;
			o2=pK_ptr[i].overlaps[0];
			for (k=1;k<=N;k++)
			 {
				r1=r2;
				o1=o2;
				j1=j2;
				j2=pK_ptr[i].flower[k];
				((r2 = pR_ptr[j2].rad) >0) ? r2=sqrt(r2) : 0;
				o2=pK_ptr[i].overlaps[k];
				o3=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
				fbest += 
				 acos(h_cos_overlap(sr,r1,r2,o3,o2,o1,&flag));
			 }
		 }
		faim = pR_ptr[i].aim;	/* get target sum */
	/* set up for model */
		N = 2*N;
		del = sin(faim/N);
		bet = sin(fbest/N);
		r2 = (bet-sr)/(bet*r-sr);	/* reference radius */
		if (r2>0) 			/* calc new label */
		 {
			t1 = 1 - r2;
			t2 = 2*del;
			t3 = t2/(sqrt(t1*t1+t2*t2*r2)+t1);
			r2 = t3*t3; 
		 }
		else
			r2 = del*del;		/* use lower limit */
		pR_ptr[i].rad = r2;		/* store new label */ 
		pR_ptr[i].curv = fbest;		/* store new anglesum */
		fbest = fbest-faim;
		cut += fbest*fbest;		/* accumulate error */
	 }
/* End Iteration Loop */

	cut = sqrt(cut);	
	key = 1;
	fact = cut/cut0;
	if (key0==1 && fact < 1.0)		/* try to extrapolate */
	 {
		cut *= fact;
		if (fabs(fact-fact0)<ftol)	/* use a super step */
			fact = fact/(1-fact);
		lmax = 1000;
		for (j=0;j<aimnum;j++)		/* find max allowable step*/
		 {
		   i = index[j];
		   r = pR_ptr[i].rad;
		   rat = r - R0[i];
		   if (rat>0)			    
			lmax = (lmax < (tr=((1-r)/rat))) ? lmax : tr;    
						/* to keep R<1 */
		   else if (rat < 0)
			lmax = (lmax < (tr= (-r/rat))) ? lmax : tr;      
						/* to keep R>0 */
		 }
/* compute new step */
		fact = (fact < 0.5*lmax) ? fact : 0.5*lmax;
/* interpolate new labels */						
		for (j=0;j<aimnum;j++)
		 {
		   i = index[j];
		   pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
		 }
		key = 0;
	 }
		
	count++;
 } /* end of main while loop */

	for (i=1;i<=p->nodecount;i++)               /* reset labels */
		if (pR_ptr[i].rad>0)
			pR_ptr[i].rad = sqrt(pR_ptr[i].rad);
	free(index);
	free(R0);
	return count;
} /* h_riffle */


	/* --------- pre 1996 riffle routine (slow, but dependable) -- */
int
old_h_riffle(p,passes) /* (Could use some reworking.) */
int passes;
struct p_data *p;
{
	int i,j,aimnum=0,*index,count=0,dummy,flag;
	float recip,accum,verr,err,cut;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern float h_radcalc();
	extern void h_anglesum_overlap();

	index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	accum=0;
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag &&
			  pR_ptr[i].aim>=0  && pR_ptr[i].aim<.001)
			pR_ptr[i].rad=(-.2);
		else if (pR_ptr[i].aim>0)
		 {
			if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001) 
				pR_ptr[i].rad=.01;
			index[aimnum]=i;
			aimnum++;
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
	 }
	if (aimnum==0) {free(index);return count;}
	recip=.333333/aimnum;
	cut=accum*recip;
	while (cut >toler && count<passes)
	 {
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			h_anglesum_overlap(p,i,pR_ptr[i].rad,
				&pR_ptr[i].curv,&flag);
			verr=pR_ptr[i].curv-pR_ptr[i].aim;
			if (fabs(verr)>cut)
			 {
			   pR_ptr[i].rad=h_radcalc(p,i,
				pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
			   count++;
			 }
		 }
		accum=0;
		for (j=0;j<aimnum;j++)
		 {
			i=index[j];
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
		cut=accum*recip;

	 } /* end of while */
	free(index);
	return count;
} /* old_h_riffle */


float
h_radcalc(p,i,s,aim,chgflag) /* returns best radius at vert i to get
anglesum aim, via secant method. s = first guess. */
int i,*chgflag;
float s,aim;
struct p_data *p;
{
	int n,j,flag;
	float bestcurv,lower,upper,upcurv,lowcurv,factor=0.5;
	extern void h_anglesum_overlap();

	h_anglesum_overlap(p,i,s,&bestcurv,&flag);

/* Modify for incompatibilities due to inv distances */
	if (flag) return s; /* current incompatibility, so don't change */

	if (bestcurv>(aim+okerr))
	 {
		lower=s*factor;
		h_anglesum_overlap(p,i,lower,&lowcurv,&flag);
		(*chgflag)++;
		if (!flag && lowcurv>aim) return lower;
		if (flag) /* binary search for lower */
		 {
		   j=1;
		   while (j<8)
		     {
		       j++;
		       lower=(s+lower)/2;
		       h_anglesum_overlap(p,i,lower,&lowcurv,&flag);
		       if (!flag && lowcurv>aim) return lower;
		       if (!flag) j=10;
		     }
		   if (j==8) return s; /* failed to find compatib lower */
		 }
	 }
	else if (bestcurv<(aim-okerr))
	 {
		upper=1-factor+s*factor;
		h_anglesum_overlap(p,i,upper,&upcurv,&flag);
		(*chgflag)++;
		if (!flag && upcurv<aim) return upper;
		if (flag) /* binary search for upper */
		 {
		   j=1;
		   while (j<8)
		     {
		       j++;
		       upper=(s+upper)/2;
		       h_anglesum_overlap(p,i,upper,&upcurv,&flag);
		       if (!flag && upcurv<aim) return upper;
		       if (!flag) j=10;
		     }
		   if (j==8) return s; /* failed to find compatib upper */
		 }
	 }
	else return s;
	(*chgflag)++;
	for (n=1;n<=iterates;n++)
	 {
		if (bestcurv>(aim+okerr)) 
		 {
		   upper=s;
		   upcurv=bestcurv;
		   s -= (bestcurv-aim)*(lower-s)/(lowcurv-bestcurv);
		 }
		else if (bestcurv<(aim-okerr))
		 {
		   lower=s;
		   lowcurv=bestcurv;
	   	   s += (aim-bestcurv)*(upper-s)/(upcurv-bestcurv);
		 }
		else return s;
		h_anglesum_overlap(p,i,s,&bestcurv,&flag);
	 }
	return s;
} /* h_radcalc */

