/*								-*- C++ -*-
 * $Id: GDI_font.cpp,v 1.4 1997-01-15 14:58:48+01 mho Exp $
 *
 * Purpose: wxWindows font handling
 *
 * Authors: Markus Holzem, Julian Smart, and Matthew Flatt
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "GDI_font.h"
#endif

#define  Uses_XLib
#define  Uses_wxFont
#define  Uses_wxHashTable
#include "wx.h"

#include <string.h>

//-----------------------------------------------------------------------------
// local data
//-----------------------------------------------------------------------------

static char *wx_font_family [] = {
    "wxDEFAULT", "wxDECORATIVE", "wxMODERN", "wxROMAN", "wxSCRIPT",
    "wxSWISS", "wxTELETYPE",
};
static char *wx_font_style [] = {
    "wxDEFAULT", "wxNORMAL", "wxSLANT", "wxITALIC",
};
static char *wx_font_weight [] = {
    "wxDEFAULT", "wxNORMAL", "wxBOLD", "wxLIGHT",
};

IMPLEMENT_DYNAMIC_CLASS(wxFont, wxObject)
IMPLEMENT_DYNAMIC_CLASS(wxFontList, wxList)

//-----------------------------------------------------------------------------
// internal representation
//-----------------------------------------------------------------------------

class wxFontRep : public wxObject {
DECLARE_DYNAMIC_CLASS(wxFontRep)
    wxFontRep(void) : scaled_xfonts(wxKEY_INTEGER), ref_count(1) {}
public:
    wxList  scaled_xfonts;
    int     point_size;
    int     family, style, weight;
    Bool    underlined;
    int     ref_count;
    int     font_id;
    char*   face_name;
#if WXDEBUG
    void Dump(ostream& str);
#endif
};

IMPLEMENT_ABSTRACT_CLASS(wxFontRep, wxObject);

#if WXDEBUG
void wxFontRep::Dump(ostream& str)
{
    str << "wxFontRep ref = " << ref_count;
    wxNode *node = scaled_xfonts.First();
    if (node) {
	str << " sizes:";
	do {
	    str << " " << node->key.integer;
	} while ( (node = node->Next()) );
    }
}
#endif // WXDEBUG

//-----------------------------------------------------------------------------
// wxFont create and destroy
//-----------------------------------------------------------------------------

wxFont::wxFont(void)
{
    __type = wxTYPE_FONT;

    rep = wxNEW wxFontRep;

    rep->family        = rep->style = rep->weight = wxDEFAULT;
    rep->underlined    = FALSE;
    rep->point_size    = 0;
    rep->font_id       = 0;
    rep->face_name     = NULL;

#if !WXGARBAGE_COLLECTION_ON
    wxTheFontList->Append(this);
#endif
}

wxFont::wxFont(wxFont& new_font)
{
    __type = wxTYPE_FONT;

    rep = new_font.rep;
    if (rep) rep->ref_count++;
}

wxFont::wxFont(int PointSize, int FontIdOrFamily, int Style, int Weight,
	       Bool Underlined, const char* Face)
{
    __type = wxTYPE_FONT;

    rep = wxNEW wxFontRep;

    if ( (rep->face_name = (Face) ? copystring(Face) : (char*)NULL) ) {
	rep->font_id = wxTheFontNameDirectory->FindOrCreateFontId(Face, FontIdOrFamily);
	rep->family  = wxTheFontNameDirectory->GetFamily(FontIdOrFamily);
    } else {
	rep->font_id = FontIdOrFamily;
	rep->family  = wxTheFontNameDirectory->GetFamily(FontIdOrFamily);
    }
    rep->style         = Style;
    rep->weight        = Weight;
    rep->point_size    = PointSize;
    rep->underlined    = Underlined;

#if !WXGARBAGE_COLLECTION_ON
    wxTheFontList->Append(this);
#endif
}

wxFont::wxFont(int PointSize, const char *Face, int Family, int Style, 
	       int Weight, Bool Underlined)
{
    __type = wxTYPE_FONT;

    rep = wxNEW wxFontRep;

    rep->font_id       = wxTheFontNameDirectory->FindOrCreateFontId(Face, Family);
    rep->face_name     = (Face) ? copystring(Face) : (char*)NULL;
    rep->family        = wxTheFontNameDirectory->GetFamily(rep->font_id);
    rep->style         = Style;
    rep->weight        = Weight;
    rep->point_size    = PointSize;
    rep->underlined    = Underlined;

#if !WXGARBAGE_COLLECTION_ON
    wxTheFontList->Append(this);
#endif
}

wxFont::~wxFont(void)
{
    FreeInternalFonts();

#if !WXGARBAGE_COLLECTION_ON
    wxTheFontList->DeleteObject(this);
#endif
}

//-----------------------------------------------------------------------------
// assignment
//-----------------------------------------------------------------------------

wxFont& wxFont::operator = (wxFont& new_font)
{
    FreeInternalFonts();
    rep = new_font.rep;
    if (rep) rep->ref_count++;

    return (*this);
}

void wxFont::operator = (wxFont* new_font)
{
    FreeInternalFonts();
    if (new_font) {
	rep = new_font->rep;
	if (rep) rep->ref_count++;
    } else {
	rep = wxNEW wxFontRep;

	rep->family        = rep->style = rep->weight = wxDEFAULT;
	rep->underlined    = FALSE;
	rep->point_size    = 0;
    }
}

//-----------------------------------------------------------------------------
// get infos about font
//-----------------------------------------------------------------------------

int wxFont::GetPointSize(void)
{
    return rep->point_size;
}

char *wxFont::GetFaceString(void)
{
    return wxTheFontNameDirectory->GetFontName(rep->font_id); 
}

char* wxFont::GetFaceName(void)
{
    return wxTheFontNameDirectory->GetFontName(rep->font_id); 
}

int wxFont::GetFamily(void)
{
    return rep->family;
}

char* wxFont::GetFamilyString(void)
{
    return wx_font_family[rep->family];
}

int wxFont::GetFontId(void)
{
    return rep->font_id; // stub
}

int wxFont::GetStyle(void)
{
    return rep->style;
}

char* wxFont::GetStyleString(void)
{
    return wx_font_style[rep->style];
}

int wxFont::GetWeight(void)
{
    return rep->weight;
}

char* wxFont::GetWeightString(void)
{
    return wx_font_weight[rep->weight];
}

Bool wxFont::GetUnderlined(void)
{
    return rep->underlined;
}

//-----------------------------------------------------------------------------
// get internal representation of font
//-----------------------------------------------------------------------------

// local help function
static XFontStruct *wxLoadQueryNearestFont(int point_size, int fontid,
					   int style, int weight, 
					   Bool underlined);

XFontStruct *wxFont::GetInternalFont(float scale)
{
    long        int_scale = long(scale * 100.0 + 0.5); // key for fontlist
    int         point_scale = (rep->point_size * 10 * int_scale) / 100;
    XFontStruct *xfont=NULL;;

    wxNode *node = rep->scaled_xfonts.Find(int_scale);
    if (node) {
	xfont = (XFontStruct*)node->Data(); // I have the XFontStruct
    } else {
	xfont = wxLoadQueryNearestFont(point_scale, rep->font_id, rep->style,
				       rep->weight, rep->underlined);
	rep->scaled_xfonts.Append(int_scale, (wxObject*)xfont);
    }
    if (!xfont)
	wxError("could not load any font", "wxFont");
    return xfont;
}

void wxFont::FreeInternalFonts(void)
{
    // unreference and free internal representation (if no references left)
    if (rep && --rep->ref_count==0) {
	wxNode *node = rep->scaled_xfonts.First();
	while (node) {
	    XFontStruct *fs   = (XFontStruct*)node->Data();
	    wxNode      *next = node->Next();
	    XFreeFont(wxAPP_DISPLAY, fs);
	    node = next;
	}
	if (rep->face_name) {
	    delete rep->face_name;
	    rep->face_name = NULL;
	}
	delete rep; rep = NULL;
    }
}

//-----------------------------------------------------------------------------
// local utilities to find a X font
//-----------------------------------------------------------------------------

static XFontStruct *wxLoadQueryFont(int point_size, int fontid, int style,
				    int weight, Bool WXUNUSED(underlined))
{
    char buffer[512];
    char *name = wxTheFontNameDirectory->GetScreenName(fontid, weight, style);

    if (!name)
	name = "-*-*-*-*-*-*-*-%d-*-*-*-*-*-*";
    sprintf(buffer, name, point_size);

    return XLoadQueryFont(wxAPP_DISPLAY, buffer);
}

static XFontStruct *wxLoadQueryNearestFont(int point_size, int fontid,
					   int style, int weight,
					   Bool underlined)
{
    XFontStruct *font;

    font = wxLoadQueryFont(point_size, fontid, style, weight, underlined);

    if (!font) {
	// search up and down by stepsize 10
	int max_size = point_size + 20 * (1 + (point_size/180));
	int min_size = point_size - 20 * (1 + (point_size/180));
	int i;

	// Search for smaller size (approx.)
	for (i=point_size-10; !font && i >= 10 && i >= min_size; i -= 10)
	    font = wxLoadQueryFont(i, fontid, style, weight, underlined);
	// Search for larger size (approx.)
	for (i=point_size+10; !font && i <= max_size; i += 10)
	    font = wxLoadQueryFont(i, fontid, style, weight, underlined);
	// Try default family
	if (!font && fontid != wxDEFAULT)
	    font = wxLoadQueryFont(point_size, wxDEFAULT, style, 
				   weight, underlined);
	// Bogus font
	if (!font)
	    font = wxLoadQueryFont(120, wxDEFAULT, wxNORMAL, wxNORMAL,
				    underlined);
    }
    return font;
}

//-----------------------------------------------------------------------------
// wxFontList
//-----------------------------------------------------------------------------

wxFontList::wxFontList(void) : wxList()
{
    // there is one representation of wxFontList
    wxTheFontList = this;

    wxNORMAL_FONT = FindOrCreateFont(12, wxMODERN, wxNORMAL, wxNORMAL);
    wxSMALL_FONT  = FindOrCreateFont(10, wxSWISS,  wxNORMAL, wxNORMAL);
    wxITALIC_FONT = FindOrCreateFont(12, wxROMAN,  wxITALIC, wxNORMAL);
    wxSWISS_FONT  = FindOrCreateFont(12, wxSWISS,  wxNORMAL, wxNORMAL);
    wxLABEL_FONT  = FindOrCreateFont(12, wxSWISS,  wxNORMAL, wxBOLD);
}

wxFontList::~wxFontList(void)
{
/*  FUM-Warning PURIFY
    wxNode *node = First();
    while (node) {
	wxFont *font = (wxFont*)node->Data();
	wxNode *next = node->Next();
	delete font;
	node = next;
    }
*/
}

wxFont *wxFontList::FindOrCreateFont(int PointSize, int FontIdOrFamily, 
				     int Style, int Weight, Bool Underline,
				     const char *Face)
{
    for (wxNode * node = First (); node; node = node->Next ()) {
	wxFont *each_font = (wxFont *) node->Data ();
	if (each_font &&
	    each_font->GetPointSize () == PointSize &&
	    each_font->GetStyle () == Style &&
	    each_font->GetWeight () == Weight &&
	    each_font->GetUnderlined () == Underline &&
	    each_font->GetFontId () == FontIdOrFamily)
	{
	    return each_font;
	}
    }

    wxFont *font = wxNEW wxFont(PointSize, FontIdOrFamily, Style, Weight, Underline, Face);

#if WXGARBAGE_COLLECTION_ON
    // otherwise done by wxFont()
    AddFont(font);
#endif

    return font;
}

wxFont *wxFontList::FindOrCreateFont(int PointSize, const char *Face, 
				     int Family, int Style, int Weight, 
				     Bool Underline)
{
    return FindOrCreateFont(PointSize,
			    wxTheFontNameDirectory->FindOrCreateFontId(Face, Family),
			    Style, Weight, Underline, Face);
}

//-----------------------------------------------------------------------------
// face names and index functions
//-----------------------------------------------------------------------------

static char *font_defaults[] = {
    "FamilyDefault", "Default",
    "FamilyRoman", "Roman",
    "FamilyDecorative", "Decorative",
    "FamilyModern", "Modern",
    "FamilyTeletype", "Teletype",
    "FamilySwiss", "Swiss",
    "FamilyScript", "Script",

    "AfmMedium", "",
    "AfmBold", "Bo",
    "AfmLight", "",
    "AfmStraight", "",
    "AfmItalic", "${AfmSlant}",
    "AfmSlant", "O",
    "AfmRoman", "Ro",
    "AfmTimes", "Times",
    "AfmHelvetica", "Helv",
    "AfmCourier", "Cour",
    
    "Afm___", "${AfmTimes,$[weight],$[style]}",

    "AfmTimes__", "${AfmTimes}${Afm$[weight]}${Afm$[style]}",
    "AfmTimesMediumStraight", "${AfmTimes}${AfmRoman}",
    "AfmTimesLightStraight", "${AfmTimes}${AfmRoman}",
    "AfmTimes_Italic", "${AfmTimes}$[weight]${AfmItalic}",
    "AfmTimes_Slant", "${AfmTimes}$[weight]${AfmItalic}",

    "AfmSwiss__", "${AfmHelvetica}${Afm$[weight]}${Afm$[style]}",
    "AfmModern__", "${AfmCourier}${Afm$[weight]}${Afm$[style]}",

    "AfmTeletype__", "${AfmModern,$[weight],$[style]}",

    "PostScriptMediumStraight", "",
    "PostScriptMediumItalic", "-Oblique",
    "PostScriptMediumSlant", "-Oblique",
    "PostScriptLightStraight", "",
    "PostScriptLightItalic", "-Oblique",
    "PostScriptLightSlant", "-Oblique",
    "PostScriptBoldStraight", "-Bold",
    "PostScriptBoldItalic", "-BoldOblique",
    "PostScriptBoldSlant", "-BoldOblique",
    
#if WX_NORMALIZED_PS_FONTS
    "PostScript___", "${PostScriptTimes,$[weight],$[style]}",
#else
    "PostScriptRoman__", "${PostScriptTimes,$[weight],$[style]}",
    "PostScript___", "LucidaSans${PostScript$[weight]$[style]}",
#endif

    "PostScriptTimesMedium", "",
    "PostScriptTimesLight", "",
    "PostScriptTimesBold", "Bold",

    "PostScriptTimes__", "Times${PostScript$[weight]$[style]}",
    "PostScriptTimesMediumStraight", "Times-Roman",
    "PostScriptTimesLightStraight", "Times-Roman",
    "PostScriptTimes_Slant", "Times-${PostScriptTimes$[weight]}Italic",
    "PostScriptTimes_Italic", "Times-${PostScriptTimes$[weight]}Italic",

    "PostScriptSwiss__", "Helvetica${PostScript$[weight]$[style]}",
    "PostScriptModern__", "Courier${PostScript$[weight]$[style]}",

    "PostScriptTeletype__", "${PostScriptModern,$[weight],$[style]}",

#if !WX_NORMALIZED_PS_FONTS
    "PostScriptScript__", "Zapf-Chancery-MediumItalic",
#endif

    "ScreenMedium", "medium",
    "ScreenBold", "bold",
    "ScreenLight", "light",
    "ScreenStraight", "r",
    "ScreenItalic", "i",
    "ScreenSlant", "o",

    "ScreenDefaultBase", "misc-fixed",
    "ScreenRomanBase", "*-times",
    "ScreenDecorativeBase", "*-helvetica",
    "ScreenModernBase", "*-courier",
    "ScreenTeletypeBase", "*-lucidatypewriter",
    "ScreenSwissBase", "*-lucida",
    "ScreenScriptBase", "*-zapfchancery",

    "ScreenStdSuffix", "-${Screen$[weight]}-${Screen$[style]}"
    "-normal-*-*-%d-*-*-*-*-*-*",

    "Screen___",
    "-${ScreenDefaultBase}${ScreenStdSuffix}",
    "ScreenRoman__",
    "-${ScreenRomanBase}${ScreenStdSuffix}",
    "ScreenDecorative__",
    "-${ScreenDecorativeBase}${ScreenStdSuffix}",
    "ScreenModern__",
    "-${ScreenModernBase}${ScreenStdSuffix}",
    "ScreenTeletype__",
    "-${ScreenTeletypeBase}${ScreenStdSuffix}",
    "ScreenSwiss__",
    "-${ScreenSwissBase}${ScreenStdSuffix}",
    "ScreenScript__",
    "-${ScreenScriptBase}${ScreenStdSuffix}",
    NULL
};

enum {wxWEIGHT_NORMAL, wxWEIGHT_BOLD,  wxWEIGHT_LIGHT, wxNUM_WEIGHTS};
enum {wxSTYLE_NORMAL,  wxSTYLE_ITALIC, wxSTYLE_SLANT,  wxNUM_STYLES};

static int WCoordinate(int w)
{
    switch (w) {
    case wxBOLD:   return wxWEIGHT_BOLD;
    case wxLIGHT:  return wxWEIGHT_LIGHT;
    case wxNORMAL:
    default:       return wxWEIGHT_NORMAL;
    }
}

static int SCoordinate(int s)
{
    switch (s) {
    case wxITALIC: return wxSTYLE_ITALIC;
    case wxSLANT:  return wxSTYLE_SLANT;
    case wxNORMAL:
    default:       return wxSTYLE_NORMAL;
    }
}

//-----------------------------------------------------------------------------
// wxSuffixMap
//-----------------------------------------------------------------------------

class wxSuffixMap {
public:
    ~wxSuffixMap(void);

    inline char *GetName(int weight, int style)
    {
	return ( map [WCoordinate(weight)] [SCoordinate(style)] );
    }

    char *map[wxNUM_WEIGHTS][wxNUM_STYLES];
    void Initialize(const char *, const char *);
};

#if !USE_RESOURCES
#define wxGetResource(a, b, c) 0
#endif

static void SearchResource(const char *prefix, const char **names, int count, char **v)
{
    int k, i, j;
    char resource[1024], **defaults, *internal;

    k = 1 << count;
    
    *v = NULL;
    internal = NULL;

    for (i = 0; i < k; i++) {
	strcpy(resource, prefix);
	for (j = 0; j < count; j++) {
	    if (!(i & (1 << j)))
		strcat(resource, names[j]);
	    else
		strcat(resource, "_");
	}
	if (wxGetResource(wxAPP_CLASS, (char *)resource, v))
	    return;
	if (!internal) {
	    defaults = font_defaults;
	    while (*defaults) {
		if (!strcmp(*defaults, resource)) {
		    internal = defaults[1];
		    break;
		}
		defaults += 2;
	    }
	}
    }
    if (internal)
	*v = copystring(internal);
}

wxSuffixMap::~wxSuffixMap(void)
{
    int k, j;

    for (k = 0; k < wxNUM_WEIGHTS; ++k)
	for (j = 0; j < wxNUM_STYLES; ++j)
	    if (map[k][j]) {
		delete[] map[k][j];
		map[k][j] = NULL;
	    }
}

void wxSuffixMap::Initialize(const char *resname, const char *devresname)
{
    const char *weight, *style;
    char *v;
    int i, j, k;
    const char *names[3];

    for (k = 0; k < wxNUM_WEIGHTS; k++) {
	switch (k) {
	case wxWEIGHT_NORMAL: weight = "Medium"; break;
	case wxWEIGHT_LIGHT:  weight = "Light"; break;
	case wxWEIGHT_BOLD:
	default:	      weight = "Bold";
	}
	for (j = 0; j < wxNUM_STYLES; j++) {
	    switch (j) {
	    case wxSTYLE_NORMAL: style = "Straight"; break;
	    case wxSTYLE_ITALIC: style = "Italic"; break;
	    case wxSTYLE_SLANT:
	    default:		 style = "Slant";
	    }
	    names[0] = resname;
	    names[1] = weight;
	    names[2] = style;

	    SearchResource(devresname, names, 3, &v);

	    /* Expand macros in the found string: */
	found:
	    int len, closer = 0, startpos = 0;
	    
	    len = (v ? strlen(v) : 0);
	    for (i = 0; i < len; i++) {
		if (v[i] == '$' && ((v[i+1] == '[') || (v[i+1] == '{'))) {
		    startpos = i;
		    closer   = (v[i+1] == '[') ? ']' : '}';
		    ++i;
		} else if (v[i] == closer) {
		    int newstrlen;
		    const char *r = NULL; Bool delete_r = FALSE;
		    char *name;
	  
		    name = v + startpos + 2;
		    v[i] = 0;

		    if (closer == '}') {
			int i, count, len;
			char **names;

			for (i = 0, count = 1; name[i]; i++)
			    if (name[i] == ',')
				count++;
	    
			len = i;

			names = wxNEW char*[count];
			names[0] = name;
			for (i = 0, count = 1; i < len; i++)
			    if (name[i] == ',') {
				names[count++] = name + i + 1;
				name[i] = 0;
			    }

			SearchResource("", (const char **)names, count, (char **)&r);
			delete_r = (r != 0);
			delete[] names;
			
			if (!r) {
			    for (i = 0; i < len; i++)
				if (!name[i])
				    name[i] = ',';
			    r = "";
			    printf("Bad resource name \"%s\" in font lookup\n", name);
			}
		    } else if (!strcmp(name, "weight")) {
			r = weight;
		    } else if (!strcmp(name, "style")) {
			r = style;
		    } else if (!strcmp(name, "family")) {
			r = resname;
		    } else {
			r = "";
			printf("Bad font macro name \"%s\"\n", name);
		    }

		    // add r to v
		    newstrlen = strlen(r);
		    char *naya = wxNEW char[startpos + newstrlen + len - i];
		    memcpy(naya, v, startpos);
		    memcpy(naya + startpos, r, newstrlen);
		    memcpy(naya + startpos + newstrlen, v + i + 1, len - i);
		    if (delete_r)
		      delete[] (char*)r;
		    delete[] v;
		    v = naya;
		    
		    goto found;
		}
	    }
	    /* We have a final value: */
	    map[k][j] = v;
	}
    }
}

//-----------------------------------------------------------------------------
// wxFontNameItem
//-----------------------------------------------------------------------------

class wxFontNameItem : public wxObject {
DECLARE_DYNAMIC_CLASS(wxFontNameItem)
public:
    wxFontNameItem(const char *name, int id, int family);
    ~wxFontNameItem();

    inline char* GetScreenName(int w, int s)     {return screen.GetName(w, s);}
    inline char* GetPostScriptName(int w, int s) {return printing.GetName(w, s);}
    inline char* GetAFMName(int w, int s)        {return afm.GetName(w, s);}
    inline char* GetName(void)                   {return name;}
    inline int   GetFamily(void)                 {return family;}
    inline int   GetId(void)                     {return id;}
    inline Bool  IsRoman(void)                   {return isroman;}
#if WXDEBUG
    void Dump(ostream& str);
#endif

    int id;
    int family;
    char *name;
    wxSuffixMap screen, printing, afm;
    Bool isroman;
};

IMPLEMENT_ABSTRACT_CLASS(wxFontNameItem, wxObject)

wxFontNameItem::wxFontNameItem(const char *Name, int Id, int Family)
{
    name   = copystring(Name);
    id     = Id;
    family = Family;

    screen.  Initialize(name, "Screen");
    printing.Initialize(name, "PostScript");
    afm.     Initialize(name, "Afm");
}

wxFontNameItem::~wxFontNameItem(void)
{
    if (name)
	delete[] name;
    name = NULL;
}

#if WXDEBUG
void wxFontNameItem::Dump(ostream& str)
{
    str << "wxFontNameItem(" << name << ")";
}
#endif

//-----------------------------------------------------------------------------
// wxFontDirectory
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxFontNameDirectory, wxObject)

wxFontNameDirectory::wxFontNameDirectory(void)
{
    // there is one representation of wxFontList
    wxTheFontNameDirectory = this;

    table = wxNEW wxHashTable(wxKEY_INTEGER, 20);
    nextFontId = -1;
    Initialize();
}

wxFontNameDirectory::~wxFontNameDirectory()
{
    // Cleanup wxFontNameItems allocated
    table->BeginFind();
    wxNode *node = table->Next();
    while (node) {
	wxFontNameItem *item = (wxFontNameItem*)node->Data();
	delete item;
	node = table->Next();
    }
    delete table;
}

int wxFontNameDirectory::GetNewFontId(void)
{
    return (nextFontId--);
}

void wxFontNameDirectory::Initialize()
{
    Initialize(wxDEFAULT,    wxDEFAULT,    "Default");
    Initialize(wxDECORATIVE, wxDECORATIVE, "Decorative");
    Initialize(wxROMAN,      wxROMAN,      "Roman");
    Initialize(wxMODERN,     wxMODERN,     "Modern");
    Initialize(wxTELETYPE,   wxTELETYPE,   "Teletype");
    Initialize(wxSWISS,      wxSWISS,      "Swiss");
    Initialize(wxSCRIPT,     wxSCRIPT,     "Script");
}

void wxFontNameDirectory::Initialize(int fontid, int family, const char *resname)
{
    char *fam, resource[256];
  
    sprintf(resource, "Family%s", resname);
    SearchResource((const char *)resource, NULL, 0, (char **)&fam);
    if (fam) {
	if      (!strcmp(fam, "Default"))	family = wxDEFAULT;
	else if (!strcmp(fam, "Roman"))		family = wxROMAN;
	else if (!strcmp(fam, "Decorative"))	family = wxDECORATIVE;
	else if (!strcmp(fam, "Modern"))	family = wxMODERN;
	else if (!strcmp(fam, "Teletype"))	family = wxTELETYPE;
	else if (!strcmp(fam, "Swiss"))		family = wxSWISS;
	else if (!strcmp(fam, "Script"))	family = wxSCRIPT;
	delete[] fam; // free resource
    }
    table->Put(fontid, wxNEW wxFontNameItem(resname, fontid, family));
}

int wxFontNameDirectory::FindOrCreateFontId(const char *name, int family)
{
    int id;

    // font exists -> return id
    if ( (id = GetFontId(name)) ) return id;
    // create new font
    Initialize(id=GetNewFontId(), family, name);
    return id;
}

char *wxFontNameDirectory::GetScreenName(int fontid, int weight, int style)
{
    wxFontNameItem *item = (wxFontNameItem*)table->Get(fontid); // find font
    if (item)
	return item->GetScreenName(weight, style);
    // font does not exist
    return NULL;
}

char *wxFontNameDirectory::GetPostScriptName(int fontid, int weight, int style)
{
    wxFontNameItem *item = (wxFontNameItem*)table->Get(fontid); // find font
    if (item)
	return item->GetPostScriptName(weight, style);
    // font does not exist
    return NULL;
}

char *wxFontNameDirectory::GetAFMName(int fontid, int weight, int style)
{
    wxFontNameItem *item = (wxFontNameItem *)table->Get(fontid); // find font
    if (item)
	return item->GetAFMName(weight, style);
    // font does not exist
    return NULL;
}

char *wxFontNameDirectory::GetFontName(int fontid)
{
    wxFontNameItem *item = (wxFontNameItem *)table->Get(fontid); // find font
    if (item)
	return item->GetName();
    // font does not exist
    return NULL;
}

int wxFontNameDirectory::GetFontId(const char *name)
{
    wxNode *node;

    table->BeginFind();

    while ( (node = table->Next()) ) {
	wxFontNameItem *item = (wxFontNameItem*)node->Data();
	if (!strcmp(name, item->name))
	    return item->id;
    }
    // font does not exist
    return 0;
}

int wxFontNameDirectory::GetFamily(int fontid)
{
    wxFontNameItem *item = (wxFontNameItem *)table->Get(fontid);
  
    if (item)
	return item->family;
    // font does not exist
    return wxDEFAULT;
}
