/*								-*- C++ -*-
 * $Id: DEV_psdc.cpp,v 1.7 1997-02-28 16:03:57+01 mho Exp $
 *
 * Purpose: postscript device context to draw on a postscript printer
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "DEV_psdc.h"
#endif

#define  Uses_XLib
#define  Uses_wxList
#define  Uses_wxMemoryDC
#define  Uses_wxWindowDC
#define  Uses_wxPostScriptDC
#define  Uses_wxPrintSetup
#include "wx.h"

#include <assert.h>
#include <limits.h>
#include <math.h>
#include <string.h>

//-----------------------------------------------------------------------------
// create and destroy wxPostScriptDC
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxPostScriptDC, wxDC)

wxPostScriptDC::wxPostScriptDC(void) : wxDC()
{
    __type = wxTYPE_DC_POSTSCRIPT;

    device = wxDEVICE_EPS;

    current_red = current_blue = current_green = 0.0;
    underline_position = underline_thickness = 0.0;
    clipping = FALSE;
    title    = NULL;
    pstream  = NULL;
    Colour   = wxColourDisplay();

    sign_x =  1.0;  // default x-axis left to right
    sign_y = -1.0;  // default y-axis bottom up -> top down
}

wxPostScriptDC::wxPostScriptDC(char *output, Bool interactive, wxWindow *parent) : wxDC()
{
    __type = wxTYPE_DC_POSTSCRIPT;

    device = wxDEVICE_EPS;

    current_red = current_blue = current_green = 0.0;
    clipping = FALSE;
    title    = NULL;
    pstream  = NULL;
    Colour   = wxColourDisplay();

    sign_x =  1.0;  // default x-axis left to right
    sign_y = -1.0;  // default y-axis bottom up -> top down
	
    Create(output, interactive, parent);
}

Bool wxPostScriptDC::Create(char *output, Bool interactive, wxWindow *parent)
{
    char tmp[256];

    if (interactive && !PrinterDialog(parent))
	return (ok = FALSE);

    if (!output) {
	if (wxThePrintSetupData->GetPrinterMode() == PS_PREVIEW
	||  wxThePrintSetupData->GetPrinterMode() == PS_PRINTER) {
	    #ifdef VMS
	    output = "preview.ps";
	    #else
	    // For PS_PRINTER action this depends on a Unix-style print spooler
	    strcpy (tmp, "/tmp/preview_");
	    wxGetUserId(tmp+strlen(tmp), 252-strlen(tmp));
	    strcat(tmp, ".ps");
	    output = tmp;
	    #endif
	} else if (wxThePrintSetupData->GetPrinterMode() == PS_FILE) {
	    output = wxSaveFileSelector ("PostScript", "ps", 0, parent);
	}
    }
    if (output) {
	wxThePrintSetupData->SetPrinterFile(output);
	return (ok = TRUE);
    }
    return (ok = FALSE);
}

wxPostScriptDC::~wxPostScriptDC(void)
{
    if (title)
	delete[] title;
    if (pstream)
	delete pstream;
    title   = NULL;
    pstream = NULL;
}

//-----------------------------------------------------------------------------
// start and end of document/page
//-----------------------------------------------------------------------------

static const char *wxPostScriptHeaderEllipse = "\
/ellipsedict 8 dict def\n\
ellipsedict /mtrx matrix put\n\
/ellipse {\n\
	ellipsedict begin\n\
	/endangle exch def\n\
	/startangle exch def\n\
	/yrad exch def\n\
	/xrad exch def\n\
	/y exch def\n\
	/x exch def\n\
	/savematrix mtrx currentmatrix def\n\
	x y translate\n\
	xrad yrad scale\n\
	0 0 1 startangle endangle arc\n\
	savematrix setmatrix\n\
	end\n\
	} def\n\
";

#if USE_SPLINES
static const char *wxPostScriptHeaderSpline = "\
/DrawSplineSection {\n\
	/y3 exch def\n\
	/x3 exch def\n\
	/y2 exch def\n\
	/x2 exch def\n\
	/y1 exch def\n\
	/x1 exch def\n\
	/xa x1 x2 x1 sub 0.666667 mul add def\n\
	/ya y1 y2 y1 sub 0.666667 mul add def\n\
	/xb x3 x2 x3 sub 0.666667 mul add def\n\
	/yb y3 y2 y3 sub 0.666667 mul add def\n\
	x1 y1 lineto\n\
	xa ya xb yb x3 y3 curveto\n\
	} def\n\
";
#endif

static const char *wxPostScriptHeaderColourImage = "\
% define 'colorimage' if it isn't defined\n\
%   ('colortogray' and 'mergeprocs' come from xwd2ps\n\
%     via xgrab)\n\
/colorimage where   % do we know about 'colorimage'?\n\
  { pop }           % yes: pop off the 'dict' returned\n\
  {                 % no:  define one\n\
    /colortogray {  % define an RGB->I function\n\
      /rgbdata exch store    % call input 'rgbdata'\n\
      rgbdata length 3 idiv\n\
      /npixls exch store\n\
      /rgbindx 0 store\n\
      0 1 npixls 1 sub {\n\
        grays exch\n\
        rgbdata rgbindx       get 20 mul    % Red\n\
        rgbdata rgbindx 1 add get 32 mul    % Green\n\
        rgbdata rgbindx 2 add get 12 mul    % Blue\n\
        add add 64 idiv      % I = .5G + .31R + .18B\n\
        put\n\
        /rgbindx rgbindx 3 add store\n\
      } for\n\
      grays 0 npixls getinterval\n\
    } bind def\n\
\n\
    % Utility procedure for colorimage operator.\n\
    % This procedure takes two procedures off the\n\
    % stack and merges them into a single procedure.\n\
\n\
    /mergeprocs { % def\n\
      dup length\n\
      3 -1 roll\n\
      dup\n\
      length\n\
      dup\n\
      5 1 roll\n\
      3 -1 roll\n\
      add\n\
      array cvx\n\
      dup\n\
      3 -1 roll\n\
      0 exch\n\
      putinterval\n\
      dup\n\
      4 2 roll\n\
      putinterval\n\
    } bind def\n\
\n\
    /colorimage { % def\n\
      pop pop     % remove 'false 3' operands\n\
      {colortogray} mergeprocs\n\
      image\n\
    } bind def\n\
  } ifelse          % end of 'false' case\n\
";

static char wxPostScriptHeaderReencodeISO[] = 
"\n/reencodeISO {\n"
"dup dup findfont dup length dict begin\n"
"{ 1 index /FID ne { def }{ pop pop } ifelse } forall\n"
"/Encoding ISOLatin1Encoding def\n"
"currentdict end definefont\n"
"} def\n"
"/ISOLatin1Encoding [\n"
"/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef\n"
"/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef\n"
"/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef\n"
"/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef\n"
"/space/exclam/quotedbl/numbersign/dollar/percent/ampersand/quoteright\n"
"/parenleft/parenright/asterisk/plus/comma/minus/period/slash\n"
"/zero/one/two/three/four/five/six/seven/eight/nine/colon/semicolon\n"
"/less/equal/greater/question/at/A/B/C/D/E/F/G/H/I/J/K/L/M/N\n"
"/O/P/Q/R/S/T/U/V/W/X/Y/Z/bracketleft/backslash/bracketright\n"
"/asciicircum/underscore/quoteleft/a/b/c/d/e/f/g/h/i/j/k/l/m\n"
"/n/o/p/q/r/s/t/u/v/w/x/y/z/braceleft/bar/braceright/asciitilde\n"
"/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef\n"
"/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef/.notdef\n"
"/.notdef/dotlessi/grave/acute/circumflex/tilde/macron/breve\n"
"/dotaccent/dieresis/.notdef/ring/cedilla/.notdef/hungarumlaut\n"
"/ogonek/caron/space/exclamdown/cent/sterling/currency/yen/brokenbar\n"
"/section/dieresis/copyright/ordfeminine/guillemotleft/logicalnot\n"
"/hyphen/registered/macron/degree/plusminus/twosuperior/threesuperior\n"
"/acute/mu/paragraph/periodcentered/cedilla/onesuperior/ordmasculine\n"
"/guillemotright/onequarter/onehalf/threequarters/questiondown\n"
"/Agrave/Aacute/Acircumflex/Atilde/Adieresis/Aring/AE/Ccedilla\n"
"/Egrave/Eacute/Ecircumflex/Edieresis/Igrave/Iacute/Icircumflex\n"
"/Idieresis/Eth/Ntilde/Ograve/Oacute/Ocircumflex/Otilde/Odieresis\n"
"/multiply/Oslash/Ugrave/Uacute/Ucircumflex/Udieresis/Yacute\n"
"/Thorn/germandbls/agrave/aacute/acircumflex/atilde/adieresis\n"
"/aring/ae/ccedilla/egrave/eacute/ecircumflex/edieresis/igrave\n"
"/iacute/icircumflex/idieresis/eth/ntilde/ograve/oacute/ocircumflex\n"
"/otilde/odieresis/divide/oslash/ugrave/uacute/ucircumflex/udieresis\n"
"/yacute/thorn/ydieresis\n"
"] def\n\n";

Bool wxPostScriptDC::StartDoc(char *message)
{
    pstream = wxNEW ofstream(wxThePrintSetupData->GetPrinterFile());
    if (!pstream || !pstream->good()) {
	wxMessageBox ("Error", "Cannot open file!", wxOK);
	return (ok = FALSE);
    }
    // set default drawing tools
    SetBrush(wxBLACK_BRUSH);
    SetPen(wxBLACK_PEN);
    SetBackground(wxWHITE_BRUSH);
    SetTextForeground(wxBLACK);
    // set origin according to paper size
    SetDeviceOrigin(0, 0);
    // number of pages and page number
    page_number = 0;
    // for %%Title: title
    if (message)
	title = copystring(message);
    // ready for drawing
    return (ok = TRUE);
}

void wxPostScriptDC::EndDoc(void)
{
    char buf1[256], buf2[256];

    if (!pstream)
	return;
    DestroyClippingRegion();

#ifdef VMS
    char *header_file = copystring("header.ps");
#else
    char *header_file = wxGetTempFileName("ps");
#endif
    delete pstream;
    pstream = wxNEW ofstream(header_file);

    // PostScript magic strings for EPS and non-EPS
    if (wxThePrintSetupData->GetPrinterOrientation() == PS_EPS)
	*pstream << "%!PS-Adobe-2.0 EPSF-2.0\n";
    else
	*pstream << "%!PS-Adobe-2.0\n";
    if (title)
	*pstream << "%%Title: " << title << "\n";
    *pstream << "%%Creator: " << wxAPP_NAME << "\n";
    *pstream << "%%CreationDate: " << wxNow() << "\n";
    wxGetEmailAddress(buf1, 255);
    wxGetUserName(buf2, 255);
    *pstream << "%%For: " << (*buf1 ? buf1 : "unknown")
	     << " ("      << (*buf2 ? buf2 : "unknown") << ")\n";

    float trans_x, trans_y, scale_x, scale_y, size_x, size_y;
    Bool  landscape
	= (wxThePrintSetupData->GetPrinterOrientation() == PS_LANDSCAPE);
    wxThePrintSetupData->GetPrinterTranslation(&trans_x, &trans_y);
    wxThePrintSetupData->GetPrinterScaling(&scale_x, &scale_y);
    GetSize(&size_x, &size_y);
    if (!landscape)
	trans_y += (size_y / scale_y);
    // Compute the bounding box.  Note that it is in the default user
    // coordinate system, thus we have to convert the values.
    float llx = (min_x + trans_x) * scale_x;
    float lly = (min_y + trans_y) * scale_y;
    float urx = (max_x + trans_x) * scale_x;
    float ury = (max_y + trans_y) * scale_y;
    // If we're landscape, our sense of "x" and "y" is reversed.
    if (landscape) {
	llx = (-max_y - trans_y) * scale_y;
	lly = ( min_x + trans_x) * scale_x;
	urx = (-min_y - trans_y) * scale_y;
	ury = ( max_x + trans_x) * scale_x;
    }
    *pstream << "%%BoundingBox: "
	     << floor(llx) << " " << floor(lly) << " "
 	     << ceil(urx)  << " " << ceil(ury)  << "\n"
	     << "%%Pages: " << page_number << "\n"
	     << "%%EndComments\n\n";
    // draw bounding box (commented out)
    *pstream << "% newpath\n"
	     << "% " << llx << " " << lly << " moveto\n"
	     << "% " << urx << " " << lly << " lineto\n"
	     << "% " << urx << " " << ury << " lineto\n"
	     << "% " << llx << " " << ury << " lineto closepath stroke\n";
    // Output scaling
    if (landscape)
	*pstream << "90 rotate\n";
    *pstream << scale_x << " " << scale_y << " scale\n"
	     << trans_x << " " << trans_y << " translate\n";
    *pstream << "%%BeginProlog\n";
    *pstream << wxPostScriptHeaderEllipse;
#if USE_SPLINES
    *pstream << wxPostScriptHeaderSpline;
#endif
    *pstream << wxPostScriptHeaderColourImage;
    *pstream << wxPostScriptHeaderReencodeISO;
    *pstream << "%%EndProlog\n";
    delete pstream;
    pstream = NULL;

    // Paste header Before wx_printer_file
    wxConcatFiles(header_file, wxThePrintSetupData->GetPrinterFile(),
		  wxThePrintSetupData->GetPrinterFile());
    wxRemoveFile(header_file);
    delete[]header_file;

    if (ok) {
	char *argv[5]; int i; char *opts; 

	switch (wxThePrintSetupData->GetPrinterMode()) {
	case PS_PREVIEW:
	    argv[0] = wxThePrintSetupData->GetPrintPreviewCommand();
	    argv[1] = "-a4";
	    argv[2] = ((wxThePrintSetupData->GetPrinterOrientation()
			== PS_LANDSCAPE) ? "-landscape" : "-portrait");
	    argv[3] = wxThePrintSetupData->GetPrinterFile();
	    argv[4] = NULL;
	    wxExecute(argv);
	    break;
	case PS_PRINTER:
	    argv[0] = wxThePrintSetupData->GetPrinterCommand();
	    i = 1;
	    opts = wxThePrintSetupData->GetPrinterOptions();
	    if (opts && *opts) argv[i++] = opts;
	    argv[i++] = wxThePrintSetupData->GetPrinterFile();
	    argv[i] = NULL;
	    wxExecute(argv);
	    break;
	case PS_FILE:
	    break;
	}
    }
}

void wxPostScriptDC::StartPage(void)
{
    if (!pstream)
	return;
    *pstream << "%%Page: " << ++page_number << "\n";
    *pstream << "matrix currentmatrix\n";
}

void wxPostScriptDC::EndPage(void)
{
    if (!pstream)
	return;
    *pstream << "showpage\n";
    *pstream << "setmatrix\n";
}

//-----------------------------------------------------------------------------
// drawing methods
//-----------------------------------------------------------------------------

Bool wxPostScriptDC::Blit(float _xdest, float _ydest, float fwidth, float fheight,
			  wxDC *_src, float xsrc, float ysrc, int WXUNUSED(rop))
{
    // I need drawables for this operation
    if (!_src->IsKindOf(CLASSINFO(wxWindowDC)) || !((wxWindowDC*)_src)->DrawArea())
	return FALSE;

    wxWindowDC *src = (wxWindowDC*)_src;

    // source coordinates
    long width  = src->XLOG2DEVREL(fwidth);
    long height = src->YLOG2DEVREL(fheight);
    long x      = src->XLOG2DEV(xsrc);
    long y      = src->YLOG2DEV(ysrc);
    // destination coordinates
    long xdest = XLOG2DEV(_xdest);
    long ydest = YLOG2DEV(_ydest + fheight);
    long wdest = (do_gdi_scaling ? XLOG2DEVREL(fwidth)  : width);
    long hdest = (do_gdi_scaling ? YLOG2DEVREL(fheight) : height);
    // PostScript setup:
    *pstream << "/origstate save def\n";
    *pstream << "20 dict begin\n";
    *pstream << "/pix " << wdest << " string def\n";
    *pstream << "/grays " << wdest << " string def\n";
    *pstream << "/npixels 0 def\n";
    *pstream << "/rgbindx 0 def\n";
    *pstream << xdest << " " << ydest << " translate\n";
    *pstream << wdest << " " << hdest << " scale\n";
    *pstream << wdest << " " << hdest << " 8\n";
    *pstream << "[" << wdest << " 0 0 " << (-hdest) << " 0 " << hdest << "]\n";
    *pstream << "{currentfile pix readhexstring pop}\n";
    *pstream << "false 3 colorimage\n";
  
    Colormap cm=wxAPP_COLOURMAP->GetColormap();
    XImage *image=NULL;
    char s[7];

    image = XGetImage(wxAPP_DISPLAY, src->DrawArea(),
		      x, y, width, height, AllPlanes, ZPixmap);

    s[2] = 0;

    #define CM_CACHE_SIZE 256
    unsigned long cachesrc[CM_CACHE_SIZE];
    int cachedest[CM_CACHE_SIZE], cache_pos = 0, all_cache = FALSE;

    for (int j = 0; j < hdest; j++) {
	for (int i = 0; i < wdest; i++) {
	    XColor xcol;
	    unsigned long spixel;
	    int pixel, k;
	    const unsigned short MAX_COLOR = 0xFFFF;
	    spixel = XGetPixel(image, i*width/wdest, j*height/hdest);
	    for (k = cache_pos; k--; )
		if (cachesrc[k] == spixel) {
		    pixel = cachedest[k];
		    goto install;
		}
	    if (all_cache)
		for (k = CM_CACHE_SIZE; k-- > cache_pos; )
		    if (cachesrc[k] == spixel) {
			pixel = cachedest[k];
			goto install;
		    }
            cachesrc[cache_pos] = xcol.pixel = spixel;
	    XQueryColor(wxAPP_DISPLAY, cm, &xcol);
	    {
		// set colour or grey value
		float r    = float(xcol.red)   / MAX_COLOR;
		float g    = float(xcol.green) / MAX_COLOR;
		float b    = float(xcol.blue)  / MAX_COLOR;
		float gray = (.3*r) + (.59*g) + (.11*b);
		if (gray > 1.0)
		    gray = 1.0;
		if (Colour) {
		    pixel = (int)(256*256*int(255*r) +
				  256*    int(255*g) +
				          int(255*b));
		} else {
		    int gr = int(255*gray);
		    pixel = 256*256*gr + 256*gr + gr;
		}
// 		cerr << "RGB: " << hex << r << " " << g << " " << b
// 		     << " = "   << gray
// 		     << " : "   << pixel << dec << endl;
	    }
	    cachedest[cache_pos] = pixel;
	    if (++cache_pos >= CM_CACHE_SIZE) {
		cache_pos = 0;
		all_cache = TRUE;
	    }
	install:      
	    wxDecToHex((pixel >> 16) & 0xff, s);
	    wxDecToHex((pixel >>  8) & 0xff, s+2);
	    wxDecToHex((pixel      ) & 0xff, s+4);
            *pstream << s;
	}
	*pstream << "\n";
    }

    if (image) XDestroyImage(image);

    *pstream << "end\n";
    *pstream << "origstate restore\n";

    CalcBoundingBox(xdest, ydest);
    CalcBoundingBox(xdest + wdest, ydest - hdest);

    return TRUE;
}

void wxPostScriptDC::Clear(void)
{
    // no need for PS
}

void wxPostScriptDC::CrossHair(float WXUNUSED(x), float WXUNUSED(y))
{
    // no need for PS
}

// constant to convert radian to degree
#define RAD2DEG 57.2957795131

void wxPostScriptDC::DrawArc(float x1, float y1, float x2, float y2,
			     float xc, float yc)
{
    if (!pstream)
	return;

    double dx = x1 - xc;
    double dy = y1 - yc;
    double radius = sqrt(dx*dx+dy*dy);
    double alpha1, alpha2;

    if (x1 == x2 && y1 == y2) {
	alpha1 = 0.0;
	alpha2 = 360.0;
    } else if (radius == 0.0) {
	alpha1 = alpha2 = 0.0;
    } else {
	alpha1 = (x1 - xc == 0) ?
	    (y1 - yc < 0) ? 90.0 : -90.0 :
	    -atan2(double(y1-yc), double(x1-xc)) * RAD2DEG;
	alpha2 = (x2 - xc == 0) ?
	    (y2 - yc < 0) ? 90.0 : -90.0 :
	    -atan2(double(y2-yc), double(x2-xc)) * RAD2DEG;
    }
    while (alpha1 <= 0)   alpha1 += 360;
    while (alpha2 <= 0)   alpha2 += 360; // adjust angles to be between
    while (alpha1 > 360)  alpha1 -= 360; // 0 and 360 degree
    while (alpha2 > 360)  alpha2 -= 360;

    if (current_brush && current_brush->GetStyle() != wxTRANSPARENT) {
	SetBrush(current_brush);
	*pstream << "newpath\n"
	         << _XLOG2DEV(xc)        << " "
		 << _YLOG2DEV(yc)        << " "
	         << _XLOG2DEVREL(radius) << " "
		 << _YLOG2DEVREL(radius) << " "
		 << alpha1               << " "
		 << alpha2               << " ellipse\n"
	         << _XLOG2DEV(xc)        << " "
		 << _YLOG2DEV(yc)        << " lineto\n"
		 << "closepath\n"
	         << "fill\n";
    }
    if (current_pen && current_pen->GetStyle() != wxTRANSPARENT) {
	SetPen(current_pen);
	*pstream << "newpath\n"
	         << _XLOG2DEV(xc)        << " "
		 << _YLOG2DEV(yc)        << " "
	         << _XLOG2DEVREL(radius) << " "
		 << _YLOG2DEVREL(radius) << " "
		 << alpha1               << " "
		 << alpha2               << " ellipse\n"
	         << "stroke\n";
    }
    CalcBoundingBox(_XLOG2DEV(xc-radius), _YLOG2DEV(yc-radius));
    CalcBoundingBox(_XLOG2DEV(xc+radius), _YLOG2DEV(yc+radius));
}

void wxPostScriptDC::DrawBitmap8(unsigned char *WXUNUSED(data),
				 int WXUNUSED(w), int WXUNUSED(h),
				 wxColourMap *WXUNUSED(cmap),
				 float WXUNUSED(x), float WXUNUSED(y))
{
    // not implemented for PostScript yet.
}

void wxPostScriptDC::DrawEllipse(float x, float y, float w, float h,
				 int startAngle, int sizeAngle)
{
    if (!pstream)
	return;

    CalcBoundingBox(_XLOG2DEV(x),   _YLOG2DEV(y));
    CalcBoundingBox(_XLOG2DEV(x+w), _YLOG2DEV(y+h));

    if (current_brush && current_brush->GetStyle () != wxTRANSPARENT) {
	SetBrush(current_brush);
	*pstream << "newpath\n"
	         << _XLOG2DEV(x+w/2)     << " " << _YLOG2DEV(y+h/2)   << " "
	         << _XLOG2DEVREL(w/2)    << " " << _YLOG2DEVREL(h/2)  << " "
		 << float(startAngle)/64 << " "
		 << float(startAngle+sizeAngle)/64
	         << " ellipse\n";
	if (sizeAngle % (360*64))
	    *pstream << _XLOG2DEV(x+w/2) << " "
		     << _YLOG2DEV(y+h/2) << " lineto\nclosepath\n";
	*pstream << "fill\n";
    }
    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen(current_pen);
	*pstream << "newpath\n"
	         << _XLOG2DEV(x+w/2)     << " " << _YLOG2DEV(y+h/2)   << " "
	         << _XLOG2DEVREL(w/2)    << " " << _YLOG2DEVREL(h/2)  << " "
		 << float(startAngle)/64 << " "
		 << float(startAngle+sizeAngle)/64
	         << " ellipse\n";
	if (sizeAngle % (360*64))
	    *pstream << _XLOG2DEV(x+w/2) << " "
		     << _YLOG2DEV(y+h/2) << " lineto\nclosepath\n";
	*pstream << "stroke\n";
    }
}

void wxPostScriptDC::DrawIcon(wxBitmap *icon, float x, float y,
			      Bool WXUNUSED(useMask))
{
    if (!pstream)
	return;

    wxMemoryDC memDC;
    memDC.SelectObject(icon);
    Blit(x, y, float(icon->GetWidth()), float(icon->GetHeight()), &memDC, 0.0, 0.0);
}

void wxPostScriptDC::DrawLine(float x1, float y1, float x2, float y2)
{
    if (!pstream)
	return;

    CalcBoundingBox(_XLOG2DEV(x1), _YLOG2DEV(y1));
    CalcBoundingBox(_XLOG2DEV(x2), _YLOG2DEV(y2));

    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	*pstream << "newpath\n"
	         << _XLOG2DEV(x1) << " " << _YLOG2DEV (y1) << " moveto\n"
	         << _XLOG2DEV(x2) << " " << _YLOG2DEV (y2) << " lineto\n"
	         << "stroke\n";
    }
}

void wxPostScriptDC::DrawLines(int n, wxPoint pts[], float xoff, float yoff)
{
    if (!pstream || n <= 0)
	return;

    for (int i=0; i<n ; ++i)
	CalcBoundingBox(_XLOG2DEV(pts[i].x+xoff), _YLOG2DEV(pts[i].y+yoff));

    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	*pstream << "newpath\n"
		 << _XLOG2DEV(pts[0].x+xoff) << " "
		 << _YLOG2DEV(pts[0].y+yoff) << " moveto\n";
	for (int i = 1; i < n; i++)	{
	    *pstream << _XLOG2DEV(pts[i].x+xoff) << " "
		     << _YLOG2DEV(pts[i].y+yoff) << " lineto\n";
	}
	*pstream << "stroke\n";
    }
}

void wxPostScriptDC::DrawLines(int n, wxIntPoint pts[], int xoff, int yoff)
{
    if (!pstream || n <= 0)
	return;

    for (int i=0; i<n ; ++i)
	CalcBoundingBox(_XLOG2DEV(float(pts[i].x+xoff)), _YLOG2DEV(float(pts[i].y+yoff)));

    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	*pstream << "newpath\n"
		 << _XLOG2DEV(pts[0].x+xoff) << " "
		 << _YLOG2DEV(pts[0].y+yoff) << " moveto\n";
	for (int i = 1; i < n; i++)	{
	    *pstream << _XLOG2DEV(pts[i].x+xoff) << " "
		     << _YLOG2DEV(pts[i].y+yoff) << " lineto\n";
	}
	*pstream << "stroke\n";
    }
}

void wxPostScriptDC::DrawLines(wxList *list, float xoff, float yoff)
{
    int n = list->Number();
    wxPoint *points = wxNEW wxPoint[n];

    int i = 0;
    for(wxNode *node = list->First(); node; node = node->Next()) {
	wxPoint *point = (wxPoint*)node->Data();
	points[i].x = point->x;
	points[i++].y = point->y;
    }
    DrawLines(n, points, xoff, yoff);
    delete []points;
}

void wxPostScriptDC::DrawPoint(float x, float y)
{
    if (!pstream)
	return;

    CalcBoundingBox(_XLOG2DEV(x), _YLOG2DEV(y));

    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen(current_pen);
	*pstream << "newpath\n"
		 << _XLOG2DEV(x)   << " " << _YLOG2DEV (y) << " moveto\n"
		 << _XLOG2DEV(x+1) << " " << _YLOG2DEV (y) << " lineto\n"
		 << "stroke\n";
    }
}

void wxPostScriptDC::DrawPolygon(int n, wxPoint pts[], float xoff, float yoff,
				 int WXUNUSED(fill))
{
    if (!pstream && n <= 0)
	return;

    for (int i=0; i<n ; ++i)
	CalcBoundingBox(_XLOG2DEV(pts[i].x+xoff), _YLOG2DEV(pts[i].y+yoff));

    if (current_brush && current_brush->GetStyle () != wxTRANSPARENT) {
	SetBrush (current_brush);

	*pstream << "newpath\n"
		 << _XLOG2DEV(pts[0].x+xoff) << " "
		 << _YLOG2DEV(pts[0].y+yoff) << " moveto\n";
	for (int i = 1; i < n; i++)	{
	    *pstream << _XLOG2DEV(pts[i].x+xoff) << " "
		     << _YLOG2DEV(pts[i].y+yoff) << " lineto\n";
	}
	*pstream << "closepath\n" << "fill\n";
    }
    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	*pstream << "newpath\n"
		 << _XLOG2DEV(pts[0].x+xoff) << " "
		 << _YLOG2DEV(pts[0].y+yoff) << " moveto\n";
	for (int i = 1; i < n; i++) {
	    *pstream << _XLOG2DEV(pts[i].x+xoff) << " "
		     << _YLOG2DEV(pts[i].y+yoff) << " lineto\n";
	}
	*pstream << "closepath\n" << "stroke\n";
    }
}

void wxPostScriptDC::DrawPolygon(wxList *list, float xoff, float yoff, int fill)
{
    int n = list->Number();
    wxPoint *points = wxNEW wxPoint[n];

    int i = 0;
    for(wxNode *node = list->First(); node; node = node->Next()) {
	wxPoint *point = (wxPoint *)node->Data();
	points[i].x = point->x;
	points[i++].y = point->y;
    }
    DrawPolygon(n, points, xoff, yoff, fill);
    delete[] points;
}

void wxPostScriptDC::DrawRectangle(float x, float y, float w, float h)
{
    if (!pstream)
	return;

    CalcBoundingBox(_XLOG2DEV(x),   _YLOG2DEV(y));
    CalcBoundingBox(_XLOG2DEV(x+w), _YLOG2DEV(y+h));

    if (current_brush && current_brush->GetStyle () != wxTRANSPARENT) {
	SetBrush (current_brush);
	*pstream << "newpath\n"
		 << _XLOG2DEV(x)   << " " << _YLOG2DEV(y)   << " moveto\n"
		 << _XLOG2DEV(x+w) << " " << _YLOG2DEV(y)   << " lineto\n"
		 << _XLOG2DEV(x+w) << " " << _YLOG2DEV(y+h) << " lineto\n"
		 << _XLOG2DEV(x)   << " " << _YLOG2DEV(y+h) << " lineto\n"
		 << "closepath\n" << "fill\n";
    }
    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	*pstream << "newpath\n"
		 << _XLOG2DEV(x)   << " " << _YLOG2DEV(y)   << " moveto\n"
		 << _XLOG2DEV(x+w) << " " << _YLOG2DEV(y)   << " lineto\n"
		 << _XLOG2DEV(x+w) << " " << _YLOG2DEV(y+h) << " lineto\n"
		 << _XLOG2DEV(x)   << " " << _YLOG2DEV(y+h) << " lineto\n"
		 << "closepath\n" << "stroke\n";
    }
}

void wxPostScriptDC::DrawRoundedRectangle(float x, float y, float w, float h, float r)
{
    if (!pstream)
	return;

    CalcBoundingBox(_XLOG2DEV(x),   _YLOG2DEV(y)  );
    CalcBoundingBox(_XLOG2DEV(x+w), _YLOG2DEV(y+h));

    if (current_brush && current_brush->GetStyle () != wxTRANSPARENT) {
	SetBrush (current_brush);
	// Draw rectangle anticlockwise
	*pstream << "newpath\n"
		 << _XLOG2DEV(x+r)   << " " << _YLOG2DEV(y+r)   << " " << _XLOG2DEVREL(r) << " 90 180 arc\n"
		 << _XLOG2DEV(x)     << " " << _YLOG2DEV(y+h-r) << " lineto\n"
		 << _XLOG2DEV(x+r)   << " " << _YLOG2DEV(y+h-r) << " " << _XLOG2DEVREL(r) << " 180 270 arc\n"
		 << _XLOG2DEV(x+w-r) << " " << _YLOG2DEV(y+h)   << " lineto\n"
		 << _XLOG2DEV(x+w-r) << " " << _YLOG2DEV(y+h-r) << " " << _XLOG2DEVREL(r) << " 270 0 arc\n"
		 << _XLOG2DEV(x+w)   << " " << _YLOG2DEV(y+r)   << " lineto\n"
		 << _XLOG2DEV(x+w-r) << " " << _YLOG2DEV(y+r)   << " " << _XLOG2DEVREL(r) << " 0 90 arc\n"
		 << _XLOG2DEV(x+r)   << " " << _YLOG2DEV(y)     << " lineto\n"
		 << "closepath\n" << "fill\n";
    }
    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	// Draw rectangle anticlockwise
	*pstream << "newpath\n"
		 << _XLOG2DEV(x+r)   << " " << _YLOG2DEV(y+r)   << " " << _XLOG2DEVREL(r) << " 90 180 arc\n"
		 << _XLOG2DEV(x)     << " " << _YLOG2DEV(y+h-r) << " lineto\n"
		 << _XLOG2DEV(x+r)   << " " << _YLOG2DEV(y+h-r) << " " << _XLOG2DEVREL(r) << " 180 270 arc\n"
		 << _XLOG2DEV(x+w-r) << " " << _YLOG2DEV(y+h)   << " lineto\n"
		 << _XLOG2DEV(x+w-r) << " " << _YLOG2DEV(y+h-r) << " " << _XLOG2DEVREL(r) << " 270 0 arc\n"
		 << _XLOG2DEV(x+w)   << " " << _YLOG2DEV(y+r)   << " lineto\n"
		 << _XLOG2DEV(x+w-r) << " " << _YLOG2DEV(y+r)   << " " << _XLOG2DEVREL(r) << " 0 90 arc\n"
		 << _XLOG2DEV(x+r)   << " " << _YLOG2DEV(y)     << " lineto\n"
		 << "closepath\n" << "stroke\n";
    }
}

void wxPostScriptDC::FloodFill(float WXUNUSED(x), float WXUNUSED(y),
			       wxColour *WXUNUSED(col),int WXUNUSED(style))
{
    // don't know how to do it for PS
}

void wxPostScriptDC::IntDrawLine(int x1, int y1, int x2, int y2)
{
    DrawLine(float(x1), float(y1), float(x2), float(y2));
}

void wxPostScriptDC::IntDrawLines(int n, wxIntPoint pts[], int xoff, int yoff)
{
    DrawLines(n, pts, xoff, yoff);
}

//-----------------------------------------------------------------------------
// drawing tools
//-----------------------------------------------------------------------------

void wxPostScriptDC::SetBackground(wxBrush *brush)
{
    current_background_brush = brush;
}

void wxPostScriptDC::SetBrush(wxBrush *brush)
{
//  wxBrush *old_brush = current_brush;
    if (!pstream || !(current_brush = brush))
	return;

    // Brush colour
    float red   = current_brush->GetColour().Red();
    float blue  = current_brush->GetColour().Blue();
    float green = current_brush->GetColour().Green();
    // set colour or grey value
    float psRed   = (float)(red / 255.0);	// convert to 0 to 1.0
    float psGreen = (float)(green / 255.0);	// convert to 0 to 1.0
    float psBlue  = (float)(blue / 255.0);	// convert to 0 to 1.0
    if (Colour) {
	*pstream << psRed << " " << psGreen << " " << psBlue
		 << " setrgbcolor\n";
	current_red   = red;
	current_blue  = blue;
	current_green = green;
    } else {
// 	float gray = (.3*psRed) + (.59*psGreen) + (.11*psBlue);
// 	*pstream << gray << " setgray\n";
	*pstream << 0.0 << " setgray\n";
    }
}

void wxPostScriptDC::SetColourMap(wxColourMap *WXUNUSED(cmap))
{
    // no need for PS
}

void wxPostScriptDC::SetLogicalFunction(int WXUNUSED(fkt))
{
    // don't know how to do it for PS
}

void wxPostScriptDC::SetPen(wxPen *pen)
{
    wxPen *old_pen = current_pen;
    if (!pstream || !(current_pen = pen))
	return;

    // Line width
    *pstream << (do_gdi_scaling ?_XLOG2DEVREL(pen->GetWidth()) : pen->GetWidth())
	     << " setlinewidth\n";
    // Line style
    if (old_pen != pen) {
	int style;
	switch (style=pen->GetStyle()) {
	default:
	    *pstream << "[] 0";
	    break;
	case wxDOT: case wxLONG_DASH: case wxSHORT_DASH: case wxDOT_DASH:
	    static char *dashdefs[4] = {
		"[2 5] 2",    // wxDOT
		"[4 8] 2",    // wxLONG_DASH
		"[4 4] 2",    // wxSHORT_DASH
		"[6 6 2 6] 4" // wxDOT_DASH
	    };
	    *pstream << dashdefs[style - wxFIRST_DASH];
	    break;
	}
	*pstream << " setdash\n";
    }
    // Line colour
    float red   = pen->GetColour().Red();
    float blue  = pen->GetColour().Blue();
    float green = pen->GetColour().Green();
    // set colour or grey value
    float psRed   = (float)(red / 255.0);	// convert to 0 to 1.0
    float psGreen = (float)(green / 255.0);	// convert to 0 to 1.0
    float psBlue  = (float)(blue / 255.0);	// convert to 0 to 1.0
    if (Colour) {
	*pstream << psRed << " " << psGreen << " " << psBlue
		 << " setrgbcolor\n";
	current_red   = red;
	current_blue  = blue;
	current_green = green;
    } else {
// 	float gray = (.3*psRed) + (.59*psGreen) + (.11*psBlue);
// 	*pstream << gray << " setgray\n";
	*pstream << 0.0 << " setgray\n";
    }
}

//-----------------------------------------------------------------------------
// text and font methods
//-----------------------------------------------------------------------------

void wxPostScriptDC::DrawText(const char *text, float x, float y,
			      Bool WXUNUSED(use16Bit))
{
    if (!pstream || !text)
	return;

    int len  = strlen(text);
    int size = 10;
    if (current_font) {
	SetFont(current_font);
	size = current_font->GetPointSize();
    }

    CalcBoundingBox(_XLOG2DEV(x),              _YLOG2DEV(y));
    CalcBoundingBox(_XLOG2DEV(x+size*len*2/3), _YLOG2DEV(y+size));

    if (current_text_fg.Ok()) {
	// Text colour
	float red   = current_text_fg.Red();
	float blue  = current_text_fg.Blue();
	float green = current_text_fg.Green();
	// set colour or grey value
	float psRed   = (float)(red / 255.0);	// convert to 0 to 1.0
	float psGreen = (float)(green / 255.0);	// convert to 0 to 1.0
	float psBlue  = (float)(blue / 255.0);	// convert to 0 to 1.0
	if (Colour) {
	    *pstream << psRed << " " << psGreen << " " << psBlue
		     << " setrgbcolor\n";
	    current_red   = red;
	    current_blue  = blue;
	    current_green = green;
	} else {
// 	    float gray = (.3*psRed) + (.59*psGreen) + (.11*psBlue);
// 	    *pstream << gray << " setgray\n";
	    *pstream << 0.0 << " setgray\n";
	}
    }
    float by = y + floor( float(size) * 2.0 / 3.0 ); // approximate baseline
    *pstream << _XLOG2DEV(x) << " " << _YLOG2DEV(by) << " moveto\n";

//     cerr << "DrawText(" << text << "," << x << "," << y << ") to : ("
// 	 << _XLOG2DEV(x) << "," << _YLOG2DEV(by) << ")" << endl
// 	 << "\torigin = (" << origin_x <<"," << origin_y << ")" << endl
// 	 << "\tscale  = (" << scale_x <<"," << scale_y << ")" << endl;

    *pstream << "(";
    for (int i = 0; i < len; i++) {
	int c = (unsigned char) text[i];
	if ( c == ')' || c == '(' || c == '\\') {
	    *pstream << "\\" << (char) c;}
	else if ( c >= 128 ) {
	    // Cope with character codes > 127
	    char tmp[5];
	    sprintf(tmp, "\\%o", c);
	    *pstream << tmp;
	} else
	    *pstream << (char) c;
    }
    *pstream << ")" << " show\n";

    float uy = y + size - underline_position;
    if (current_font->GetUnderlined() && CanGetTextExtent()) {
	float w, h;
	GetTextExtent(text, &w, &h);
	*pstream << "gsave " << _XLOG2DEV(x) << " "      << _YLOG2DEV(uy)
		 << " moveto\n"
		 << underline_thickness << " setlinewidth "
		 << _XLOG2DEV(x+w) << " " << _YLOG2DEV(uy)
		 << " lineto stroke grestore\n";
   }
}

float wxPostScriptDC::GetCharHeight(void)
{
    if (current_font)
	return (float)current_font->GetPointSize();
    return 12.0;
}

float wxPostScriptDC::GetCharWidth(void)
{
    if (current_font)
	return (float)current_font->GetPointSize();
    return 12.0;
}

void wxPostScriptDC::GetTextExtent (const char *string, float *x, float *y,
				    float *descent, float *externalLeading,
				    wxFont *theFont, Bool WXUNUSED(use16))
{
    if (string == NULL) {
	*x = *y = 0.0;
	return;
    }

  wxFont *fontToUse = theFont;
  if (!fontToUse)
    fontToUse = current_font;
  
  if (!fontToUse) {
    wxError("set a font before calling GetTextExtent", "wxPostScriptDC");
    *x = *y = 1.0;
    if (descent)
      *descent = 0.0;
    if (externalLeading)
      *externalLeading = 0.0;
    return;
  }
  
  if (!pstream)
    return;
#if !USE_AFM_FOR_POSTSCRIPT
  // Provide a VERY rough estimate (avoid using it)
  int width = 12;
  int height = 12;

  if (fontToUse)
    {
      height = fontToUse->GetPointSize ();
      width = height;
    }
  *x = (float) strlen (string) * width;
  *y = (float) height;
  if (descent)
    *descent = 0.0;
  if (externalLeading)
    *externalLeading = 0.0;
#else
  // +++++ start of contributed code +++++
  
  // ************************************************************
  // method for calculating string widths in postscript:
  // read in the AFM (adobe font metrics) file for the
  // actual font, parse it and extract the character widths
  // and also the descender. this may be improved, but for now
  // it works well. the AFM file is only read in if the
  // font is changed. this may be chached in the future.
  // calls to GetTextExtent with the font unchanged are rather
  // efficient!!!
  //
  // for each font and style used there is an AFM file necessary.
  // currently i have only files for the roman font family.
  // i try to get files for the other ones!
  //
  // CAVE: the size of the string is currently always calculated
  //       in 'points' (1/72 of an inch). this should later on be
  //       changed to depend on the mapping mode.
  // CAVE: the path to the AFM files must be set before calling this
  //       function. this is usually done by a call like the following:
  //       wxSetAFMPath("d:\\wxw161\\afm\\");
  //
  // example:
  //
  //    wxPostScriptDC dc(NULL, TRUE);
  //    if (dc.Ok()){
  //      wxSetAFMPath("d:\\wxw161\\afm\\");
  //      dc.StartDoc("Test");
  //      dc.StartPage();
  //      float w,h;
  //      dc.SetFont(new wxFont(10, wxROMAN, wxNORMAL, wxNORMAL));
  //      dc.GetTextExtent("Hallo",&w,&h);
  //      dc.EndPage();
  //      dc.EndDoc();
  //    }
  //
  // by steve (stefan.hammes@urz.uni-heidelberg.de)
  // created: 10.09.94
  // updated: 14.05.95

  assert(fontToUse && "void wxPostScriptDC::GetTextExtent: no font defined");
  assert(x && "void wxPostScriptDC::GetTextExtent: x == NULL");
  assert(y && "void wxPostScriptDC::GetTextExtent: y == NULL");

  // these static vars are for storing the state between calls
  static int lastFamily= INT_MIN;
  static int lastSize= INT_MIN;
  static int lastStyle= INT_MIN;
  static int lastWeight= INT_MIN;
  static int lastDescender = INT_MIN;
  static int lastWidths[256]; // widths of the characters

  // get actual parameters
  const int Family = fontToUse->GetFamily();
  const int Size =   fontToUse->GetPointSize();
  const int Style =  fontToUse->GetStyle();
  const int Weight = fontToUse->GetWeight();

  // if we have another font, read the font-metrics
  if(Family!=lastFamily||Size!=lastSize||Style!=lastStyle||Weight!=lastWeight){
    // store actual values
    lastFamily = Family;
    lastSize =   Size;
    lastStyle =  Style;
    lastWeight = Weight;

    // read in new font metrics **************************************

    // 1. construct filename ******************************************
    /* MATTHEW: [2] Use wxTheFontNameDirectory */
    char *name;

    name = wxTheFontNameDirectory->GetAFMName(Family, Weight, Style);
    if (!name)
      name = "unknown";

    // get the directory of the AFM files
    char afmName[256];
    afmName[0] = 0;
    if (wxGetAFMPath())
      strcpy(afmName,wxGetAFMPath());

    // 2. open and process the file **********************************

    // a short explanation of the AFM format:
    // we have for each character a line, which gives its size
    // e.g.:
    //
    //   C 63 ; WX 444 ; N question ; B 49 -14 395 676 ;
    //
    // that means, we have a character with ascii code 63, and width 
    // (444/1000 * fontSize) points.
    // the other data is ignored for now!
    //
    // when the font has changed, we read in the right AFM file and store the
    // character widths in an array, which is processed below (see point 3.).
    
        // new elements JC Sun Aug 25 23:21:44 MET DST 1996

    
    strcat(afmName,name);
    strcat(afmName,".afm");
    FILE *afmFile = fopen(afmName,"r");
    if(afmFile==NULL){
      wxDebugMsg("GetTextExtent: can't open AFM file '%s'\n",afmName);
      wxDebugMsg("               using approximate values\n");
      int i;
      for (i=0; i<256; i++) lastWidths[i] = 500; // an approximate value
      lastDescender = -150; // dito.
    }else{
      int i;
      // init the widths array
      for(i=0; i<256; i++) lastWidths[i]= INT_MIN;
      // some variables for holding parts of a line
      char cString[10],semiString[10],WXString[10],descString[20];
      char upString[30], utString[30], encString[50];
      char line[256];
      int ascii,cWidth;
      // read in the file and parse it
      while(fgets(line,sizeof(line),afmFile)!=NULL){
        // A.) check for descender definition
        if(strncmp(line,"Descender",9)==0){
          if((sscanf(line,"%s%d",descString,&lastDescender)!=2)
	     || (strcmp(descString,"Descender")!=0)) {
	    wxDebugMsg("AFM-file '%s': line '%s' has error (bad descender)\n",
		       afmName,line);
          }
        }
            // JC 1.) check for UnderlinePosition
        else if(strncmp(line,"UnderlinePosition",17)==0){
          if((sscanf(line,"%s%f",upString,&underline_position)!=2)
	     || (strcmp(upString,"UnderlinePosition")!=0)) {
	    wxDebugMsg("AFM-file '%s': line '%s' has error (bad UnderlinePosition)\n",
		       afmName,line);
          }
        }
	// JC 2.) check for UnderlineThickness
        else if(strncmp(line,"UnderlineThickness",18)==0){
          if((sscanf(line,"%s%f",utString,&underline_thickness)!=2)
	     || (strcmp(utString,"UnderlineThickness")!=0)) {
	    wxDebugMsg("AFM-file '%s': line '%s' has error (bad UnderlineThickness)\n",
		       afmName,line);
          }
        }
	// JC 3.) check for EncodingScheme
        else if(strncmp(line,"EncodingScheme",14)==0){
          if((sscanf(line,"%s%s",utString,encString)!=2)
	     || (strcmp(utString,"EncodingScheme")!=0)) {
	    wxDebugMsg("AFM-file '%s': line '%s' has error (bad EncodingScheme)\n",
		       afmName,line);
          }
          else if (strncmp(encString, "AdobeStandardEncoding", 21))
          {
	    wxDebugMsg("AFM-file '%s': line '%s' has error (unsupported EncodingScheme %s)\n",
		       afmName,line, encString);
          }
        }
            // B.) check for char-width
        else if(strncmp(line,"C ",2)==0){
          if(sscanf(line,"%s%d%s%s%d",
              cString,&ascii,semiString,WXString,&cWidth)!=5){
             wxDebugMsg("AFM-file '%s': line '%s' has an error (bad character width)\n",afmName,line);
          }
          if(strcmp(cString,"C")!=0 || strcmp(semiString,";")!=0 ||
             strcmp(WXString,"WX")!=0){
             wxDebugMsg("AFM-file '%s': line '%s' has a format error\n",afmName,line);
          }
          //printf("            char '%c'=%d has width '%d'\n",ascii,ascii,cWidth);
          if(ascii>=0 && ascii<256){
            lastWidths[ascii] = cWidth; // store width
          }else{
	    /* MATTHEW: this happens a lot; don't print an error */
            // wxDebugMsg("AFM-file '%s': ASCII value %d out of range\n",afmName,ascii);
          }
        }
        // C.) ignore other entries.
      }
      fclose(afmFile);
    }
        // hack to compute correct values for german 'Umlaute'
        // the correct way would be to map the character names
        // like 'adieresis' to corresp. positions of ISOEnc and read
        // these values from AFM files, too. Maybe later ...
    lastWidths[196] = lastWidths['A'];  // 
    lastWidths[228] = lastWidths['a'];  // 
    lastWidths[214] = lastWidths['O'];  // 
    lastWidths[246] = lastWidths['o'];  // 
    lastWidths[220] = lastWidths['U'];  // 
    lastWidths[252] = lastWidths['u'];  // 
    lastWidths[223] = lastWidths[251];  // 
  }

      // JC: calculate UnderlineThickness/UnderlinePosition
  underline_position  = underline_position * fontToUse->GetPointSize() / 1000.0f;
  underline_thickness = underline_thickness * fontToUse->GetPointSize() / 1000.0f;
  
  // 3. now the font metrics are read in, calc size *******************
  // this is done by adding the widths of the characters in the
  // string. they are given in 1/1000 of the size!

  float widthSum=0.0;
  float height=(float)Size; // by default
  unsigned char *p;
  for(p=(unsigned char *)string; *p; p++){
    if(lastWidths[*p]== INT_MIN){
      wxDebugMsg("GetTextExtent: undefined width for character '%c' (%d)\n",
                 *p,*p);
      widthSum += lastWidths[' ']/1000.0F * Size; // assume space
    }else{
      widthSum += (lastWidths[*p]/1000.0F)*Size;
    }
  }
  // add descender to height (it is usually a negative value)
  if(lastDescender!=INT_MIN){
    height += ((-lastDescender)/1000.0F) * Size; /* MATTHEW: forgot scale */
  }
  
  // return size values
  *x = widthSum;
  *y = height;

  // return other parameters
  if (descent){
    if(lastDescender!=INT_MIN){
      *descent = ((-lastDescender)/1000.0F) * Size; /* MATTHEW: forgot scale */
    }else{
      *descent = 0.0;
    }
  }

  // currently no idea how to calculate this!
  // if (externalLeading) *externalLeading = 0.0;
  if (externalLeading)
    *externalLeading = 0.0;

  // ----- end of contributed code -----
#endif
}

void wxPostScriptDC::SetFont(wxFont *font)
{
    if (!pstream || !(current_font = font))
	return;
    char *name = wxTheFontNameDirectory->GetPostScriptName(font->GetFamily(),
							   font->GetWeight(),
							   font->GetStyle());
    if (!name)
	name = "Times-Roman";

    *pstream << "/" << name << " reencodeISO def\n"
	     << "/" << name << " findfont\n"
	     << (do_gdi_scaling ? _YLOG2DEVREL(font->GetPointSize())
				: font->GetPointSize())
	     << " scalefont setfont\n";
}

void wxPostScriptDC::SetTextForeground(wxColour *col)
{
    if (col)
	current_text_fg = *col;
}

void wxPostScriptDC::SetTextBackground(wxColour *col)
{
    if (col)
	current_text_bg = *col;
}

//-----------------------------------------------------------------------------
// clipping region
//-----------------------------------------------------------------------------

void wxPostScriptDC::GetClippingRegion(float *x, float *y, float *w, float *h)
{
    *x = *y = 0;
    *w = *h = -1;
}

void wxPostScriptDC::SetClippingRegion(float x, float y, float w, float h)
{
    if (!pstream || clipping)
	return;

    clipping = TRUE;
    *pstream << "gsave\n"
	     << "newpath\n"
	     << _XLOG2DEV(x)   << " " << _YLOG2DEV(y)   << " moveto\n"
	     << _XLOG2DEV(x+w) << " " << _YLOG2DEV(y)   << " lineto\n"
	     << _XLOG2DEV(x+w) << " " << _YLOG2DEV(y+h) << " lineto\n"
	     << _XLOG2DEV(x)   << " " << _YLOG2DEV(y+h) << " lineto\n"
	     << "closepath clip newpath\n";
}

void wxPostScriptDC::DestroyClippingRegion(void)
{
    if (!pstream || !clipping)
	return;

    clipping = FALSE;
    *pstream << "grestore\n";
}

//-----------------------------------------------------------------------------
// ask, if a part of the DC is exposed
//-----------------------------------------------------------------------------

int wxPostScriptDC::IsExposed(float WXUNUSED(x), float WXUNUSED(y))
{
    return wxEXPOSED_FULL; // has always to be redrawn completely
}

int wxPostScriptDC::IsExposed(float WXUNUSED(x), float WXUNUSED(y),
			      float WXUNUSED(w), float WXUNUSED(h))
{
    return wxEXPOSED_FULL; // has always to be redrawn completely
}

//-----------------------------------------------------------------------------
// methods unique to wxPostScriptDC
//-----------------------------------------------------------------------------

void wxPostScriptDC::GetSize(float *width, float *height)
{
    char *paperType = wxThePrintSetupData->GetPaperName();

    if (!paperType)
	paperType = "A4 210 x 297 mm";
    wxPrintPaperType *paper = wxThePrintPaperDatabase->FindPaperType(paperType);
    if (paper) {
	*width  = (float)paper->widthPixels;
	*height = (float)paper->heightPixels;
    } else {
	*width  = 595;
	*height = 842;
    }

    // swap if landscape printing
    if (wxThePrintSetupData->GetPrinterOrientation()==PS_LANDSCAPE) {
	float help = *width; *width = *height; *height = help;
    }
}

void wxPostScriptDC::GetSizeMM(float *width, float *height)
{
    char *paperType = wxThePrintSetupData->GetPaperName();

    if (!paperType)
	paperType = "A4 210 x 297 mm";
    wxPrintPaperType *paper = wxThePrintPaperDatabase->FindPaperType(paperType);
    if (paper) {
	*width  = (float)paper->widthMM;
	*height = (float)paper->heightMM;
    } else {
	*width  = 210;
	*height = 297;
    }
    // swap if landscape printing
    if (wxThePrintSetupData->GetPrinterOrientation()==PS_LANDSCAPE) {
	float help = *width; *width = *height; *height = help;
    }
}

//-----------------------------------------------------------------------------
// scale and origin methods
//-----------------------------------------------------------------------------

void wxPostScriptDC::SetAxisOrientation(Bool x_left_right, Bool y_bottom_up)
{
    sign_x = (x_left_right ? 1.0 : -1.0);
    sign_y = (y_bottom_up  ? 1.0 : -1.0);
    ComputeScaleAndOrigin();
}

void wxPostScriptDC::SetDeviceOrigin(float x, float y)
{
    wxDC::SetDeviceOrigin(x, y);
}

void wxPostScriptDC::StartUnzoomedDrawing(float x, float y)
{
    save_origin_x = origin_x;
    save_origin_y = origin_y;
    save_scale_x  = scale_x;
    save_scale_y  = scale_y;
    save_sign_x   = sign_x;
    save_sign_y   = sign_y;
    // set temporary scale and origin
    origin_x = -XLOG2DEV(x);
    origin_y = +YLOG2DEV(y);
    scale_x  = 1;
    scale_y  = 1;
    sign_x   = 1;
    sign_y   = -1;
}

//-----------------------------------------------------------------------------
// wxPostScriptDC::DrawOpenSpline(wxList *pts)
//-----------------------------------------------------------------------------

#if USE_SPLINES

void wxPostScriptDC::DrawOpenSpline(wxList *pts)
{
    wxNode  *node;
    double  a, b, c, d, x1, y1, x2, y2, x3, y3;
    wxPoint *p, *q;
    Bool    draw = FALSE;

    if (current_pen && current_pen->GetStyle () != wxTRANSPARENT) {
	SetPen (current_pen);
	draw = TRUE;
    }

    node = pts->First(); p = (wxPoint*)node->Data();
    x1 = p->x; y1 = p->y;
    node = node->Next(); p = (wxPoint *)node->Data();
    c = p->x; d = p->y;
    x3 = a = (float)(x1 + c) / 2;
    y3 = b = (float)(y1 + d) / 2;

    if (draw)
	*pstream << "newpath "
		 << _XLOG2DEV(x1) << " " << _YLOG2DEV(y1) << " moveto "
		 << _XLOG2DEV(x3) << " " << _YLOG2DEV(y3) << " lineto\n";
    CalcBoundingBox(_XLOG2DEV(x1), _YLOG2DEV(y1));
    CalcBoundingBox(_XLOG2DEV(x3), _YLOG2DEV(y3));

    while ((node = node->Next()) != NULL) {
	q = (wxPoint *)node->Data();

	x1 = x3;   y1 = y3;
	x2 = c;    y2 = d;
	c  = q->x; d  = q->y;
	x3 = (float)(x2 + c) / 2;
	y3 = (float)(y2 + d) / 2;

	if (draw)
	    *pstream << _XLOG2DEV(x1) << " " << _YLOG2DEV(y1) << " "
		     << _XLOG2DEV(x2) << " " << _YLOG2DEV(y2) << " "
		     << _XLOG2DEV(x3) << " " << _YLOG2DEV(y3) << " DrawSplineSection\n";
	
	CalcBoundingBox(_XLOG2DEV(x1), _YLOG2DEV(y1));
	CalcBoundingBox(_XLOG2DEV(x3), _YLOG2DEV(y3));
    }
    /*
     * At this point, (x2,y2) and (c,d) are the position of the 
     * next-to-last and last point respectively, in the point list
     */
    *(pstream) << _XLOG2DEV(c) << " " << _YLOG2DEV(d) << " lineto stroke\n";
}

#endif
