/*
 * Copyright (c) 1997 Stanford University
 *
 * The use of this software for revenue-generating purposes may require a
 * license from the owners of the underlying intellectual property.
 * Specifically, the SRP protocol may not be used for revenue-generating
 * purposes without a license.
 *
 * Within that constraint, permission to use, copy, modify, and distribute
 * this software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notices and this permission
 * notice appear in all copies of the software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <pwd.h>
#include <sys/stat.h>

#include "t_defines.h"
#include "t_pwd.h"
#include "t_read.h"
#include "t_sha.h"

struct t_pw *
t_openpw(fp)
     FILE * fp;
{
  struct t_pw * tpw;
  char close_flag = 0;

  if(fp == NULL) {
    if((fp = fopen(DEFAULT_PASSWD, "r")) == NULL)
      return NULL;
    close_flag = 1;
  }
  else
    close_flag = 0;

  if((tpw = malloc(sizeof(struct t_pw))) == NULL)
    return NULL;
  tpw->instream = fp;
  tpw->close_on_exit = close_flag;

  return tpw;
}

struct t_pw *
t_openpwbyname(pwname)
     t_const char * pwname;
{
  FILE * fp;
  struct t_pw * t;

  if(pwname == NULL)
    return t_openpw(NULL);

  if((fp = fopen(pwname, "r")) == NULL)
    return NULL;

  t = t_openpw(fp);
  t->close_on_exit = 1;
  return t;
}

void
t_closepw(tpw)
     struct t_pw * tpw;
{
  if(tpw->close_on_exit)
    fclose(tpw->instream);
  free(tpw);
}

void
t_rewindpw(tpw)
     struct t_pw * tpw;
{
  rewind(tpw->instream);
}

struct t_pwent *
t_getpwent(tpw)
     struct t_pw * tpw;
{
  char indexbuf[16];
  char passbuf[MAXB64PARAMLEN];
  char saltstr[MAXB64SALTLEN];

  while(1) {
    if(t_nextfield(tpw->instream, tpw->userbuf, MAXUSERLEN) > 0 &&
       t_nextfield(tpw->instream, passbuf, MAXB64PARAMLEN) > 0 &&
       (tpw->pebuf.password.len = t_fromb64(tpw->pwbuf, passbuf)) > 0 &&
       t_nextfield(tpw->instream, saltstr, MAXB64SALTLEN) > 0 &&
       (tpw->pebuf.salt.len = t_fromb64(tpw->saltbuf, saltstr)) > 0 &&
       t_nextfield(tpw->instream, indexbuf, 16) > 0 &&
       (tpw->pebuf.index = atoi(indexbuf)) > 0) {
      tpw->pebuf.name = tpw->userbuf;
      tpw->pebuf.password.data = tpw->pwbuf;
      tpw->pebuf.salt.data = tpw->saltbuf;
      t_nextline(tpw->instream);
      return &tpw->pebuf;
    }
    else if(t_nextline(tpw->instream) < 0)
      return NULL;
  }
}

struct t_pwent *
t_getpwbyname(tpw, user)
     struct t_pw * tpw;
     t_const char * user;
{
  char indexbuf[16];
  char passbuf[MAXB64PARAMLEN];
  char saltstr[MAXB64SALTLEN];
  char username[MAXUSERLEN];

  t_rewindpw(tpw);

  while(t_nextfield(tpw->instream, username, MAXUSERLEN) > 0) {
    if(strcmp(user, username) == 0)
      if(t_nextfield(tpw->instream, passbuf, MAXB64PARAMLEN) > 0 &&
	 (tpw->pebuf.password.len = t_fromb64(tpw->pwbuf, passbuf)) > 0 &&
	 t_nextfield(tpw->instream, saltstr, MAXB64SALTLEN) > 0 &&
	 (tpw->pebuf.salt.len = t_fromb64(tpw->saltbuf, saltstr)) > 0 &&
	 t_nextfield(tpw->instream, indexbuf, 16) > 0 &&
	 (tpw->pebuf.index = atoi(indexbuf)) > 0) {
	strcpy(tpw->userbuf, username);
	tpw->pebuf.name = tpw->userbuf;
	tpw->pebuf.password.data = tpw->pwbuf;
	tpw->pebuf.salt.data = tpw->saltbuf;
	t_nextline(tpw->instream);
	return &tpw->pebuf;
      }
    if(t_nextline(tpw->instream) < 0)
      return NULL;
  }
  return NULL;
}

struct t_pwent *
t_makepwent(tpw, user, pass, salt, confent)
     struct t_pw * tpw;
     t_const char * user;
     t_const char * pass;
     t_const struct t_num * salt;
     t_const struct t_confent * confent;
{
  BigInteger x, v, n, g;
  char hexbuf[MAXHEXPARAMLEN];
  unsigned char dig[SHA_DIGESTSIZE];
  SHA1_CTX ctxt;

  tpw->pebuf.name = tpw->userbuf;
  tpw->pebuf.password.data = tpw->pwbuf;
  tpw->pebuf.salt.data = tpw->saltbuf;

  strncpy(tpw->pebuf.name, user, MAXUSERLEN);
  tpw->pebuf.index = confent->index;

  if(salt) {
    tpw->pebuf.salt.len = salt->len;
    memcpy(tpw->pebuf.salt.data, salt->data, salt->len);
  }
  else {
    memset(dig, 0, SALTLEN);		/* salt is 80 bits */
    tpw->pebuf.salt.len = SALTLEN;
    do {
      t_random(tpw->pebuf.salt.data, SALTLEN);
    } while(memcmp(tpw->pebuf.salt.data, dig, SALTLEN) == 0);
    if(tpw->pebuf.salt.data[0] == 0)
      tpw->pebuf.salt.data[0] = 0xff;
  }

  n = BigIntegerFromBytes(confent->modulus.data, confent->modulus.len);
  g = BigIntegerFromBytes(confent->generator.data, confent->generator.len);
  v = BigIntegerFromInt(0);

  SHA1Init(&ctxt);
  SHA1Update(&ctxt, user, strlen(user));
  SHA1Update(&ctxt, ":", 1);
  SHA1Update(&ctxt, pass, strlen(pass));
  SHA1Final(dig, &ctxt);

  SHA1Init(&ctxt);
  SHA1Update(&ctxt, tpw->pebuf.salt.data, tpw->pebuf.salt.len);
  SHA1Update(&ctxt, dig, sizeof(dig));
  SHA1Final(dig, &ctxt);

  /* x = H(s, H(u, ':', p)) */
  x = BigIntegerFromBytes(dig, sizeof(dig));

  BigIntegerModExp(v, g, x, n);
  tpw->pebuf.password.len = BigIntegerToBytes(v, tpw->pebuf.password.data);

  BigIntegerFree(v);
  BigIntegerFree(x);
  BigIntegerFree(g);
  BigIntegerFree(n);

  return &tpw->pebuf;
}

void
t_putpwent(ent, fp)
     t_const struct t_pwent * ent;
     FILE * fp;
{
  char strbuf[MAXB64PARAMLEN];
  char saltbuf[MAXB64SALTLEN];

  fprintf(fp, "%s:%s:%s:%d\n", ent->name,
	  t_tob64(strbuf, ent->password.data, ent->password.len),
	  t_tob64(saltbuf, ent->salt.data, ent->salt.len), ent->index);
}

static int
t_pwcopy(pwdest, pwsrc, diff)
     FILE * pwdest;
     FILE * pwsrc;
     struct t_pwent * diff;
{
  struct t_pw * src;
  struct t_pwent * ent;

  if((src = t_openpw(pwsrc)) == NULL)
    return -1;

  while((ent = t_getpwent(src)) != NULL)
    if(diff && strcmp(diff->name, ent->name) == 0) {
      t_putpwent(diff, pwdest);
      diff = NULL;
    }
    else
      t_putpwent(ent, pwdest);

  if(diff)
    t_putpwent(diff, pwdest);
}

int
t_changepw(pwname, diff)
     t_const char * pwname;
     t_const struct t_pwent * diff;
{
  char bakfile[256], bakfile2[256];
  struct stat st;
  FILE * passfp;
  FILE * bakfp;

  if(pwname == NULL)
    pwname = DEFAULT_PASSWD;
  sprintf(bakfile, "%s.bak", pwname);
  sprintf(bakfile2, "%s.sav", pwname);

  if((passfp = fopen(pwname, "r")) == NULL || fstat(fileno(passfp), &st) < 0)
    return -1;

  if((bakfp = fopen(bakfile2, "w")) == NULL &&
    (unlink(bakfile2) < 0 || (bakfp = fopen(bakfile2, "w")) == NULL))
    return -1;

  fchmod(fileno(bakfp), st.st_mode & 0777);

  t_pwcopy(bakfp, passfp, diff);

  fclose(bakfp);
  fclose(passfp);

  unlink(bakfile);
  link(pwname, bakfile);
  unlink(pwname);
  link(bakfile2, pwname);
  unlink(bakfile2);

  return 0;
}

int
t_verifypw(user, pass)
     t_const char * user;
     t_const char * pass;
{
  struct t_pw * tpw;
  struct t_pwent * ent;
  struct t_conf * tconf;
  struct t_confent * tcent;
  struct passwd * syspwent;
  struct t_pw temp_pw;
  struct t_pwent * testent;
  char pathname[256];
  int rval;

  tpw = t_openpw(NULL);
  if(tpw == NULL || (ent = t_getpwbyname(tpw, user)) == NULL) {
    if(tpw != NULL)
      t_closepw(tpw);

#ifdef USE_HOMEDIR
    syspwent = getpwnam(user);		/* Find user's homedir */
    if(syspwent == NULL)
      return -1;
    sprintf(pathname, "%s/.tpasswd", syspwent->pw_dir);
    tpw = t_openpwbyname(pathname);
    if(tpw == NULL)
      return -1;
    
    strcat(pathname, ".conf");		/* Look up entries */
    if((ent = t_getpwbyname(tpw, user)) == NULL ||
       (tconf = t_openconfbyname(pathname)) == NULL) {
      t_closepw(tpw);
      return -1;
    }
#else
    return -1;
#endif
  }
  else {
    if((tconf = t_openconf(NULL)) == NULL) {
      t_closepw(tpw);
      return -1;
    }
  }

  if((tcent = t_getconfbyindex(tconf, ent->index)) == NULL) {
    t_closepw(tpw);
    t_closeconf(tconf);
    return -1;
  }

  testent = t_makepwent(&temp_pw, user, pass, &ent->salt, tcent);

  if(ent->password.len == testent->password.len &&
     memcmp(ent->password.data, testent->password.data, ent->password.len) == 0)
    rval = 1;
  else
    rval = 0;

  t_closepw(tpw);
  t_closeconf(tconf);
  return rval;
}
