"""Code to start and maintain empire connection"""

#    Copyright (C) 1998 Kevin O'Connor
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys
import socket
import select
import string
import re
import traceback

import empDb

# Key Ideas:

# What is contained within this file:

# This file contains the underlying socket code.  Code to handle all the
# incoming data, and broker all the outgoing data is contained within the
# class EmpIOQueue.  EmpIOQueue uses a number of other classes to handle
# the incoming/outgoing data.  This file also contains these other classes,
# and contains the basic building block for the chained display classes.  A
# quick glance at the other files in this distribution will show that no
# other file references any of the low-level empire protocols
# (EG. C_PROMPT) - all that protocol stuff is handled here.

# The purpose of the low-level data managers:

# The "data managers" are helper classes for EmpIOQueue.  These helper
# classes, essentially, manage output from the server.  They strip the
# low-level empire protocol stuff from the information, and transmit the
# data using a variety of classes and methods.  The classes AsyncHandler,
# LoginHandler, NormalHandler, and DummyHandler, are these "data managers".
# It is important to note that AsyncHandler and LoginHandler have special
# properties.  Both of these classes will normally have exactly one
# instance associated with them.  These single instances are associated
# with EmpIOQueue at its instantiation, and are stored in the variables
# defParse and loginParser.  To the contrary, the NormalHandler and
# DummyHandler classes will generally have multiple instances associated
# with them - there will be one instance per queued command.

# The purpose of the chained display classes:

# Associated with many of the server commands will be client parsers.  (For
# example, the server command "read" will be transmitted to the class
# empDb.ParseRead.)  These classes are referred to as "chained" because
# they generally form a linked-list of parsers.  More than one parser might
# have interest in the server output of any given command, so a chain is
# formed where each low-level parser transmits the information to the next
# parser.  The chaining is performed by the individual parsers - thus, a
# lower level parser might decide to not send information up the chain,
# thus hiding info.  (Which, BTW, is the basis for the "null" command -
# "null" just associates the given command with a parser that discards its
# input.)  At the highest level of this parse chain, is generally the main
# viewer.  The global variable viewer, must support the same methods of an
# ordinary display class.  (In addition to its special methods - see
# AsyncHandler/LoginHandler for more info.)

# The use of the Queue Flags:

# The queue flags describe the current state of the command queue.  They
# determine if the connection is open/closed/paused/etc..  The more
# interesting flags, however, are the per-command transmission flags.
# These flags (QU_BURST, QU_SYNC, and QU_FULLSYNC) determine how each
# command is transmitted to the server.  With QU_BURST, the command is
# transmitted immediately; no checks are made to determine if the command
# will be sent to a sub-prompt or a command line.  (Note: This doesn't work
# yet.)  With QU_FULLSYNC, the command is not sent until the previous
# command is fully processed, completed, and dequeued.  With QU_SYNC (the
# most interesting and complex mode), the command is guaranteed to be sent
# to a command-line, but can be sent prior to the completed processing of
# the previous command.  This is done by forcing the socket reader to
# pre-scan all the server data.  When a C_PROMPT protocol is detected, the
# socket reader will transmit the next command immediately.  This
# pre-scanning of the server data can significantly reduce the connection's
# latency.  Certain commands can cause a Tk redraw, which may take several
# seconds - by sending the next command prior to trudging through the
# chained display classes, it is possible to negotiate with the server just
# prior to performing significant client work.


# Empire Protocol IDs
C_CMDOK	 = "0"
C_DATA	 = "1"
C_INIT	 = "2"
C_EXIT	 = "3"
C_FLUSH	 = "4"
C_NOECHO = "5"
C_PROMPT = "6"
C_ABORT	 = "7"
C_REDIR	 = "8"
C_PIPE	 = "9"
C_CMDERR = "a"
C_BADCMD = "b"
C_EXECUTE= "c"
C_FLASH	 = "d"
C_INFORM = "e"
C_LAST	 = "e"

# Queue flags
QU_BURST = 0
QU_SYNC = 1
QU_FULLSYNC = 3

QU_DISCONNECT = 8
QU_OFFLINE = 9
QU_CONNECTING = 10

class EmpIOQueue:
    """Broker all input and output to/from server.

    This class generally has only one instance associated with it.  It is
    used mainly as a code/data container.
    """

    def __init__(self, async, login):
	global empQueue
	empQueue = self

	# Storage area for partial socket reads.  (Reads that contain a
	# string that is not terminated by a newline.)  This info has to be
	# stored somewhere until the next read reveals the trailing
	# information.
	self.InpBuf = ""

	# Storage for the command queue.  Each command is queued by
	# associating it with a "data manager" class
	# (NormalHandler/DummyHandler), and placing it in the queue stored
	# in FuncList.  The integers FLWaitLev and FLSentLev are indexes
	# into this queue.  FuncList[FLWaitLev] points to the command that
	# is currently receiving data.  FuncList[FLSentLev] points to the
	# next command that needs to be sent to the server.
	self.FuncList = []
	self.FLWaitLev = self.FLSentLev = 0

	# Special data manager classes.  defParser (which is associated
	# with the class AsyncHandler) is used for all asynchronous data -
	# data that is received when there are exactly zero commands
	# waiting on the queue.  loginParser (which is associated with the
	# class LoginHandler) is used when the queue needs to reestablish a
	# server connection.
	self.defParser = async
	self.loginParser = login

	self.flags = QU_OFFLINE

    def SendNow(self, cmd):
	"""Immediately send CMD to socket."""
	# Paranoia check
	if string.find(cmd, "\n") != -1:
	    # Ugh, there should be no newlines in the cmd
	    viewer.flash("Send error - embedded newline: " + cmd)
	    cmd = cmd[:string.find(cmd, "\n")]
##	print "## " + cmd
	try:
	    self.socket.send(cmd+"\n")
	except socket.error, e:
	    self.loginParser.Disconnect()
	    viewer.flash("Socket write error: " + str(e))

    def GetStatusMsg(self):
	"""Return a string that describes the current queue."""
	if self.flags >= QU_DISCONNECT:
	    msg = {QU_DISCONNECT:'Disconnected',
		   QU_OFFLINE:'Off-line',
		   QU_CONNECTING:'Connecting',
		   }[self.flags]
	else:
	    tot = len(self.FuncList)
	    sent = self.FLSentLev
	    msg = ""
	    if tot != 0: msg = msg + str(tot)
	    if sent != 0: msg = msg + "/" + str(sent)
	    if msg == "": msg = "Idle"
	return msg

    def doFlags(self):
	"""Set the queue flags depending on the current state of the queue."""
	if self.flags >= QU_DISCONNECT:
	    return
	ls = self.FLSentLev
##	lt = len(self.FuncList)
##	if lt == 0:
##	    self.flags = QU_BURST
##	elif lt == ls:
##	    self.flags = self.FuncList[ls-1].postFlags
	if len(self.FuncList) == ls or ls == 0:
	    self.flags = QU_BURST
	else:
	    self.flags = (self.FuncList[ls-1].postFlags
			  | self.FuncList[ls].preFlags)

    def sendCommand(self):
	"""Pop a command from the queue.  (Sending it along to the server.)"""
	while 1:
	    if self.FLSentLev >= len(self.FuncList):
		break
	    cmd = self.FuncList[self.FLSentLev].command
##	    if callable(cmd):
##		cmd = cmd()
	    self.FLSentLev = self.FLSentLev + 1
	    if self.FLSentLev == 1:
##		print "force begin"
		try: self.FuncList[0].start()
		except: flashException()
		if cmd == None:
		    del self.FuncList[:1]
		    self.FLSentLev = self.FLSentLev - 1
		    self.FLWaitLev = self.FLWaitLev - 1
	    if cmd == None:
		self.FLWaitLev = self.FLWaitLev + 1
	    else:
		self.SendNow(cmd)
	    self.doFlags()
	    if self.flags != QU_BURST:
		break
##	print "leave send @ " + str(
##	    (len(self.FuncList), self.FLWaitLev, self.FLSentLev)), self.flags

    def AddHandler(self, handler):
	"""Queue command for sequential execution."""
##	print "add " + str(handler.command) + " @ " + str(
##	    (len(self.FuncList), self.FLWaitLev, self.FLSentLev)), self.flags
	if self.flags == QU_OFFLINE:
	    self.loginParser.Connect()
	self.FuncList.append(handler)
	if len(self.FuncList)-1 == self.FLSentLev:
	    self.doFlags()
	    if (self.flags == QU_BURST
		or (self.flags == QU_SYNC and self.FLWaitLev == self.FLSentLev)
		or (self.flags == QU_FULLSYNC and len(self.FuncList) == 1)):
##		print "force send @ " + str(
##		    (len(self.FuncList), self.FLWaitLev, self.FLSentLev)), self.flags
		self.sendCommand()

    def prependHandler(self, handler):
	"""Prepend command to the queue.

	This is similar to AddHandler, except this will make HANDLER the
	next command to be sent.  This is useful for commands that are
	marked as QU_FULLSYNC, and wish to send a command immediately
	following its execution.
	"""
	if len(self.FuncList) == self.FLSentLev:
	    return self.AddHandler(handler)
	self.FuncList.insert(self.FLSentLev, handler)
	self.doFlags()

    def HandleInput(self):
	"""Parse input from socket; send all data to function list.

	This method does all the actual reading of the socket.  It reads
	and stores the server information, and delegates complete lines of
	data to the data-managers.  It also checks for the special sequence
	C_PROMPT which always indicates the end of a command.

	This method is made significantly more complex because of the
	heterogeneous mixture of command priorities.  Tracking of
	FLWaitLev, and FLSentLev is an ardeous task.
	"""
	cache = []
	error = ""
	# HACK!  Obscure python optimization - make local copies of global
	# variables.
	(__C_PROMPT, __QU_SYNC, __QU_FULLSYNC, __QU_DISCONNECT,
	 __len, select__select, string__count, string__split,
	 self__InpBuf, self__FuncList, self__sendCommand,
	 self__socket, self__socket__recv) = (
	     C_PROMPT, QU_SYNC, QU_FULLSYNC, QU_DISCONNECT,
	     len, select.select, string.count, string.split,
	     self.InpBuf, self.FuncList, self.sendCommand,
	     self.socket, self.socket.recv)
	# Dummy exception - used for unusual flow control
	StopRead = "Internal Break"
	while 1:
	    # Make sure socket is available and readable.
	    if self.flags >= __QU_DISCONNECT:
		# Socket disconnected sometime during the cycle
		if error:
		    # It was an error
		    viewer.flash(error)
		# Remove all sent items from the queue
		del self__FuncList[:self.FLSentLev]
		self.FLSentLev = self.FLWaitLev = 0
		# Attempt to reconnect if neccessary
		if self__FuncList and self.flags == QU_OFFLINE:
		    self.loginParser.Connect()
		break
	    try:
		# Begin unconventional flow control.
		# StopRead is used here to "break out" of the read if any
		# of the socket conditions are "bad".  It is possible to do
		# this more conventionally, but I find it "ugly", and less
		# intuitive.
		sts = select__select([self__socket], [], [self__socket], 0)
		if sts[2]:
		    error = "Exceptional condition on socket!"
		    self.loginParser.Disconnect()
		    raise StopRead
		if not sts[0]:
		    # Nothing left to read
		    if not cache:
			# cache is empty - break from main loop and return.
			break
		    # There is info on the cache, process it first.
		    raise StopRead
		# Socket is Ok to read - now read a large block.
		try:
		    tmp = self__socket__recv(4096)
		except socket.error, e:
		    error = "Socket read exception: (%s)." % (e,)
		    self.loginParser.Disconnect()
		    raise StopRead
		if not tmp:
		    # Ughh.  This should not occur.  Ideally an exception
		    # should be raised..
		    error = "Zero read on socket!"
		    self.loginParser.Disconnect()
		    raise StopRead
		# If a prompt is encountered anywhere in the buffered
		# data, send the next command immediately.
		cnt = string__count(tmp, "\n"+__C_PROMPT)
		if not self__InpBuf and tmp[:1] == __C_PROMPT:
		    cnt = cnt + 1
		self.FLWaitLev = self.FLWaitLev + cnt
		if (self.flags == __QU_SYNC
		    and self.FLWaitLev == self.FLSentLev):
## 		    print ("pre send @ " + str(
## 			(len(self.FuncList), self.FLWaitLev,
## 			 self.FLSentLev)), self.flags)
		    self__sendCommand()
		# Convert input to line cache
		l = __len(cache)
		cache[l:] = string__split(tmp, "\n")
		cache[l] = self__InpBuf + cache[l]
		self__InpBuf = cache[-1]
		del cache[-1]
	    except StopRead:
		# End of unconventional flow control.  If StopRead is
		# raised, process what is on the cache.  (We are guaranteed
		# to exit at the start of the next loop if the socket was
		# closed, because the queue flags are checked.)
		pass

	    while cache:
		data = cache[0]
		del cache[0]
##		print "-- %s" % (data,)

		if not self__FuncList:
		    # Asyncrounous line of data
		    self.defParser.line(data)
		    continue
		# call the handler
		try: self__FuncList[0].line(data)
		except: flashException()
## 		if data[:1] == C_FLUSH:
		if data[:1] != __C_PROMPT:
		    continue
		# handle prompt line
		while 1:
		    if (self.flags == __QU_FULLSYNC
			and self.FLSentLev == 1):
## 			print "planned send @ " + str(
## 			    (len(self__FuncList), self.FLWaitLev,
## 			     self.FLSentLev)), self.flags
			self__sendCommand()
		    del self__FuncList[0]
		    self.FLWaitLev = self.FLWaitLev - 1
		    self.FLSentLev = self.FLSentLev - 1
		    if not self__FuncList:
			break
## 		    print "begin"
		    try: self__FuncList[0].start()
		    except: flashException()
		    if self__FuncList[0].command != None:
			break

		if self.flags == __QU_SYNC:
		    break
	# Processing has ended.	 Copy local variables back to their originals.
	self.InpBuf = self__InpBuf

    def fileno(self):
	"""Return the socket descriptor."""
	return self.socket.fileno()

class LoginHandler:
    """Aid EmpIOQueue with all login/logout requests.

    This class requires the global viewer class to contain the class
    viewer.loginHandler.  The class viewer.loginHandler must contain the
    following methods/objects:
    login_error(), login_success(), connect_success(), connect_terminate(),
    and login_kill
    """

    def __init__(self, callback, username, first):
	self.callback = callback
	callback.loginHandler = self
	self.first = first
	self.username = username
##	self.pos = 0

    postFlags = QU_SYNC

##     def start(self):
##	pass

##     preFlags = QU_BURST

    command = '\t\t-=-'

    def Connect(self):
	"""Connect to specified host/port as coun/repr."""
	ldb = empDb.megaDB['login']
	if self.first:
	    self.callback.login_error("Enter connect information.")
	    return
	try:
	    empQueue.socket = socket.socket(socket.AF_INET,
					    socket.SOCK_STREAM)
	    empQueue.socket.connect(ldb['host'], ldb['port'])
	except socket.error, e:
	    self.callback.login_error("Connect error: " + str(e))
	else:
	    self.pos = 0
	    empQueue.flags = QU_SYNC
##	    empQueue.prependHandler(self)
	    empQueue.FuncList[:0] = [ self ]
##	    empQueue.FLWaitLev = empQueue.FLWaitLev + 1
	    empQueue.FLSentLev = empQueue.FLSentLev + 1
##	    empQueue.doFlags()
	    self.callback.connect_success()

    def Disconnect(self):
	"""Break the connection with the server."""
	if empQueue.flags >= QU_DISCONNECT:
	    # Already disconnected
	    return
	self.callback.connect_terminate()
	empQueue.socket.close()
	del empQueue.socket
	empQueue.flags = QU_OFFLINE

    def line(self, line):
	"""EmpIOQueue Handler: Process a line of data."""
	proto = line[:1]
	msg = line[1:]

	# Ughh..  It appears we can get these async messages at any time..
	if proto == C_FLASH or proto == C_INFORM:
	    empQueue.defParser.line(line)

	ldb = empDb.megaDB['login']
	if (self.pos == 0):
	    if proto != C_INIT:
		self.callback.login_error("[%s]%s" % (proto, msg))
		return
	    self.pos = 1
	    empQueue.SendNow("user %s" % self.username)
	elif (self.pos == 1):
	    if proto != C_CMDOK:
		self.callback.login_error("[%s]%s" % (proto, msg))
		return
	    self.pos = 2
	    empQueue.SendNow("coun %s" % ldb['coun'])
	elif (self.pos == 2):
	    if proto != C_CMDOK:
		self.callback.login_error("[%s]%s" % (proto, msg))
		return
	    self.pos = 3
	    empQueue.SendNow("pass %s" % ldb['repr'])
	elif (self.pos == 3):
	    if proto != C_CMDOK:
		self.callback.login_error("[%s]%s" % (proto, msg))
		return
	    if self.callback.login_kill:
		self.pos = 4
		empQueue.SendNow("kill")
	    else:
		self.pos = 5
		empQueue.SendNow("play")
	elif (self.pos == 4):
	    if proto != C_EXIT:
		self.callback.login_error("[%s]%s" % (proto, msg))
		return
	    self.pos = 5
	    empQueue.SendNow("play")
	elif (self.pos == 5):
	    if proto != C_INIT:
		self.callback.login_error("[%s]%s" % (proto, msg))
		return
	    # Login successfull!
	    self.pos = 99
	    self.callback.login_success()
	    self.out = NormalHandler(self.command, viewer)
	else:
	    self.out.line(line)
    def retry(self):
	"""Try to connect to the server again.

	Calling this function is the natural conclusion of a call to the
	login_error() method described above.
	"""
	self.first = 0
	if empQueue.flags >= QU_DISCONNECT:
	    self.Connect()
	else:
	    self.pos = 0
	    self.line(C_INIT)

class AsyncHandler:
    """Handle asyncrounous server data.

    Data received that is not associated with a command is processed by
    this class.  The class requires the global viewer class to support the
    following methods:

    flash(), and inform()
    """
    def line(self, line):
	"""EmpIOQueue Handler: Process a line of data."""
	if line[:1] == C_DATA:
	    viewer.flash(line[2:])
	elif line[:1] == C_EXIT:
	    viewer.flash("Server exiting (%s)." % (line[2:],))
	    empQueue.loginParser.Disconnect()
##	    # HACK++
##	    if empQueue.FuncList:
##		del empQueue.FuncList[:1]
##		empQueue.FLSentLev = empQueue.FLSentLev - 1
	elif line[:1] == C_INFORM:
	    empDb.checkUpdated('prompt', 'inform', line[2:])
	    viewer.inform()
	elif line[:1] == C_FLASH:
	    viewer.flash(line[2:])
	else:
	    viewer.flash('PTkEI: Bad protocol "%s"' % (line,))

class NormalHandler:
    """Handle server data that is received during most normal operations.

    This class will invoke the chained display class associated with the
    command.  It will also detect built-in parsers from empDb.py, and bind
    those parsers with the display chain.
    """
    # The server sends telegram/annoucement/etc. messages to the client as
    # if they were normal C_DATA messages.  Store these messages
    # seperately, and send them all at once using the viewer.flash method.
    # This way, the individual parsers dont have to worry about detecting
    # these messages.
    msgqueue = []

#You have a new telegram waiting ...
#You have three new announcements waiting ...
#You lost your capital... better designate one
    teleMatch = re.compile(r"You have .* new telegrams? waiting \.\.\.$")
    annoMatch = re.compile(r"You have .* new announcements? waiting \.\.\.$")

    def __init__(self, command, disp, pre=None, post=None):
	self.command = command
	self.out = disp
	if pre == None:
	    pre = 1
	if post == None:
	    post = pre
	self.preFlags, self.postFlags = pre, post

    def start(self):
	"""EmpIOQueue Handler: Previous command completed - start this command."""
	# Check for lowlevel parsers
	try:
	    parser = empDb.standardParsers[
		string.split(self.command)[0]]
	except (KeyError, IndexError):
	    pass
	else:
	    self.out = parser(self.out)
	# Dummy message used to indicate start of command.
	self.out.Begin(self.command)

    def line(self, line):
	"""EmpIOQueue Handler: Process a line of data."""
	proto = line[:1]
	msg = line[2:]
	if proto == C_DATA:
	    if ("You lost your capital... better designate one" == msg
		or self.teleMatch.match(msg) or self.annoMatch.match(msg)):
		self.msgqueue.append(msg)
	    else:
		# If there are messages on the msgqueue, then we have
		# been spoofed - send them along to the parsers normally.
		for i in self.msgqueue:
		    self.out.data(i)
		self.out.data(msg)
		del self.msgqueue[:]
	elif proto == C_PROMPT:
	    # If there are messages on the msgqueue, then flash them out.
	    for i in self.msgqueue:
		viewer.flash(i)
	    del self.msgqueue[:]
	    ndb = empDb.megaDB['prompt']
	    pinfo = map(string.atoi, string.split(msg))
	    empDb.checkUpdated('prompt', 'minutes', pinfo[0])
	    empDb.checkUpdated('prompt', 'BTU', pinfo[1])
	    self.out.End(self.command)
	elif proto == C_FLUSH:
	    self.out.flush(msg)
##	elif msg[0] == C_REDIR:
##	    print "PE: Server Redirect requested:", msg[2:]
##	elif msg[0] == C_PIPE:
##	    print "PE: Server Pipe requested:", msg[2:]
##	elif msg[0] == C_EXECUTE:
##	    print "PE: Server Execute requested:", msg[2:]
	else:
	    empQueue.defParser.line(line)

class DummyHandler:
    """Dummy class - useful as a place marker.

    This class does not handle input/output; it is placed within the queue
    to note a specific location.  When this class' start() method is
    invoked, it genearlly triggers a client smart command.
    """
    def __init__(self, command, disp, pre=None, post=None):
	self.command = None
	self.id = command
	self.out = disp
	if pre == None:
	    pre = QU_BURST
	if post == None:
	    post = QU_SYNC
	self.preFlags, self.postFlags = pre, post

    def start(self):
	"""EmpIOQueue Handler: Previous command completed - start this command."""
	self.out.Begin(self.id)
	self.out.End(self.id)

def flashException():
    """Send an exception message via the flash method.

    This should be used in a block: try: ... except: flashException()."""

    viewer.flash('Internal error! Exception %s with detail "%s".'
		 % tuple(sys.exc_info()[:2]))
    traceback.print_exc()

def doNothing(*args, **kw):
    """Do absolutely nothing.  (Used as a dummy function.)"""
    pass

class baseDisp:
    """Base class for the chained display classes.  (Does little by itself.)

    Basically, this class just insures that all the sub-classes support the
    standard chained display class protocols:

    Begin() - notes the beginning of a command.
    data() - transmits a line of data.
    flush() - notes a sub-prompt.
    Answer() - is not yet implemented!  In theory, it would note answers to
    sub-prompts.
    updateDB() - isn't used by the queue scheme at all; however, it is used
    by the Tk and database code to force a redraw.
    End() - notes the termination of a command.

    This class also defines self.out - a handy reference to the next parser
    in the display chain.

    Note: Because the global viewer class is generally at the top of the
    chain, it must support all of the above methods.  (But because this
    class is used for chaining, the global viewer is generally not an
    ancestor of this class.)
    """
    def __init__(self, disp):
	# Establish defaults for all the standard display class commands.
	self.out = disp
	for i in ('Begin', 'data', 'flush', 'Answer',
		  'updateDB', 'End'):
	    if not hasattr(self, i):
		setattr(self, i, getattr(disp, i))

def EmpData(username, first):
    """Initialize EmpIOQueue with default values."""
    return EmpIOQueue(AsyncHandler(),
		      LoginHandler(viewer.loginCallback, username, first))
