// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <ctype.h>
#include <stdlib.h>
}
#include <qregexp.h>
#include "Url.h"

void Url::parseUrl( QString url )
{
	QString tmp( url.copy() );
	int idx1, idx2;
	
	// Search for a fragment identifier.
	idx1 = tmp.find( '#' );
	if ( idx1 >= 0 ) {
		_fragment = tmp.right( tmp.length() - idx1 - 1 );
		tmp.remove( idx1, tmp.length() );
//		printf( "Url::parseUrl() -- fragment = '%s', url = '%s'\n", (const char*)_fragment, (const char*)tmp );
	}
	
	// Search for a scheme (method).
	QRegExp regexp( "^[a-zA-Z0-9\\+\\.\\-]+:" );
	idx1 = tmp.find( regexp );
	if ( idx1 >= 0 ) {
		idx2 = tmp.find( ':' );
		_method = tmp.left( idx2 );
		tmp.remove( 0, idx2 + 1 );
//		printf( "Url::parseUrl() -- method = '%s', url = '%s'\n", (const char*)_method, (const char*)tmp )
	}
	
	// Search for a network location/login.
	if ( tmp.left( 2 ) == "//" ) {
		tmp.remove( 0, 2 );
		idx1 = tmp.find( '/' );
		QString netloc;
		netloc = tmp.left( idx1 );
		tmp.remove( 0, idx1 );
//		printf( "Url::parseUrl() -- netloc = '%s', url = '%s'\n", (const char*)netloc, (const char*)tmp );
		
		if ( ( idx1 = netloc.find( '@' ) ) >= 0 ) {
			QString login = netloc.left( idx1 );
			netloc.remove( 0, idx1 + 1 );
//			printf( "Url::parseUrl() -- login = '%s', netloc = '%s'\n", (const char*)login, (const char*)netloc );
			
			if ( ( idx1 = login.find( ':' ) ) >= 0 ) {
				_user = login.left( idx1 );
				_password = login.right( login.length() - idx1 - 1 );
			} else {
				_user = login;
			}
//			printf( "Url::parseUrl() -- user = '%s', password = '%s'\n", (const char*)_user, (const char*)_password );
		}
		
		if ( ( idx1 = netloc.find( ':' ) ) >= 0 ) {
			QString port = netloc.right( netloc.length() - idx1 - 1 );
			netloc.remove( idx1, netloc.length() );
			_port = atoi( (const char*)port );
//			printf( "Url::parseUrl() -- port = %d, netloc = '%s'\n", _port, (const char*)netloc );
		}
		
		_hostname = netloc;
//		printf( "Url::parseUrl() -- hostname = '%s'\n", (const char*)_hostname );
	}
	
	// Search for a query.
	if ( ( idx1 = tmp.find( '?' ) ) >= 0 ) {
		_query = tmp.right( tmp.length() - idx1 - 1 );
		tmp.remove( idx1, tmp.length() );
//		printf( "Url::parseUrl() -- query = '%s', url = '%s'\n", (const char*)_query, (const char*)tmp );
	}
	
	// Search for parameters.
	if ( ( idx1 = tmp.find( ';' ) ) >= 0 ) {
		_parameters = tmp.right( tmp.length() - idx1 - 1 );
		tmp.remove( idx1, tmp.length() );
//		printf( "Url::parseUrl() -- parameters = '%s', url = '%s'\n", (const char*)_parameters, (const char*)tmp );
	}
	
	// The rest is the path.
	_path = tmp;
//	printf( "Url::parseUrl() -- path = '%s'\n", (const char*)_path );
	
	if ( !_method.length() ) _method = 0;
	if ( !_user.length() ) _user = 0;
	if ( !_password.length() ) _password = 0;
	if ( !_hostname.length() ) _hostname = 0;
	if ( !_path.length() ) _path = 0;
	if ( !_fragment.length() ) _fragment = 0;
	if ( !_query.length() ) _query = 0;
	if ( !_parameters.length() ) _parameters = 0;
}

Url::Url( const QString absUrl ) : _port( -1 )
{
	parseUrl( absUrl );
}

Url::Url( const char* absUrl ) : _port( -1 )
{
	QString tmp( absUrl );
	parseUrl( tmp );
}

Url::Url( const Url* absUrl, const QString relUrl ) : _port( -1 )
{
	parseUrl( relUrl );
	
	if ( !absUrl ) return;
	
	if ( !url() ) {
		*this = *absUrl;
		return;
	}
	
	if ( method() ) return;
	
	_method = absUrl->method();
	
	if ( !hostname() ) {
		_user     = absUrl->user();
		_password = absUrl->password();
		_hostname = absUrl->hostname();
		_port     = absUrl->port();
		
		if ( !path() || ( _path.left( 1 ) != "/" ) ) {
			if ( !path() ) {
				_path = absUrl->path();
				if ( !parameters() ) {
					_parameters = absUrl->parameters();
					if ( !query() ) {
						_query = absUrl->query();
					}
				}
			} else {
				// Step 6...
				QString newPath;
				int idx1 = 0;
				newPath = absUrl->path().copy();
				if ( ( idx1 = newPath.findRev( '/' ) ) >= 0 ) {
					newPath.truncate( idx1 + 1 );
				} else {
					newPath.truncate( 0 );
				}
				newPath += path();
				
				idx1 = 0;
				if ( newPath.left( 1 ) == "/" ) idx1++;
				while ( idx1 < int( newPath.length() ) ) {
					if ( newPath.mid( idx1, 2 ) == "./" ) {
						newPath.remove( idx1, 2 );
//						printf( "Url::Url() -- (1)path = '%s'\n", (const char*)newPath );
					} else {
						idx1 = newPath.find( '/', idx1 + 1 );
						if ( idx1 < 0 ) idx1 = newPath.length();
						else idx1++;
					}
				}
				
				if ( newPath.right( 2 ) == "/." ) {
					newPath.truncate( newPath.length() - 1 );
//					printf( "Url::Url() -- (2)path = '%s'\n", (const char*)newPath );
				}
				
				bool found = TRUE;
				while ( found ) {
					found = FALSE;
					idx1 = 0;
					if ( newPath.left( 1 ) == '/' ) idx1++;
					while ( !found && ( idx1 < int( newPath.length() ) ) ) {
						int idx2 = newPath.find( '/', idx1 + 1 );
						if ( ( idx2 > idx1 ) && ( newPath.mid( idx1, 3 ) != "../" ) && ( newPath.mid( idx2, 4 ) == "/../" ) ) {
							found = TRUE;
							newPath.remove( idx1, idx2 - idx1 + 4 );
//							printf( "Url::Url() -- (3)path = '%s'\n", (const char*)newPath );
						} else {
							if ( idx2 < 0 ) {
								idx1 = newPath.length();
							} else {
								idx1 = idx2 + 1;
							}
						}
					}
				}
				
				if ( newPath.right( 3 ) == "/.." ) {
					idx1 = newPath.findRev( '/', newPath.length() - 4 ) + 1;
					if ( newPath.mid( idx1, 3 ) != "../" ) {
						newPath.truncate( idx1 );
//						printf( "Url::Url() -- (4)path = '%s'\n", (const char*)newPath );
					}
				}
				
				_path = newPath;
			}
		}
	}
}

QString Url::url() const
{
	QString tmp;

	if ( method() ) {
		tmp += method();
		tmp += ":";
	}
	
	if ( hostname() ) {
		tmp += "//";
		if ( user() ) {
			tmp += user();
			if ( password() ) {
				tmp += ":";
				tmp += password();
			}
			tmp += "@";
		}
		tmp += hostname();
		if ( port() >= 0 ) {
			tmp += ":";
			tmp += QString().setNum( port() );
		}
		if ( path() && ( path()[ 0 ] != '/' ) ) {
//			printf( "Url::url() -- path() = '%s'\n", path() );
			tmp += "/";
		}
	}

	tmp += path();
	
	if ( parameters() ) {
		tmp += ";";
		tmp += parameters();
	}
	
	if ( query() ) {
		tmp += "?";
		tmp += query();
	}
	
	if ( fragment() ) {
		tmp += "#";
		tmp += fragment();
	}

	return tmp;
}

void Url::encode( QString& str )
{
    uint    i;
    uchar   ch;
    QString escaped;
    for ( i = 0; i < str.length(); i++ ) {
        ch = str[i];
        if ( ( ch < 32 )       ||
             ( ch > 126 )      ||
             ( isspace( ch ) ) ||
             ( strchr( "<>\"#%{}|\\^~[]'`;/?:@=&", ch ) ) ) {
            // Escape the character.
            escaped.sprintf( "%%%02X", ch );
            str.replace( i, 1, escaped );
            i += 2;
        }
    }
}
