// qps.C
//
// qps -- Qt-based visual process status monitor
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997, 1998

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/utsname.h>
#include <signal.h>
#include <errno.h>
#include <sched.h>

#include "qps.h"
#include "dialogs.h"
#include "scheddlg.h"
#include "lookup.h"
#include "icon.xpm"
#include <qpopmenu.h>
#include <qmenubar.h>
#include <qkeycode.h>
#include <qapp.h>
#include <qfont.h>
#include <qpainter.h>
#include <qaccel.h>
#include <qtooltip.h>
#include <qmsgbox.h>
#include <qbitmap.h>
#include <qclipbrd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define QPS_VERSION "1.4.4"

// default values of settings, overridden by $HOME/.qps-settings if present
bool Qps::show_cmd_path = TRUE;
bool Qps::show_infobar = TRUE;
bool Qps::show_mem_bar = TRUE;
bool Qps::show_swap_bar = TRUE;
bool Qps::show_cpu_bar = TRUE;
bool Qps::show_load_graph = TRUE;
bool Qps::phosphor_graph = TRUE;
bool Qps::load_in_icon = TRUE;
bool Qps::auto_save_options = TRUE;
bool Qps::hostname_lookup = TRUE;
bool Qps::pids_to_selection = TRUE;
bool Qps::cumulative = FALSE;

int Qps::swaplimit = 10;	// default: warn when less than 10% swap left
bool Qps::swaplim_percent = TRUE;

Qps::Qps(QWidget *parent, const char *name)
   : QWidget(parent, name)
{
    m_process = new QPopupMenu;
    m_process->insertItem("Renice...", MENU_RENICE);
    m_process->connectItem(MENU_RENICE, this, SLOT(menu_renice()));
    m_process->setAccel(ALT + Key_R, MENU_RENICE);
    m_process->insertItem("Change Scheduling...", MENU_SCHED);
    m_process->connectItem(MENU_SCHED, this, SLOT(menu_sched()));
    m_process->setAccel(ALT + Key_S, MENU_SCHED);
    m_process->insertSeparator();
    m_process->insertItem("View Details", MENU_DETAILS);
    m_process->connectItem(MENU_DETAILS, this, SLOT(menu_info()));
    m_process->setAccel(ALT + Key_I, MENU_DETAILS);
    m_process->insertSeparator();
    m_process->insertItem("Find Parent", MENU_PARENT);
    m_process->connectItem(MENU_PARENT, this, SLOT(menu_parent()));
    m_process->setAccel(ALT + Key_P, MENU_PARENT);
    m_process->insertItem("Find Children", MENU_CHILD);
    m_process->connectItem(MENU_CHILD, this, SLOT(menu_children()));
    m_process->setAccel(ALT + Key_C, MENU_CHILD);
    m_process->insertSeparator();
    m_process->insertItem("Quit", this, SLOT(save_quit()), ALT + Key_Q);

    QPopupMenu *menu_signals = make_signal_menu();

    m_signal = new QPopupMenu;
    m_signal->insertItem("Terminate", MENU_SIGTERM);
    m_signal->connectItem(MENU_SIGTERM, this, SLOT(sig_term()));
    m_signal->setAccel(ALT + Key_T, MENU_SIGTERM);
    m_signal->insertItem("Hangup", MENU_SIGHUP);
    m_signal->connectItem(MENU_SIGHUP, this, SLOT(sig_hup()));
    m_signal->setAccel(ALT + Key_H, MENU_SIGHUP);
    m_signal->insertItem("Interrupt", MENU_SIGINT);
    m_signal->connectItem(MENU_SIGINT, this, SLOT(sig_int()));
    m_signal->insertSeparator();
    m_signal->insertItem("Kill", MENU_SIGKILL);
    m_signal->connectItem(MENU_SIGKILL, this, SLOT(sig_kill()));
    m_signal->setAccel(ALT + Key_K, MENU_SIGKILL);
    m_signal->insertItem("Other", menu_signals, MENU_OTHERS);
    connect(menu_signals, SIGNAL(activated(int)), SLOT(other_menu(int)));

    QPopupMenu *popup_signals = make_signal_menu();

    m_popup = new QPopupMenu;
    m_popup->insertItem("Renice...", this, SLOT(menu_renice()));
    m_popup->insertItem("Scheduling...", this, SLOT(menu_sched()));
    m_popup->insertSeparator();
    m_popup->insertItem("View Details", this, SLOT(menu_info()));
    m_popup->insertSeparator();
    m_popup->insertItem("Find Parent", this, SLOT(menu_parent()));
    m_popup->insertItem("Find Children", this, SLOT(menu_children()));
    m_popup->insertSeparator();
    m_popup->insertItem("Terminate", this, SLOT(sig_term()));
    m_popup->insertItem("Hangup", this, SLOT(sig_hup()));
    m_popup->insertItem("Interrupt", this, SLOT(sig_int()));
    m_popup->insertSeparator();
    m_popup->insertItem("Kill", this, SLOT(sig_kill()));
    m_popup->insertItem("Other Signals", popup_signals);
    connect(popup_signals, SIGNAL(activated(int)), SLOT(other_menu(int)));

    m_fields = new QPopupMenu;
    // m_fields will be filled with all non-displayed fields when used
    connect(m_fields, SIGNAL(activated(int)), SLOT(add_fields_menu(int)));

    m_headpopup = new QPopupMenu;
    m_headpopup->insertItem("Remove Field", this, SLOT(menu_remove_field()));
    m_headpopup->insertItem("Add Field...", m_fields);

    m_view = new CheckMenu;
    m_view->setCheckable(TRUE);
    m_view->insertItem("All Processes", Procview::ALL);
    m_view->insertItem("Your Processes", Procview::OWNED);
    m_view->insertItem("Non-Root Processes", Procview::NROOT );
    m_view->insertItem("Running Processes", Procview::RUNNING);
    m_view->insertSeparator();
    m_view->insertItem("User Fields", Procview::USER);
    m_view->insertItem("Jobs Fields", Procview::JOBS);
    m_view->insertItem("Memory Fields", Procview::MEM);
    m_view->insertSeparator();
    m_view->insertItem("Select Fields...", MENU_CUSTOM);
    m_view->connectItem(MENU_CUSTOM, this, SLOT(menu_custom()));
    connect(m_view, SIGNAL(activated(int)), SLOT(view_menu(int)));

    m_options = new QPopupMenu;
    m_options->insertItem("Update Period...", this, SLOT(menu_update()));
    m_options->insertSeparator();
    m_options->insertItem("", MENU_PATH);	// text will be set later
    m_options->connectItem(MENU_PATH, this, SLOT(menu_toggle_path()));
    m_options->insertItem("", MENU_INFOBAR);	// text will be set later
    m_options->connectItem(MENU_INFOBAR, this, SLOT(menu_toggle_infobar()));
    m_options->insertItem("", MENU_CUMUL);	// text will be set later
    m_options->connectItem(MENU_CUMUL, this, SLOT(menu_toggle_cumul()));
    m_options->insertSeparator();
    m_options->insertItem("Preferences...", MENU_PREFS);
    m_options->connectItem(MENU_PREFS, this, SLOT(menu_prefs()));
    m_options->insertSeparator();
    m_options->insertItem("Save Settings Now", this, SLOT(menu_savenow()));

    QPopupMenu *help = new QPopupMenu;
    help->insertItem("License", this, SLOT(license()));
    help->insertSeparator();
    help->insertItem("About qps", this, SLOT(about()));

    menu = new QMenuBar(this);
    menu->insertItem("Process", m_process);
    menu->insertItem("Signal", m_signal);
    menu->insertItem("View", m_view);
    menu->insertItem("Options", m_options);
    menu->insertSeparator();
    menu->insertItem("Help", help);
    menu->setSeparator(QMenuBar::InWindowsStyle);

    proc = new Proc();
    procview = new Procview(proc);
    procview->refresh();

    default_icon = 0;
    default_icon_set = FALSE;

    infobar = new Infobar(this, "infobar");
    infobar->setGeometry(0, menu->height(), width(), 56);

    pstable = new Pstable(this, procview);
    int ly = infobar->y() + infobar->height();
    pstable->setGeometry(0, ly, width(), height() - ly);

    QAccel *acc = new QAccel(this);
    // misc. accelerators
    acc->connectItem(acc->insertItem(Key_Space),
		     this, SLOT(forced_update()));
    acc->connectItem(acc->insertItem(Key_Return),
		     this, SLOT(forced_update()));
    acc->connectItem(acc->insertItem(Key_Q),
		     this, SLOT(save_quit()));
    acc->connectItem(acc->insertItem(ALT + Key_W),
		     this, SLOT(save_quit()));
    acc->connectItem(acc->insertItem(CTRL + Key_Z),
		     this, SLOT(iconify_window()));
    acc->connectItem(acc->insertItem(SHIFT + ALT + Key_C),
		     this, SLOT(select_dynasty()));

    connect(infobar->update_button(), SIGNAL(clicked()),
	    SLOT(forced_update()));
    connect(pstable, SIGNAL(selection_changed()),
	    SLOT(update_selection_status()));
    connect(pstable, SIGNAL(doubleClicked(int)),
	    SLOT(open_details(int)));
    connect(pstable, SIGNAL(rightClickedRow(QPoint)),
	    this, SLOT(context_row_menu(QPoint)));
    connect(pstable, SIGNAL(rightClickedHeading(QPoint, int)),
	    this, SLOT(context_heading_menu(QPoint, int)));

    field_win = 0;
    prefs_win = 0;

    selection_items_enabled = TRUE;

    details.setAutoDelete(TRUE);

    resize(700, 300);		// default size, can be overridden
    set_update_period(5000);	// also default
    update_load_time = 0;
    read_settings();
    pstable->recompute_table_widths(0);
    infobar->configure();	// make settings take effect in status bar
    infobar->refresh(TRUE);
    update_menu_status();
    infobar_visibility();

    startTimer(update_period);
}

// destructor isn't actually needed, but prevents some gcc bugs
Qps::~Qps()
{}

// build signal menu (used in two places)
QPopupMenu *Qps::make_signal_menu()
{
    QPopupMenu *m = new QPopupMenu;
    m->insertItem("SIGQUIT (quit)", MENU_SIGQUIT);
    m->insertItem("SIGILL (illegal instruction)", MENU_SIGILL);
    m->insertItem("SIGABRT (abort)", MENU_SIGABRT);
    m->insertItem("SIGFPE (floating point exception)", MENU_SIGFPE);
    m->insertItem("SIGSEGV (segmentation violation)", MENU_SIGSEGV);
    m->insertItem("SIGPIPE (broken pipe)", MENU_SIGPIPE);
    m->insertItem("SIGALRM (timer signal)", MENU_SIGALRM);
    m->insertItem("SIGUSR1 (user-defined 1)", MENU_SIGUSR1);
    m->insertItem("SIGUSR2 (user-defined 2)", MENU_SIGUSR2);
    m->insertItem("SIGCHLD (child death)", MENU_SIGCHLD);
    m->insertItem("SIGCONT (continue)", MENU_SIGCONT);
    m->insertItem("SIGSTOP (stop)", MENU_SIGSTOP);
    m->insertItem("SIGTSTP (stop from tty)", MENU_SIGTSTP);
    m->insertItem("SIGTTIN (tty input)", MENU_SIGTTIN);
    m->insertItem("SIGTTOU (tty output)", MENU_SIGTTOU);
    return m;
}


void Qps::resizeEvent(QResizeEvent *)
{
    infobar->resize(width(), infobar->height());
    pstable->resize(width(), height() - pstable->y());
}

void Qps::timerEvent(QTimerEvent *)
{
    killTimers();		// avoid accumulation of timer events if slow
    timer_refresh();
    startTimer(update_period);
}

void Qps::closeEvent(QCloseEvent *)
{
    save_quit();
}

void Qps::save_quit()
{
    if(auto_save_options)
	write_settings();
    qApp->quit();
}

void Qps::timer_refresh()
{
    // don't update load more often than necessary
    bool update_load = FALSE;
    update_load_time -= update_period;
    if(update_load_time <= 0) {
	update_load = TRUE;
	update_load_time = load_update_period;
	Procinfo::read_loadavg();
	if(load_in_icon)
	    set_load_icon();
	else
	    set_default_icon();
    }
    if(isVisible()) {
	procview->refresh();
	if(show_infobar)
	    infobar->refresh(update_load);
	pstable->recompute_table_widths();
	pstable->transfer_selection();
	pstable->repaint_changed();
	update_menu_selection_status();
    } else {
	if(update_load)
	    infobar->update_load();
    }
    refresh_details();
}

// update table due to a configuration change
// update_col is leftmost column that has changed
void Qps::update_table(int update_col)
{
    pstable->recompute_table_widths(update_col);
    pstable->transfer_selection();
    pstable->set_sortcol();
    pstable->repaint_statically_changed(update_col);
}

void Qps::set_default_icon()
{
    if(!default_icon_set) {
	if(!default_icon)
	    default_icon = new QPixmap((const char**)icon_xpm);
	setIcon(*default_icon);
	default_icon_set = TRUE;
    }
}

// avoid a fvwm2/Qt 1.30 problem and create a filled mask for the icon
void Qps::make_filled_mask(QPixmap *pm)
{
    QBitmap bm(pm->size());
    bm.fill(color1);
    pm->setMask(bm);
}

void Qps::set_load_icon()
{
    QPixmap *pm = infobar->load_icon(icon_width, icon_height);
    make_filled_mask(pm);
    setIcon(*pm);
    default_icon_set = FALSE;
}

void Qps::refresh_details()
{	
    details.first();
    Details *d = 0;
    Procinfo::invalidate_sockets();
    while((d = details.current()) != 0) {
	if(d->isVisible())
	    d->refresh();
	details.next();
    }
}

void Qps::forced_update()
{
    // kludge: to force the "update" button to the "down" state, we temporarily
    // change it to a toggle button
    infobar->update_button()->setToggleButton(TRUE);
    infobar->update_button()->setOn(TRUE);
    QApplication::flushX(); 

    killTimers();
    timer_refresh();

    startTimer(update_period);
    infobar->update_button()->setOn(FALSE);
    infobar->update_button()->setToggleButton(FALSE);
}

// update the menu status
void Qps::update_menu_status()
{
    update_menu_selection_status();
    for(int i = Procview::ALL; i <= Procview::RUNNING; i++)
	m_view->setItemChecked(i, i == procview->viewproc);
    for(int i = Procview::USER; i <= Procview::MEM; i++)
	m_view->setItemChecked(i, i == procview->viewfields);

    m_options->changeItem(show_cmd_path
			  ? "Hide Command Path" : "Show Command Path",
			  MENU_PATH);
    m_options->changeItem(show_infobar
			  ? "Hide Status Bar" : "Show Status Bar",
			  MENU_INFOBAR);
    m_options->changeItem(cumulative
			  ? "Exclude Child Times" : "Include Child Times",
			  MENU_CUMUL);
}

void Qps::update_menu_selection_status()
{
    bool enabled = (pstable->numSelected() > 0);
    if(enabled != selection_items_enabled) {
	m_process->setItemEnabled(MENU_RENICE, enabled);
	m_process->setItemEnabled(MENU_SCHED, enabled);
	m_process->setItemEnabled(MENU_DETAILS, enabled);
	m_process->setItemEnabled(MENU_PARENT, enabled);
	m_process->setItemEnabled(MENU_CHILD, enabled);
	m_signal->setItemEnabled(MENU_SIGTERM, enabled);
	m_signal->setItemEnabled(MENU_SIGHUP, enabled);
	m_signal->setItemEnabled(MENU_SIGINT, enabled);
	m_signal->setItemEnabled(MENU_SIGKILL, enabled);
	m_signal->setItemEnabled(MENU_OTHERS, enabled);
	selection_items_enabled = enabled;

    }
}

// update various states according to current selection
// called whenever selection is changed interactively (in the table)
void Qps::update_selection_status()
{
    update_menu_selection_status();
    if(pstable->numSelected() > 0 && pids_to_selection) {
	// set the X11 selection to "PID1 PID2 PID3 ..."
	QString s, num;
	int n = pstable->numRows();
	for(int i = 0; i < n; i++) {
	    if(pstable->isSelected(i)) {
		num.sprintf("%d ", procview->procs[i]->pid);
		s.append(num);
	    }
	}
	s[s.length() - 1] = '\0';

	// important: this mustn't be called non-interactively since Qt uses
	// the selection time of the last mouse or keyboard event
	QApplication::clipboard()->setText(s);
    }
}

void Qps::sig_term()
{
    send_to_selected(SIGTERM);
}

void Qps::sig_hup()
{
    send_to_selected(SIGHUP);
}

void Qps::sig_int()
{
    send_to_selected(SIGINT);
}

void Qps::sig_kill()
{
    send_to_selected(SIGKILL);
}

void Qps::other_menu(int id)
{
    switch(id) {
    case MENU_SIGQUIT:	send_to_selected(SIGQUIT); break;
    case MENU_SIGILL:	send_to_selected(SIGILL); break;
    case MENU_SIGABRT:	send_to_selected(SIGABRT); break;
    case MENU_SIGFPE:	send_to_selected(SIGFPE); break;
    case MENU_SIGSEGV:	send_to_selected(SIGSEGV); break;
    case MENU_SIGPIPE:	send_to_selected(SIGPIPE); break;
    case MENU_SIGALRM:	send_to_selected(SIGALRM); break;
    case MENU_SIGUSR1:	send_to_selected(SIGUSR1); break;
    case MENU_SIGUSR2:	send_to_selected(SIGUSR2); break;
    case MENU_SIGCHLD:	send_to_selected(SIGCHLD); break;
    case MENU_SIGCONT:	send_to_selected(SIGCONT); break;
    case MENU_SIGSTOP:	send_to_selected(SIGSTOP); break;
    case MENU_SIGTSTP:	send_to_selected(SIGTSTP); break;
    case MENU_SIGTTIN:	send_to_selected(SIGTTIN); break;
    case MENU_SIGTTOU:	send_to_selected(SIGTTOU); break;
    }
}

void Qps::view_menu(int id)
{
    if(id >= Procview::ALL && id <= Procview::RUNNING) {
	Procview::procstates state = (Procview::procstates)id;
	if(procview->viewproc != state) {
	    procview->viewproc = state;
	    procview->rebuild();
	    pstable->recompute_table_widths();
	    pstable->transfer_selection();
	    pstable->topAndRepaint();
	    update_menu_status();
	}
    } else if(id >= Procview::USER && id <= Procview::MEM) {
	Procview::fieldstates state = (Procview::fieldstates)id;
	if(procview->viewfields != state) {
	    procview->viewfields = state;
	    procview->set_fields();
	    pstable->set_sortcol();
	    pstable->recompute_table_widths();
	    pstable->transfer_selection();
	    pstable->topAndRepaint();
	    update_menu_status();
	    if(field_win)
		field_win->update_boxes();
	}
    }
}

void Qps::about()
{
    QPixmap icon((const char**)icon_xpm);
    QString s(80);
    s.sprintf("qps " QPS_VERSION " - A Visual Process Manager\n\n"
	      "using Qt library %s\n\n"
	      "Author: Mattias Engdegrd\n(f91-men@nada.kth.se)\n\n"
	      "http://www.nada.kth.se/~f91-men/qps/",
	      qVersion());
    MessageDialog::message("About qps", s, "OK", &icon);
}

void Qps::license()
{
    QPixmap icon((const char**)icon_xpm);
    MessageDialog::message("qps license",
			   "This program is free software, and you are"
			   " welcome\nto redistribute it under certain"
			   " conditions.\n"
			   "See the GNU General Public License\n"
			   "distributed in the file COPYING for details.",
			   "OK", &icon);
}

void Qps::menu_custom()
{
    if(field_win) {
	field_win->show();
	field_win->raise();
    } else {
	field_win = new FieldSelect(procview, proc);
	field_win->show();
	connect(field_win, SIGNAL(added_field(int)),
		this, SLOT(field_added(int)));
	connect(field_win, SIGNAL(removed_field(int)),
		this, SLOT(field_removed(int)));
    }
}

void Qps::field_added(int index)
{
    Category *newcat = proc->cats[index];
    procview->add_cat(newcat);
    int newcol = -1;
    for(int i = 0; i < procview->cats.size(); i++)
	if(procview->cats[i] == newcat) {
	    newcol = i; break;
	}

    update_table(newcol);
    procview->viewfields = Procview::CUSTOM;
    update_menu_status();
}

void Qps::field_removed(int index)
{
    // don't remove last field
    if(procview->cats.size() == 1) {
	field_win->update_boxes();
	return;		// don't remove
    }
    int delcol = -1;
    for(int i = 0; i < procview->cats.size(); i++) {
	Category *c = procview->cats[i];
	if(c->index == index) {
	    procview->remove_cat(i);
	    delcol = i;
	    break;
	}
    }
    update_table(delcol);
    procview->viewfields = Procview::CUSTOM;
    update_menu_status();
}

void Qps::menu_info()
{
    for(int i = 0; i < procview->procs.size(); i++) {
	Procinfo *p = procview->procs[i];
	if(p->selected)
	    open_details(i);
    }
}

void Qps::open_details(int row)
{
    Procinfo *p = procview->procs[row];
    if(p->details)
	p->details->raise();
    else {
	Details *d = new Details(p, this);
	details.append(d);
	d->show();
	connect(d, SIGNAL(closed(Details *)),
		this, SLOT(details_closed(Details *)));
    }
}

void Qps::details_closed(Details *d)
{
    details.removeRef(d);	// deletes window
}

// find parents of selected processes
void Qps::menu_parent()
{
    locate_relatives(&Procinfo::ppid, &Procinfo::pid);
}

void Qps::menu_children()
{
    locate_relatives(&Procinfo::pid, &Procinfo::ppid);
}

// Find processes whose attribute b is equal to the attribute a of
// selected processes. Center around topmost found.
// This is quadratic in worst case (shouldn't be a problem)
void Qps::locate_relatives(int Procinfo::*a, int Procinfo::*b)
{
    Svec<int> relatives;
    const int infinity = 2000000000;
    int topmost = infinity;
    for(int i = 0; i < procview->procs.size(); i++) {
	Procinfo *p = procview->procs[i];
	if(p->selected) {
	    pstable->setSelected(i, FALSE);
	    for(int j = 0; j < procview->procs.size(); j++) {
		Procinfo *q = procview->procs[j];
		if(p->*a == q->*b) {
		    relatives.add(j);
		    if(j < topmost)
			topmost = j;
		}
	    }
	}
    }
    for(int i = 0; i < relatives.size(); i++)
	pstable->setSelected(relatives[i], TRUE);
    if(topmost < infinity)
	pstable->centerVertically(topmost);
}

// select all (direct and indirect) offsprings of currently selected
// processes, without deselecting them
void Qps::select_dynasty()
{
    Svec<int> family;
    for(int i = 0; i < procview->procs.size(); i++)
	if(pstable->isSelected(i))
	    family.add(i);
    for(int i = 0, j = family.size(); i < j;) {
	for(int k = 0; k < procview->procs.size(); k++) {
	    Procinfo *p = procview->procs[k];
	    for(int m = i; m < j; m++) {
		Procinfo *q = procview->procs[family[m]];
		if(q->pid == p->ppid)
		    family.add(k);
	    }
	}
	i = j;
	j = family.size();
    }
    const int infinity = 2000000000;
    int topmost = infinity;
    for(int i = 0; i < family.size(); i++) {
	pstable->setSelected(family[i], TRUE);
	if(family[i] < topmost)
	    topmost = family[i];
    }
    if(topmost < infinity)
	pstable->centerVertically(topmost);
}

// change the update period, recomputing the averaging factor
void Qps::set_update_period(int milliseconds)
{
    update_period = milliseconds;
    Procview::avg_factor =
	exp(-(float)update_period / Procview::cpu_avg_time);
}

// called when right button is clicked in table
void Qps::context_row_menu(QPoint p)
{
    m_popup->popup(p);
}

// called when right button is clicked in heading
void Qps::context_heading_menu(QPoint p, int col)
{
    // rebuild the submenu: only include non-displayed fields
    m_fields->clear();
    // we use the fact that the fields are always in the same order
    for(int i = 0, j = 0; i < proc->cats.size(); i++) {
	if(j >= procview->cats.size() || proc->cats[i] != procview->cats[j])
	    m_fields->insertItem(proc->cats[i]->name, i);
	else
	    j++;
    }
    m_headpopup->setItemEnabled(1, procview->cats.size() < proc->cats.size());
    context_col = col;
    m_headpopup->popup(p);
}

// called when field is added from heading context menu
void Qps::add_fields_menu(int id)
{
    field_added(id);
    if(field_win)
	field_win->update_boxes();
}

void Qps::menu_remove_field()
{
    if(procview->cats.size() >= 2) {
	field_removed(procview->cats[context_col]->index);
	if(field_win)
	    field_win->update_boxes();
    }
}

void Qps::menu_update()
{
    QString txt(10);
    for(;;) {
	if(update_period % 1000 == 0)
	    txt.sprintf("%d s", update_period / 1000);
	else
	    txt.sprintf("%d ms", update_period);
	ValueDialog vd("qps: change update period", "New update period:",
		       (const char*)txt);
	if(vd.exec()) {
	    QString s = vd.ed_result;
	    int i = 0;
	    while(s[i] >= '0' && s[i] <= '9' || s[i] == '.') i++;
	    float period = (i > 0) ? s.left(i).toFloat() : -1;
	    s = s.mid(i, 3).stripWhiteSpace();
	    if(s.length() == 0 || s == "s")
		period *= 1000;
	    else if(s == "min")
		period *= 60000;
	    else if(s != "ms")
		period = -1;
	    if(period < 0) {
		MessageDialog::message("Invalid input",
				       "The time between updates should be a"
				       " number, optionally followed\n"
				       "by a unit (ms, s or min). If no unit"
				       " is given, seconds is assumed.",
				       "OK", MessageDialog::warningIcon());
		continue;
	    } else {
		set_update_period((int)period);
		killTimers();
		startTimer(update_period);
	    }
	}
	return;
    }
}

void Qps::menu_toggle_path()
{	
    show_cmd_path = !show_cmd_path;
    update_menu_status();
    int col = procview->findCol(F_CMDLINE);
    if(col != -1)
	update_table(col);
}

void Qps::menu_prefs()
{
    if(prefs_win) {
	prefs_win->show();
	prefs_win->raise();
    } else {
	prefs_win = new Preferences();
	prefs_win->show();
	connect(prefs_win, SIGNAL(prefs_change()),
		this, SLOT(config_change()));
	connect(infobar, SIGNAL(config_change()),
		prefs_win, SLOT(update_boxes()));
    }
}

void Qps::config_change()
{
    infobar->configure();
    details.first();
    Details *d = 0;
    while((d = details.current()) != 0) {
	d->config_change();
	details.next();
    }
}

void Qps::menu_savenow()
{
    write_settings();
}

// update the visibility of the infobar according to show_infobar
void Qps::infobar_visibility()
{
    update_menu_status();
    if(show_infobar) {
	int ly = infobar->y() + infobar->height();
	pstable->setGeometry(0, ly, width(), height() - ly);
	infobar->show();
    } else {
	infobar->hide();
	int ly = menu->height();
	pstable->setGeometry(0, ly, width(), height() - ly);
    }
}

void Qps::menu_toggle_infobar()
{	
    show_infobar = !show_infobar;
    infobar_visibility();
}

void Qps::menu_toggle_cumul()
{
    cumulative = !cumulative;
    update_menu_status();
    int col = procview->findCol(F_TIME);
    if(col == pstable->sortedCol()) {
	procview->rebuild();
	pstable->transfer_selection();
	pstable->topAndRepaint();
    } else if(col != -1)
	update_table(col);
}

void Qps::menu_renice()
{
    if(pstable->numSelected() == 0)
	return;
    int defnice = -1000;

    // use nice of first selected process as default, and check permission
    bool possible = TRUE;
    int euid = geteuid();
    Procinfo *p = 0;
    for(int i = 0; i < procview->procs.size(); i++) {
	p = procview->procs[i];
	if(p->selected) {
	    if(defnice == -1000)
		defnice = p->nice;
	    if(euid != 0 && euid != p->uid && euid != p->euid)
		possible = FALSE;
	}
    }
    if(!possible) {
	QString s(100);
	s.sprintf("You do not have permission to renice the\n"
		  "selected process%s.\n"
		  "Only the process owner and the super-user\n"
		  "are allowed to do that.",
		  (pstable->numSelected() == 1) ? "" : "es");
	MessageDialog::message("Permission denied", s, "OK",
			       MessageDialog::warningIcon());
	return;
    }

    int new_nice;
    for(;;) {
	SliderDialog vd("qps: renice process", "New nice value:",
			defnice, -20, 20, "(faster)", "(slower)");
	if(!vd.exec())
	    return;
	bool ok;
	new_nice = vd.ed_result.toInt(&ok);
	if(ok && new_nice >= -20 && new_nice <= 20)
	    break;
	else {
	    MessageDialog::message("Invalid input",
				   "The nice value should be\n"
				   "in the range -20 to 20.", "OK",
				   MessageDialog::warningIcon());
	}
    }
    int nicecol = procview->findCol(F_NICE);
    int statcol = procview->findCol(F_STAT);

    // do the actual renicing
    for(int i = 0; i < procview->procs.size(); i++) {
	Procinfo *p = procview->procs[i];
	if(p->selected) {
	    if(setpriority(PRIO_PROCESS, p->pid, new_nice) < 0) {
		QString s(100);
		switch(errno) {
		case EPERM:
		    // this shouldn't happen, but (e)uid could be changed...
		    s.sprintf("You do not have permission to renice"
			      " process %d (%s).\n"
			      "Only the process owner and the super-user are"
			      " allowed to do that.",
			      p->pid, (const char*)p->comm);
		    MessageDialog::message("Permission denied", s, "OK",
					   MessageDialog::warningIcon());
		    break;
		case EACCES:
		    MessageDialog::message("Permission denied",
					   "Only the super-user may lower"
					   " the nice value of a process.",
					   "OK",
					   MessageDialog::warningIcon());
		    return;
		}
	    } else {
		p->nice = new_nice; // don't wait for update
		if(nicecol != -1)
		    pstable->updateCell(i, nicecol);
		if(statcol != -1)
		    pstable->updateCell(i, statcol);
	    }
	}
    }
}

void Qps::menu_sched()
{
    if(pstable->numSelected() == 0)
	return;
    if(geteuid() != 0) {
	MessageDialog::message("Permission denied",
			       "Only the super-user may change the\n"
			       "scheduling policy and static priority.",
			       "OK", MessageDialog::warningIcon());
	return;
    }

    // provide reasonable defaults (first selected process)
    Procinfo *p = 0;
    for(int i = 0; i < procview->procs.size(); i++) {
	p = procview->procs[i];
	if(p->selected)
	    break;
    }
    int pol = p->get_policy();
    int pri = p->get_rtprio();
    SchedDialog sd(pol, pri);
    if(!sd.exec())
	return;
    if(sd.out_policy == SCHED_OTHER)
	sd.out_prio = 0;	// only allowed value
    int plcycol = procview->findCol(F_PLCY);
    int rpricol = procview->findCol(F_RPRI);
    for(int i = 0; i < procview->procs.size(); i++) {
	Procinfo *p = procview->procs[i];
	if(p->selected) {
	    struct sched_param sp;
	    sp.sched_priority = sd.out_prio;
	    if(sched_setscheduler(p->pid, sd.out_policy, &sp) < 0) {
		QString s(100);
		if(errno == EPERM) {
		    s.sprintf("You do not have permission to change the\n"
			      "scheduling and/or priority of"
			      " process %d (%s).\n"
			      "Only the super-user may do that.",
			      p->pid, (const char*)p->comm);
		    MessageDialog::message("Permission denied", s, "OK",
					   MessageDialog::warningIcon());
		    break;
		}
	    } else {
		p->policy = sd.out_policy; // don't wait for update
		p->rtprio = sd.out_prio;
		if(plcycol != -1)
		    pstable->updateCell(i, plcycol);
		if(rpricol != -1)
		    pstable->updateCell(i, rpricol);
	    }
	}
    }
}

void Qps::iconify_window()
{
    iconify();
}

void Qps::send_to_selected(int sig)
{
    for(int i = 0; i < procview->procs.size(); i++) {
	Procinfo *p = procview->procs[i];
	if(p->selected)
	    sendsig(p, sig);
    }
    earlier_refresh();		// in case we killed one
}

void Qps::sendsig(Procinfo *p, int sig)
{
    if(kill(p->pid, sig) < 0) {
	// if the process is gone, do nothing - no need to alert the user
	if(errno == EPERM) {
	    QString str(100);
	    str.sprintf("You do not have permission to send a signal to"
			" process %d (%s).\n"
			"Only the super-user and the owner of the process"
			" may send signals to it.",
			p->pid, (const char*)p->comm);
	    MessageDialog::message("Permission denied", str, "OK",
				   MessageDialog::warningIcon());
	}
    }
}

// make next timer_refresh happen a little earlier to remove processes that
// might have died after a signal_
void Qps::earlier_refresh()
{
    const int delay = 500;	// wait no more than this (ms)
    if(update_period > delay) {
	killTimers();
	startTimer(delay);
    }
}

// If the file format is changed in any way, QPS_FILE_VERSION must be
// incremented to prevent version mismatches and core dumps

#define QPS_FILE_VERSION 12	// version of .qps-settings file format

void Qps::read_settings()
{
    int x, y, w, h;
    int procsel, fieldsel;
    int index;
    int ver;
    char name[128];
    strcpy(name, getenv("HOME"));
    strcat(name, "/.qps-settings");
    FILE *f = fopen(name, "r");
    if(!f) return;
    fscanf(f, "%d", &ver);	// file version must agree
    if(ver == QPS_FILE_VERSION) {
	fscanf(f, "%d %d %d %d %d %d", &x, &y, &w, &h, &procsel, &fieldsel);
	setGeometry(x, y, w, h);
	procview->viewproc = (Procview::procstates)procsel;
	procview->viewfields = (Procview::fieldstates)fieldsel;
	procview->cats.clear();
	while(fscanf(f, "%d", &index), index >= 0)
	    procview->add_cat(proc->cats[index]);
	int sortindex, rev, showpath, showinfo, interval, autosave;
	int membar, swapbar, cpubar, loadgr, phosphor, icon, lookup, selection;
	int cumul, percent;
	fscanf(f, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
	       &sortindex, &rev, &showpath, &showinfo, &autosave,
	       &membar, &swapbar, &cpubar, &loadgr, &phosphor, &icon,
	       &lookup, &selection, &cumul, &percent,
	       &swaplimit, &interval);
	set_update_period(interval);
	procview->sortcat = proc->cats[sortindex];
	procview->reversed = rev;
	show_cmd_path = showpath;
	show_infobar = showinfo;
	auto_save_options = autosave;
	show_mem_bar = membar;
	show_swap_bar = swapbar;
	show_cpu_bar = cpubar;
	show_load_graph = loadgr;
	phosphor_graph = phosphor;
	load_in_icon = icon;
	hostname_lookup = lookup;
	pids_to_selection = selection;
	cumulative = cumul;
	swaplim_percent = percent;

	procview->rebuild();
	pstable->recompute_table_widths();
	update_menu_status();
	pstable->set_sortcol();
	config_change();
    }
    fclose(f);
}

// write geometry, visible fields and other settings to $HOME/.qps-settings

void Qps::write_settings()
{
    char name[128];
    strcpy(name, getenv("HOME"));
    strcat(name, "/.qps-settings");
    FILE *f = fopen(name, "w");
    if(!f) return;
    fprintf(f, "%d\n%d %d %d %d\n%d %d\n",
	    QPS_FILE_VERSION,
	    pos().x(), pos().y(), width(), height(),
	    procview->viewproc, procview->viewfields);
    for(int i = 0; i < procview->cats.size(); i++)
	fprintf(f, "%d ", procview->cats[i]->index);
    fprintf(f, "-1 %d %d\n%d %d %d %d %d %d %d %d %d %d %d %d %d\n%d\n%d\n",
	    procview->sortcat->index, (int)procview->reversed,
	    (int)show_cmd_path,
	    (int)show_infobar,
	    (int)auto_save_options,
	    (int)show_mem_bar,
	    (int)show_swap_bar,
	    (int)show_cpu_bar,
	    (int)show_load_graph,
	    (int)phosphor_graph,
	    (int)load_in_icon,
	    (int)hostname_lookup,
	    (int)pids_to_selection,
	    (int)cumulative,
	    (int)swaplim_percent,
	    swaplimit,
	    update_period);
    fclose(f);
}

// return host name with domain stripped
QString short_hostname()
{
    struct utsname u;
    uname(&u);
    char *p = strchr(u.nodename, '.');
    if(p) *p = '\0';
    QString s(u.nodename);
    return s;
}

bool opt_eq(char *arg, char *opt)
{
    if(arg[0] == '-') {
	arg++;
	if(arg[0] == '-')
	    arg++;
	return strcmp(arg, opt) == 0;
    } else
	return FALSE;
}

// print some help to stdout and exit
void print_help(char *cmdname)
{
    fprintf(stderr, "Usage: %s [options]\nOptions:\n"
	    "  -display <disp>\tX11 display to use\n"
	    "  -geometry <geom>\tgeometry of main window\n"
	    "  -title <title>\ttitle of main window\n"
	    "  -style <style>\tGUI style; <style> is one of {Motif,Windows}\n"
	    "  -version\t\tdisplay version and exit\n",
	    cmdname);
}

int main(int argc, char **argv, char **envp)
{
    Lookup::initproctitle(argv, envp);
    for(int i = 1; i < argc; i++) {
	if(opt_eq(argv[i], "version")) {
	    fprintf(stderr, "qps version " QPS_VERSION
		    ", using Qt library %s\n",
		    qVersion());
	    exit(1);
	} else if(opt_eq(argv[i], "help") || opt_eq(argv[i], "h")) {
	    print_help(argv[0]);
	    exit(1);
	}
    }
    QApplication app(argc, argv);

    // gross hack: if the font is too wide, then it's probably a 100dpi font
    // so we use a 9pt one instead (9pt at 100dpi == 12pt at 75dpi)
    // the right way to handle this would be proper geometry management instead
    QFont f("Helvetica", 12, QFont::Bold);
    QFontMetrics fm(f);
    if(fm.width('0') > 7)
	f.setPointSize(9);
    app.setFont(f, TRUE);
    f.setBold(FALSE);		// same font for tooltips, but not bold
    QToolTip::setFont(f);

    Qps q;
    QString cap;
    cap.sprintf("qps@%s%s", (const char *)short_hostname(),
		(geteuid() == 0) ? " (root)" : "");
    q.setCaption(cap);
    app.setMainWidget(&q);

    q.show();

    return app.exec();
}

