#!/usr/bin/perl -w
#

require 5.002; # Perl 5.001 doesn't work

# Standard Perl Library modules
use strict; # must define all variables, no barewords, no hard references
use vars qw($opt_f); # allow '-f' command line option
use Getopt::Std; # read in command-line options
use File::Basename; # basename command

# Personal modules (in `pwd`)
use Text::TeX; # TeX parser package
use MakePreamble; # package to split off LaTeX preamble & translate it to LyX
use CleanTeX; # package to clean TeX file for Lyxifying
use BasicLyX; # package to translate clean TeX to Basic LyX
use LastLyX; # package to print out LyX file once all translation is done

######
# Taken from the test file for TeX.pm
# Prints a message on failure
BEGIN { $| = 1; print "..\n"; }
my $TeXloaded;
END {print "fatal failure\n" unless $TeXloaded;}
$TeXloaded = 1;


######
# Global variables
my @suffixlist = '\.(ltx|latex|tex)'; # allowed suffixes for TeX file

my $LyXFormat = "2.15"; #What to print in \lyxformat command in .lyx file


##################### SUBROUTINES  ###########################################
# Gets the script's arguments.
# Returns the input file name, as well as the 'pathbase', which
#    is the input file name (including path) without the '.tex' ending.
#    It will be used for building temporary and output file names
sub getargs {
    getopts('f') or die "error in GetOpt!"; # set $opt_f if '-f' on command line
    my $InFileName = $ARGV[0];
    die "No filename specified.\n" unless defined $InFileName;
    die "Cannot find input file $InFileName.\n"   unless -e $InFileName;

    # Get file names, set up for making different temporary files
    # fileparse_set_fstype("MSDOS") for DOS support!
    my ($basename, $path, $suffix) = fileparse($InFileName, @suffixlist);
    die "invalid file suffix" unless $suffix;
    my $PathBase = $path . $basename; # all but suffix

    # Check for files that already exist
    my $pname = $PathBase . ".preamble";
    my $lname = $PathBase . ".lyx";
    if ($opt_f) {
	warn "Will overwrite Preamble file $pname\n" if -e $pname;
	warn "Will overwrite LyX file $lname\n" if -e $lname;
    } else {
	die "Preamble file $pname already exists\n" if -e $pname;
	die "LyX file $lname already exists\n" if -e $lname;
    }

    return ($InFileName, $PathBase);
}


##############################################################################
# MAIN
#

my ($InFileName, $PreambleName, $OutFileName, $filenum);
my $PathBase; # InFileName without .tex ending. For building output filenames

#get arguments to reLyX, including TeX file name
($InFileName, $PathBase) = &getargs;

# Split the preamble off of the rest of the file
# Copy until \begin{document} into foo.preamble
# Copy rest into foo.relyx1
$PreambleName = $PathBase . ".preamble";
$filenum = 1;
$OutFileName = $PathBase . ".relyx" . $filenum;
&MakePreamble::split_preamble($InFileName, $PreambleName, $OutFileName);

# Now read and translate the LaTeX preamble into LyX
# Simply store the LyX preamble in a string, for later
my $LyXPreamble = &MakePreamble::translate_preamble($PreambleName, $LyXFormat);


# Clean the TeX file (not including its preamble)
$InFileName = $OutFileName;
$OutFileName = $PathBase . ".relyx" . ++$filenum; # foo.relyx1 -> foo.relyx2
&CleanTeX::call_parser($InFileName, $OutFileName);


# Now convert basic constructs in the cleaned TeX file to LyX constructs
#    ("basic" means things that are not class-specific, e.g.)
$InFileName = $OutFileName;
$OutFileName = $PathBase . ".relyx" . ++$filenum; # foo.relyx2 -> foo.relyx3
&BasicLyX::call_parser($InFileName, $OutFileName);

# Someday, there will be calls to class-specific translation routines here...

# Finally, print out the actual LyX file including the preamble
$InFileName = $OutFileName;
$OutFileName = $PathBase . ".lyx";
&LastLyX::last_lyx($InFileName, $OutFileName, $LyXPreamble);

warn "Finished successfully!\n";
exit;
